/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog;

import android.annotation.SuppressLint;
import android.content.Context;
import android.text.Editable;
import android.text.InputFilter;
import android.text.Spanned;
import android.text.TextWatcher;
import android.text.method.ScrollingMovementMethod;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.View;
import android.view.WindowManager;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputMethodManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppTheme;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.UIMarqueeTextView;
import com.foxit.uiextensions.utils.UIToast;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.foxit.uiextensions.utils.AppResource.getString;

public class UITextEditDialog extends UIDialog {
    public static final int MAX_FILE_NAME_LENGTH = 255;

    TextView mPromptTextView;
    EditText mInputEditText;
    ImageView mClearInputText;
    Button mOkButton;
    Button mCancelButton;
    String mEditPattern;
    View mCuttingLine;
    View mTitleViewDivider;
    View mFramgentLayout;
    View mDivider;
    UIMarqueeTextView mTvTitle;
    boolean mCheckEditEmail;
    boolean mCheckEditEmpty;

    int mTopHeight;
    int mBottomHeight;
    View mEditContentView;
    View mBottomView;
    private Context mContext;
    private DisplayMetrics mMetrics;

    public UITextEditDialog(Context context) {
        this(context, UIDialog.WITH_INPUT);
    }

    @SuppressLint("ClickableViewAccessibility")
    public UITextEditDialog(Context context, @ShowType int showType) {
        super(context, R.layout.fx_dialog_tv_edittext,
                AppTheme.getDialogTheme(),
                AppDisplay.getUITextEditDialogWidth());
        mContext = context.getApplicationContext();
        mTopHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_toolbar_height_phone);
        LinearLayout.LayoutParams titleParams = (LinearLayout.LayoutParams) mTitleView.getLayoutParams();
        titleParams.height = mTopHeight;
        mTitleView.setLayoutParams(titleParams);

        mBottomView = mContentView.findViewById(R.id.fx_dialog_bottom_ll);
        mBottomHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_dialog_button_height);
        LinearLayout.LayoutParams bottomParams = (LinearLayout.LayoutParams) mBottomView.getLayoutParams();
        bottomParams.height = mBottomHeight;
        mBottomView.setLayoutParams(bottomParams);

        mPromptTextView = (TextView) mContentView.findViewById(R.id.fx_dialog_textview);
        mTvTitle = mContentView.findViewById(R.id.fx_dialog_title);
        mTitleViewDivider = mContentView.findViewById(R.id.fx_dialog_title_divider);
        mInputEditText = (EditText) mContentView.findViewById(R.id.fx_dialog_edittext);
        mClearInputText = mContentView.findViewById(R.id.fx_dialog_clear_edittext);
        mEditContentView = mContentView.findViewById(R.id.fx_dialog_content);
        mFramgentLayout = mContentView.findViewById(R.id.fx_dialog_fragment_layout);
        mFramgentLayout.setVisibility(showType == UIDialog.WITH_INPUT ? View.VISIBLE : View.GONE);
        mDivider = mContentView.findViewById(R.id.fx_dialog_divider);

        mMetrics = mContext.getResources().getDisplayMetrics();
        if (AppDisplay.isPad()) {
            usePadDimes();
        }
        mOkButton = (Button) mContentView.findViewById(R.id.fx_dialog_ok);
        mOkButton.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
        mCuttingLine = mContentView.findViewById(R.id.fx_dialog_button_cutting_line);
        mCancelButton = (Button) mContentView.findViewById(R.id.fx_dialog_cancel);
        mCancelButton.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
        mCancelButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mDialog.dismiss();
            }
        });

        if (AppDisplay.isPad()) { // SDKRD-9313
            mInputEditText.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }
        mInputEditText.setOnKeyListener(new View.OnKeyListener() {
            @Override
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if (KeyEvent.KEYCODE_ENTER == keyCode && event.getAction() == KeyEvent.ACTION_DOWN) {
                    InputMethodManager inputManager = (InputMethodManager) v.getContext().getSystemService(Context.INPUT_METHOD_SERVICE);
                    inputManager.hideSoftInputFromWindow(v.getWindowToken(), 0);
                    return true;
                }
                return false;
            }
        });

        mClearInputText.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mInputEditText.setText("");
            }
        });

        TextWatcher textChangeWatcher = new TextWatcher() {

            @Override
            public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            }

            @Override
            public void onTextChanged(CharSequence s, int start, int before, int count) {
                String text1 = mInputEditText.getText().toString();
                String text2 = stringFilter(text1);
                mClearInputText.setVisibility(text2.length() > 0 ? View.VISIBLE : View.GONE);

                if (!text1.equals(text2)) {
                    mInputEditText.setText(text2);
                    mInputEditText.setSelection(mInputEditText.length());
                }
                if (mCheckEditEmail) {
                    if (AppUtil.isEmailFormatForRMS(text2)) {
                        mOkButton.setEnabled(true);
                    } else {
                        mOkButton.setEnabled(false);
                    }
                }
            }

            @Override
            public void afterTextChanged(Editable s) {
                mOkButton.setEnabled(!mCheckEditEmpty || !AppUtil.isEmpty(s));
            }

            private String stringFilter(String input) {
                if (mEditPattern == null || mEditPattern.length() == 0)
                    return input;
                Pattern pattern = Pattern.compile(mEditPattern);
                Matcher matcher = pattern.matcher(input);
                return matcher.replaceAll("");
            }
        };

        mInputEditText.addTextChangedListener(textChangeWatcher);
        mDialog.getWindow().setBackgroundDrawableResource(android.R.color.transparent);
    }

    public void setDarkModeColor() {
        mTvTitle.setTextColor(mContext.getResources().getColor(R.color.t4));
        mPromptTextView.setTextColor(mContext.getResources().getColor(R.color.t4));
        mTitleViewDivider.setBackgroundColor(mContext.getResources().getColor(R.color.p3));
        mFramgentLayout.setBackground(mContext.getResources().getDrawable(R.drawable.edit_text_input_bg));
        mInputEditText.setHintTextColor(mContext.getResources().getColor(R.color.t3));
        mInputEditText.setTextColor(mContext.getResources().getColor(R.color.t4));
        mDivider.setBackgroundColor(mContext.getResources().getColor(R.color.p2));
        mCancelButton.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
        mCancelButton.setBackground(mContext.getResources().getDrawable(R.drawable.rd_list_item_bg));
        mCuttingLine.setBackgroundColor(mContext.getResources().getColor(R.color.p2));
        mOkButton.setTextColor(ThemeUtil.getToolbarTextColor(mContext));
        mOkButton.setBackground(mContext.getResources().getDrawable(R.drawable.rd_list_item_bg));
        mContentView.setBackground(mContext.getResources().getDrawable(R.drawable.dialog_radian));
    }

    private void usePadDimes() {
        try {
            ((LinearLayout.LayoutParams) mTitleView.getLayoutParams()).leftMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mTitleView.getLayoutParams()).rightMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mPromptTextView.getLayoutParams()).leftMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mPromptTextView.getLayoutParams()).rightMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mInputEditText.getLayoutParams()).leftMargin = AppDisplay.dp2px(24);
            ((LinearLayout.LayoutParams) mInputEditText.getLayoutParams()).rightMargin = AppDisplay.dp2px(24);
        } catch (Exception e) {
            if (e.getMessage() != null) {
                Log.e("UITextEditDialog", e.getMessage());
            } else {
                Log.e("UITextEditDialog", "usePadDimes_has_an_error");
            }
        }
    }

    public TextView getPromptTextView() {
        return mPromptTextView;
    }

    public EditText getInputEditText() {
        return mInputEditText;
    }

    public Button getOKButton() {
        return mOkButton;
    }

    public Button getCancelButton() {
        return mCancelButton;
    }

    public void setEditVisible(boolean visible) {
        mFramgentLayout.setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    public void setPattern(String pattern) {
        mEditPattern = pattern;
    }

    public void setBottomItemWeight(int leftWeight, int rightWeight) {
        LinearLayout.LayoutParams leftParams = (LinearLayout.LayoutParams) mCancelButton.getLayoutParams();
        leftParams.weight = leftWeight;
        mCancelButton.setLayoutParams(leftParams);

        LinearLayout.LayoutParams rightParams = (LinearLayout.LayoutParams) mOkButton.getLayoutParams();
        rightParams.weight = rightWeight;
        mOkButton.setLayoutParams(rightParams);
    }

    @Override
    public void show() {
        if (mOkButton.getVisibility() == View.VISIBLE && mCancelButton.getVisibility() == View.VISIBLE) {
            mCuttingLine.setVisibility(View.VISIBLE);
            mOkButton.setBackgroundResource(R.drawable.dialog_right_button_background_selector);
            mCancelButton.setBackgroundResource(R.drawable.dialog_left_button_background_selector);
        } else {
            mCuttingLine.setVisibility(View.GONE);
            mOkButton.setBackgroundResource(R.drawable.dialog_button_background_selector);
            mCancelButton.setBackgroundResource(R.drawable.dialog_button_background_selector);
        }
        super.show();
    }

    public boolean isShowing() {
        return mDialog.isShowing();
    }

    private int mHeight = -100;

    public void setHeight(int height) {
        mHeight = height;
//        if (mContentView.getMeasuredHeight() >= height ||
//                (mContentView.getMeasuredHeight() == 0 && AppDisplay.isLandscape() && !AppDisplay.isPad())) {
//            mPromptTextView.setMaxLines(6);
//        } else {
//            mPromptTextView.setMaxLines(15);
//
//            mContentView.measure(0, 0);
//            if (mContentView.getMeasuredHeight() >= height)
//                mPromptTextView.setMaxLines(6);
//        }

        mPromptTextView.setMovementMethod(ScrollingMovementMethod.getInstance());
        if (mPromptTextView.getScrollX() != 0 || mPromptTextView.getScrollY() != 0) {
            mPromptTextView.post(new Runnable() {
                @Override
                public void run() {
                    mPromptTextView.scrollTo(0, 0);
                }
            });
        }

        int dialogHeight;
        WindowManager.LayoutParams params = mDialog.getWindow().getAttributes();
        if (mHeight >= getDialogHeight() ||
                (mHeight <= 0 && mHeight != WindowManager.LayoutParams.WRAP_CONTENT && mHeight != WindowManager.LayoutParams.MATCH_PARENT)) {
            dialogHeight = getDialogHeight();

            if (mContentView.getMeasuredHeight() > dialogHeight) {
                int contentHeight = dialogHeight - mTopHeight - mBottomHeight - AppDisplay.dp2px(2);
                LinearLayout.LayoutParams layoutParams = (LinearLayout.LayoutParams) mEditContentView.getLayoutParams();
                layoutParams.height = contentHeight;
                mEditContentView.setLayoutParams(layoutParams);
            } else {
                LinearLayout.LayoutParams layoutParams = (LinearLayout.LayoutParams) mEditContentView.getLayoutParams();
                layoutParams.height = LinearLayout.LayoutParams.WRAP_CONTENT;
                mEditContentView.setLayoutParams(layoutParams);
            }
        } else {
            dialogHeight = height;

            int contentHeight = dialogHeight - mTopHeight - mBottomHeight - AppDisplay.dp2px(2);
            LinearLayout.LayoutParams layoutParams = (LinearLayout.LayoutParams) mEditContentView.getLayoutParams();
            layoutParams.height = contentHeight;
            mEditContentView.setLayoutParams(layoutParams);
        }
        params.height = dialogHeight;
        mDialog.getWindow().setGravity(Gravity.CENTER);
        mDialog.getWindow().setAttributes(params);
    }

    public int getDialogHeight() {
        return AppDisplay.getDialogHeight();
    }

    public void setTitleDividerVisible(boolean visible) {
        mContentView.findViewById(R.id.fx_dialog_title_divider).setVisibility(visible ? View.VISIBLE : View.GONE);
    }

    public void setLengthFilters(int maxLength) {
        setLengthFilters(maxLength, AppResource.getString(mContext, R.string.edit_name_length_limited));
    }

    public void setLengthFilters(final int maxLength, final String propmpt) {
        mInputEditText.setFilters(new InputFilter[]{new InputFilter() {
            @Override
            public CharSequence filter(CharSequence source, int start, int end, Spanned dest, int dstart, int dend) {
                int keep = maxLength - (dest.length() - (dend - dstart));
                if (!AppUtil.isEmpty(propmpt) && keep < end - start) {
                    UIToast.getInstance(mContext).show(propmpt);
                }

                if (keep <= 0) {
                    return "";
                } else if (keep >= end - start) {
                    return null; // keep original
                } else {
                    keep += start;
                    if (Character.isHighSurrogate(source.charAt(keep - 1))) {
                        --keep;
                        if (keep == start) {
                            return "";
                        }
                    }
                    return source.subSequence(start, keep);
                }
            }
        }});
    }

    public static UITextEditDialog quickCreate(Context context, int titleResId, int contentResId) {
        UITextEditDialog dialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
        TextView titleText = (TextView) dialog.getTitleView();
        titleText.setText(getString(context, titleResId));
        TextView promptTextView = dialog.getPromptTextView();
        promptTextView.setText(getString(context, contentResId));
        return dialog;
    }

    @Override
    public void dismiss() {
        super.dismiss();
    }

    public void setCheckInputEmpty(boolean check) {
        mCheckEditEmpty = check;
    }

}
