/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls.dialog;


import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Point;
import android.os.Bundle;
import android.view.View;
import android.view.Window;
import android.view.WindowManager;

import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.google.android.material.bottomsheet.BottomSheetBehavior;
import com.google.android.material.bottomsheet.BottomSheetDialog;
import com.google.android.material.bottomsheet.BottomSheetDialogFragment;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.FragmentActivity;

public class UIBottomSheetFragment extends BottomSheetDialogFragment {

    public static UIBottomSheetFragment newInstance(FragmentActivity act, View contentView) {
        UIBottomSheetFragment bottomFragment = new UIBottomSheetFragment();
        bottomFragment.init(act, contentView, TAG);
        return bottomFragment;
    }

    public static UIBottomSheetFragment newInstance(FragmentActivity act, View contentView, String name) {
        UIBottomSheetFragment bottomFragment = new UIBottomSheetFragment();
        bottomFragment.init(act, contentView, name);
        return bottomFragment;
    }

    private static final String TAG = "BOTTOM_SHEET_NAME";
    private View mContentView;
    private FragmentActivity mAct;
    private BottomSheetDialog mBottomDialog;
    private Context mContext;
    private DialogInterface.OnDismissListener mDismissListener;

    private int mWidth;
    private int mHeight;
    private int mLocation;
    private boolean mIsShowing = false;
    private boolean mShowOnKeyboard = false;
    private Point mPosition = new Point();

    private String mSheetName;

    private void init(FragmentActivity act, View contentView, String name) {
        mAct = act;
        mContext = act.getApplicationContext();
        mContentView = contentView;
        mSheetName = name;
    }

    @NonNull
    @Override
    public Dialog onCreateDialog(@Nullable Bundle savedInstanceState) {
        mBottomDialog = (BottomSheetDialog) super.onCreateDialog(savedInstanceState);
        if (mBottomDialog.getWindow() != null) {
            if (mShowOnKeyboard) {
                mBottomDialog.getWindow().setFlags(WindowManager.LayoutParams.FLAG_ALT_FOCUSABLE_IM,
                        WindowManager.LayoutParams.FLAG_ALT_FOCUSABLE_IM);
            } else {
                mBottomDialog.getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN,
                        WindowManager.LayoutParams.FLAG_FULLSCREEN);
            }

            if (SystemUiHelper.getInstance().isFullScreenMode(getActivity()) || SystemUiHelper.getInstance().isFullScreen()) {
                SystemUiHelper.getInstance().hideStatusBar(mBottomDialog.getWindow());
            }
        }

        AppUtil.removeViewFromParent(mContentView);
        mBottomDialog.setDismissWithAnimation(true);
        mBottomDialog.setCanceledOnTouchOutside(true);
        if (mContentView != null)
            mBottomDialog.setContentView(mContentView);
        mBottomDialog.getBehavior().setSkipCollapsed(true);
        mBottomDialog.getBehavior().addBottomSheetCallback(mSheetCallback);
        if (AppDisplay.getActivityHeight() < mHeight)
            mBottomDialog.getBehavior().setPeekHeight(AppDisplay.getActivityHeight() / 2);
        else
            mBottomDialog.getBehavior().setState(BottomSheetBehavior.STATE_EXPANDED);
        setWindowAttributes(getDialog());
        return mBottomDialog;
    }

    private BottomSheetBehavior.BottomSheetCallback mSheetCallback = new BottomSheetBehavior.BottomSheetCallback() {
        @Override
        public void onStateChanged(@NonNull View bottomSheet, int newState) {
            if (newState == BottomSheetBehavior.STATE_HIDDEN) {
                dismiss();
                mBottomDialog.getBehavior().setState(BottomSheetBehavior.STATE_COLLAPSED);
            }
        }

        @Override
        public void onSlide(@NonNull View bottomSheet, float slideOffset) {
        }
    };

    @Override
    public void onResume() {
        super.onResume();
        setWindowAttributes(getDialog());
    }

    @Override
    public void onDestroy() {
        super.onDestroy();
        mIsShowing = false;
    }

    @Override
    public void onDetach() {
        super.onDetach();
        mIsShowing = false;
    }

    @Override
    public void dismiss() {
        if (mBottomDialog != null)
            mBottomDialog.getBehavior().removeBottomSheetCallback(mSheetCallback);
        super.dismissAllowingStateLoss();

        mIsShowing = false;
        AppDialogManager.getInstance().remove(this);
    }

    @Override
    public void dismissAllowingStateLoss() {
        if (mBottomDialog != null)
            mBottomDialog.getBehavior().removeBottomSheetCallback(mSheetCallback);
        super.dismissAllowingStateLoss();

        mIsShowing = false;
        AppDialogManager.getInstance().remove(this);
    }

    public void setOnDismissListener(DialogInterface.OnDismissListener dismissListener) {
        mDismissListener = dismissListener;
    }

    public void setWidth(int width) {
        if (mWidth == width) return;
        mWidth = width;
        setWindowAttributes(getDialog());
    }

    public void setHeight(int height) {
        if (mHeight == height) return;
        mHeight = height;
        setWindowAttributes(getDialog());
    }

    public void update(int width, int height) {
        mWidth = width;
        mHeight = height;
        setWindowAttributes(getDialog());
    }

    public void update(int x, int y, int width, int height) {
        mPosition.x = x;
        mPosition.y = y;
        mWidth = width;
        mHeight = height;
        setWindowAttributes(getDialog());
    }

    public void showAtLocation(int gravity, int x, int y) {
        mLocation = gravity;
        mPosition.x = x;
        mPosition.y = y;
        setWindowAttributes(getDialog());

        if (!isShowing()) {
            mIsShowing = true;
            AppDialogManager.getInstance().showAllowManager(this, mAct.getSupportFragmentManager(), mSheetName, null);
        }
    }

    private void setWindowAttributes(Dialog dialog) {
        if (dialog == null)
            return;

        Window window = dialog.getWindow();
        if (window != null) {
            window.setGravity(mLocation);
            WindowManager.LayoutParams lp = window.getAttributes();
            if (lp != null) {
                lp.x = mPosition.x;
                lp.y = mPosition.y;

                lp.width = mWidth;
                lp.height = mHeight;
                window.setAttributes(lp);
            }
        }
    }

    public boolean isShowing() {
        return mIsShowing;
    }

    public void setShowOnKeyboard(boolean flag) {
        mShowOnKeyboard = flag;
    }

    @Override
    public void onDismiss(@NonNull DialogInterface dialog) {
        super.onDismiss(dialog);
        if (mDismissListener != null)
            mDismissListener.onDismiss(dialog);
    }
}

