/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.controls;


import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.ObjectAnimator;
import android.animation.ValueAnimator;
import android.animation.ValueAnimator.AnimatorUpdateListener;
import android.content.Context;
import android.content.res.TypedArray;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PixelFormat;
import android.graphics.RectF;
import android.os.Build;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.animation.AccelerateInterpolator;
import android.widget.TextView;

import com.foxit.uiextensions.R;

import androidx.annotation.ColorInt;

public class ToastSeekBar extends View {
    private static final int DEFAULT_TOUCH_TARGET_SIZE = 40;

    private Paint paint;

    private float width = 750;
    private int minProgress = 0;
    private int maxProgress = 100;
    @ColorInt
    private int progressBackColor = Color.BLACK;
    private float progressBackHeight = 10;

    @ColorInt
    private int progressColor = Color.GREEN;
    private float progressHeight = 20;
    private int progress = 50;
    private float mThumbNormalRadius = 14;
    private float mThumbPressedRadius = 24;
    @ColorInt
    private int mThumbColor = Color.BLUE;
    private float mTextBackRadius = 10;

    private boolean mIsCenterState = false;
    private float mThumbRadius = mThumbNormalRadius;
    private float progressPosition;
    private boolean isTouchLegal = false;
    private ObjectAnimator mAnimator;
    private final RectF mBackRectF;
    private final RectF mProgressRectF;
    private OnSeekBarProgressListener mOnSeekBarProgressListener;
    private OnSeekBarFinishedListener mOnSeekBarFinishedListener;
    private WindowManager mWindowManager;
    private WindowManager.LayoutParams mLayoutParams;
    private TextView mView;
    private float mCenterRadius = 4;
    private Context mContext;

    public ToastSeekBar(Context context) {
        this(context, null);
    }

    public ToastSeekBar(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public ToastSeekBar(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        mContext = context;
        paint = new Paint();

        paint.setAntiAlias(true);
        if (attrs != null) {
            TypedArray styledAttrs = getContext().obtainStyledAttributes(attrs,
                    R.styleable.CenterSeekBar, 0, 0);
            maxProgress = styledAttrs.getInteger(R.styleable.CenterSeekBar_maxProgress, 100);
            minProgress = styledAttrs.getInteger(R.styleable.CenterSeekBar_minProgress, 0);
            minProgress = styledAttrs.getInteger(R.styleable.CenterSeekBar_minProgress, 0);
            width = styledAttrs.getDimension(R.styleable.CenterSeekBar_width, 800);
            mIsCenterState = styledAttrs.getBoolean(R.styleable.CenterSeekBar_centerState, false);
            progressBackColor = styledAttrs.getColor(R.styleable.CenterSeekBar_backColor, Color.BLACK);
            progressBackHeight = styledAttrs.getDimension(R.styleable.CenterSeekBar_backHeight, 10);
            progressColor = styledAttrs.getColor(R.styleable.CenterSeekBar_progressColor, Color.GREEN);
            progressHeight = styledAttrs.getDimension(R.styleable.CenterSeekBar_progressHeight, progressBackHeight);
            progress = styledAttrs.getInteger(R.styleable.CenterSeekBar_progress, 50);
            mThumbRadius = mThumbNormalRadius = styledAttrs.getDimension(R.styleable.CenterSeekBar_thumbNormalRadius, 14);
            mThumbPressedRadius = styledAttrs.getDimension(R.styleable.CenterSeekBar_thumbPressRadius, 24);
            mThumbColor = styledAttrs.getColor(R.styleable.CenterSeekBar_thumbColor, Color.BLUE);
            mTextBackRadius = styledAttrs.getDimension(R.styleable.CenterSeekBar_textBackRadius, 10);
            styledAttrs.recycle();
        }
        mAnimator = getTargetAnimator(false);
        mBackRectF = new RectF();
        mProgressRectF = new RectF();


        mWindowManager = (WindowManager) context.getSystemService(Context.WINDOW_SERVICE);
        mLayoutParams = new WindowManager.LayoutParams();
        initLayoutParams(mLayoutParams);
    }


    private void initLayoutParams(WindowManager.LayoutParams mLayoutParams) {
        mLayoutParams.gravity = Gravity.START | Gravity.TOP;
        mLayoutParams.width = ViewGroup.LayoutParams.WRAP_CONTENT;
        mLayoutParams.height = ViewGroup.LayoutParams.WRAP_CONTENT;
        mLayoutParams.format = PixelFormat.TRANSLUCENT;
        mLayoutParams.flags = WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL
                | WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE
                | WindowManager.LayoutParams.FLAG_SHOW_WHEN_LOCKED;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N_MR1) {
            mLayoutParams.type = WindowManager.LayoutParams.TYPE_APPLICATION;
        } else {
            mLayoutParams.type = WindowManager.LayoutParams.TYPE_TOAST;
        }
    }

    public void setProgressColor(int progressColor) {
        try {
            this.progressColor = progressColor;
            invalidate();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public int getMax() {
        return maxProgress;
    }

    public ToastSeekBar setCenterModeEnable(boolean enable) {
        if (mIsCenterState) {
            if (!enable) {
                if (progress < 0) {
                    progress = -progress;
                }
            }
        }
        mIsCenterState = enable;
        invalidate();
        return this;
    }

    public ToastSeekBar setProgress(int progress) {
        if (progress < maxProgress && progress > minProgress) {
            this.progress = progress;
        } else {
            progress = minProgress;
        }
        return this;
    }

    public int getProgress() {
        return this.progress;
    }

    public ToastSeekBar setOnSeekBarProgressListener(OnSeekBarProgressListener l) {
        mOnSeekBarProgressListener = l;
        return this;
    }

    public ToastSeekBar setOnSeekBarFinishedListener(OnSeekBarFinishedListener l) {
        mOnSeekBarFinishedListener = l;
        return this;
    }

    public void setToastView(TextView view) {
        mView = view;
    }

    public void drawBubble(int x, int y) {
        mLayoutParams.x = x;
        mLayoutParams.y = y;
        if (mView.getParent() != null) {
            mWindowManager.updateViewLayout(mView, mLayoutParams);
        } else {
            mView.setVisibility(INVISIBLE);
            mWindowManager.addView(mView, mLayoutParams);
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        width = getWidth() / 5.0f * 3;
        int centerX = getWidth() / 2;
        int centerY = 2 * getHeight() / 4;
        float startX = centerX - width / 2;
        paint.setColor(progressBackColor);
        paint.setStrokeWidth(progressBackHeight);
        mBackRectF.left = startX;
        mBackRectF.top = centerY - progressBackHeight / 2;
        mBackRectF.right = startX + width;
        mBackRectF.bottom = centerY + progressBackHeight / 2;
        canvas.drawRoundRect(mBackRectF, mTextBackRadius, mTextBackRadius, paint);

        paint.setStyle(Paint.Style.FILL);
        paint.setColor(progressColor);

        paint.setStrokeWidth(progressHeight);
        paint.setColor(progressColor);
        if (mIsCenterState) {
            startX = centerX;
            progressPosition = startX + (int) ((progress * (width / 2f) / (maxProgress - minProgress)));
        } else {
            progressPosition = startX + ((progress * width / (maxProgress - minProgress)));
        }
        mProgressRectF.top = centerY - progressBackHeight / 2;
        mProgressRectF.bottom = centerY + progressBackHeight / 2;
        if (progress > 0) {
            mProgressRectF.left = startX;
            mProgressRectF.right = progressPosition;
        } else {
            mProgressRectF.left = progressPosition;
            mProgressRectF.right = startX;
        }
        canvas.drawRoundRect(mProgressRectF, mTextBackRadius, mTextBackRadius, paint);
//        paint.setColor(mThumbColor);
//        canvas.drawCircle(centerX, centerY, mCenterRadius, paint);

        paint.setColor(mThumbColor);
        int pPosition = (int) progressPosition;
        canvas.drawCircle((float) pPosition, centerY, mThumbRadius, paint);
        String text = progress +"%";
        mView.setText(text);
        drawBubble((int) (progressPosition - mView.getWidth() / 2f), (int) getY() - (int) mThumbPressedRadius *2 - 5);

    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        if (!isEnabled())
            return false;
        switch (event.getAction()) {
            case MotionEvent.ACTION_DOWN:
                checkTouchingTarget(event);
                break;
            case MotionEvent.ACTION_MOVE:
                if (isTouchLegal) {
                    if (mView.getVisibility() == View.INVISIBLE) {
                        mView.setVisibility(VISIBLE);
                    }
                    progress = (int) clamp((int) event.getX());
                    invalidate();
                    if (mOnSeekBarProgressListener != null) {
                        mOnSeekBarProgressListener.onProgress(progress);
                    }
                }
                break;
            case MotionEvent.ACTION_UP:
                mView.setVisibility(INVISIBLE);
                if (isTouchLegal) {
                    mAnimator.cancel();
                    mAnimator = getTargetAnimator(false);
                    mAnimator.start();
                    if (mOnSeekBarFinishedListener != null) {
                        mOnSeekBarFinishedListener.onFinished(progress);

                    }
                }
                break;
        }
        return true;
    }

    public void reset(int progress) {
        this.progress = progress;
        invalidate();
    }

    private void checkTouchingTarget(MotionEvent event) {
        if (isTouchingTarget(event)) {
            mAnimator.cancel();
            mAnimator = getTargetAnimator(true);
            mAnimator.start();
        }
    }

    private boolean isTouchingTarget(MotionEvent event) {
        isTouchLegal = event.getX() > progressPosition - DEFAULT_TOUCH_TARGET_SIZE
                && event.getX() < progressPosition + DEFAULT_TOUCH_TARGET_SIZE;
        return isTouchLegal;
    }

    private ObjectAnimator getTargetAnimator(boolean touching) {
        final ObjectAnimator anim = ObjectAnimator.ofFloat(this,
                "mThumbRadius",
                mThumbRadius,
                touching ? mThumbPressedRadius : mThumbNormalRadius);
        anim.addUpdateListener(new AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator animation) {
                postInvalidate();
            }
        });
        anim.addListener(new AnimatorListenerAdapter() {
            @Override
            public void onAnimationEnd(Animator animation) {
                anim.removeAllListeners();
            }
        });
        anim.setInterpolator(new AccelerateInterpolator());
        return anim;
    }

    public void setMThumbRadius(float mThumbRadius) {
        this.mThumbRadius = mThumbRadius;
    }

    private float clamp(int value) {
        int centerX = getWidth() / 2;
        float min = centerX - width / 2;
        float max = centerX + width / 2;
        if (mIsCenterState) {
            if (value > centerX) {
                if (value >= max) {
                    return maxProgress;
                } else {
                    return (int) ((maxProgress - minProgress) * (value - centerX) / (width / 2f));
                }
            } else if (value < centerX) {
                if (value <= min) {
                    return -maxProgress;
                } else {
                    return (int) ((maxProgress - minProgress) * (value - centerX) / (width / 2f));
                }
            } else {
                return minProgress;
            }
        } else {
            if (value >= max) {
                return maxProgress;
            } else if (value <= min) {
                return minProgress;
            } else {
                return (maxProgress - minProgress) * (value - min) / width;
            }
        }
    }

    public interface OnSeekBarProgressListener {
        void onProgress(int progress);
    }

    public interface OnSeekBarFinishedListener {
        void onFinished(int progress);
    }
}