/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.browser;

import android.animation.ValueAnimator;
import android.annotation.SuppressLint;
import android.content.Context;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewTreeObserver;
import android.widget.LinearLayout;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.utils.AppUtil;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

public class PullListView extends LinearLayout {
    private static final float MIN_MOVE_DISTANCE = 5;
    private static final float PULL_RATE = 0.4f;

    private static final int PULL_DOWN = 1;
    private static final int RECOVER = 2;
    private static final int NONE = 3;

    private LinearLayout mHeaderView;
    private LinearLayout mTopView;
    private RecyclerView mRecyclerView;

    private int mHeaderHeight;
    private int mTopMaigin;
    private int mRecyclerViewOffset;
    private int mState = NONE;
    private float mDownY = 0;
    private boolean mScrollable = true;

    public PullListView(Context context, AttributeSet attrs) {
        super(context, attrs);

        View contentView = LayoutInflater.from(context).inflate(R.layout.rd_pull_view, this, false);
        addView(contentView);
        setOrientation(VERTICAL);

        mHeaderView = contentView.findViewById(R.id.pull_header_view);
        mTopView = contentView.findViewById(R.id.pull_top_view);
        mRecyclerView = contentView.findViewById(R.id.pull_recycler_view);
        mRecyclerView.addOnScrollListener(new RecyclerView.OnScrollListener() {
            @Override
            public void onScrolled(@NonNull RecyclerView recyclerView, int dx, int dy) {
                super.onScrolled(recyclerView, dx, dy);
                mRecyclerViewOffset = mRecyclerView.computeVerticalScrollOffset();
            }
        });
    }

    public void addHeaderView(final View view) {
        AppUtil.removeViewFromParent(view);
        mHeaderView.addView(view, 0);

        mHeaderView.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                view.getViewTreeObserver().removeOnGlobalLayoutListener(this);

                mHeaderHeight = mHeaderView.getHeight();
                LayoutParams params = (LayoutParams) mHeaderView.getLayoutParams();
                params.topMargin = -mHeaderHeight;
                mHeaderView.setLayoutParams(params);

                mTopMaigin = -mHeaderHeight;
            }
        });
    }

    public void addTopView(View view) {
        AppUtil.removeViewFromParent(view);
        mTopView.addView(view);
    }

    public void hideHeaderView() {
        mHeaderView.post(new Runnable() {
            @Override
            public void run() {
                mHeaderHeight = mHeaderView.getHeight();
                mTopMaigin = -mHeaderHeight;

                LayoutParams params = (LayoutParams) mHeaderView.getLayoutParams();
                params.topMargin = mTopMaigin;
                mHeaderView.setLayoutParams(params);
            }
        });
    }

    public void showHeaderView() {
        mHeaderView.post(new Runnable() {
            @Override
            public void run() {
                mHeaderHeight = mHeaderView.getHeight();
                mTopMaigin = 0;

                LayoutParams params = (LayoutParams) mHeaderView.getLayoutParams();
                showAnimation(params.topMargin, mTopMaigin);
            }
        });
    }

    public RecyclerView getRecyclerView() {
        return mRecyclerView;
    }

    public void setScrollable(boolean srollable) {
        mScrollable = srollable;
    }

    @Override
    public boolean onInterceptTouchEvent(MotionEvent ev) {
        if (mScrollable) {
            boolean intercepted = false;
            float y = ev.getY();
            switch (ev.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    intercepted = false;
                    mDownY = y;
                    break;
                case MotionEvent.ACTION_MOVE:
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL:
                    if (mState == PULL_DOWN || mState == RECOVER)
                        intercepted = true;

                    if (mState == NONE) {
                        LayoutParams layoutParams = (LayoutParams) mHeaderView.getLayoutParams();
                        if (layoutParams.topMargin >= 0 && Math.abs(y - mDownY) > MIN_MOVE_DISTANCE) {
                            mState = PULL_DOWN;
                            intercepted = true;
                        } else if (mRecyclerViewOffset == 0 && y - mDownY > MIN_MOVE_DISTANCE) {
                            mState = PULL_DOWN;
                            intercepted = true;
                        } else {
                            intercepted = false;
                        }
                    }
                    break;
                default:
                    break;
            }
            if (intercepted)
                return true;
        }
        return super.onInterceptTouchEvent(ev);
    }

    @SuppressLint("ClickableViewAccessibility")
    @Override
    public boolean onTouchEvent(MotionEvent ev) {
        if (mScrollable) {
            float y = ev.getY();
            switch (ev.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    LayoutParams params = (LayoutParams) mHeaderView.getLayoutParams();
                    mTopMaigin = params.topMargin;
                    mDownY = y;
                    break;
                case MotionEvent.ACTION_MOVE:
                    float offset = y - mDownY;
                    LayoutParams params2 = (LayoutParams) mHeaderView.getLayoutParams();
                    params2.topMargin = mTopMaigin + (int) (offset * PULL_RATE);
                    mHeaderView.setLayoutParams(params2);
                    break;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL:
                    if (mState == PULL_DOWN) {
                        mState = RECOVER;
                    }
                    float offset2 = y - mDownY;
                    int start = mTopMaigin + (int) (offset2 * PULL_RATE);
                    if (start > 0)
                        mTopMaigin = 0;
                    else
                        mTopMaigin = -mHeaderHeight;
                    showAnimation(start, mTopMaigin);
                    break;
                default:
                    break;
            }
            return true;
        }
        return super.onTouchEvent(ev);
    }

    private void showAnimation(int start, final int end) {
        ValueAnimator animator = ValueAnimator.ofInt(start, end);
        animator.setDuration(300).start();
        animator.addUpdateListener(new ValueAnimator.AnimatorUpdateListener() {
            @Override
            public void onAnimationUpdate(ValueAnimator animation) {
                int i = (int) animation.getAnimatedValue();
                LayoutParams params = (LayoutParams) mHeaderView.getLayoutParams();
                params.topMargin = i;
                mHeaderView.setLayoutParams(params);

                if (i == end) {
                    mState = NONE;
                }
            }
        });
    }

}
