/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.textmarkup.highlight;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.view.MotionEvent;
import android.view.View;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Range;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.TextPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Highlight;
import com.foxit.sdk.pdf.annots.QuadPoints;
import com.foxit.sdk.pdf.annots.QuadPointsArray;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotContent;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.common.UIMagnifierView;
import com.foxit.uiextensions.annots.textmarkup.TextMarkupContent;
import com.foxit.uiextensions.annots.textmarkup.TextMarkupContentAbs;
import com.foxit.uiextensions.annots.textmarkup.TextSelector;
import com.foxit.uiextensions.annots.textmarkup.TextmarkupConstants;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;

public class HighlightToolHandler implements ToolHandler {
    private Paint mPaint;
    private Paint mAreaPaint;

    private int mColor;
    private int mOpacity;
    private int mAreaColor;
    private int mAreaOpacity;

    private int mCurrentIndex = -1;
    private RectF mBBoxRect;

    private PropertyBar mPropertyBar;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;

    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiextensionsManager;
    private Context mContext;
    private final TextSelector mTextSelector;
    private ToolItemBean mCurToolItem;

    private boolean mIsContinuousCreate = true;
    private boolean mbAreaTool;

    public HighlightToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
        mTextSelector = new TextSelector(pdfViewCtrl);
        mPaint = new Paint();
        mPaint.setAntiAlias(true);
        mPaint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.MULTIPLY));
        mAreaPaint = new Paint();
        mAreaPaint.setAntiAlias(true);
        mAreaPaint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.MULTIPLY));
        mBBoxRect = new RectF();

        this.mContext = context;
        mUiextensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mPropertyBar = mUiextensionsManager.getMainFrame().getPropertyBar();

        Config config = mUiextensionsManager.getConfig();
        mColor = config.uiSettings.annotations.highlight.color;
        double opacity = config.uiSettings.annotations.highlight.opacity;
        mOpacity = AppDmUtil.opacity100To255((int) (opacity * 100));
        mPaint.setColor(AppDmUtil.calColorByMultiply(mColor, mOpacity));

        mAreaColor = config.uiSettings.annotations.areaHighlight.color;
        double opacity2 = config.uiSettings.annotations.areaHighlight.opacity;
        mAreaOpacity = AppDmUtil.opacity100To255((int) (opacity2 * 100));
        mAreaPaint.setColor(AppDmUtil.calColorByMultiply(mAreaColor, mAreaOpacity));
        mDefaultSizeDimension = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_48dp);
    }

    protected void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    void resetPropertyBar() {
        int[] pbColors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        long supportProperty = PropertyBar.PROPERTY_COLOR | PropertyBar.PROPERTY_OPACITY;
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, pbColors, 0, pbColors.length);
        mPropertyBar.setColors(pbColors);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, getColor());
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, AppDmUtil.opacity255To100(getOpacity()));
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.reset(supportProperty);
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_HIGHLIGHT;
    }

    private UIMagnifierView mMagnifierView;

    @Override
    public void onActivate() {
        mTextSelector.clear();
        mBBoxRect.setEmpty();
        resetPropertyBar();
        initMagnifierView();
    }

    private RectF mTmpRectF = new RectF();
    private Rect mTmpRoundRect = new Rect();
    private RectF mAreaDrawRect = new RectF();
    private PointF mStartPoint = new PointF(0, 0);
    private PointF mStopPoint = new PointF(0, 0);
    private RectF mSelectRect = new RectF();
    private int mPageView_W;
    private int mPageView_H;

    @Override
    public void onDeactivate() {
        mTextSelector.clear();
        mBBoxRect.setEmpty();
        mSelectRect.setEmpty();
        mAreaDrawRect.setEmpty();
        mCurrentIndex = -1;
        mbAreaTool = false;
        removeMagnifierView();
    }

    private void onTouchMove(int x, int y) {
        if (x < 0) x = 0;
        if (y < 0) y = 0;
        if (x > mPageView_W) x = mPageView_W;
        if (y > mPageView_H) y = mPageView_H;

        if (x >= mStartPoint.x && y >= mStartPoint.y) {
            //4
            mSelectRect.set(mStartPoint.x, mStartPoint.y, x, y);
        }

        if (x >= mStartPoint.x && y <= mStartPoint.y) {
            //2
            mSelectRect.set(mStartPoint.x, y, x, mStartPoint.y);
        }

        if (x <= mStartPoint.x && y >= mStartPoint.y) {
            //3
            mSelectRect.set(x, mStartPoint.y, mStartPoint.x, y);
        }

        if (x <= mStartPoint.x && y <= mStartPoint.y) {
            //1
            mSelectRect.set(x, y, mStartPoint.x, mStartPoint.y);
        }
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiextensionsManager.getDocumentManager().getCurrentAnnot() != null){
            return mUiextensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        boolean handled;
        if (mbAreaTool){
            handled = onHighlightToolTouch(pageIndex, motionEvent);
            mSelectRect.setEmpty();
            mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
            return handled;
        }else {
            mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
            handled = onHighlightToolTouch(pageIndex, motionEvent);
        }
        return handled;
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiextensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!mbAreaTool && !handled){
            handled = handleHighlightTapConfirmed(pageIndex, motionEvent);
        }else if (mbAreaTool){
            handled = true;
        }
        return handled;
    }

    private boolean handleHighlightTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        PDFPage page = mUiextensionsManager.getDocumentManager().getPage(pageIndex, false);
        if (page == null) {
            return true;
        }
        try {
            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);
            PointF pagePt = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
            PointF pdfPt = new PointF();
            mPdfViewCtrl.convertPageViewPtToPdfPt(pagePt, pdfPt, pageIndex);
            Range range = textPage.getWordAtPos(pdfPt.x,pdfPt.y,10);
            if (range.getSegmentCount() == 1){
                int start = range.getSegmentStart(0);
                int end = range.getSegmentEnd(0);
                mTextSelector.start(page, start);
                mTextSelector.update(page, end);
                if (mTextSelector.getRectFList().size() > 0){
                    mTextSelector.setContents(mTextSelector.getText(page));
                    mCurrentIndex = pageIndex;
                    addAnnot(mCurrentIndex, true, null, null);
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return true;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinuousCreate;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinuousCreate = continueAddAnnot;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mCurrentIndex != pageIndex) return;

        if (mbAreaTool) {
            canvas.save();
            canvas.drawRect(mSelectRect, mAreaPaint);
            canvas.restore();
        } else {
            Rect clipRect = canvas.getClipBounds();
            for (RectF rect : mTextSelector.getRectFList()) {
                mPdfViewCtrl.convertPdfRectToPageViewRect(rect, mTmpRectF, mCurrentIndex);
                mTmpRectF.round(mTmpRoundRect);
                if (mTmpRoundRect.intersect(clipRect)) {
                    canvas.save();
                    canvas.drawRect(mTmpRoundRect, mPaint);
                    canvas.restore();
                }
            }
        }
    }

    protected void setPaint(int color, int opacity) {
        if (mbAreaTool) {
            mAreaColor = color;
            mAreaOpacity = opacity;
            mAreaPaint.setColor(AppDmUtil.calColorByMultiply(mAreaColor, mAreaOpacity));
        } else {
            mColor = color;
            mOpacity = opacity;
            mPaint.setColor(AppDmUtil.calColorByMultiply(mColor, mOpacity));
        }
        setToolItemColor(color, opacity, mCurToolItem);
    }

    protected void setUpdateDefaultPaint(boolean isAreaTool, int color, int opacity){
        if (isAreaTool) {
            mAreaColor = color;
            mAreaOpacity = opacity;
            mAreaPaint.setColor(AppDmUtil.calColorByMultiply(mAreaColor, mAreaOpacity));
        } else {
            mColor = color;
            mOpacity = opacity;
            mPaint.setColor(AppDmUtil.calColorByMultiply(mColor, mOpacity));
        }
    }

    private void setToolItemColor(int color, int opacity, ToolItemBean mCurToolItem) {
        if(mCurToolItem==null){
            return;
        }
        mCurToolItem.property.color = color;
        mCurToolItem.property.opacity = opacity;
        if(mCurToolItem.toolItem==null){
            return;
        }
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
    }

    int getOpacity() {
        return mbAreaTool ? mAreaOpacity : mOpacity;
    }

    int getColor() {
        return mbAreaTool ? mAreaColor : mColor;
    }
    public int getAreaLastColor() {
        return mAreaColor;
    }

    public int getAreaLastOpacity() {
        return mAreaOpacity;
    }

    protected void addAnnot(final int pageIndex, final boolean addUndo, final AnnotContent contentSupplier, final Event.Callback result) {
        int color = getColor();
        int opacity = getOpacity();

        final Highlight annot;
        try {
            PDFPage page = mUiextensionsManager.getDocumentManager().getPage(pageIndex, false);
            annot = (Highlight) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Highlight, AppUtil.toFxRectF(new RectF())), Annot.e_Highlight);
            if (annot == null) {
                if (!misFromSelector && !mIsContinuousCreate) {
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
                }
                return;
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
            return;
        }

        final HighlightAddUndoItem undoItem = new HighlightAddUndoItem(mPdfViewCtrl);
        undoItem.mColor = color;
        undoItem.quadPointsArray = new QuadPointsArray();
        QuadPoints quadPoint;
        if (contentSupplier instanceof TextMarkupContent) {
            TextMarkupContent tmContent = (TextMarkupContent) contentSupplier;
            undoItem.setCurrentValue(tmContent);
            ArrayList<PointF> pointsList = tmContent.getQuadPoints();

            if (pointsList != null && pointsList.size() >= 4) {
                for (int i = 0; i < pointsList.size() / 4; i++) {
                    quadPoint = new QuadPoints(AppUtil.toFxPointF(pointsList.get(4 * i).x, pointsList.get(4 * i).y),
                            AppUtil.toFxPointF(pointsList.get(4 * i + 1).x, pointsList.get(4 * i + 1).y),
                            AppUtil.toFxPointF(pointsList.get(4 * i + 2).x, pointsList.get(4 * i + 2).y),
                            AppUtil.toFxPointF(pointsList.get(4 * i + 3).x, pointsList.get(4 * i + 3).y));
                    undoItem.quadPointsArray.add(quadPoint);
                }
            } else {
                RectF rect = new RectF(contentSupplier.getBBox());
                if (Math.abs(rect.width()) > 0 && Math.abs(rect.height()) > 0) {
                    quadPoint = new QuadPoints();
                    quadPoint.setFirst(AppUtil.toFxPointF(rect.left, rect.top));
                    quadPoint.setSecond(AppUtil.toFxPointF(rect.right, rect.top));
                    quadPoint.setThird(AppUtil.toFxPointF(rect.left, rect.bottom));
                    quadPoint.setFourth(AppUtil.toFxPointF(rect.right, rect.bottom));
                    undoItem.quadPointsArray.add(quadPoint);
                }
            }
        } else if (contentSupplier instanceof TextMarkupContentAbs) {
            TextMarkupContentAbs tmSelector = (TextMarkupContentAbs) contentSupplier;
            for (int i = 0; i < tmSelector.getTextSelector().getRectFList().size(); i++) {
                RectF rect = tmSelector.getTextSelector().getRectFList().get(i);
                quadPoint = new QuadPoints();
                quadPoint.setFirst(AppUtil.toFxPointF(rect.left, rect.top));
                quadPoint.setSecond(AppUtil.toFxPointF(rect.right, rect.top));
                quadPoint.setThird(AppUtil.toFxPointF(rect.left, rect.bottom));
                quadPoint.setFourth(AppUtil.toFxPointF(rect.right, rect.bottom));

                undoItem.quadPointsArray.add(quadPoint);
            }

            undoItem.mColor = color;
            undoItem.mOpacity = opacity / 255f;
            if (mUiextensionsManager.getDocumentManager().withAddPopupAnnotPermission(annot))
                undoItem.mContents = tmSelector.getContents();
        } else if (mTextSelector != null) {
            if (mbAreaTool) {
                RectF rect = new RectF();
                mPdfViewCtrl.convertPageViewRectToPdfRect(mAreaDrawRect, rect, pageIndex);
                quadPoint = new QuadPoints();
                quadPoint.setFirst(AppUtil.toFxPointF(rect.left, rect.top));
                quadPoint.setSecond(AppUtil.toFxPointF(rect.right, rect.top));
                quadPoint.setThird(AppUtil.toFxPointF(rect.left, rect.bottom));
                quadPoint.setFourth(AppUtil.toFxPointF(rect.right, rect.bottom));
                undoItem.quadPointsArray.add(quadPoint);
            } else {
                for (int i = 0; i < mTextSelector.getRectFList().size(); i++) {
                    RectF rF = mTextSelector.getRectFList().get(i);
                    quadPoint = new QuadPoints();
                    quadPoint.setFirst(AppUtil.toFxPointF(rF.left, rF.top));
                    quadPoint.setSecond(AppUtil.toFxPointF(rF.right, rF.top));
                    quadPoint.setThird(AppUtil.toFxPointF(rF.left, rF.bottom));
                    quadPoint.setFourth(AppUtil.toFxPointF(rF.right, rF.bottom));
                    undoItem.quadPointsArray.add(quadPoint);
                }
            }
            undoItem.mColor = color;
            undoItem.mOpacity = opacity / 255f;
            if (mUiextensionsManager.getDocumentManager().withAddPopupAnnotPermission(annot))
                undoItem.mContents = mTextSelector.getContents();
            undoItem.mFlags = Annot.e_FlagPrint;
        } else {
            if (result != null) {
                result.result(null, false);
            }
            return;
        }

        undoItem.mNM = (contentSupplier != null && !AppUtil.isEmpty( contentSupplier.getNM())) ? contentSupplier.getNM() : AppDmUtil.randomUUID(null);
        undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
        undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
        undoItem.mAuthor = (contentSupplier != null && contentSupplier.getAuthor() != null) ? contentSupplier.getAuthor() : ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
        undoItem.mPageIndex = pageIndex;
        undoItem.mSubject = "Highlight";
        if (contentSupplier instanceof TextMarkupContent) {
            if (contentSupplier.getIntent() != null && contentSupplier.getIntent().equals(TextmarkupConstants.AREA_HIGHLIGHT)) {
                undoItem.mbAreaHi = true;
                undoItem.mIntent = TextmarkupConstants.AREA_HIGHLIGHT;
            }
        } else {
            if (mbAreaTool) {
                undoItem.mbAreaHi = true;
                undoItem.mIntent = TextmarkupConstants.AREA_HIGHLIGHT;
            }
        }
        HighlightEvent event = new HighlightEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, annot, mPdfViewCtrl);
        EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (success) {
                    try {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(mPdfViewCtrl.getDoc().getPage(pageIndex), annot);
                        if (addUndo) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(undoItem);
                        }
                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            RectF rectF = new RectF(AppUtil.toRectF(annot.getRect()));
                            invalidate(mPdfViewCtrl, pageIndex, rectF);
                        }
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }

                    mTextSelector.clear();
                    mBBoxRect.setEmpty();
                    mAreaDrawRect.setEmpty();
                    mSelectRect.setEmpty();
                    mStartPoint.set(0, 0);
                    mStopPoint.set(0, 0);
                    mCurrentIndex = -1;
                    if (!misFromSelector) {
                        if (!mIsContinuousCreate) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
                        }
                    }
                    misFromSelector = false;
                }

                if (result != null) {
                    result.result(null, success);
                }
            }
        });

        mPdfViewCtrl.addTask(task);
    }

    private void invalidateTouch(PDFViewCtrl pdfViewCtrl, int pageIndex, RectF rectF) {
        if (rectF == null) return;
        RectF rBBox = new RectF(rectF);
        pdfViewCtrl.convertPageViewRectToDisplayViewRect(rBBox, rBBox, pageIndex);
        RectF rCalRectF = AppUtil.calculateRect(mBbox, mBBoxRect);
        rCalRectF.roundOut(mInvalidateRect);
        pdfViewCtrl.invalidate(mInvalidateRect);
        mBBoxRect.set(rBBox);
    }

    private RectF mBbox = new RectF();
    private Rect mInvalidateRect = new Rect();

    private void invalidate(PDFViewCtrl pdfViewCtrl, int pageIndex, RectF rectF) {
        if (rectF == null || !pdfViewCtrl.isPageVisible(pageIndex)) return;
        mBbox.set(rectF);
        mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, mBbox, pageIndex);
        mBbox.roundOut(mInvalidateRect);
        pdfViewCtrl.refresh(pageIndex, mInvalidateRect);
    }

    protected void removeProbarListener() {
        mPropertyChangeListener = null;
    }

    private boolean misFromSelector = false;

    public void setFromSelector(boolean b) {
        misFromSelector = b;
    }

    private IToolSupply mToolSupply;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new HighlightToolSupply(mContext);
        return mToolSupply;
    }

    private class HighlightToolSupply extends ToolSupplyImpl {

        public HighlightToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            if (toolType == ToolConstants.Highlight)
                return R.drawable.comment_tool_highlight_bg;
            else if (toolType == ToolConstants.Area_Highlight)
                return R.drawable.comment_tool_area_highlight_bg;
            return 0;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            if (toolType == ToolConstants.Highlight)
                return R.drawable.comment_tool_highlight_src;
            else if (toolType == ToolConstants.Area_Highlight)
                return R.drawable.comment_tool_area_highlight_src;
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = toolType;
            if (toolType == ToolConstants.Area_Highlight) {
                property.color = mAreaColor;
                property.opacity = mAreaOpacity;
            } else {
                property.color = mColor;
                property.opacity = mOpacity;
            }
            return property;
        }

        @Override
        public void setPropertyBar(ToolProperty property) {
            super.setPropertyBar(property);
            setPaint(property.color,property.opacity);
        }

        @Override
        public String getToolName(int toolType) {
            return toolType == ToolConstants.Area_Highlight ? JsonConstants.TYPE_AREA_HIGHLIGHT : JsonConstants.TYPE_HIGHLIGHT;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                mbAreaTool = itemBean.type == ToolConstants.Area_Highlight;
                if(mUiextensionsManager.getMainFrame().getCurrentTab()== ToolbarItemConfig.ITEM_COMMENT_TAB) {
                    if(mbAreaTool){
                        mUiextensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_AreaHighlight);
                    }else{
                        mUiextensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_Highlight);
                    }
                }

                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }

                if (mbAreaTool) {
                    mAreaColor = property.color;
                    mAreaOpacity = property.opacity;
                    mAreaPaint.setColor(AppDmUtil.calColorByMultiply(mAreaColor, mAreaOpacity));
                } else {
                    mColor = property.color;
                    mOpacity = property.opacity;
                    mPaint.setColor(AppDmUtil.calColorByMultiply(mColor, mOpacity));
                }
                mUiextensionsManager.setCurrentToolHandler(HighlightToolHandler.this);
            } else {
                mCurToolItem = null;
                mbAreaTool = false;

                if (mUiextensionsManager.getCurrentToolHandler() == HighlightToolHandler.this)
                    mUiextensionsManager.setCurrentToolHandler(null);
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;
            mbAreaTool = itemBean.type == ToolConstants.Area_Highlight;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }

            if (mbAreaTool) {
                mAreaColor = property.color;
                mAreaOpacity = property.opacity;
            } else {
                mColor = property.color;
                mOpacity = property.opacity;
            }
            HighlightToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mbAreaTool = false;
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

    public void setAreaTool(boolean value) {
        mbAreaTool = value;
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiextensionsManager.defaultTouchEvent(pageIndex, motionEvent);
        if (!handled && motionEvent.getActionMasked() != MotionEvent.ACTION_DOWN)
            handled = onHighlightToolTouch(pageIndex, motionEvent);
        return handled;
    }

    private PointF mLastDownPoint;
    private boolean isHighlightMoving = false;
    private boolean onHighlightToolTouch(int pageIndex, MotionEvent motionEvent) {
        try {
            PDFPage page = mUiextensionsManager.getDocumentManager().getPage(pageIndex, false);
            if (page == null) {
                return false;
            }
            TextPage textPage = null;
            if (!mbAreaTool)
                textPage = new TextPage(page, TextPage.e_ParseTextNormal);
            PointF pagePt = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
            float x = pagePt.x;
            float y = pagePt.y;

            PointF pdfPt = new PointF();
            mPdfViewCtrl.convertPageViewPtToPdfPt(pagePt, pdfPt, pageIndex);
            int action = motionEvent.getAction();
            switch (action) {
                case MotionEvent.ACTION_DOWN: {
                    if (mLastDownPoint == null)
                        mLastDownPoint = new PointF();
                    mLastDownPoint.set(pagePt);
                    mCurrentIndex = pageIndex;
                    if (mbAreaTool) {
                        mStartPoint.x = x;
                        mStartPoint.y = y;
                        mStopPoint.x = x;
                        mStopPoint.y = y;
                        mSelectRect.set(x, y, 0, 0);

                        mPageView_W = mPdfViewCtrl.getPageViewWidth(pageIndex);
                        mPageView_H = mPdfViewCtrl.getPageViewHeight(pageIndex);
                    } else {
                        isHighlightMoving = false;
                        if (textPage != null) {
                            int index = textPage.getIndexAtPos(pdfPt.x, pdfPt.y, 10);
                            if (index >= 0) {
                                mTextSelector.start(page, index);
                            }
                        }
                        if (mMagnifierView != null) {
                            mMagnifierView.onTouchEvent(motionEvent);
                            mMagnifierView.setVisibility(View.VISIBLE);
                        }
                    }
                    break;
                }
                case MotionEvent.ACTION_MOVE: {
                    if (mCurrentIndex != pageIndex) return true;

                    if (mbAreaTool) {
                        if (!mStopPoint.equals(x, y)) {
                            onTouchMove((int) x, (int) y);
                            mAreaDrawRect.set(mSelectRect);
                            invalidateTouch(mPdfViewCtrl, pageIndex, mAreaDrawRect);
                            mStopPoint.set(x, y);
                        }
                    } else {
                        if (AppDmUtil.distanceOfTwoPoints(mLastDownPoint,pagePt) < 2)
                            break;
                        isHighlightMoving = true;
                        if (textPage != null) {
                            int index = textPage.getIndexAtPos(pdfPt.x, pdfPt.y, 10);
                            if (index >= 0) {
                                mTextSelector.update(page, index);
                            }
                            RectF pageViewRect = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(mTextSelector.getBbox(), pageViewRect, pageIndex);
                            invalidateTouch(mPdfViewCtrl, pageIndex, pageViewRect);
                        }
                        if (mMagnifierView != null) {
                            mMagnifierView.onTouchEvent(motionEvent);
                        }
                    }
                    break;
                }
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL: {
                    if (mMagnifierView != null) {
                        mMagnifierView.setVisibility(View.GONE);
                    }
                    if (mbAreaTool) {
                        if (!mStartPoint.equals(mStopPoint.x, mStopPoint.y) && Math.abs(mStopPoint.length() - mStartPoint.length()) > 8) {
                            addAnnot(mCurrentIndex, true, null, null);
                        }
                    } else {
                        if (mTextSelector.getRectFList().size() == 0) {
                            isHighlightMoving = false;
                            break;
                        }
                        if (textPage != null && mTextSelector.getStart() == mTextSelector.getEnd() && !isHighlightMoving){
                            Range range = textPage.getWordAtPos(pdfPt.x,pdfPt.y,10);
                            if (range.getSegmentCount() == 1){
                                int start = range.getSegmentStart(0);
                                int end = range.getSegmentEnd(0);
                                mTextSelector.start(page, start);
                                mTextSelector.update(page, end);
                            }
                        }
                        mTextSelector.setContents(mTextSelector.getText(page));
                        addAnnot(mCurrentIndex, true, null, null);
                        isHighlightMoving = false;
                    }
                    return true;
                }
                default:
                    break;
            }
        } catch (PDFException e1) {
            e1.printStackTrace();
        }
        return true;
    }

    private int mDefaultSizeDimension;

    private void adjustPointOrNot(int pageViewWidth, int pageViewHeight, int dx) {
        float startPointOffset = dx / -2f;

        int offset = 0;
        float minStartX = offset - startPointOffset;
        float maxStartX = pageViewWidth - offset + startPointOffset;
        float maxStartY = pageViewHeight - offset + startPointOffset;
        if (mStartPoint.x < minStartX){
            mStartPoint.x = minStartX;
        }
        if (mStartPoint.x > maxStartX){
            mStartPoint.x = maxStartX;
        }
        if (mStartPoint.y < minStartX){
            //noinspection SuspiciousNameCombination
            mStartPoint.y = minStartX;
        }
        if (mStartPoint.y > maxStartY){
            mStartPoint.y = maxStartY;
        }
        mStopPoint.x = mStartPoint.x;
        mStopPoint.y = mStartPoint.y;
        mStartPoint.offset(startPointOffset,startPointOffset);
        mStopPoint.offset(-startPointOffset,-startPointOffset);
    }

    private void initMagnifierView() {
        MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
        if (mMagnifierView == null) {
            mMagnifierView = new UIMagnifierView(mContext.getApplicationContext());
        }
        mMagnifierView.setTargetView(mPdfViewCtrl);
        mMagnifierView.setVisibility(View.GONE);
        mainFrame.getContentView().addView(mMagnifierView);
    }

    private void removeMagnifierView() {
        if (mMagnifierView != null) {
            MainFrame mainFrame = (MainFrame) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getMainFrame();
            mainFrame.getContentView().removeView(mMagnifierView);
            mMagnifierView.setTargetView(null);
        }
    }
}
