/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.stamp.customstamp;


import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.annots.stamp.StampConstants;
import com.foxit.uiextensions.annots.stamp.StampUtil;
import com.foxit.uiextensions.controls.ToastSeekBar;
import com.foxit.uiextensions.controls.dialog.BaseDialogFragment;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;

import java.io.File;

import androidx.annotation.NonNull;

public class ImageOpacityFragment extends BaseDialogFragment {
    public static final String TAG = ImageOpacityFragment.class.getSimpleName();

    private IPositiveItemClickCallback mItemClickCallback;
    private PDFViewCtrl mViewCtrl;
    private ImageStampBean mStampBean;
    private Bitmap mTempThumbnail;

    private int mOperaType;
    private boolean mShouldDeleteCache;

    public void init(int type, PDFViewCtrl viewCtrl, ImageStampBean stampBean) {
        mOperaType = type;
        mViewCtrl = viewCtrl;
        mStampBean = stampBean;
    }

    @Override
    protected View onCreateView(LayoutInflater inflater, ViewGroup container) {
        if (mOperaType == StampConstants.MODIFY_STAMP) {
            File file = new File(mStampBean.mCacheFile);
            if (file.exists()) {
                mTempThumbnail = StampUtil.createBitmapFromPath(mStampBean.mCacheFile, 3);
            } else {
                mTempThumbnail = BitmapFactory.decodeResource(getResources(), R.drawable.custom_stamp_image_empty_icon);
            }
        } else {
            mTempThumbnail = StampUtil.createBitmapFromPath(mStampBean.mCacheFile, 3);
        }

        View contentView = inflater.inflate(R.layout.fx_stamp_custom_opacity, container, false);
        final ImageView opacityIv = contentView.findViewById(R.id.custom_stamp_opacity_iv);
        opacityIv.setImageAlpha(AppDmUtil.opacity100To255(mStampBean.mOpacity));
        opacityIv.setImageBitmap(mTempThumbnail);

        final ToastSeekBar toastSeekBar = contentView.findViewById(R.id.custom_stamp_opacity_seek_bar);
        toastSeekBar.setProgressColor(ThemeConfig.getInstance(getContext()).getPrimaryColor());
        View toastView = View.inflate(getActivity(), R.layout.fx_stamp_custom_opacity_toast, null);
        TextView textView = toastView.findViewById(R.id.stamp_custom_image_opacity);
        textView.setTextColor(AppResource.getColor(mContext, R.color.ux_color_ff2e2e2e));
        toastSeekBar.setToastView(textView);
        toastSeekBar.setOnSeekBarProgressListener(new ToastSeekBar.OnSeekBarProgressListener() {
            @Override
            public void onProgress(int progress) {
                opacityIv.setImageAlpha(AppDmUtil.opacity100To255(progress));
            }
        });
        toastSeekBar.reset(mStampBean.mOpacity);

        mShouldDeleteCache = mOperaType == StampConstants.ADD_STAMP && mStampBean.mCacheFile.contains("custom_stamps");
        contentView.findViewById(R.id.custom_stamp_opacity_cancel).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dismiss();
            }
        });
        contentView.findViewById(R.id.custom_stamp_opacity_done).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mShouldDeleteCache = false;

                try {
                    if (mOperaType == StampConstants.ADD_STAMP) {
                        if (!mStampBean.mCacheFile.contains("custom_stamps")) {
                            String newPath = StampUtil.getImageCacheDir(getContext()) + AppFileUtil.getFileName(mStampBean.mCacheFile);
                            AppFileUtil.copyFile(mStampBean.mCacheFile, newPath);
                            mStampBean.mCacheFile = newPath;
                        }

                        String cachePath = mStampBean.mCacheFile;
                        String cachePDFPath = StampUtil.convertImageToPDF(cachePath, toastSeekBar.getProgress());
                        PDFDoc doc = new PDFDoc(cachePDFPath);
                        int errorCode = doc.load(null);
                        if (errorCode == Constants.e_ErrSuccess) {
                            PDFPage page = doc.getPage(0);
                            mStampBean.mWidth = (int) page.getWidth();
                            mStampBean.mHeight = (int) page.getHeight();
                            page.delete();
                            doc.delete();
                        } else {
                            int[] imageInfo = StampUtil.getImageInfo(cachePath);
                            mStampBean.mWidth = imageInfo[0];
                            mStampBean.mHeight = imageInfo[1];
                        }
                        mStampBean.mCachePDFFile = cachePDFPath;
                        mStampBean.mThumbnailBitmap = StampUtil.getThumbnailBitmap(getContext(), cachePath);
                    } else {
                        if (mStampBean.mOpacity != toastSeekBar.getProgress()) {
                            mStampBean.mModifiedTime = System.currentTimeMillis();
                            //mStampBean.mCachePDFFile = StampUtil.convertImageToPDF(mStampBean.mCacheFile, toastSeekBar.getProgress());
                        }
                    }
                    mStampBean.mOpacity = toastSeekBar.getProgress();
                } catch (PDFException ignored) {
                }

                if (mItemClickCallback != null)
                    mItemClickCallback.onClick(v, mStampBean);
                dismiss();
            }
        });
        return contentView;
    }

    @NonNull
    @Override
    protected PDFViewCtrl getPDFViewCtrl() {
        return mViewCtrl;
    }

    @Override
    protected void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
    }

    @Override
    public void onDetach() {
        super.onDetach();
        if (mShouldDeleteCache) {
            AppFileUtil.deleteFile(mStampBean.mCacheFile);
        }
        if (mTempThumbnail != null) {
            mTempThumbnail.recycle();
            mTempThumbnail = null;
        }
    }

    public void setPositiveItemClickCallback(IPositiveItemClickCallback callback) {
        mItemClickCallback = callback;
    }

    public interface IPositiveItemClickCallback {
        void onClick(View v, ImageStampBean stampBean);
    }

}
