/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.stamp.customstamp;


import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Color;
import android.graphics.Point;
import android.graphics.Rect;
import android.os.Environment;
import android.text.TextUtils;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Progressive;
import com.foxit.sdk.common.Renderer;
import com.foxit.sdk.common.fxcrt.Matrix2D;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Widget;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.stamp.StampConstants;
import com.foxit.uiextensions.annots.stamp.StampUtil;
import com.foxit.uiextensions.utils.AppFileUtil;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

public class DrawStampThumbnailTask extends Task {
    private final Context mContext;
    private final UIExtensionsManager mUiExtensionsManager;
    private TextStampBean mTextStampBean;
    private ImageStampBean mImageStampBean;

    private final int mType;
    private final Rect mBmpArea;
    private final Point mViewSize;
    private Bitmap mBmp;

    public DrawStampThumbnailTask(Context context, UIExtensionsManager uiExtensionsManager,
                                  final TextStampBean stampBean, final IDrawStampThumbnailCallback<TextStampBean> callBack) {
        super(new CallBack() {
            @Override
            public void result(Task task) {
                stampBean.mIsRendering = false;
                DrawStampThumbnailTask task1 = (DrawStampThumbnailTask) task;
                if (task1.mStatus == STATUS_FINISHED) {
                    callBack.onResult(task1, stampBean, ((DrawStampThumbnailTask) task).mBmp);
                }
            }
        });
        mType = StampConstants.CUSTOM_TEXT_STAMP;
        mUiExtensionsManager = uiExtensionsManager;
        mContext = context;
        mTextStampBean = stampBean;
        mViewSize = new Point(100, 30);
        mBmpArea = new Rect(0, 0, mViewSize.x, mViewSize.y);
        mPriority = PRIORITY_PATCH;

        mTextStampBean.mIsRendering = true;
    }

    public DrawStampThumbnailTask(Context context, UIExtensionsManager uiExtensionsManager,
                                  final ImageStampBean stampBean, final IDrawStampThumbnailCallback<ImageStampBean> callBack) {
        super(new CallBack() {
            @Override
            public void result(Task task) {
                stampBean.mIsRendering = false;
                DrawStampThumbnailTask task1 = (DrawStampThumbnailTask) task;
                if (task1.mStatus == STATUS_FINISHED) {
                    callBack.onResult(task1, stampBean, ((DrawStampThumbnailTask) task).mBmp);
                }
            }
        });

        mType = StampConstants.CUSTOM_IMAGE_STAMP;
        mUiExtensionsManager = uiExtensionsManager;
        mContext = context;
        mImageStampBean = stampBean;
        mViewSize = new Point(160, 75);
        mBmpArea = new Rect(0, 0, mViewSize.x, mViewSize.y);
        mPriority = PRIORITY_PATCH;

        mImageStampBean.mIsRendering = true;
    }

    @Override
    protected void prepare() {
        if (mType == StampConstants.CUSTOM_TEXT_STAMP) {
            if (mBmp == null) {
                if (mBmpArea.width() == 0 || mBmpArea.height() == 0) {
                    mStatus = STATUS_ERROR;
                    return;
                }
                mBmp = Bitmap.createBitmap(mBmpArea.width(), mBmpArea.height(),
                        Bitmap.Config.RGB_565);
            }
        }
    }

    @Override
    protected void execute() {
        if (mStatus != STATUS_REDAY)
            return;
        mStatus = STATUS_RUNNING;
        if (mBmpArea.width() == 0 || mBmpArea.height() == 0) {
            mStatus = STATUS_ERROR;
            return;
        }

        if (mType == StampConstants.CUSTOM_TEXT_STAMP) {
            renderPage();
        } else {
            mBmp = getImageThumbnail();
        }
    }

    public BaseStampBean getThumbnailItem() {
        if (mType == StampConstants.CUSTOM_TEXT_STAMP)
            return mTextStampBean;
        else
            return mImageStampBean;
    }

    private void renderPage() {
        FileOutputStream fos = null;
        InputStream is = null;
        try {
            if (mTextStampBean == null) {
                mErr = Constants.e_ErrParam;
                mStatus = STATUS_ERROR;
                return;
            }

            String assetsPath = "CustomStamps/";
            String stampFileName;
            if (mTextStampBean.mColor == StampConstants.Custom_Colors[0]) {
                stampFileName = assetsPath + "custom_red.pdf";
            } else if (mTextStampBean.mColor == StampConstants.Custom_Colors[1]) {
                stampFileName = assetsPath + "custom_green.pdf";
            } else if (mTextStampBean.mColor == StampConstants.Custom_Colors[2]) {
                stampFileName = assetsPath + "custom_blue.pdf";
            } else if (mTextStampBean.mColor == StampConstants.Custom_Colors[3]) {
                stampFileName = assetsPath + "custom_purple.pdf";
            } else {
                stampFileName = assetsPath + "custom_red.pdf";
            }

            String cachePath = AppFileUtil.getDiskCachePath(mContext);
            if (TextUtils.isEmpty(cachePath)) {
                cachePath = Environment.getExternalStorageDirectory().getPath() + "/FoxitSDK/";
            } else if (!cachePath.endsWith("/")) {
                cachePath += "/";
            }

            String cacheFile = cachePath + stampFileName;
            File file = new File(cacheFile);
            if (!file.exists()) {
                File dir = new File(cachePath + assetsPath);
                if (!dir.exists()) {
                    dir.mkdirs();
                }

                is = mContext.getAssets().open(stampFileName);
                byte[] buffer = new byte[1 << 13];
                fos = new FileOutputStream(file);
                int n;
                while (-1 != (n = is.read(buffer))) {
                    fos.write(buffer, 0, n);
                }
            }

            PDFDoc pdfDoc = new PDFDoc(cacheFile);
            int errorCode = pdfDoc.load(null);
            if (errorCode != Constants.e_ErrSuccess) {
                mErr = Constants.e_ErrNotFound;
                mStatus = STATUS_ERROR;
                return;
            }

            PDFPage page = pdfDoc.getPage(0);
            if (!page.isParsed()) {
                Progressive progressive = page.startParse(PDFPage.e_ParsePageNormal, null, false);
                int state = Progressive.e_ToBeContinued;
                while (state == Progressive.e_ToBeContinued) {
                    state = progressive.resume();
                }
            }

            if (mTextStampBean.mShowAuthor || mTextStampBean.mShowDate || mTextStampBean.mShowTime) {
                //title field -- on the top
                setValue(page.getAnnot(1), mTextStampBean.mText);
                //date field -- on the bottom
                setValue(page.getAnnot(0), StampUtil.getCustomDateString(mUiExtensionsManager.getAnnotAuthor(), mTextStampBean));
            } else {
                //title field -- on the center
                setValue(page.getAnnot(2), mTextStampBean.mText);
            }

            mBmp.eraseColor(Color.WHITE);
            Matrix2D matrix = page.getDisplayMatrix(0, 0, mViewSize.x, mViewSize.y, 0);
            Renderer render = new Renderer(mBmp, true);
            Progressive progressive = render.startRender(page, matrix, null);
            int state = Progressive.e_ToBeContinued;
            while (state == Progressive.e_ToBeContinued) {
                state = progressive.resume();
            }

            mErr = Constants.e_ErrSuccess;
            mStatus = STATUS_FINISHED;
        } catch (PDFException e) {
            mErr = e.getLastError();
            mStatus = STATUS_ERROR;
        } catch (IOException e) {
            mErr = Constants.e_ErrUnknown;
            mStatus = STATUS_ERROR;
        } finally {
            try {
                if (is != null) {
                    is.close();
                }
                if (fos != null) {
                    fos.flush();
                    fos.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private void setValue(Annot annot, String value) throws PDFException {
        Widget widget = new Widget(annot);
        widget.getField().setValue(value);
        widget.resetAppearanceStream();
    }

    private Bitmap getImageThumbnail() {
        mBmp = StampUtil.getThumbnailBitmap(mContext, mImageStampBean.mCacheFile);
        mErr = Constants.e_ErrSuccess;
        mStatus = STATUS_FINISHED;
        return mBmp;
    }

}
