/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.stamp.adapter;


import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.drawable.GradientDrawable;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.ImageView;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.stamp.IStampItemClickListener;
import com.foxit.uiextensions.annots.stamp.customstamp.DrawStampThumbnailTask;
import com.foxit.uiextensions.annots.stamp.customstamp.IDrawStampThumbnailCallback;
import com.foxit.uiextensions.annots.stamp.customstamp.ImageStampBean;
import com.foxit.uiextensions.browser.adapter.SuperAdapter;
import com.foxit.uiextensions.browser.adapter.viewholder.SuperViewHolder;
import com.foxit.uiextensions.modules.panel.bean.BaseBean;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;

import java.util.ArrayList;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.GridLayoutManager;

public class ImageStampAdapter extends SuperAdapter<ImageStampBean> {
    private static final int MAX_CACHE_SIZE = 40;
    private IStampItemClickListener<ImageStampBean> mItemClickListener;
    private final UIExtensionsManager mUiExtensionsManager;
    private GridLayoutManager mGridLayoutManager;
    private PDFViewCtrl mViewCtrl;

    private final ArrayList<DrawStampThumbnailTask> mTaskList;
    private final ArrayList<ImageStampBean> mCacheList;
    private final List<ImageStampBean> mItems;
    private final List<ImageStampBean> mSelectedItems;
    private ImageStampBean mLastSelectedItem = null;
    private boolean mIsEditState = false;

    public ImageStampAdapter(Context context, PDFViewCtrl viewCtrl, List<ImageStampBean> items) {
        super(context);
        mItems = items;
        mSelectedItems = new ArrayList<>();
        mCacheList = new ArrayList<>();
        mTaskList = new ArrayList<>();
        mViewCtrl = viewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) viewCtrl.getUIExtensionsManager();
    }

    public void addImageItem(ImageStampBean stampBean) {
        int index = mItems.indexOf(stampBean);
        if (index == -1) {
            mItems.add(stampBean);
        }
        notifyUpdateData();
    }

    public void setGridLayoutManager(GridLayoutManager gridLayoutManager) {
        this.mGridLayoutManager = gridLayoutManager;
    }

    public List<ImageStampBean> getItems() {
        return mItems;
    }

    public void removeSelectedItems() {
        if (mSelectedItems.contains(mLastSelectedItem)) {
            mLastSelectedItem = null;
        }
        for (ImageStampBean stampBean : mSelectedItems) {
            AppFileUtil.deleteFile(stampBean.mCacheFile);
            updateCacheList(stampBean, false);
        }

        mItems.removeAll(mSelectedItems);
        mSelectedItems.clear();
        notifyUpdateData();
    }

    public void selectAll(boolean selectAll) {
        mSelectedItems.clear();
        for (ImageStampBean stampBean : mItems) {
            stampBean.mSelected = selectAll;
            if (selectAll)
                mSelectedItems.add(stampBean);
        }
        notifyUpdateData();
    }

    public boolean isSelectAll() {
        return mSelectedItems.size() == mItems.size();
    }

    public List<ImageStampBean> getSelectedItems() {
        return mSelectedItems;
    }

    public ImageStampBean getLastSelectedItem() {
        return mLastSelectedItem;
    }

    public void setSelectedItem(ImageStampBean stampBean) {
        if (stampBean == mLastSelectedItem) return;

        if (mLastSelectedItem != null) {
            mLastSelectedItem.mSelected = false;
            int index = mItems.indexOf(mLastSelectedItem);
            notifyItemChanged(index);
            mLastSelectedItem = null;
        }

        if (stampBean != null) {
            stampBean.mSelected = true;
            int index = mItems.indexOf(stampBean);
            notifyItemChanged(index);
            mLastSelectedItem = stampBean;
        }
    }

    public void setSelectedItem(long createTime) {
        if (mLastSelectedItem != null) {
            if (mLastSelectedItem.mCreateTime == createTime) return;

            mLastSelectedItem.mSelected = false;
            int index = mItems.indexOf(mLastSelectedItem);
            notifyItemChanged(index);
            mLastSelectedItem = null;
        }

        if (createTime > 0) {
            for (int i = 0; i < mItems.size(); i++) {
                ImageStampBean stampBean = mItems.get(i);
                if (stampBean.mCreateTime == createTime) {
                    stampBean.mSelected = true;
                    notifyItemChanged(i);
                    mLastSelectedItem = stampBean;
                    break;
                }
            }
        }
    }

    public void setEditState(boolean isEditState) {
        mIsEditState = isEditState;
        if (isEditState) {
            if (mLastSelectedItem != null)
                mLastSelectedItem.mSelected = false;
        } else {
            if (mLastSelectedItem != null)
                mLastSelectedItem.mSelected = true;
            if (mSelectedItems.size() > 0) {
                for (ImageStampBean stampBean : mSelectedItems) {
                    if (stampBean != mLastSelectedItem) {
                        stampBean.mSelected = false;
                    }
                }
                mSelectedItems.clear();
            }
        }
        notifyUpdateData();
    }

    public boolean isEmpty() {
        return mItems.size() == 0;
    }

    @Override
    public void notifyUpdateData() {
        notifyDataSetChanged();
    }

    @Override
    public ImageStampBean getDataItem(int position) {
        return mItems.get(position);
    }

    @NonNull
    @Override
    public SuperViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        View itemView = LayoutInflater.from(parent.getContext()).inflate(R.layout.fx_stamp_image_item_layout, parent, false);
        return new ImageStampViewHolder(itemView);
    }

    @Override
    public int getItemCount() {
        return mItems.size();
    }

    class ImageStampViewHolder extends SuperViewHolder {
        private final View mItemView;
        private final ImageView mIconView;
        private final CheckBox mSelectedView;

        public ImageStampViewHolder(View itemView) {
            super(itemView);
            mItemView = itemView.findViewById(R.id.cusom_image_item_view);
            mIconView = itemView.findViewById(R.id.cusom_image_item_iv);
            mSelectedView = itemView.findViewById(R.id.cusom_image_item_checkview);

            mItemView.setOnClickListener(this);
            mSelectedView.setOnClickListener(this);
        }

        @Override
        public void bind(BaseBean data, int position) {
            ImageStampBean stampBean = (ImageStampBean) data;
            if (stampBean.mSelected) {
                GradientDrawable drawable = new GradientDrawable();
                drawable.setCornerRadius(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_margin_4dp));
                drawable.setColor(ThemeConfig.getInstance(getContext()).getB2());
                drawable.setStroke(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_list_divider_height),
                        ThemeConfig.getInstance(getContext()).getPrimaryColor());
                mItemView.setBackground(drawable);
            } else {
                if (mIsEditState) {
                    GradientDrawable drawable = new GradientDrawable();
                    drawable.setCornerRadius(AppResource.getDimensionPixelSize(getContext(), R.dimen.ux_margin_4dp));
                    drawable.setColor(ThemeConfig.getInstance(getContext()).getB2());
                    mItemView.setBackground(drawable);
                } else {
                    mItemView.setBackground(null);
                }
            }
            int picRotation = AppUtil.isEmpty(stampBean.mCacheFile) ? 0 : AppFileUtil.readPictureDegree(stampBean.mCacheFile);
            mIconView.setRotation(picRotation);
            mIconView.setImageAlpha(AppDmUtil.opacity100To255(stampBean.mOpacity));
            mSelectedView.setChecked(stampBean.mSelected);
            mSelectedView.setVisibility(mIsEditState ? View.VISIBLE : View.GONE);

            if (stampBean.mThumbnailBitmap != null) {
                mIconView.setImageBitmap(stampBean.mThumbnailBitmap);
            } else {
                if (stampBean.mIsRendering)
                    return;
                DrawStampThumbnailTask drawTask = new DrawStampThumbnailTask(getContext(),
                        mUiExtensionsManager, stampBean, new IDrawStampThumbnailCallback<ImageStampBean>() {
                    @Override
                    public void onResult(DrawStampThumbnailTask task, ImageStampBean stampBean, Bitmap bitmap) {
                        if (stampBean.mThumbnailBitmap != null && !stampBean.mThumbnailBitmap.isRecycled())
                            stampBean.mThumbnailBitmap.recycle();
                        stampBean.mThumbnailBitmap = bitmap;
                        mIconView.setImageBitmap(bitmap);
                        mIconView.invalidate();
                        updateCacheList(stampBean, true);
                        removeTask(task);
                    }
                });
                addTask(drawTask);
            }
        }

        @Override
        public void onClick(View v) {
            int position = getAdapterPosition();
            if (position == -1) return;
            int id = v.getId();
            if (id == R.id.cusom_image_item_view) {
                if (mIsEditState) {
                    ImageStampBean selectedBean = mItems.get(position);
                    if (mItemClickListener != null)
                        mItemClickListener.onItemClick(true, position, selectedBean);
                } else {
                    ImageStampBean selectedBean = mItems.get(position);
                    if (mLastSelectedItem == selectedBean) {
                        if (mItemClickListener != null)
                            mItemClickListener.onItemClick(false, position, selectedBean);
                        return;
                    }

                    if (mLastSelectedItem != null) {
                        mLastSelectedItem.mSelected = false;
                        int index = mItems.indexOf(mLastSelectedItem);
                        notifyItemChanged(index);
                    }
                    selectedBean.mSelected = true;
                    notifyItemChanged(position);
                    mLastSelectedItem = selectedBean;
                    if (mItemClickListener != null)
                        mItemClickListener.onItemClick(false, position, selectedBean);
                }
            } else if (id == R.id.cusom_image_item_checkview) {
                ImageStampBean itemBean = mItems.get(position);
                if (mSelectedItems.contains(itemBean)) {
                    itemBean.mSelected = false;
                    mSelectedItems.remove(itemBean);
                } else {
                    itemBean.mSelected = true;
                    mSelectedItems.add(itemBean);
                }
                notifyItemChanged(position);
                if (mItemClickListener != null)
                    mItemClickListener.onCheckedChanged(position, mSelectedItems);
            }
        }
    }

    public void setItemClickListener(IStampItemClickListener<ImageStampBean> itemClickListener) {
        mItemClickListener = itemClickListener;
    }

    public void release(){
        for (int i = 0; i < mTaskList.size(); i++) {
            DrawStampThumbnailTask task = mTaskList.get(i);
            mViewCtrl.removeTask(task);
        }
        mTaskList.clear();
    }

    private void addTask(DrawStampThumbnailTask task) {
        synchronized (mTaskList) {
            if (mTaskList.size() >= MAX_CACHE_SIZE) {
                DrawStampThumbnailTask oldTask = null;
                for (DrawStampThumbnailTask thumbnailTask : mTaskList) {
                    if (!isItemVisible((ImageStampBean) task.getThumbnailItem())) {
                        oldTask = thumbnailTask;
                        break;
                    }
                }
                if (oldTask == null) {
                    oldTask = mTaskList.get(0);
                }

                mViewCtrl.removeTask(oldTask);
                mTaskList.remove(oldTask);
                oldTask.getThumbnailItem().mIsRendering = false;
            }
            mTaskList.add(task);
            mViewCtrl.addTask(task);
        }
    }

    private void removeTask(DrawStampThumbnailTask task) {
        synchronized (mTaskList) {
            mTaskList.remove(task);
        }
    }

    private void updateCacheList(ImageStampBean stampBean, boolean add) {
        if (add) {
            if (mCacheList.contains(stampBean))
                return;
            if (mCacheList.size() >= MAX_CACHE_SIZE) {
                ImageStampBean item = mCacheList.get(0);
                item.mThumbnailBitmap = null;
                mCacheList.remove(0);
            }
            mCacheList.add(stampBean);
        } else {
            if (mCacheList.contains(stampBean)) {
                mCacheList.remove(stampBean);
                stampBean.mThumbnailBitmap = null;
            }
        }
    }

    private boolean isItemVisible(ImageStampBean stampBean) {
        if (mGridLayoutManager == null) return false;

        int position = mItems.indexOf(stampBean);
        return position >= mGridLayoutManager.findFirstVisibleItemPosition() && position <= mGridLayoutManager.findLastVisibleItemPosition();
    }

}
