/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.stamp;

import android.app.Activity;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.ColorStateList;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Build;
import android.text.Editable;
import android.text.InputFilter;
import android.text.Spanned;
import android.text.TextWatcher;
import android.util.SparseArray;
import android.util.TypedValue;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.core.widget.NestedScrollView;
import androidx.core.widget.TextViewCompat;
import androidx.fragment.app.FragmentActivity;
import androidx.fragment.app.FragmentManager;
import androidx.recyclerview.widget.DefaultItemAnimator;
import androidx.recyclerview.widget.GridLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import androidx.viewpager.widget.ViewPager;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Library;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Stamp;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotContent;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.stamp.adapter.CommonStampAdapter;
import com.foxit.uiextensions.annots.stamp.adapter.ImageStampAdapter;
import com.foxit.uiextensions.annots.stamp.adapter.TextStampAdapter;
import com.foxit.uiextensions.annots.stamp.customstamp.BaseStampBean;
import com.foxit.uiextensions.annots.stamp.customstamp.DrawStampThumbnailTask;
import com.foxit.uiextensions.annots.stamp.customstamp.IDrawStampThumbnailCallback;
import com.foxit.uiextensions.annots.stamp.customstamp.ImageOpacityFragment;
import com.foxit.uiextensions.annots.stamp.customstamp.ImageStampBean;
import com.foxit.uiextensions.annots.stamp.customstamp.TextStampBean;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.ViewPagerAdapter;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverFrag;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.sheetmenu.ISheetMenu;
import com.foxit.uiextensions.controls.dialog.sheetmenu.UISheetMenu;
import com.foxit.uiextensions.controls.panel.PanelContentViewPage;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.HomeToolBar;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.pdfreader.config.ActRequestCode;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.ActManager;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppIntentUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;
import com.google.android.material.tabs.TabLayout;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;


public class StampToolHandler implements ToolHandler, View.OnClickListener {
    private final Context mContext;
    private StampPropertyDialog mPropertyDlg;
    private final PDFViewCtrl mPdfViewCtrl;
    private final UIExtensionsManager mUiExtensionsManager;

    private boolean mIsContinuousCreate = true;
    private int mStampType = 0;
    private BaseStampBean mCurSelectedCustomStamp;

    // add custom text stamp view begin
    private TabLayout mTabLayout;
    private View mStampsRootView;
    private PanelContentViewPage mViewPager;
    private TextView mTitleTv;
    private TextView mEditTv;

    private View mAddTextStampRootView;
    private LinearLayout mCustomColorView;
    private EditText mCustomTextEdit;
    private RelativeLayout mThumbnailColorView;
    private TextView mThumbnailCustomTextTv;
    private TextView mThumbnailDateTv;
    private ImageView mClearCustomTextIv;
    private RelativeLayout mAuthorCheckLayout;
    private ImageView mAuthorCheckIv;
    private RelativeLayout mDateCheckLayout;
    private ImageView mDateCheckIv;
    private RelativeLayout mTimeCheckLayout;
    private ImageView mTimeCheckIv;

    private View mCustomStampsRootView;
    private NestedScrollView mCustomStampsContentView;
    private TextView mTextStampTitleView;
    private TextView mImageStampTitleView;
    private LinearLayout mEmptyStampsView;
    private LinearLayout mAddStampsBottombar;
    private ImageView mAddImageStampsIv;
    private RelativeLayout mDeleteStampsBottombar;
    private ImageView mDeleteStampsIv;
    private TextView mSelectedAllTv;
    private RecyclerView mTextStampsRV;
    private RecyclerView mImageStampsRV;
    private TextStampAdapter mTextStampAdapter;
    private ImageStampAdapter mImageStampAdapter;
    private TextStampAdapter mDynamicStampAdapter;

    private TextStampBean mCurEditTextStampBean;
    private ISheetMenu mImageSheetMenu;
    private UITextEditDialog mPromptDialog;

    private final String mCustomStampCacheDir;
    private int mEditState = StampConstants.EDIT_NONE;
    private int mSelectedColor;
    private int mSpanCount = 2;
    private int mHorSpacing;
    private int mVerSpacing;
    private int mStampItemWidth;
    private int mStampItemHeight;
    private int mCurrentTab;
    private final int mProperbarWidth;
    private String mCaptureImgPath;
    private final List<RecyclerView> mStampRecyclerViews = new ArrayList<>();
    private final SparseArray<CommonStampAdapter> mStampAdapters = new SparseArray<>();
    private List<TextStampBean> mCustomTextStamps;
    private List<ImageStampBean> mCustomImageStamps;
    private UIMatchDialog mAddTextStampDialog;
    // end custom stamp text

    public StampToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
        mContext = context;
        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mCustomStampCacheDir = StampUtil.getImageCacheDir(mContext);
        mProperbarWidth = AppDisplay.getDialogWidth();

        initAnnotIconProvider();
    }

    protected void initAnnotIconProvider() {
        Library.setAnnotIconProviderCallback(DynamicStampIconProvider.getInstance(mContext));
    }

    protected PropertyBar getPropertyBar() {
        return mPropertyDlg;
    }

    private void initDisplayItems() {
        mStampItemWidth = AppResource.getDimensionPixelSize(mContext, R.dimen.stamp_normal_selected_width);
        mStampItemHeight = AppResource.getDimensionPixelSize(mContext, R.dimen.stamp_normal_selected_height);
        mHorSpacing = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_4dp);
        updateSpanCount();

        if (mStampsRootView == null) {
            mStampRecyclerViews.clear();
            if (mCustomTextStamps == null)
                mCustomTextStamps = StampUtil.getTextStamps(mContext);
            if (mCustomImageStamps == null)
                mCustomImageStamps = StampUtil.getImageStamps(mContext);


            Context context = mContext;
            if (Build.VERSION.SDK_INT <= 19) {
                Activity activity = ActManager.getInstance().getCurrentActivity();
                if (activity == null) {
                    activity = this.mUiExtensionsManager.getMainFrame().getAttachedActivity();
                }
               context = activity != null ? activity : mContext;
            }
            mStampsRootView = View.inflate(context, R.layout.fx_stamp_root_layout, null);
            initTitleBar(mStampsRootView);
            initTabView(mStampsRootView);
            initContentView(mStampsRootView);
        }

        if (mStampType > 21) {
            if (mCurSelectedCustomStamp instanceof TextStampBean) {
                mImageStampAdapter.setSelectedItem(null);
                mTextStampAdapter.setSelectedItem(mCurSelectedCustomStamp.mCreateTime);
            } else if (mCurSelectedCustomStamp instanceof ImageStampBean) {
                mTextStampAdapter.setSelectedItem(null);
                mImageStampAdapter.setSelectedItem(mCurSelectedCustomStamp.mCreateTime);
            }
            mViewPager.setCurrentItem(3);
        } else if (mStampType > 16) {
            mTextStampAdapter.setSelectedItem(null);
            mImageStampAdapter.setSelectedItem(null);
            mViewPager.setCurrentItem(2);
        } else if (mStampType > 11) {
            mTextStampAdapter.setSelectedItem(null);
            mImageStampAdapter.setSelectedItem(null);
            mViewPager.setCurrentItem(1);
        } else {
            mTextStampAdapter.setSelectedItem(null);
            mImageStampAdapter.setSelectedItem(null);
            mViewPager.setCurrentItem(0);
        }
        notifyUpdateAdapterData();
        updateCustomStampView();
    }

    private void initTitleBar(View rootView) {
        LinearLayout topLayout = rootView.findViewById(R.id.stamp_topbar_layout);
        TextView closeTv = topLayout.findViewById(R.id.stamp_close_tv);
        closeTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        closeTv.setOnClickListener(this);

        mTitleTv = topLayout.findViewById(R.id.stamp_title_tv);
        mEditTv = topLayout.findViewById(R.id.stamp_edit_tv);
        mEditTv.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mEditTv.setOnClickListener(this);
    }

    private void switchEditMode(boolean isEdit) {
        if (isEdit) {
            mEditState = StampConstants.EDIT_STATE;
            mEditTv.setText(AppResource.getString(mContext, R.string.fx_string_done));
            mDeleteStampsBottombar.setVisibility(View.VISIBLE);
            mAddStampsBottombar.setVisibility(View.GONE);
            updateDeleteBottombar();
        } else {
            mEditState = StampConstants.EDIT_NONE;
            mEditTv.setText(AppResource.getString(mContext, R.string.fx_string_edit));
            mDeleteStampsBottombar.setVisibility(View.INVISIBLE);
            mAddStampsBottombar.setVisibility(View.VISIBLE);
        }
        mTextStampAdapter.setEditState(isEdit);
        mImageStampAdapter.setEditState(isEdit);
    }

    private void updateDeleteBottombar() {
        mDeleteStampsIv.setEnabled(mTextStampAdapter.getSelectedItems().size() > 0 || mImageStampAdapter.getSelectedItems().size() > 0);
        if (mTextStampAdapter.isSelectAll() && mImageStampAdapter.isSelectAll())
            mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
        else
            mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));
    }

    private void updateCustomStampView() {
        if (mTextStampAdapter.isEmpty() && mImageStampAdapter.isEmpty()) {
            mEditTv.setEnabled(false);
            mEmptyStampsView.setVisibility(View.VISIBLE);
            mCustomStampsContentView.setVisibility(View.GONE);
            if (isEditMode()) {
                switchEditMode(false);
            }
        } else {
            mTextStampTitleView.setVisibility(mTextStampAdapter.isEmpty() ? View.GONE : View.VISIBLE);
            mTextStampsRV.setVisibility(mTextStampAdapter.isEmpty() ? View.GONE : View.VISIBLE);
            mImageStampTitleView.setVisibility(mImageStampAdapter.isEmpty() ? View.GONE : View.VISIBLE);
            mImageStampsRV.setVisibility(mImageStampAdapter.isEmpty() ? View.GONE : View.VISIBLE);

            mEditTv.setEnabled(true);
            mEmptyStampsView.setVisibility(View.GONE);
            mCustomStampsContentView.setVisibility(View.VISIBLE);
        }
    }

    private void initTabView(View rootView) {
        mTabLayout = rootView.findViewById(R.id.stamp_tabbar_layout);
        mTabLayout.setTabMode(TabLayout.MODE_FIXED);
        mTabLayout.setTabGravity(TabLayout.GRAVITY_FILL);
        mTabLayout.setSelectedTabIndicatorColor(ThemeConfig.getInstance(mContext).getPrimaryColor());

        mTabLayout.addOnTabSelectedListener(new TabLayout.OnTabSelectedListener() {
            @Override
            public void onTabSelected(TabLayout.Tab tab) {
                int position = tab.getPosition();
                if (mCurrentTab != position) {
                    mCurrentTab = position;
                    mViewPager.setCurrentItem(position);
                }
            }

            @Override
            public void onTabUnselected(TabLayout.Tab tab) {
            }

            @Override
            public void onTabReselected(TabLayout.Tab tab) {
            }
        });

        addTab(R.drawable.comment_tool_stamp_standard_tab, StampConstants.Standard_Stamps);
        addTab(R.drawable.comment_tool_stamp_sign_tab, StampConstants.SignHere_Stamps);
        addTab(R.drawable.comment_tool_stamp_dynamic_tab, StampConstants.Dynamic_Stamps);
        addTab(R.drawable.comment_tool_stamp_custom_tab, StampConstants.Custom_Stamps);
    }

    private void initContentView(View rootView) {
        mViewPager = rootView.findViewById(R.id.stamp_content_viewpager);
        ArrayList<View> views = new ArrayList<>();
        ViewPagerAdapter viewPagerAdapter = new ViewPagerAdapter(views);
        mViewPager.setAdapter(viewPagerAdapter);
        mViewPager.addOnPageChangeListener(new ViewPager.OnPageChangeListener() {
            @Override
            public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {
            }

            @Override
            public void onPageSelected(int position) {
                if (position == StampConstants.Standard_Stamps) {
                    mEditTv.setVisibility(View.INVISIBLE);
                    mTitleTv.setText(AppResource.getString(mContext, R.string.annot_stamp_standard));
                } else if (position == StampConstants.SignHere_Stamps) {
                    mEditTv.setVisibility(View.INVISIBLE);
                    mTitleTv.setText(AppResource.getString(mContext, R.string.annot_stamp_signhere));
                } else if (position == StampConstants.Dynamic_Stamps) {
                    mEditTv.setVisibility(View.INVISIBLE);
                    mTitleTv.setText(AppResource.getString(mContext, R.string.annot_stamp_dynamic));
                } else {
                    mEditTv.setVisibility(View.VISIBLE);
                    mTitleTv.setText(AppResource.getString(mContext, R.string.custom_stamp_tab_title));
                }

                if (isEditMode()) {
                    switchEditMode(false);
                }

                if (mCurrentTab != position) {
                    mCurrentTab = position;
                    TabLayout.Tab tab = mTabLayout.getTabAt(position);
                    if (tab != null)
                        tab.select();
                }
            }

            @Override
            public void onPageScrollStateChanged(int state) {
            }
        });

        views.add(getCommonStamps(StampConstants.Standard_Stamps, StampUtil.getStandardStampBeans()));
        views.add(getCommonStamps(StampConstants.SignHere_Stamps, StampUtil.getSignStampBeans()));
        views.add(getDynamicStamps());
        views.add(getCustomStamps());
        viewPagerAdapter.notifyDataSetChanged();
    }

    private View getCommonStamps(final int type, List<CommonStampAdapter.CommonStampItemBean> stamps) {
        View contentListView = View.inflate(mContext, R.layout.common_recyclerview_layout, null);
        contentListView.setPadding(0, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp), 0, 0);

        RecyclerView recyclerView = contentListView.findViewById(R.id.rd_recyclerview_list);
        recyclerView.setHasFixedSize(true);
        recyclerView.setItemAnimator(new DefaultItemAnimator());
        SpacesItemDecoration spacesItemDecoration = new SpacesItemDecoration();
        recyclerView.addItemDecoration(spacesItemDecoration);
        GridLayoutManager gridLayoutManager = new GridLayoutManager(mContext, mSpanCount);
        recyclerView.setLayoutManager(gridLayoutManager);

        CommonStampAdapter adapter = new CommonStampAdapter(mContext, stamps);
        adapter.setItemClickListener(new IStampItemClickListener<CommonStampAdapter.CommonStampItemBean>() {
            @Override
            public void onItemClick(boolean isEditMode, int position, CommonStampAdapter.CommonStampItemBean bean) {
                if (type == StampConstants.Standard_Stamps) {
                    mStampType = position;
                } else if (type == StampConstants.SignHere_Stamps) {
                    mStampType = position + 12;
                }
                onStampItemSelected(mStampType);
            }

            @Override
            public void onCheckedChanged(int position, List<CommonStampAdapter.CommonStampItemBean> selectedItems) {
            }
        });
        recyclerView.setAdapter(adapter);

        mStampRecyclerViews.add(recyclerView);
        mStampAdapters.put(type, adapter);
        return contentListView;
    }

    private View getDynamicStamps() {
        View contentListView = View.inflate(mContext, R.layout.common_recyclerview_layout, null);
        contentListView.setPadding(0, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp), 0, 0);
        RecyclerView recyclerView = contentListView.findViewById(R.id.rd_recyclerview_list);
        recyclerView.setHasFixedSize(true);
        recyclerView.setItemAnimator(new DefaultItemAnimator());
        SpacesItemDecoration spacesItemDecoration = new SpacesItemDecoration();
        recyclerView.addItemDecoration(spacesItemDecoration);
        GridLayoutManager gridLayoutManager = new GridLayoutManager(mContext, mSpanCount);
        recyclerView.setLayoutManager(gridLayoutManager);

        mDynamicStampAdapter = new TextStampAdapter(mContext, mPdfViewCtrl, StampUtil.getDynamicStampBeans());
        mDynamicStampAdapter.setItemClickListener(new IStampItemClickListener<TextStampBean>() {
            @Override
            public void onItemClick(boolean isEditMode, int position, TextStampBean itemBean) {
                mStampType = position + 17;
                onStampItemSelected(mStampType);
            }

            @Override
            public void onCheckedChanged(int position, List<TextStampBean> selectedItems) {
            }
        });
        recyclerView.setAdapter(mDynamicStampAdapter);
        mStampRecyclerViews.add(recyclerView);
        return contentListView;
    }

    private void notifyUpdateAdapterData() {
        for (int i = 0; i < mStampAdapters.size(); i++) {
            CommonStampAdapter stampAdapter = mStampAdapters.valueAt(i);
            if (stampAdapter != null) {
                int key = mStampAdapters.keyAt(i);
                if (StampConstants.Standard_Stamps == key) {
                    stampAdapter.setSelectedPosition(mStampType);
                } else if (StampConstants.SignHere_Stamps == key) {
                    stampAdapter.setSelectedPosition(mStampType - 12);
                }
            }
        }
        if (mDynamicStampAdapter != null)
            mDynamicStampAdapter.setSelectedItem(mStampType - 17);
    }

    private View getCustomStamps() {
        if (mCustomStampsRootView == null) {
            mCustomStampsRootView = View.inflate(mContext, R.layout.fx_stamp_custom_layout, null);

            mTextStampsRV = mCustomStampsRootView.findViewById(R.id.custom_text_stamps_listview);
            mTextStampsRV.setHasFixedSize(true);
            mTextStampsRV.setNestedScrollingEnabled(false);
            mTextStampsRV.setItemAnimator(new DefaultItemAnimator());
            SpacesItemDecoration textSpacesItem = new SpacesItemDecoration();
            mTextStampsRV.addItemDecoration(textSpacesItem);
            GridLayoutManager textLayoutManager = new GridLayoutManager(mContext, mSpanCount);
            mTextStampsRV.setLayoutManager(textLayoutManager);
            mTextStampAdapter = new TextStampAdapter(mContext, mPdfViewCtrl, mCustomTextStamps);
            mTextStampAdapter.setItemClickListener(new IStampItemClickListener<TextStampBean>() {
                @Override
                public void onItemClick(boolean isEditMode, int position, TextStampBean itemBean) {
                    if (isEditMode) {
                        showTextStampDialog(StampConstants.MODIFY_STAMP, itemBean);
                    } else {
                        drawTextStampThumbnail(itemBean);

                        mStampType = StampConstants.CUSTOM_TEXT_STAMP;
                        mCurSelectedCustomStamp = itemBean;
                        if (mCurToolItem != null)
                            mCurToolItem.property.mTag = itemBean.mCreateTime;
                        onStampItemSelected(mStampType);
                    }
                }

                @Override
                public void onCheckedChanged(int position, List<TextStampBean> selectedItems) {
                    updateDeleteBottombar();
                }
            });
            mTextStampsRV.setAdapter(mTextStampAdapter);
            mStampRecyclerViews.add(mTextStampsRV);

            mImageStampsRV = mCustomStampsRootView.findViewById(R.id.custom_image_stamps_listview);
            mImageStampsRV.setHasFixedSize(true);
            mImageStampsRV.setNestedScrollingEnabled(false);
            mImageStampsRV.setItemAnimator(new DefaultItemAnimator());
            SpacesItemDecoration imageSpacesItem = new SpacesItemDecoration();
            mImageStampsRV.addItemDecoration(imageSpacesItem);
            GridLayoutManager imageLayoutManager = new GridLayoutManager(mContext, mSpanCount);
            mImageStampsRV.setLayoutManager(imageLayoutManager);
            mImageStampAdapter = new ImageStampAdapter(mContext, mPdfViewCtrl, mCustomImageStamps);
            mImageStampAdapter.setGridLayoutManager(imageLayoutManager);
            mImageStampAdapter.setItemClickListener(new IStampItemClickListener<ImageStampBean>() {
                @Override
                public void onItemClick(boolean isEditMode, int position, ImageStampBean itemBean) {
                    if (isEditMode) {
                        showImageStampDialog(StampConstants.MODIFY_STAMP, itemBean);
                    } else {
                        mCurSelectedCustomStamp = itemBean;
                        mStampType = StampConstants.CUSTOM_IMAGE_STAMP;
                        if (mCurToolItem != null)
                            mCurToolItem.property.mTag = itemBean.mCreateTime;
                        onStampItemSelected(mStampType);
                    }
                }

                @Override
                public void onCheckedChanged(int position, List<ImageStampBean> selectedItems) {
                    updateDeleteBottombar();
                }
            });
            mImageStampsRV.setAdapter(mImageStampAdapter);
            mStampRecyclerViews.add(mImageStampsRV);

            mCustomStampsContentView = mCustomStampsRootView.findViewById(R.id.custom_stamps_scrollview);
            mTextStampTitleView = mCustomStampsContentView.findViewById(R.id.custom_text_stamps_title);
            mImageStampTitleView = mCustomStampsContentView.findViewById(R.id.custom_image_stamps_title);
            mEmptyStampsView = mCustomStampsRootView.findViewById(R.id.custom_stamp_no_content_ll);
            ImageView emptyStampsIv = mCustomStampsRootView.findViewById(R.id.custom_stamp_no_content_iv);
            emptyStampsIv.setColorFilter(ThemeConfig.getInstance(mContext).getPrimaryColor());

            mAddStampsBottombar = mCustomStampsRootView.findViewById(R.id.custom_add_stamps_bottom_view);
            ImageView addTextStampsIv = mCustomStampsRootView.findViewById(R.id.add_custom_stamps_from_text);
            ThemeUtil.setTintList(addTextStampsIv, ThemeUtil.getPrimaryIconColor(mContext));
            mAddImageStampsIv = mCustomStampsRootView.findViewById(R.id.add_custom_stamps_from_image);
            ThemeUtil.setTintList(mAddImageStampsIv, ThemeUtil.getPrimaryIconColor(mContext));
            mAddImageStampsIv.setOnClickListener(this);
            addTextStampsIv.setOnClickListener(this);

            mDeleteStampsBottombar = mCustomStampsRootView.findViewById(R.id.custom_edit_stamps_bottom_view);
            mSelectedAllTv = mCustomStampsRootView.findViewById(R.id.custom_select_all_stamps);
            mDeleteStampsIv = mCustomStampsRootView.findViewById(R.id.custom_stamps_delete_view);
            ThemeUtil.setTintList(mDeleteStampsIv, ThemeUtil.getPrimaryIconColor(mContext));
            mSelectedAllTv.setOnClickListener(this);
            mDeleteStampsIv.setOnClickListener(this);
        }
        return mCustomStampsRootView;
    }

    @Override
    public void onClick(View v) {
        int id = v.getId();
        if (id == R.id.stamp_edit_tv) { // edit custom stamp
            switchEditMode(!isEditMode());
        } else if (id == R.id.stamp_close_tv) { // close stamp property dialog
            if (isEditMode()) {
                switchEditMode(false);
            }
            if (mPropertyDlg != null) {
                mPropertyDlg.dismiss();
            }
        } else if (id == R.id.add_custom_stamps_from_text) { // add text stamp
            TextStampBean stampBean = new TextStampBean();
            stampBean.mColor = StampConstants.Custom_Colors[0];
            stampBean.mText = "";
            showTextStampDialog(StampConstants.ADD_STAMP, stampBean);
        } else if (id == R.id.add_custom_stamps_from_image) { // add image stamp
            if (mImageSheetMenu == null) {
                mImageSheetMenu = UISheetMenu.newInstance((FragmentActivity) mUiExtensionsManager.getAttachedActivity());
                if (AppDisplay.isPad())
                    mImageSheetMenu.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_pad_more_menu_width));
                mImageSheetMenu.addSheetItem(ISheetMenu.CAMERA_MENU);
                mImageSheetMenu.addSheetItem(ISheetMenu.GALLERY_MENU);
            }

            mImageSheetMenu.setSheetItemClickListener(new ISheetMenu.OnSheetItemClickListener() {
                @Override
                public void onItemClick(int type) {
                    mImageSheetMenu.dismiss();

                    if (type == ISheetMenu.CAMERA_MENU) {
                        mCaptureImgPath = mCustomStampCacheDir + "camera/" + System.currentTimeMillis();

                        AppIntentUtil.selectImageFromCamera(mUiExtensionsManager,
                                mCaptureImgPath, ActRequestCode.REQ_CAMERA_PERMISSION, ActRequestCode.REQ_FILE_FROM_CAMERA);
                    } else if (type == ISheetMenu.GALLERY_MENU) {
                        AppIntentUtil.selectImageFromGallery(mUiExtensionsManager.getAttachedActivity(), ActRequestCode.REQ_FILE_FROM_GALLERY);
                    }
                }
            });
            showImageMenu(v);
        } else if (id == R.id.custom_text_stamp_clear_iv) { // clear custom stamp text
            mCustomTextEdit.setText("");
        } else if (id == R.id.custom_text_stamp_author_select_layout) { // show author switch
            setSelectedButtonState(mAuthorCheckLayout, mAuthorCheckIv, !mAuthorCheckLayout.isSelected());
            mThumbnailDateTv.setText(getCustomDateString(mCurEditTextStampBean));
            mAddTextStampDialog.setRightButtonEnable(!AppUtil.isEmpty(mCustomTextEdit.getText()));
            requestThumbnailTextLayout();
        } else if (id == R.id.custom_text_stamp_date_select_layout) { // show date switch
            setSelectedButtonState(mDateCheckLayout, mDateCheckIv, !mDateCheckLayout.isSelected());
            mThumbnailDateTv.setText(getCustomDateString(mCurEditTextStampBean));
            mAddTextStampDialog.setRightButtonEnable(!AppUtil.isEmpty(mCustomTextEdit.getText()));
            requestThumbnailTextLayout();
        } else if (id == R.id.custom_text_stamp_time_select_layout) { // show time switch
            setSelectedButtonState(mTimeCheckLayout, mTimeCheckIv, !mTimeCheckLayout.isSelected());
            mThumbnailDateTv.setText(getCustomDateString(mCurEditTextStampBean));
            mAddTextStampDialog.setRightButtonEnable(!AppUtil.isEmpty(mCustomTextEdit.getText()));
            requestThumbnailTextLayout();
        } else if (id == R.id.custom_select_all_stamps) { //select all stamps
            if (mTextStampAdapter.isSelectAll() && mImageStampAdapter.isSelectAll()) {
                mDeleteStampsIv.setEnabled(false);
                mTextStampAdapter.selectAll(false);
                mImageStampAdapter.selectAll(false);
                mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_select_all));

            } else {
                mDeleteStampsIv.setEnabled(true);
                mTextStampAdapter.selectAll(true);
                mImageStampAdapter.selectAll(true);
                mSelectedAllTv.setText(AppResource.getString(mContext, R.string.fx_string_deselect_all));
            }
        } else if (id == R.id.custom_stamps_delete_view) {//delete selected stamps
            if (mTextStampAdapter.isSelectAll() && mImageStampAdapter.isSelectAll()) {
                if (mPromptDialog == null) {
                    final Context context = mUiExtensionsManager.getAttachedActivity();
                    String title = AppResource.getString(mContext, R.string.menu_more_confirm);
                    String prompt = AppResource.getString(mContext, R.string.clear_all_custom_stamps);
                    mPromptDialog = new UITextEditDialog(context, UIDialog.NO_INPUT);
                    mPromptDialog.setTitle(title);
                    mPromptDialog.getPromptTextView().setText(prompt);
                }
                mPromptDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        removeCustomStampsFromHomeBar(true);
                        mTextStampAdapter.removeSelectedItems();
                        mImageStampAdapter.removeSelectedItems();
                        if (mCurSelectedCustomStamp != null) {
                            mCurSelectedCustomStamp = null;
                            mStampType = 0;
                            onStampItemSelected(mStampType, false);
                        }
                        updateCustomStampView();
                        updateDeleteBottombar();

                        mPromptDialog.dismiss();
                    }
                });
                mPromptDialog.show();

            } else {
                removeCustomStampsFromHomeBar(false);
                mTextStampAdapter.removeSelectedItems();
                mImageStampAdapter.removeSelectedItems();
                if (mCurSelectedCustomStamp != null) {
                    mCurSelectedCustomStamp = null;
                    mStampType = 0;
                    onStampItemSelected(mStampType, false);
                }
                updateCustomStampView();
                updateDeleteBottombar();
            }
        }
    }

    private void removeCustomStampsFromHomeBar(boolean clearAll) {
        List<BaseStampBean> selectedItems = new ArrayList<>();
        selectedItems.addAll(mTextStampAdapter.getSelectedItems());
        selectedItems.addAll(mImageStampAdapter.getSelectedItems());
        HomeToolBar homeBar = (HomeToolBar) mUiExtensionsManager.getMainFrame().getToolbar(ToolbarItemConfig.ITEM_HOME_TAB);
        if (homeBar != null)
            homeBar.removeCustomStamps(clearAll, selectedItems);
    }

    private void requestThumbnailTextLayout() {
        if (isShowCustomDateView()) {
            mThumbnailDateTv.setVisibility(View.VISIBLE);

            mThumbnailCustomTextTv.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_14sp));
            TextViewCompat.setAutoSizeTextTypeUniformWithConfiguration(mThumbnailCustomTextTv, 1,
                    14, 1, TypedValue.COMPLEX_UNIT_SP);
            LinearLayout.LayoutParams lp = (LinearLayout.LayoutParams) mThumbnailCustomTextTv.getLayoutParams();
            lp.height = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_20dp);
            mThumbnailCustomTextTv.setLayoutParams(lp);
        } else {
            mThumbnailDateTv.setVisibility(View.GONE);

            mThumbnailCustomTextTv.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_30sp));
            TextViewCompat.setAutoSizeTextTypeUniformWithConfiguration(mThumbnailCustomTextTv, 1,
                    30, 1, TypedValue.COMPLEX_UNIT_SP);
            LinearLayout.LayoutParams lp = (LinearLayout.LayoutParams) mThumbnailCustomTextTv.getLayoutParams();
            lp.height = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_40dp);
            mThumbnailCustomTextTv.setLayoutParams(lp);
        }
    }

    private boolean isShowCustomDateView() {
        return mAuthorCheckLayout.isSelected() || mDateCheckLayout.isSelected() || mTimeCheckLayout.isSelected();
    }

    private String getCustomDateString(TextStampBean stampBean) {
        StringBuilder sb = new StringBuilder();
        if (mAuthorCheckLayout.isSelected()) {
            sb.append(mUiExtensionsManager.getAnnotAuthor());
        }
        if (mTimeCheckLayout.isSelected()) {
            if (stampBean.mCreateTime == 0) {
                stampBean.mCreateTime = new Date().getTime();
                stampBean.mModifiedTime = new Date().getTime();
            }
            if (mAuthorCheckLayout.isSelected())
                sb.append(",");
            sb.append(StampUtil.formatTime(stampBean.mCreateTime));
        }
        if (mDateCheckLayout.isSelected()) {
            if (stampBean.mCreateTime == 0) {
                stampBean.mCreateTime = new Date().getTime();
                stampBean.mModifiedTime = new Date().getTime();
            }
            if (mAuthorCheckLayout.isSelected() || mTimeCheckLayout.isSelected())
                sb.append(",");
            sb.append(StampUtil.formatDate(stampBean.mCreateTime));
        }
        return sb.toString();
    }

    private class SpacesItemDecoration extends RecyclerView.ItemDecoration {
        @Override
        public void getItemOffsets(@NonNull Rect outRect, @NonNull View view,
                                   @NonNull RecyclerView parent, @NonNull RecyclerView.State state) {
            if (mSpanCount > 0) {
                int position = parent.getChildAdapterPosition(view);
                int spanIndex = position % mSpanCount;

                outRect.left = mVerSpacing - spanIndex * mVerSpacing / mSpanCount;
                outRect.right = (spanIndex + 1) * mVerSpacing / mSpanCount;

                outRect.top = mHorSpacing;
                outRect.bottom = mHorSpacing;
            } else {
                outRect.setEmpty();
            }
        }
    }

    private void addTab(int icon, int type) {
        // icon view
        ImageView iconView = new ImageView(mContext);
        iconView.setId(R.id.rd_panel_tab_item);
        iconView.setImageResource(icon);
        ColorStateList colorStateList = ThemeUtil.getPrimaryIconColor(mContext);
        ThemeUtil.setTintList(iconView, colorStateList);

        RelativeLayout.LayoutParams iconLayoutParams = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.WRAP_CONTENT);
        iconLayoutParams.addRule(RelativeLayout.CENTER_IN_PARENT, RelativeLayout.TRUE);

        RelativeLayout tabItemView = new RelativeLayout(mContext);
        tabItemView.addView(iconView, iconLayoutParams);

        TabLayout.Tab tab = mTabLayout.newTab();
        tab.setTag(type);
        tab.setCustomView(tabItemView);
        mTabLayout.addTab(tab);
    }

    private View getTextStampView(int type, final TextStampBean stampBean) {
        mCurEditTextStampBean = stampBean;
        mSelectedColor = stampBean.mColor;

        if (mAddTextStampRootView == null) {
            mAddTextStampRootView = View.inflate(mUiExtensionsManager.getAttachedActivity(), R.layout.fx_stamp_custom_text_layout, null);
            mClearCustomTextIv = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_clear_iv);
            mClearCustomTextIv.setVisibility(View.GONE);
            mClearCustomTextIv.setOnClickListener(this);

            mCustomTextEdit = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_input_edit);
            if (AppDisplay.isPad()) { // SDKRD-9313
                mCustomTextEdit.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            }
            mCustomTextEdit.addTextChangedListener(new TextWatcher() {
                @Override
                public void beforeTextChanged(CharSequence s, int start, int count, int after) {
                }

                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {
                    mClearCustomTextIv.setVisibility(AppUtil.isEmpty(s) ? View.GONE : View.VISIBLE);
                }

                @Override
                public void afterTextChanged(Editable s) {
                    if (AppUtil.isEmpty(s)) {
                        mThumbnailCustomTextTv.setText(AppResource.getString(mContext, R.string.add_custom_text_stamp_prompt));
                        if (mAddTextStampDialog != null)
                            mAddTextStampDialog.setRightButtonEnable(false);
                    } else {
                        if (mAddTextStampDialog != null)
                            mAddTextStampDialog.setRightButtonEnable(true);
                        mThumbnailCustomTextTv.setText(s);
                    }
                }
            });
            mCustomTextEdit.setOnFocusChangeListener(new View.OnFocusChangeListener() {
                @Override
                public void onFocusChange(View v, boolean hasFocus) {
                    if (hasFocus) {
                        mClearCustomTextIv.setVisibility(AppUtil.isEmpty(mCustomTextEdit.getText()) ? View.GONE : View.VISIBLE);
                    } else {
                        mClearCustomTextIv.setVisibility(View.GONE);
                    }
                }
            });
            mCustomTextEdit.setFilters(new InputFilter[]{new InputFilter() {

                @Override
                public CharSequence filter(CharSequence source, int start, int end, Spanned dest, int dstart, int dend) {
                    for (int index = start; index < end; index++) {

                        char charAt = source.charAt(index);
                        if (StampUtil.isEmojiCharacter(charAt)) {
                            return "";
                        }
                    }
                    return null;
                }
            }, new InputFilter.LengthFilter(37)});

            mAuthorCheckLayout = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_author_select_layout);
            mAuthorCheckIv = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_author_select_iv);
            mDateCheckLayout = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_date_select_layout);
            mDateCheckIv = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_date_select_iv);
            mTimeCheckLayout = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_time_select_layout);
            mTimeCheckIv = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_date_time_iv);
            ThemeUtil.setBackgroundTintList(mAuthorCheckLayout, getSelectedButtonColorStateList());
            ThemeUtil.setBackgroundTintList(mDateCheckLayout, getSelectedButtonColorStateList());
            ThemeUtil.setBackgroundTintList(mTimeCheckLayout, getSelectedButtonColorStateList());
            mAuthorCheckLayout.setOnClickListener(this);
            mDateCheckLayout.setOnClickListener(this);
            mTimeCheckLayout.setOnClickListener(this);

            mThumbnailCustomTextTv = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_thumbnail_author);
            mThumbnailCustomTextTv.setText(AppResource.getString(mContext, R.string.add_custom_text_stamp_prompt));
            TextViewCompat.setAutoSizeTextTypeUniformWithConfiguration(mThumbnailCustomTextTv, 1,
                    14, 1, TypedValue.COMPLEX_UNIT_SP);
            mThumbnailDateTv = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_thumbnail_date);
            TextViewCompat.setAutoSizeTextTypeUniformWithConfiguration(mThumbnailDateTv, 1,
                    8, 1, TypedValue.COMPLEX_UNIT_SP);
            mThumbnailColorView = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_thumbnail_color);

            mCustomColorView = mAddTextStampRootView.findViewById(R.id.custom_text_stamp_color_view);
            for (int i = 0; i < StampConstants.Custom_Colors.length; i++) {
                int color = StampConstants.Custom_Colors[i];
                final ImageView colorItem = new ImageView(mContext);
                colorItem.setImageResource(StampConstants.Custom_Colors_ResIds[i]);
                int padding = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_2dp);
                colorItem.setPadding(padding, padding, padding, padding);
                colorItem.setBackgroundResource(R.drawable.custom_stamp_oval_border_bg);
                colorItem.setTag(color);

                colorItem.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        for (int i = 0; i < mCustomColorView.getChildCount(); i++) {
                            View childView = mCustomColorView.getChildAt(i);
                            if (childView == colorItem) {
                                mSelectedColor = StampConstants.Custom_Colors[i];
                                childView.getBackground().setColorFilter(new PorterDuffColorFilter(
                                        ThemeConfig.getInstance(mContext).getPrimaryColor(), PorterDuff.Mode.SRC_IN));
                                mThumbnailColorView.getBackground().setColorFilter(
                                        new PorterDuffColorFilter(StampConstants.Custom_Colors[i], PorterDuff.Mode.SRC_IN));
                            } else {
                                childView.getBackground().setColorFilter(new PorterDuffColorFilter(
                                        AppResource.getColor(mContext, R.color.ux_color_translucent), PorterDuff.Mode.SRC_IN));
                            }
                        }

                        if (mSelectedColor != mCurEditTextStampBean.mColor) {
                            mAddTextStampDialog.setRightButtonEnable(!AppUtil.isEmpty(mCustomTextEdit.getText()));
                        }
                    }
                });

                LinearLayout.LayoutParams lp = new LinearLayout.LayoutParams(
                        LinearLayout.LayoutParams.WRAP_CONTENT, LinearLayout.LayoutParams.WRAP_CONTENT);
                if (i < StampConstants.Custom_Colors.length - 1)
                    lp.rightMargin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp);
                mCustomColorView.addView(colorItem, lp);
            }
        }

        if (type == StampConstants.ADD_STAMP) {
            mCustomTextEdit.setText("");
            mThumbnailCustomTextTv.setText(AppResource.getString(mContext, R.string.add_custom_text_stamp_prompt));
        } else {
            mCustomTextEdit.setText(stampBean.mText);
            mCustomTextEdit.setSelection(mCustomTextEdit.length());
            mThumbnailCustomTextTv.setText(stampBean.mText);
        }
        setSelectedButtonState(mAuthorCheckLayout, mAuthorCheckIv, stampBean.mShowAuthor);
        setSelectedButtonState(mDateCheckLayout, mDateCheckIv, stampBean.mShowDate);
        setSelectedButtonState(mTimeCheckLayout, mTimeCheckIv, stampBean.mShowTime);

        mThumbnailDateTv.setText(getCustomDateString(stampBean));
        requestThumbnailTextLayout();
        mThumbnailColorView.getBackground().setColorFilter(
                new PorterDuffColorFilter(stampBean.mColor, PorterDuff.Mode.SRC_IN));
        for (int i = 0; i < mCustomColorView.getChildCount(); i++) {
            View childView = mCustomColorView.getChildAt(i);
            if ((Integer) childView.getTag() == stampBean.mColor) {
                childView.getBackground().setColorFilter(new PorterDuffColorFilter(
                        ThemeConfig.getInstance(mContext).getPrimaryColor(), PorterDuff.Mode.SRC_IN));
                mThumbnailColorView.getBackground().setColorFilter(
                        new PorterDuffColorFilter(stampBean.mColor, PorterDuff.Mode.SRC_IN));
            } else {
                childView.getBackground().setColorFilter(new PorterDuffColorFilter(
                        AppResource.getColor(mContext, R.color.ux_color_translucent), PorterDuff.Mode.SRC_IN));
            }
        }

        return mAddTextStampRootView;
    }

    private void showTextStampDialog(int type, final TextStampBean stampBean) {
        View contentView = getTextStampView(type, stampBean);
        if (mAddTextStampDialog == null) {
            mAddTextStampDialog = new UIMatchDialog(mUiExtensionsManager.getAttachedActivity());
            mAddTextStampDialog.setContentView(contentView);
            mAddTextStampDialog.setBackButtonStyle(MatchDialog.TEXT_BACK);
            mAddTextStampDialog.setRightButtonVisible(View.VISIBLE);
            mAddTextStampDialog.setTitle(AppResource.getString(mContext, R.string.add_custom_text_stamp_title));
            mAddTextStampDialog.setTitlePosition(BaseBar.TB_Position.Position_CENTER);
            mAddTextStampDialog.setStyle(MatchDialog.DLG_TITLE_STYLE_BG_DEFAULT);
        }
        int titleId = isEditMode() ? R.string.edit_custom_text_stamp_title : R.string.add_custom_text_stamp_title;
        mAddTextStampDialog.setTitle(AppResource.getString(mContext, titleId));
        mAddTextStampDialog.setRightButtonEnable(false);
        mAddTextStampDialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
            }

            @Override
            public void onTitleRightButtonClick() {
                stampBean.mShowAuthor = mAuthorCheckLayout.isSelected();
                stampBean.mShowDate = mDateCheckLayout.isSelected();
                stampBean.mShowTime = mTimeCheckLayout.isSelected();
                stampBean.mText = mThumbnailCustomTextTv.getText().toString();
                stampBean.mColor = mSelectedColor;
                stampBean.mModifiedTime = System.currentTimeMillis();
                mTextStampAdapter.addTextItem(stampBean);

                updateCustomStampView();
                mAddTextStampDialog.dismiss();
            }
        });
        mAddTextStampDialog.resetWH();
        mAddTextStampDialog.showDialog();
    }

    private void showImageStampDialog(String path) {
        ImageStampBean stampBean = new ImageStampBean();
        stampBean.mCacheFile = path;
        stampBean.mOpacity = 100;
        stampBean.mCreateTime = System.currentTimeMillis();
        stampBean.mModifiedTime = System.currentTimeMillis();
        showImageStampDialog(StampConstants.ADD_STAMP, stampBean);
    }

    private void showImageStampDialog(int type, ImageStampBean imageStampBean) { // type 0 -- add image stamp, type 1 -- modify stamp
        FragmentActivity act = (FragmentActivity) mUiExtensionsManager.getAttachedActivity();
        FragmentManager fragmentManager = act.getSupportFragmentManager();
        ImageOpacityFragment opacityFragment = (ImageOpacityFragment) fragmentManager.findFragmentByTag(ImageOpacityFragment.TAG);
        if (opacityFragment == null)
            opacityFragment = new ImageOpacityFragment();
        opacityFragment.init(type, mPdfViewCtrl, imageStampBean);
        opacityFragment.setPositiveItemClickCallback(new ImageOpacityFragment.IPositiveItemClickCallback() {
            @Override
            public void onClick(View v, ImageStampBean stampBean) {
                mImageStampAdapter.addImageItem(stampBean);
                if (!mTextStampAdapter.isEmpty())
                    mTextStampAdapter.notifyUpdateData();
                updateCustomStampView();
            }
        });
        AppDialogManager.getInstance().showAllowManager(opacityFragment, fragmentManager, ImageOpacityFragment.TAG, null);
    }

    private void resetPropertyBar(View propertyView) {
        if (mTextStampAdapter.isEditSate()) {
            mEditState = StampConstants.EDIT_STATE;
        } else {
            mEditState = StampConstants.EDIT_NONE;
        }

        if (mPropertyDlg == null) {
            mPropertyDlg = new StampPropertyDialog(mUiExtensionsManager.getAttachedActivity());
            mPropertyDlg.setTitleVisible(false);
        }
        mPropertyDlg.setContentView(propertyView);
        mPropertyDlg.setOnKeyListener(new DialogInterface.OnKeyListener() {
            @Override
            public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                if (keyCode == KeyEvent.KEYCODE_BACK && event.getAction() == KeyEvent.ACTION_DOWN) {
                    if (mEditState == StampConstants.EDIT_STATE) {
                        switchEditMode(false);
                        return true;
                    }
                }
                return false;
            }
        });

        mPropertyDlg.setOnShowListener(new DialogInterface.OnShowListener() {
            @Override
            public void onShow(DialogInterface dialog) {
                AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                    @Override
                    public void run() {
                        mTextStampAdapter.notifyUpdateData();
                        mImageStampAdapter.notifyUpdateData();
                    }
                });
            }
        });

        mPropertyDlg.setOnDLDismissListener(new MatchDialog.DismissListener() {
            @Override
            public void onDismiss() {
                if (isEditMode()) {
                    switchEditMode(false);
                }
            }
        });
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_STAMP;
    }

    @Override
    public void onActivate() {
        initDisplayItems();
        resetPropertyBar(mStampsRootView);
    }

    @Override
    public void onDeactivate() {
        if (isEditMode()) {
            switchEditMode(false);
        }

        if (mPropertyDlg != null && mPropertyDlg.isShowing()) {
            mPropertyDlg.dismiss();
        }
    }

    private RectF getStampRectOnPageView(PointF point, int pageIndex) {
        PointF pageViewPt = new PointF(point.x, point.y);
        float offsetX = 49.5f;
        float offsetY = 15.5f;
        if (mCurSelectedCustomStamp instanceof ImageStampBean) {
            ImageStampBean stampBean = (ImageStampBean) mCurSelectedCustomStamp;

            float image_half_width = stampBean.mWidth / 2.0f;
            float image_half_height = stampBean.mHeight / 2.0f;
            if (image_half_width < offsetX) {
                offsetX = stampBean.mWidth / 2.0f;
                offsetY =stampBean.mHeight / 2.0f;
            } else {
                float scaleW = offsetX / image_half_width;
                offsetY = image_half_height * scaleW;
            }
        }
        offsetX = thicknessOnPageView(pageIndex, offsetX);
        offsetY = thicknessOnPageView(pageIndex, offsetY);

        RectF pageViewRect = new RectF(pageViewPt.x - offsetX, pageViewPt.y - offsetY, pageViewPt.x + offsetX, pageViewPt.y + offsetY);
        if (pageViewRect.left < 0) {
            pageViewRect.offset(-pageViewRect.left, 0);
        }
        if (pageViewRect.right > mPdfViewCtrl.getPageViewWidth(pageIndex)) {
            pageViewRect.offset(mPdfViewCtrl.getPageViewWidth(pageIndex) - pageViewRect.right, 0);
        }
        if (pageViewRect.top < 0) {
            pageViewRect.offset(0, -pageViewRect.top);
        }
        if (pageViewRect.bottom > mPdfViewCtrl.getPageViewHeight(pageIndex)) {
            pageViewRect.offset(0, mPdfViewCtrl.getPageViewHeight(pageIndex) - pageViewRect.bottom);
        }
        return pageViewRect;
    }

    private RectF mLastStampRect = new RectF(0, 0, 0, 0);
    private RectF mStampRect = new RectF(0, 0, 0, 0);
    private boolean mTouchCaptured = false;
    private int mLastPageIndex = -1;

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent e) {
        boolean handled = mUiExtensionsManager.defaultTouchEvent(pageIndex, e);
        if (mTouchCaptured) {
            handled = onStampToolTouch(pageIndex, e);
        }
        return handled;
    }

    private boolean onStampToolTouch(int pageIndex, MotionEvent e) {
        PointF point = new PointF(e.getX(), e.getY());
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
        int action = e.getAction();
        mStampRect = getStampRectOnPageView(point, pageIndex);
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (!mTouchCaptured && mLastPageIndex == -1 || mLastPageIndex == pageIndex) {
                    mTouchCaptured = true;
                    mLastStampRect = new RectF(mStampRect);
                    if (mLastPageIndex == -1) {
                        mLastPageIndex = pageIndex;
                    }
                }
                return true;
            case MotionEvent.ACTION_MOVE:
                if (!mTouchCaptured || mLastPageIndex != pageIndex)
                    break;
                onStampMove(pageIndex);
                return true;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (!mIsContinuousCreate) {
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
                RectF pdfRect = new RectF();
                mPdfViewCtrl.convertPageViewRectToPdfRect(mStampRect, pdfRect, pageIndex);
                createAnnot(pdfRect, pageIndex);
                return true;
            default:
        }
        return true;
    }

    private void onStampMove(int pageIndex) {
        RectF rect = new RectF(mLastStampRect);
        rect.union(mStampRect);
        rect.inset(-10, -10);
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rect, rect, pageIndex);
        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rect));
        mLastStampRect = new RectF(mStampRect);
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
        onStampToolTouch(pageIndex, motionEvent);
        onStampMove(pageIndex);
        return true;
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!handled) {
            onStampToolTouch(pageIndex, motionEvent);
            MotionEvent upEvent = MotionEvent.obtain(motionEvent);
            upEvent.setAction(MotionEvent.ACTION_UP);
            onStampToolTouch(pageIndex, upEvent);
            upEvent.recycle();
        }
        return true;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinuousCreate;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinuousCreate = continueAddAnnot;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (!mTouchCaptured || pageIndex != mLastPageIndex)
            return;
        Paint paint = new Paint();
        paint.setAlpha(100);

        Bitmap bitmap = null;
        if (mStampType == StampConstants.CUSTOM_TEXT_STAMP ||
                mStampType == StampConstants.CUSTOM_IMAGE_STAMP) {
            if (mCurSelectedCustomStamp != null)
                bitmap = mCurSelectedCustomStamp.mThumbnailBitmap;
        } else {
            bitmap = BitmapFactory.decodeResource(mContext.getResources(), StampUtil.getStampIconByType(mStampType));
        }

        if (bitmap == null || mStampRect == null)
            return;
        canvas.drawBitmap(bitmap, null, mStampRect, paint);
    }

    private RectF mPageViewThickness = new RectF(0, 0, 0, 0);

    private float thicknessOnPageView(int pageIndex, float thickness) {
        mPageViewThickness.set(0, 0, thickness, thickness);
        mPdfViewCtrl.convertPdfRectToPageViewRect(mPageViewThickness, mPageViewThickness, pageIndex);
        return Math.abs(mPageViewThickness.width());
    }

    private float thicknessOnDoc(int pageIndex, float thickness) {
        mPageViewThickness.set(0, 0, thickness, thickness);
        mPdfViewCtrl.convertPageViewRectToPdfRect(mPageViewThickness, mPageViewThickness, pageIndex);
        return Math.abs(mPageViewThickness.width());
    }

    public void AI_addStamp(int type, PointF point, final int pageIndex) {
        if (mPdfViewCtrl.isPageVisible(pageIndex)) {

            try {
                RectF rectF = getStampRectOnPageView(point, pageIndex);
                mPdfViewCtrl.convertPageViewRectToPdfRect(rectF, rectF, pageIndex);
                final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
                final Stamp annot = (Stamp) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Stamp, AppUtil.toFxRectF(rectF)), Annot.e_Stamp);

                final StampAddUndoItem undoItem = new StampAddUndoItem(mPdfViewCtrl);
                undoItem.mPageIndex = pageIndex;
                undoItem.mStampType = type;
                undoItem.mNM = AppDmUtil.randomUUID(null);
                undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
                undoItem.mFlags = Annot.e_FlagPrint;
                undoItem.mSubject = StampUtil.getStampNameByType(type);
                undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mBBox = new RectF(rectF);
                int rotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
                undoItem.mRotation = (rotation == 0 ? rotation : 4 - rotation) * 90;
                undoItem.mIconName = undoItem.mSubject;

                StampEvent event = new StampEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, annot, mPdfViewCtrl);
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (mUiExtensionsManager.getCurrentToolHandler() != StampToolHandler.this) {
                            mCurSelectedCustomStamp = null;
                        }

                        if (success) {
                            mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, annot);
                            mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                try {
                                    RectF viewRect = AppUtil.toRectF(annot.getRect());
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                                    Rect rect = new Rect();
                                    viewRect.roundOut(rect);
                                    viewRect.union(mLastStampRect);
                                    rect.inset(-10, -10);
                                    mPdfViewCtrl.refresh(pageIndex, rect);
                                    mLastStampRect.setEmpty();
                                } catch (PDFException e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                        mTouchCaptured = false;
                        mLastPageIndex = -1;
                    }
                });
                mPdfViewCtrl.addTask(task);
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }
    private void createAnnot(final RectF rectF, final int pageIndex) {
        if (mPdfViewCtrl.isPageVisible(pageIndex)) {

            try {
                final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
                final Stamp annot = (Stamp) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Stamp, AppUtil.toFxRectF(rectF)), Annot.e_Stamp);

                final StampAddUndoItem undoItem = new StampAddUndoItem(mPdfViewCtrl);
                undoItem.mPageIndex = pageIndex;
                undoItem.mStampType = mStampType;
//                undoItem.mDsip = mDsip;
                undoItem.mNM = AppDmUtil.randomUUID(null);
                undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
                undoItem.mFlags = Annot.e_FlagPrint;
                undoItem.mSubject = StampUtil.getStampNameByType(mStampType);
                undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mBBox = new RectF(rectF);
                int rotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
                undoItem.mRotation = (rotation == 0 ? rotation : 4 - rotation) * 90;
                if (mStampType == StampConstants.CUSTOM_TEXT_STAMP || mStampType == StampConstants.CUSTOM_IMAGE_STAMP) {
                    if (mCurSelectedCustomStamp.mModifiedTime == 0) {
                        mCurSelectedCustomStamp.mModifiedTime = mCurSelectedCustomStamp.mCreateTime;
                    }
                    String iconName = String.valueOf(mCurSelectedCustomStamp.mModifiedTime);
                    undoItem.mIconName = iconName;
                    DynamicStampIconProvider.getInstance(mContext).addCustomStamp(iconName, mCurSelectedCustomStamp.cloneBean());
                } else {
                    undoItem.mIconName = undoItem.mSubject;
                }

                StampEvent event = new StampEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, annot, mPdfViewCtrl);
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (mUiExtensionsManager.getCurrentToolHandler() != StampToolHandler.this) {
                            mCurSelectedCustomStamp = null;
                        }

                        if (success) {
                            mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, annot);
                            mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                try {
                                    RectF viewRect = AppUtil.toRectF(annot.getRect());
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                                    Rect rect = new Rect();
                                    viewRect.roundOut(rect);
                                    viewRect.union(mLastStampRect);
                                    rect.inset(-10, -10);
                                    mPdfViewCtrl.refresh(pageIndex, rect);
                                    mLastStampRect.setEmpty();
                                } catch (PDFException e) {
                                    e.printStackTrace();
                                }

                            }
                        }

                        mTouchCaptured = false;
                        mLastPageIndex = -1;
                    }
                });
                mPdfViewCtrl.addTask(task);
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    protected void addAnnot(final int pageIndex, AnnotContent content, final boolean addUndo, final Event.Callback result) {
        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
            RectF bboxRect = content.getBBox();

            try {
                final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
                final Stamp annot = (Stamp) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Stamp, AppUtil.toFxRectF(bboxRect)), Annot.e_Stamp);

                final StampAddUndoItem undoItem = new StampAddUndoItem(mPdfViewCtrl);
                undoItem.setCurrentValue(content);
                undoItem.mPageIndex = pageIndex;
                undoItem.mStampType = StampUtil.getStampTypeByNameForReview(((StampAnnotContent) content).getIcon(), ((StampAnnotContent) content).getStampStream());
                undoItem.mSubject = StampUtil.getStampNameByType(undoItem.mStampType);
                undoItem.mFlags = Annot.e_FlagPrint;
                undoItem.mIconName = undoItem.mSubject;

                int rotation = ((StampAnnotContent) content).getRotation();
                undoItem.mRotation = (rotation == 0 ? rotation : 4 - rotation) * 90;

                StampEvent event = new StampEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, annot, mPdfViewCtrl);
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, annot);
                            if (addUndo) {
                                mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                            }
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                try {
                                    RectF viewRect = new RectF(AppUtil.toRectF(annot.getRect()));
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                                    Rect rect = new Rect();
                                    viewRect.roundOut(rect);
                                    rect.inset(-10, -10);
                                    mPdfViewCtrl.refresh(pageIndex, rect);
                                } catch (PDFException e) {
                                    e.printStackTrace();
                                }
                            }
                        }
                        if (result != null) {
                            result.result(null, true);
                        }
                    }
                });
                mPdfViewCtrl.addTask(task);
            } catch (PDFException e) {
                e.printStackTrace();
                if (result != null) {
                    result.result(null, false);
                }
            }
        }
    }

    private int mLastStampStyle;
    private IToolSupply mToolSupply;
    private ToolItemBean mCurToolItem;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new StampToolSupply(mContext);
        return mToolSupply;
    }

    private class StampToolSupply extends ToolSupplyImpl {

        public StampToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.comment_tool_stamp_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty toolProperty = new ToolProperty();
            toolProperty.type = ToolConstants.Stamp;
            toolProperty.style = 0;
            toolProperty.color = AppResource.getColor(mContext, R.color.p3);
            return toolProperty;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_STAMP;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if (mUiExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_COMMENT_TAB) {
                    mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_Stamp);
                }
                mLastStampStyle = mStampType;

                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mStampType = property.style;
                if (mStampType == StampConstants.CUSTOM_TEXT_STAMP) {
                    TextStampBean stampBean = getTextStampBean((Long) property.mTag);
                    if (stampBean.mThumbnailBitmap == null) {
                        drawTextStampThumbnail(stampBean);
                    }
                    mCurSelectedCustomStamp = stampBean;
                } else if (mStampType == StampConstants.CUSTOM_IMAGE_STAMP) {
                    ImageStampBean stampBean = getImageStampBean((Long) property.mTag);
                    if (stampBean.mThumbnailBitmap == null) {
                        drawImageStampThumbnail(stampBean);
                    }
                    mCurSelectedCustomStamp = stampBean;
                }
                mUiExtensionsManager.setCurrentToolHandler(StampToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == StampToolHandler.this) {
                    mStampType = mLastStampStyle;
                    mCurSelectedCustomStamp = null;
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;
            mLastStampStyle = mStampType;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mStampType = property.style;
            if (mStampType == StampConstants.CUSTOM_TEXT_STAMP) {
                TextStampBean stampBean = getTextStampBean((Long) property.mTag);
                if (stampBean.mThumbnailBitmap == null) {
                    drawTextStampThumbnail(stampBean);
                }
                mCurSelectedCustomStamp = stampBean;
            } else if (mStampType == StampConstants.CUSTOM_IMAGE_STAMP) {
                ImageStampBean stampBean = getImageStampBean((Long) property.mTag);
                if (stampBean.mThumbnailBitmap == null) {
                    drawImageStampThumbnail(stampBean);
                }
                mCurSelectedCustomStamp = stampBean;
            }

            initDisplayItems();
            StampToolHandler.this.resetPropertyBar(mStampsRootView);
            mPropertyDlg.setOnDLDismissListener(new MatchDialog.DismissListener() {
                @Override
                public void onDismiss() {
                    if (isEditMode()) {
                        switchEditMode(false);
                    }
                    mStampType = mLastStampStyle;
                    mCurToolItem = null;
                    mCurSelectedCustomStamp = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            if (mPropertyDlg == null) {
                mPropertyDlg = new StampPropertyDialog(mUiExtensionsManager.getAttachedActivity());
                mPropertyDlg.setTitleVisible(false);
            }
            return mPropertyDlg;
        }
    }

    void onConfigurationChanged(Configuration newConfig) {
        if (mPropertyDlg != null) {
            mPropertyDlg.resetWH();
            if (mPropertyDlg.isShowing()) {
                mPropertyDlg.showDialog();
            }
        }
        if (mAddTextStampDialog != null) {
            mAddTextStampDialog.resetWH();
            if (mAddTextStampDialog.isShowing()) {
                mAddTextStampDialog.showDialog();
            }
        }
        AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
            @Override
            public void run() {
                if (mImageSheetMenu != null && mImageSheetMenu.isShowing()) {
                    showImageMenu(mAddImageStampsIv);
                }

                updateSpanCount();
            }
        }, 380);
    }

    private void updateSpanCount() {
        int width;
        if (AppDisplay.isPad()) {
            width = mProperbarWidth;
        } else {
            width = mUiExtensionsManager.getRootView().getWidth();
        }
        int margin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_8dp);
        mSpanCount = Math.max(1, (width - margin) / (margin + mStampItemWidth + 2));
        mVerSpacing = (width - mStampItemWidth * mSpanCount) / (mSpanCount + 1);

        if (mStampsRootView != null) {
            for (final RecyclerView recyclerView : mStampRecyclerViews) {
                GridLayoutManager gridLayoutManager = (GridLayoutManager) recyclerView.getLayoutManager();
                if (gridLayoutManager != null) {
                    if (gridLayoutManager.getSpanCount() == mSpanCount)
                        continue;

                    gridLayoutManager.setSpanCount(mSpanCount);
                    gridLayoutManager.requestLayout();
                } else {
                    gridLayoutManager = new GridLayoutManager(mContext, mSpanCount);
                    recyclerView.setLayoutManager(gridLayoutManager);
                }

                if (recyclerView.getAdapter() != null) {
                    recyclerView.getAdapter().notifyDataSetChanged();
                }
            }
        }
    }

    private void setSelectedButtonState(RelativeLayout checkLayout, ImageView checkView, boolean isSelected) {
        checkLayout.setSelected(isSelected);
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) checkView.getLayoutParams();
        if (params == null) return;
        params.removeRule(isSelected ? RelativeLayout.ALIGN_PARENT_LEFT : RelativeLayout.ALIGN_PARENT_RIGHT);
        params.addRule(isSelected ? RelativeLayout.ALIGN_PARENT_RIGHT : RelativeLayout.ALIGN_PARENT_LEFT, RelativeLayout.TRUE);
        checkView.setLayoutParams(params);
    }

    private ColorStateList getSelectedButtonColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.p1);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.p1);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        if (verifyPermissions(grantResults)) {
            if (requestCode == ActRequestCode.REQ_CAMERA_PERMISSION) {
                AppIntentUtil.selectImageFromCamera(mUiExtensionsManager,
                        mCaptureImgPath, ActRequestCode.REQ_CAMERA_PERMISSION, ActRequestCode.REQ_FILE_FROM_CAMERA);
            }
        } else {
            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_permission_denied));
        }
    }

    private void showImageMenu(View view) {
        Rect rect = new Rect();
        view.getGlobalVisibleRect(rect);

        int[] location = new int[2];
        view.getLocationOnScreen(location);
        int screen_x = location[0];
        int screen_y = location[1];
        if (AppDevice.isChromeOs(mUiExtensionsManager.getAttachedActivity())) {
            mUiExtensionsManager.getRootView().getLocationOnScreen(location);
            screen_x -= location[0];
            screen_y -= location[1];
            rect.set(screen_x, screen_y, screen_x + rect.width(), screen_y + rect.height());
        } else {
            view.getLocationInWindow(location);
            int window_x = location[0];
            int window_y = location[1];
            rect.offset(screen_x - window_x, screen_y - window_y);
        }
        mImageSheetMenu.show(mUiExtensionsManager.getRootView(), rect, UIPopoverFrag.ARROW_BOTTOM);
    }

    private boolean verifyPermissions(int[] grantResults) {
        for (int grantResult : grantResults) {
            if (grantResult != PackageManager.PERMISSION_GRANTED) {
                return false;
            }
        }
        return true;
    }

    void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
        if (resultCode == Activity.RESULT_OK) {
            String path = null;
            if (requestCode == ActRequestCode.REQ_FILE_FROM_GALLERY && data != null) {
                path = AppFileUtil.getFilePathFromUri(mContext, data.getData());

                path = AppFileUtil.saveToScopedCache(path, mCustomStampCacheDir, AppFileUtil.getFileName(path));
                showImageStampDialog(path);
            } else if (requestCode == ActRequestCode.REQ_FILE_FROM_CAMERA) {
                path = mCaptureImgPath;
                showImageStampDialog(path);
            }
        }
    }

    private boolean isEditMode() {
        return mEditState == StampConstants.EDIT_STATE;
    }

    void updateTheme() {
        if (isEditMode()) {
            switchEditMode(false);
        }
        mStampsRootView = null;
        mAddTextStampRootView = null;
        mCustomStampsRootView = null;
        if (mImageSheetMenu != null) {
            mImageSheetMenu.updateTheme();
            mImageSheetMenu.dismiss();
        }
        if (mPromptDialog != null) {
            mPromptDialog.dismiss();
            mPromptDialog = null;
        }
        if (mAddTextStampDialog != null) {
            mAddTextStampDialog.dismiss();
            mAddTextStampDialog = null;
        }
        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
            @Override
            public void run() {
                initDisplayItems();
                resetPropertyBar(mStampsRootView);
            }
        });
    }

    void saveCustomStamps() {
        if (mTextStampAdapter != null) {
            StampUtil.saveTextStamps(mContext, mTextStampAdapter.getItems());
        }

        if (mImageStampAdapter != null) {
            StampUtil.saveImageStamps(mContext, mImageStampAdapter.getItems());
        }
    }

    private TextStampBean getTextStampBean(Long createTime) {
        if (createTime == null) {
            createTime = System.currentTimeMillis();
        }
        if (mCustomTextStamps == null)
            mCustomTextStamps = StampUtil.getTextStamps(mContext);

        for (TextStampBean stampBean : mCustomTextStamps) {
            if (stampBean.mCreateTime == createTime)
                return stampBean;
        }

        TextStampBean defaultBean = new TextStampBean();
        defaultBean.mCreateTime = System.currentTimeMillis();
        defaultBean.mModifiedTime = System.currentTimeMillis();
        defaultBean.mColor = StampConstants.Custom_Colors[0];
        defaultBean.mText = AppResource.getString(mContext, R.string.add_custom_text_stamp_prompt);
        return defaultBean;
    }

    private ImageStampBean getImageStampBean(Long createTime) {
        if (createTime == null) {
            createTime = System.currentTimeMillis();
        }
        if (mCustomImageStamps == null)
            mCustomImageStamps = StampUtil.getImageStamps(mContext);

        for (ImageStampBean stampBean : mCustomImageStamps) {
            if (stampBean.mCreateTime == createTime)
                return stampBean;
        }
        return new ImageStampBean();
    }

    private void drawTextStampThumbnail(TextStampBean stampBean) {
        DrawStampThumbnailTask drawTask = new DrawStampThumbnailTask(mContext, mUiExtensionsManager, stampBean, new IDrawStampThumbnailCallback<TextStampBean>() {
            @Override
            public void onResult(DrawStampThumbnailTask task, TextStampBean bean, Bitmap bitmap) {
                if (bean.mThumbnailBitmap != null && !bean.mThumbnailBitmap.isRecycled())
                    bean.mThumbnailBitmap.recycle();
                bean.mThumbnailBitmap = bitmap;
                mPdfViewCtrl.removeTask(task);
            }
        });
        mPdfViewCtrl.addTask(drawTask);
    }

    private void drawImageStampThumbnail(ImageStampBean stampBean) {
        if (stampBean.mThumbnailBitmap == null) {
            stampBean.mThumbnailBitmap = StampUtil.getThumbnailBitmap(mContext, stampBean.mCacheFile);
        }
    }

    private int getDialogHeight() {
        return mStampItemHeight * 7 + mHorSpacing * 2 * 7;
    }

    private void onStampItemSelected(int stampType) {
        onStampItemSelected(stampType, true);
    }

    private void onStampItemSelected(int stampType, boolean hideProperty) {
        if (stampType == StampConstants.CUSTOM_TEXT_STAMP) {
            mImageStampAdapter.setSelectedItem(null);
        } else if (stampType == StampConstants.CUSTOM_IMAGE_STAMP) {
            mTextStampAdapter.setSelectedItem(null);
        } else {
            mCurSelectedCustomStamp = null;
            mTextStampAdapter.setSelectedItem(null);
            mImageStampAdapter.setSelectedItem(null);
        }
        notifyUpdateAdapterData();

        if (mCurToolItem != null) {
            mCurToolItem.property.style = mStampType;
            if (mCustomPropertyListener != null)
                mCustomPropertyListener.onValueChanged(PropertyBar.PROPERTY_STAMP, mStampType);
            if (mPropertyDlg.getCustomPropertyChangeListener() != null)
                mPropertyDlg.getCustomPropertyChangeListener().onValueChanged(PropertyBar.PROPERTY_STAMP, mStampType);
        }
        if (hideProperty)
            mPropertyDlg.dismiss();
    }

}
