/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.square;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.view.MotionEvent;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Square;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

public class SquareToolHandler implements ToolHandler {

    private Context mContext;
    int mColor;
    int mFillColor;
    int mOpacity;
    float mThickness;
    private int mControlPtEx = 5;// Refresh the scope expansion width
    private float mCtlPtLineWidth = 2;
    private float mCtlPtRadius = 5;

    private boolean mTouchCaptured = false;
    private int mLastPageIndex = -1;

    private PointF mStartPoint = new PointF(0, 0);
    private PointF mStopPoint = new PointF(0, 0);
    private PointF mDownPoint = new PointF(0, 0);// whether moving point

    private RectF mNowRect = new RectF(0, 0, 0, 0);
    /**
     * one of mCreateList to draw the Square
     */
    private Paint mPaint;
    private Paint mFillPaint;
    private Paint mLastAnnotPaint;

    /**
     * toolbar
     */
    private PropertyBar mPropertyBar;
    private PDFViewCtrl mPdfViewCtrl;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;
    private UIExtensionsManager mUiExtensionsManager;
    private boolean mIsContinuousCreate = true;

    public SquareToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
        mContext = context;
        mControlPtEx = AppDisplay.dp2px( mControlPtEx);

        mPaint = new Paint();
        mPaint.setStyle(Style.STROKE);
        mPaint.setAntiAlias(true);
        mPaint.setDither(true);

        mFillPaint = new Paint();
        mFillPaint.setStyle(Style.FILL);
        mFillPaint.setAntiAlias(true);
        mFillPaint.setDither(true);

        mLastAnnotPaint = new Paint();
        mLastAnnotPaint.setStyle(Style.STROKE);
        mLastAnnotPaint.setAntiAlias(true);
        mLastAnnotPaint.setDither(true);

        mColor = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
        mFillColor = PropertyBar.PB_FILLCOLORS_TOOL_DEFAULT[0];
        mOpacity = 100;
        mThickness = 5.0f;

        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mPropertyBar = mUiExtensionsManager.getMainFrame().getPropertyBar();
        mDefaultSizeDimension = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_48dp);
    }

    void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_SQUARE;
    }

    @Override
    public void onActivate() {
        mLastPageIndex = -1;
        mCtlPtRadius = 5;
        mCtlPtRadius = AppDisplay.dp2px( mCtlPtRadius);
        resetPropertyBar();
    }

    @Override
    public void onDeactivate() {
    }

    private Rect mTempRectInTouch = new Rect(0, 0, 0, 0);
    private Rect mInvalidateRect = new Rect(0, 0, 0, 0);

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultTouchEvent(pageIndex, motionEvent);
        if (!handled && motionEvent.getActionMasked() != MotionEvent.ACTION_DOWN)
            handled = onSquareToolTouch(pageIndex, motionEvent);
        return handled;
    }

    private boolean onSquareToolTouch(int pageIndex, MotionEvent motionEvent) {
        PointF disPoint = new PointF(motionEvent.getX(), motionEvent.getY());
        PointF pvPoint = new PointF();
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(disPoint, pvPoint, pageIndex);
        float x = pvPoint.x;
        float y = pvPoint.y;

        int action = motionEvent.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (!mTouchCaptured && mLastPageIndex == -1 || mLastPageIndex == pageIndex) {
                    mTouchCaptured = true;
                    mStartPoint.x = x;
                    mStartPoint.y = y;
                    mStopPoint.x = x;
                    mStopPoint.y = y;
                    mDownPoint.set(x, y);
                    mTempRectInTouch.setEmpty();
                    if (mLastPageIndex == -1) {
                        mLastPageIndex = pageIndex;
                    }
                }
                return true;
            case MotionEvent.ACTION_MOVE:
                if (!mTouchCaptured || mLastPageIndex != pageIndex)
                    break;
                if (!mDownPoint.equals(x, y)) {
                    mStopPoint.x = x;
                    mStopPoint.y = y;
                    float thickness = thicknessOnPageView(pageIndex, mThickness);
                    float deltaXY = thickness / 2 + mCtlPtLineWidth + mCtlPtRadius * 2 + 2;// Judging border value
                    float line_k = (y - mStartPoint.y) / (x - mStartPoint.x);
                    float line_b = mStartPoint.y - line_k * mStartPoint.x;
                    if (y <= deltaXY && line_k != 0) {
                        // whether created annot beyond a PDF page(pageView)
                        mStopPoint.y = deltaXY;
                        mStopPoint.x = (mStopPoint.y - line_b) / line_k;
                    } else if (y >= (mPdfViewCtrl.getPageViewHeight(pageIndex) - deltaXY) && line_k != 0) {
                        mStopPoint.y = (mPdfViewCtrl.getPageViewHeight(pageIndex) - deltaXY);
                        mStopPoint.x = (mStopPoint.y - line_b) / line_k;
                    }
                    if (mStopPoint.x <= deltaXY) {
                        mStopPoint.x = deltaXY;
                    } else if (mStopPoint.x >= mPdfViewCtrl.getPageViewWidth(pageIndex) - deltaXY) {
                        mStopPoint.x = mPdfViewCtrl.getPageViewWidth(pageIndex) - deltaXY;
                    }

                    getDrawRect(mStartPoint.x, mStartPoint.y, mStopPoint.x, mStopPoint.y);

                    mInvalidateRect.set((int) mNowRect.left, (int) mNowRect.top, (int) mNowRect.right, (int) mNowRect.bottom);
                    mInvalidateRect.inset((int) (-mThickness * 12f - mControlPtEx), (int) (-mThickness * 12f - mControlPtEx));
                    if (!mTempRectInTouch.isEmpty()) {
                        mInvalidateRect.union(mTempRectInTouch);
                    }
                    mTempRectInTouch.set(mInvalidateRect);
                    RectF _rect = AppDmUtil.rectToRectF(mInvalidateRect);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(_rect, _rect, pageIndex);
                    mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(_rect));
                    mDownPoint.set(x, y);
                }
                return true;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (!mTouchCaptured || mLastPageIndex != pageIndex)
                    break;
                if (!mStartPoint.equals(mStopPoint.x, mStopPoint.y) && Math.abs(mStopPoint.length() - mStartPoint.length()) > 8) {
                    createAnnot();
                } else {
                    int pageViewWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
                    int pageViewHeight = mPdfViewCtrl.getPageViewHeight(pageIndex);
                    float scale = Math.max(1f * pageViewWidth / mPdfViewCtrl.getDisplayViewWidth(), 1f * pageViewHeight / mPdfViewCtrl.getDisplayViewHeight());
                    adjustPointOrNot(pageViewWidth, pageViewHeight, (int) (mDefaultSizeDimension * scale));
                    getDrawRect(mStartPoint.x, mStartPoint.y, mStopPoint.x, mStopPoint.y);
                    createAnnot();
                }
                return true;
            default:
                return true;
        }

        return true;
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null){
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
        return onSquareToolTouch(pageIndex, motionEvent);
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!handled){
            addSquareByDefault(pageIndex, motionEvent);
        }
        return true;
    }

    private int mDefaultSizeDimension;

    private void addSquareByDefault(int pageIndex, MotionEvent motionEvent) {
        int pageViewWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
        int pageViewHeight = mPdfViewCtrl.getPageViewHeight(pageIndex);
        float scale = Math.max(1f * pageViewWidth / mPdfViewCtrl.getDisplayViewWidth(), 1f * pageViewHeight / mPdfViewCtrl.getDisplayViewHeight());
        onSquareToolTouch(pageIndex, motionEvent);
        adjustPointOrNot(pageViewWidth, pageViewHeight, (int) (mDefaultSizeDimension * scale));
        getDrawRect(mStartPoint.x, mStartPoint.y, mStopPoint.x, mStopPoint.y);
        MotionEvent upEvent = MotionEvent.obtain(motionEvent);
        upEvent.setAction(MotionEvent.ACTION_UP);
        upEvent.setLocation(mStopPoint.x, mStopPoint.y);
        onSquareToolTouch(pageIndex, upEvent);
        upEvent.recycle();
    }

    private void adjustPointOrNot(int pageViewWidth, int pageViewHeight, int dx) {
        float startPointOffset = dx / -2f;
        int offset = (int) (mThickness + mCtlPtRadius * 2);
        float minStartX = offset - startPointOffset;
        float maxStartX = pageViewWidth - offset + startPointOffset;
        float maxStartY = pageViewHeight - offset + startPointOffset;
        if (mStartPoint.x < minStartX){
            mStartPoint.x = minStartX;
        }
        if (mStartPoint.x > maxStartX){
            mStartPoint.x = maxStartX;
        }
        if (mStartPoint.y < minStartX){
            //noinspection SuspiciousNameCombination
            mStartPoint.y = minStartX;
        }
        if (mStartPoint.y > maxStartY){
            mStartPoint.y = maxStartY;
        }
        mStopPoint.x = mStartPoint.x;
        mStopPoint.y = mStartPoint.y;
        mStartPoint.offset(startPointOffset,startPointOffset);
        mStopPoint.offset(-startPointOffset,-startPointOffset);
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinuousCreate;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinuousCreate = continueAddAnnot;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mStartPoint == null || mStopPoint == null) {
            return;
        }

        if (mLastPageIndex == pageIndex) {
            canvas.save();
            mFillPaint.setColor(mFillColor);
            canvas.drawRect(mNowRect, mFillPaint);

            setPaint(pageIndex);
            canvas.drawRect(mNowRect, mPaint);
            canvas.restore();
        }
    }

    private void setPaint(int pageIndex) {
        mPaint.setColor(mColor);
        mPaint.setAlpha(AppDmUtil.opacity100To255(mOpacity));
        mPaint.setAntiAlias(true);
        PointF tranPt = new PointF(thicknessOnPageView(pageIndex, mThickness), thicknessOnPageView(pageIndex, mThickness));
        mPaint.setStrokeWidth(tranPt.x);
    }

    private RectF mPageViewThickness = new RectF(0, 0, 0, 0);

    private float thicknessOnPageView(int pageIndex, float thickness) {
        mPageViewThickness.set(0, 0, thickness, thickness);
        mPdfViewCtrl.convertPdfRectToPageViewRect(mPageViewThickness, mPageViewThickness, pageIndex);
        return Math.abs(mPageViewThickness.width());
    }

    private RectF getBBox(int pageIndex) {
        RectF bboxRect = new RectF();
        mTempRect.set(mNowRect);

        mTempRect.inset(-thicknessOnPageView(pageIndex, mThickness) / 2f, -thicknessOnPageView(pageIndex, mThickness) / 2f);

        mPdfViewCtrl.convertPageViewRectToPdfRect(mTempRect, mTempRect, pageIndex);
        bboxRect.left = mTempRect.left;
        bboxRect.right = mTempRect.right;
        bboxRect.top = mTempRect.top;
        bboxRect.bottom = mTempRect.bottom;

        return bboxRect;
    }

    private RectF mTempRect = new RectF(0, 0, 0, 0);

    private void createAnnot() {
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            RectF bboxRect = getBBox(mLastPageIndex);

            try {
                final PDFPage page = mPdfViewCtrl.getDoc().getPage(mLastPageIndex);
                final Square newAnnot = (Square) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Square, AppUtil.toFxRectF(bboxRect)), Annot.e_Square);

                final SquareAddUndoItem undoItem = new SquareAddUndoItem(mPdfViewCtrl);
                undoItem.mPageIndex = mLastPageIndex;
                undoItem.mColor = mColor;
                undoItem.mFillColor = mFillColor;
                undoItem.mNM = AppDmUtil.randomUUID(null);
                undoItem.mOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
                undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
                undoItem.mBorderStyle = BorderInfo.e_Solid;
                undoItem.mLineWidth = mThickness;
                undoItem.mFlags = 4;
                undoItem.mSubject = "Rectangle";
                undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mBBox = new RectF(bboxRect);

                SquareEvent event = new SquareEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, newAnnot, mPdfViewCtrl);
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, newAnnot);
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(undoItem);
                            if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {

                                try {
                                    RectF viewRect = AppUtil.toRectF(newAnnot.getRect());
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, mLastPageIndex);
                                    Rect rect = new Rect();
                                    viewRect.roundOut(rect);
                                    rect.inset(-10, -10);
                                    mPdfViewCtrl.refresh(mLastPageIndex, rect);
                                } catch (PDFException e) {
                                    e.printStackTrace();
                                }

                                mTouchCaptured = false;
                                mLastPageIndex = -1;
                                mDownPoint.set(0, 0);
                                if (!mIsContinuousCreate) {
                                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
                                }
                            }
                        }
                    }
                });
                mPdfViewCtrl.addTask(task);

            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    private void getDrawRect(float x1, float y1, float x2, float y2) {
        float minx = Math.min(x1, x2);
        float miny = Math.min(y1, y2);
        float maxx = Math.max(x1, x2);
        float maxy = Math.max(y1, y2);

        mNowRect.left = minx;
        mNowRect.top = miny;
        mNowRect.right = maxx;
        mNowRect.bottom = maxy;
    }

    protected void changeCurrentColor(int currentColor) {
        mColor = currentColor;
        setProItemColor(currentColor);
    }

    protected void changeCurrentFillColor(int currentFillColor) {
        mFillColor = currentFillColor;
        setProItemFillColor(currentFillColor);
    }

    protected void changeCurrentOpacity(int opacity) {
        mOpacity = opacity;
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
    }

    protected void changeCurrentThickness(float thickness) {
        mThickness = thickness;
        if (mCurToolItem == null) return;
        mCurToolItem.property.lineWidth = thickness;
    }

    protected void removePropertyListener() {
        mPropertyChangeListener = null;
    }

    private void setProItemColor(int color) {
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
    }

    private void setProItemFillColor(int fillColor) {
        if (mCurToolItem == null) return;
        mCurToolItem.property.fillColor = fillColor;
    }

    void resetPropertyBar() {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        colors[0] = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
        mPropertyBar.setColors(colors);
        int[] fillColors = new int[PropertyBar.PB_FILLCOLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_FILLCOLORS_TOOL_DEFAULT, 0, fillColors, 0, fillColors.length);
        fillColors[0] = PropertyBar.PB_FILLCOLORS_TOOL_DEFAULT[0];
        mPropertyBar.setFillColors(fillColors);

        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_FILL_COLOR, mFillColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_LINEWIDTH, mThickness);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_LINEWIDTH, AppResource.getString(mContext, R.string.pb_border_thickness));
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_FILL_COLOR, AppResource.getString(mContext, R.string.pb_fill_color));
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_COLOR, AppResource.getString(mContext, R.string.pb_border_color));
        mPropertyBar.setArrowVisible(true);
        mPropertyBar.reset(getSupportedProperties());
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
    }

    private long getSupportedProperties() {
        return PropertyBar.PROPERTY_COLOR
                | PropertyBar.PROPERTY_FILL_COLOR
                | PropertyBar.PROPERTY_OPACITY
                | PropertyBar.PROPERTY_LINEWIDTH;
    }

    private IToolSupply mToolSupply;
    private ToolItemBean mCurToolItem;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new SquareToolSupply(mContext);
        return mToolSupply;
    }

    public int getColor() {
        return mColor;
    }

    public int getFillColor() {
        return mFillColor;
    }

    public int getOpacity() {
        return mOpacity;
    }

    public float getThickness() {
        return mThickness;
    }

    private class SquareToolSupply extends ToolSupplyImpl {

        public SquareToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.drawing_tool_square;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = ToolConstants.Rectangle;
            property.color = mColor;
            property.fillColor = mFillColor;
            property.opacity = mOpacity;
            property.lineWidth = mThickness;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_SQUARE;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if(mUiExtensionsManager.getMainFrame().getCurrentTab()== ToolbarItemConfig.ITEM_DRAWING_TAB) {
                    mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_DrawingBar_Rectangle);
                }

                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mColor = property.color;
                mFillColor = property.fillColor;
                mOpacity = property.opacity;
                mThickness = property.lineWidth;
                mUiExtensionsManager.setCurrentToolHandler(SquareToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == SquareToolHandler.this) {
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mColor = property.color;
            mFillColor = property.fillColor;
            mOpacity = property.opacity;
            mThickness = property.lineWidth;

            SquareToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

}
