/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.redaction;


import android.app.Activity;
import android.content.Context;
import android.graphics.Canvas;
import android.graphics.PorterDuff;
import android.graphics.Typeface;
import android.text.TextUtils;
import android.util.TypedValue;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Toast;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.addon.Redaction;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.fxcrt.RectF;
import com.foxit.sdk.common.fxcrt.RectFArray;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.DefaultAppearance;
import com.foxit.sdk.pdf.annots.QuadPoints;
import com.foxit.sdk.pdf.annots.QuadPointsArray;
import com.foxit.sdk.pdf.annots.Redact;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.AnnotUndoItem;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.config.uisettings.annotations.annots.RedactConfig;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.BaseDialogFragment;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.controls.toolbar.IBaseItem;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.controls.toolbar.impl.BaseBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BaseItemImpl;
import com.foxit.uiextensions.controls.toolbar.impl.BottomBarImpl;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyBtnImpl;
import com.foxit.uiextensions.controls.toolbar.impl.TopBarImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.modules.SearchModule;
import com.foxit.uiextensions.modules.SearchView;
import com.foxit.uiextensions.modules.more.RedactPageFragment;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;

import java.util.ArrayList;
import java.util.List;

import androidx.fragment.app.FragmentActivity;

import static com.foxit.uiextensions.annots.form.FormFillerUtil.getSupportFont;

public class RedactModule implements Module, PropertyBar.PropertyChangeListener {
    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private RedactAnnotHandler mAnnotHandler;
    private RedactToolHandler mToolHandler;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;
    private BaseItemImpl mApplyAllItem;
    private boolean mSelectRedact = false;
    private RedactPageFragment mRedactPage;
    private BaseItemImpl mSelectItem;
    private BaseItemImpl mBackItem;
    private RedactProperty mRedactProperty;

    public RedactModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return MODULE_NAME_REDACT;
    }

    @Override
    public boolean loadModule() {
        mToolHandler = new RedactToolHandler(mContext, mPdfViewCtrl);
        mAnnotHandler = new RedactAnnotHandler(mContext, mPdfViewCtrl);
        mAnnotHandler.setToolHandler(mToolHandler);
        mAnnotHandler.setPropertyChangeListener(this);
        mToolSupply = new RedacToolSupply(mContext, R.drawable.more_menu_redaction);
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            initRedactPageFragment();
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            RedactConfig config = uiExtensionsManager.getConfig().uiSettings.annotations.redaction;
            mRedactProperty = new RedactProperty(config);

            uiExtensionsManager.registerToolHandler(mToolHandler);
            uiExtensionsManager.registerAnnotHandler(mAnnotHandler);
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerToolHandlerChangedListener(mToolHandlerChangedListener);
            uiExtensionsManager.registerStateChangeListener(mStateChangeListener);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
            uiExtensionsManager.addCreatePropertyChangedListener(mAnnotHandler.getType(), mCreatePropertyChangedListener);
            uiExtensionsManager.getToolsManager().addToolItem(UIToolBaseBar.USAGE_TYPE_PROTECT, ToolConstants.ProtectTypeRedaction, mToolSupply);
        }
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        return true;
    }

    private IToolSupply mToolSupply;
     class RedacToolSupply extends ToolSupplyBtnImpl {
         public RedacToolSupply(Context context, int ImgResId) {
             super(context, ImgResId);
         }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            return null;
        }

        @Override
        public String getToolName(int toolType) {
            return AppResource.getString(mContext, R.string.fx_string_redaction);
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            onActivity();
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {

        }

        @Override
        public PropertyBar getPropertyBar() {
            return null;
        }
    };

    private void initRedactPageFragment() {
        if (mRedactPage != null) return;
        mRedactPage = new RedactPageFragment();
        mRedactPage.init(mPdfViewCtrl);
        mRedactPage.setDismissListener(new BaseDialogFragment.DismissListener() {
            @Override
            public void onDismiss() {
                mRedactPage = null;
            }
        });
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.unregisterToolHandler(mToolHandler);
            uiExtensionsManager.unregisterAnnotHandler(mAnnotHandler);
            uiExtensionsManager.unregisterToolHandlerChangedListener(mToolHandlerChangedListener);
            uiExtensionsManager.unregisterStateChangeListener(mStateChangeListener);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
            uiExtensionsManager.removeCreatePropertyChangedListener(mAnnotHandler.getType());
            ((UIExtensionsManager) mUiExtensionsManager).getToolsManager().removeToolItem(UIToolBaseBar.USAGE_TYPE_PROTECT, ToolConstants.ProtectTypeRedaction, mToolSupply);
        }
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mAnnotHandler.removePropertyBarListener();
        setSelectRedact(false);
        return true;
    }

    /**
     * Get the property when adding redact.
     *
     * @return the property when adding redact {@link RedactProperty} .
     */
    public RedactProperty getToolProperty() {
        return mRedactProperty;
    }

    /**
     * Set the property when adding redact.
     *
     * @param property the property when adding redact {@link RedactProperty} .
     */
    public void setToolProperty(RedactProperty property) {
        mRedactProperty = property;

        mToolHandler.onApplyFillColorChanged(mRedactProperty.applyFillColor);
        mToolHandler.onFontColorChanged(mRedactProperty.fontColor);
        mToolHandler.onFontSizeChanged(mRedactProperty.fontSize);
        mToolHandler.onFontNameChanged(mRedactProperty.fontName);
    }

    @Override
    public void onValueChanged(long property, int value) {
        AnnotHandler currentAnnotHandler = ((UIExtensionsManager) mUiExtensionsManager).getCurrentAnnotHandler();
        if (currentAnnotHandler == mAnnotHandler) {
            if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
                mAnnotHandler.onApplyFillColorChanged(value);
            } else if (property == PropertyBar.PROPERTY_FILL_COLOR) {
                mAnnotHandler.onFontColorValueChanged(value);
            }
        }
    }

    @Override
    public void onValueChanged(long property, float value) {
        AnnotHandler currentAnnotHandler = ((UIExtensionsManager) mUiExtensionsManager).getCurrentAnnotHandler();
        if (currentAnnotHandler == mAnnotHandler && property == PropertyBar.PROPERTY_FONTSIZE) {
            mAnnotHandler.onFontSizeValueChanged(value);
        }
    }

    @Override
    public void onValueChanged(long property, String value) {
        AnnotHandler currentAnnotHandler = ((UIExtensionsManager) mUiExtensionsManager).getCurrentAnnotHandler();
        if (currentAnnotHandler == mAnnotHandler) {
            if (property == PropertyBar.PROPERTY_FONTNAME) {
                mAnnotHandler.onFontValueChanged(value);
            } else if (property == PropertyBar.PROPERTY_OVERLAY_TEXT) {
                mAnnotHandler.onOverlayTextChanged(value);
            }
        }
    }

    private PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    public ToolHandler getToolHandler() {
        return mToolHandler;
    }

    private UIExtensionsManager.ToolHandlerChangedListener mToolHandlerChangedListener = new UIExtensionsManager.ToolHandlerChangedListener() {
        @Override
        public void onToolHandlerChanged(ToolHandler oldToolHandler, ToolHandler newToolHandler) {
            UIExtensionsManager extensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            if (newToolHandler == mToolHandler && oldToolHandler != mToolHandler) {
                IPanelManager panelManager = extensionsManager.getPanelManager();
                if (panelManager != null && panelManager.isShowingPanel()) {
                    panelManager.hidePanel();
                }
                resetTopBar();
                resetBottomBar();
            } else if (extensionsManager.getState() == ReadStateConfig.STATE_REDACT && newToolHandler == null
                    && oldToolHandler != null) {
                extensionsManager.setCurrentToolHandler(getToolHandler());
            }
        }
    };

    private IStateChangeListener mStateChangeListener = new IStateChangeListener() {
        @Override
        public void onStateChanged(int oldState, int newState) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            if ((newState != ReadStateConfig.STATE_REDACT && newState != ReadStateConfig.STATE_SEARCH && oldState == ReadStateConfig.STATE_REDACT) ||
                    (newState != ReadStateConfig.STATE_REDACT && newState != ReadStateConfig.STATE_SEARCH && oldState == ReadStateConfig.STATE_SEARCH)) {
                MainFrame mainFrame = (MainFrame) uiExtensionsManager.getMainFrame();
                if (mainFrame.isShowFullScreenUI()) {
                    mainFrame.hideToolbars();
                }

                if (uiExtensionsManager.getCurrentToolHandler() == mToolHandler)
                    uiExtensionsManager.setCurrentToolHandler(null);
                setSelectRedact(false);
            } else if (oldState != ReadStateConfig.STATE_REDACT && newState == ReadStateConfig.STATE_REDACT) {
                if (uiExtensionsManager.getCurrentToolHandler() != mToolHandler) {
                    uiExtensionsManager.setCurrentToolHandler(mToolHandler);
                }
            }
        }
    };

    private void resetTopBar() {
        final UIExtensionsManager extensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        TopBarImpl topBar = (TopBarImpl) (extensionsManager).getMainFrame().getCustomTopbar();
        topBar.removeAllItems();
        topBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
        topBar.setShowSolidLineColor(mContext.getResources().getColor(R.color.p1));
        IBaseItem titleItem = new BaseItemImpl(mContext);
        titleItem.setDisplayStyle(IBaseItem.ItemType.Item_Text);
        titleItem.setText(AppResource.getString(mContext.getApplicationContext(), R.string.fx_string_redaction));
        titleItem.setTextColorResource(R.color.t4);
        titleItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_16sp));
        titleItem.setTypeface(Typeface.defaultFromStyle(Typeface.BOLD));
        titleItem.setEllipsize(TextUtils.TruncateAt.END);
        titleItem.setMaxWidth(AppDisplay.getActivityWidth() / 3);

        mBackItem = new BaseItemImpl(mContext);
        mBackItem.setDisplayStyle(IBaseItem.ItemType.Item_Text);
        mBackItem.setText(R.string.fx_string_close);
        mBackItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mBackItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_15sp));
        mBackItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (AppUtil.isFastDoubleClick()) return;
                exitRedact();
            }
        });
        mApplyAllItem = new BaseItemImpl(mContext);
        mApplyAllItem.setText(AppResource.getString(mContext, R.string.menu_more_apply_all));
        mApplyAllItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
        mApplyAllItem.setTextSize(TypedValue.COMPLEX_UNIT_PX, AppResource.getDimensionPixelSize(mContext, R.dimen.ux_text_size_15sp));
        mApplyAllItem.setEllipsize(TextUtils.TruncateAt.END);
        mApplyAllItem.setMaxWidth(AppDisplay.getActivityWidth() / 3 - AppDisplay.dp2px(16));
        mApplyAllItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (AppUtil.isFastDoubleClick()) return;

                extensionsManager.startHideToolbarsTimer();
                if (extensionsManager.getState() == ReadStateConfig.STATE_PANZOOM) {
                    extensionsManager.changeState(ReadStateConfig.STATE_REDACT);
                }

                if (extensionsManager.getDocumentManager().getCurrentAnnot() != null) {
                    extensionsManager.getDocumentManager().setCurrentAnnot(null);
                }
                UIAnnotRedaction.applyAll(mPdfViewCtrl, null);
            }
        });
        mApplyAllItem.setEnable(AppAnnotUtil.hasModuleLicenseRight(Constants.e_ModuleNameRedaction)
                && !extensionsManager.getDocumentManager().isSign()
                && extensionsManager.canModifyContents()
                && extensionsManager.isEnableModification()
                && extensionsManager.getDocumentManager().withAllPermission(null)
                && extensionsManager.getDocumentManager().withModifyPermission()
                && extensionsManager.getDocumentManager().withDeletePermission());
        topBar.setStartMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
        topBar.setEndMargin(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_16dp));
        topBar.addView(mBackItem, BaseBar.TB_Position.Position_LT);
        topBar.addView(titleItem, BaseBar.TB_Position.Position_CENTER);
        topBar.addView(mApplyAllItem, BaseBar.TB_Position.Position_RB);
        topBar.setMiddleButtonCenter(true);
    }

    public void exitRedact() {
        UIExtensionsManager extensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        extensionsManager.setCurrentToolHandler(null);
        setSelectRedact(false);
        extensionsManager.getPDFViewCtrl().invalidate();
    }

    private void resetBottomBar() {
        BaseBarImpl bottomBar = (BaseBarImpl) ((UIExtensionsManager) mUiExtensionsManager).getMainFrame().getCustomBottombar();
        bottomBar.removeAllItems();
        if (!((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().withAddPermission()){
            bottomBar.setVisible(false);
            return;
        }
        bottomBar.setVisible(true);
        bottomBar.setItemInterval(mContext.getResources().getDimensionPixelSize(R.dimen.ux_margin_20dp));
        if (AppDisplay.isPad()) {
            bottomBar.setBackgroundResource(R.drawable.redaction_tool_bar_bg);
            bottomBar.setBackGroundColorFilter(ThemeConfig.getInstance(mContext).getB2(), PorterDuff.Mode.SRC_IN);
            bottomBar.setWidth(AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_208dp));
            ViewGroup.MarginLayoutParams marginLayoutParams = (ViewGroup.MarginLayoutParams) bottomBar.getContentView().getLayoutParams();
            marginLayoutParams.bottomMargin = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_44dp);
            if (bottomBar instanceof BottomBarImpl) {
                ((BottomBarImpl) bottomBar).setShowSolidLineColor(mContext.getResources().getColor(R.color.ux_color_translucent));
            }
        } else {
            bottomBar.setBackgroundColor(ThemeConfig.getInstance(mContext).getB2());
            if (bottomBar instanceof BottomBarImpl) {
                ((BottomBarImpl) bottomBar).setShowSolidLineColor(mContext.getResources().getColor(R.color.p1));
            }
        }
        mSelectItem = new BaseItemImpl(mContext);
        mSelectItem.setDisplayStyle(IBaseItem.ItemType.Item_Image);
        mSelectItem.setImageResource(R.drawable.more_menu_redaction_select);
        mSelectItem.setId(R.id.id_more_menu_redaction_select);
        mSelectItem.setBackgroundResource(R.color.ux_color_translucent);
        mSelectItem.setImageTextBackgroundResouce(AppDisplay.isPad() ? R.drawable.redaction_tool_item_select_bg : R.drawable.tool_view_select_bg);
        mSelectItem.setImageTintList(ThemeUtil.getEnableIconColor(mContext));
        mSelectItem.setOnItemClickListener(new IBaseItem.OnItemClickListener() {
            @Override
            public void onClick(IBaseItem item, View v) {
                ((UIExtensionsManager) mUiExtensionsManager).startHideToolbarsTimer();

                boolean enableRedact = !mSelectRedact;
                setSelectRedact(enableRedact);
                mSelectItem.setSelected(enableRedact);
                if (enableRedact) {
                    UIToast.getInstance(mContext).show(R.string.menu_more_redact_select_toast_content, Toast.LENGTH_SHORT);
                }
            }
        });
        IBaseItem pageItem = new BaseItemImpl(mContext);
        pageItem.setDisplayStyle(IBaseItem.ItemType.Item_Image);
        pageItem.setImageResource(R.drawable.more_menu_redaction_page);
        pageItem.setId(R.id.id_more_menu_redaction_page);
        pageItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
        pageItem.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (AppUtil.isFastDoubleClick()) return;
                ((UIExtensionsManager) mUiExtensionsManager).startHideToolbarsTimer();

                UIToast.getInstance(mContext).show(R.string.menu_more_redact_page_toast_content, Toast.LENGTH_SHORT);
                showRedactPages();
            }
        });
        bottomBar.addView(mSelectItem, BaseBar.TB_Position.Position_CENTER);
        bottomBar.addView(pageItem, BaseBar.TB_Position.Position_CENTER);
        SearchModule searchModule = ((SearchModule) ((UIExtensionsManager) mUiExtensionsManager).getModuleByName(Module.MODULE_NAME_SEARCH));
        if (searchModule != null) {
            final IBaseItem searchItem = new BaseItemImpl(mContext);
            searchItem.setDisplayStyle(IBaseItem.ItemType.Item_Image);
            searchItem.setImageResource(R.drawable.more_menu_redaction_search);
            searchItem.setId(R.id.id_more_menu_redaction_search);
            searchItem.setImageTintList(ThemeUtil.getPrimaryIconColor(mContext));
            searchItem.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (AppUtil.isFastDoubleClick()) {
                        return;
                    }
                    ((UIExtensionsManager) mUiExtensionsManager).startHideToolbarsTimer();
                    SearchModule searchModule = ((SearchModule) ((UIExtensionsManager) mUiExtensionsManager).getModuleByName(Module.MODULE_NAME_SEARCH));
                    SearchView searchView = searchModule.getSearchView();
                    searchView.launchRedactSearchView();
                    searchView.show();
                }
            });
            bottomBar.addView(searchItem, BaseBar.TB_Position.Position_CENTER);
        }
    }

    private void showRedactPages() {
        if (mUiExtensionsManager == null) {
            return;
        }

        if (((UIExtensionsManager) mUiExtensionsManager).getState() == ReadStateConfig.STATE_PANZOOM) {
            ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_NORMAL);
        }

        if (((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().getCurrentAnnot() != null) {
            ((UIExtensionsManager) mUiExtensionsManager).getDocumentManager().setCurrentAnnot(null);
        }

        Activity activity = ((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity();
        if (activity == null) {
            return;
        }

        if (!(activity instanceof FragmentActivity)) {
            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.the_attached_activity_is_not_fragmentActivity));
            return;
        }

        FragmentActivity act = (FragmentActivity) activity;
        String fragmentTag = RedactPageFragment.class.getSimpleName();
        RedactPageFragment contentFragment = (RedactPageFragment) act.getSupportFragmentManager().findFragmentByTag(fragmentTag);
        if (contentFragment == null) {
            initRedactPageFragment();
            contentFragment = mRedactPage;
        }
        AppDialogManager.getInstance().showAllowManager(contentFragment, act.getSupportFragmentManager(), fragmentTag, null);
    }

    public void setSelectRedact(boolean active) {
        mSelectRedact = active;
        mToolHandler.setSelectRedact(active);
    }

    public boolean noSelectRedact() {
        return !mSelectRedact;
    }

    public void redactPages(int[] indexes) {
        PDFDoc doc = mPdfViewCtrl.getDoc();
        if (doc == null) return;
        try {
            Redaction redaction = new Redaction(doc);
            RectFArray rectFArray;
            RectF rectF;
            UIExtensionsManager extensionsManager = ((UIExtensionsManager) mUiExtensionsManager);
            RedactConfig config = extensionsManager.getConfig().uiSettings.annotations.redaction;
            final List<Annot> redactList = new ArrayList<>();
            final RedactUndoItem addBatchRedactUndo = new RedactAddUndoItem(mPdfViewCtrl);
            addBatchRedactUndo.mUndoItemList = new ArrayList<>();
            for (int index : indexes) {
                PDFPage page = doc.getPage(index);
                RectF mediaBoxRect = page.getBox(PDFPage.e_MediaBox);
                mediaBoxRect.normalize();
                rectF = page.getBox(PDFPage.e_CropBox);
                rectF.normalize();
                if (rectF.isEmpty()) {
                    rectF = mediaBoxRect;
                } else {
                    // Some PDF comes with crop box bigger than media box, which doesn't make sense,
                    // but we need to handle it by clipping the crop box with media box
                    rectF.intersect(mediaBoxRect);
                }
                rectFArray = new RectFArray();
                rectFArray.add(rectF);
                Redact redactAnnot = redaction.markRedactAnnot(doc.getPage(index), rectFArray);
                redactList.add(redactAnnot);
                addBatchRedactUndo.mUndoItemList.add(createUndoItem(config, index, rectFArray, null));
            }
            RedactEvent event = new RedactEvent(EditAnnotEvent.EVENTTYPE_ADD, addBatchRedactUndo, redactList, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        int index;
                        try {
                            for (Annot redact : redactList) {
                                index = redact.getPage().getIndex();
                                if (mPdfViewCtrl.isPageVisible(index)) {
                                    try {
                                        android.graphics.RectF annotRectF = AppUtil.toRectF(redact.getRect());
                                        mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, index);
                                        mPdfViewCtrl.refresh(index, AppDmUtil.rectFToRect(annotRectF));
                                    } catch (PDFException e) {
                                        e.printStackTrace();
                                    }
                                }
                            }
                            DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
                            documentManager.addUndoItem(addBatchRedactUndo);
                            documentManager.onAnnotsAdded(mPdfViewCtrl.getDoc());
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    public void redactTextContent(List<SearchView.SearchResult> searchResults) {
        PDFDoc doc = mPdfViewCtrl.getDoc();
        if (doc == null) return;
        int size = searchResults.size();
        try {
            Redaction redaction = new Redaction(doc);
            int startPageIndex = -1;
            int key;
            SearchView.SearchResult item;
            UIExtensionsManager extensionsManager = ((UIExtensionsManager) mUiExtensionsManager);
            RedactConfig config = extensionsManager.getConfig().uiSettings.annotations.redaction;
            final List<Annot> redactList = new ArrayList<>();
            final RedactUndoItem addBatchRedactUndo = new RedactAddUndoItem(mPdfViewCtrl);
            addBatchRedactUndo.mUndoItemList = new ArrayList<>();
            QuadPoints quadPoint;
            QuadPointsArray quadPointsArray;
            for (int i = 0; i < size; i++) {
                item = searchResults.get(i);
                if (startPageIndex == -1) {
                    startPageIndex = item.mPageIndex;
                } else if (item.mPageIndex < startPageIndex) {
                    startPageIndex = item.mPageIndex;
                }
                Redact redact = redaction.markRedactAnnot(doc.getPage(item.mPageIndex), item.getFxRectFArray());
                quadPointsArray = new QuadPointsArray();
                for (android.graphics.RectF rect : item.mRects) {
                    quadPoint = new QuadPoints();
                    quadPoint.setFirst(AppUtil.toFxPointF(rect.left, rect.top));
                    quadPoint.setSecond(AppUtil.toFxPointF(rect.right, rect.top));
                    quadPoint.setThird(AppUtil.toFxPointF(rect.left, rect.bottom));
                    quadPoint.setFourth(AppUtil.toFxPointF(rect.right, rect.bottom));
                    quadPointsArray.add(quadPoint);
                }
                redact.setQuadPoints(quadPointsArray);
                redactList.add(redact);
                addBatchRedactUndo.mUndoItemList.add(createUndoItem(config, item.mPageIndex, item.getFxRectFArray(), quadPointsArray));
            }
            RedactEvent event = new RedactEvent(EditAnnotEvent.EVENTTYPE_ADD, addBatchRedactUndo, redactList, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        int index;
                        try {
                            for (Annot redact : redactList) {
                                index = redact.getPage().getIndex();
                                if (mPdfViewCtrl.isPageVisible(index)) {
                                    try {
                                        android.graphics.RectF annotRectF = AppUtil.toRectF(redact.getRect());
                                        mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, index);
                                        mPdfViewCtrl.refresh(index, AppDmUtil.rectFToRect(annotRectF));
                                    } catch (PDFException e) {
                                        e.printStackTrace();
                                    }
                                }
                            }
                            DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
                            documentManager.addUndoItem(addBatchRedactUndo);
                            documentManager.onAnnotsAdded(mPdfViewCtrl.getDoc());
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    public AnnotUndoItem createUndoItem(RedactConfig config, int pageIndex, RectFArray rectFArray, QuadPointsArray quadPointsArray) {
        RedactAddUndoItem undo = new RedactAddUndoItem(mPdfViewCtrl);
        undo.mRectFArray = rectFArray;
        if (quadPointsArray != null)
            undo.mQuadPointsArray = quadPointsArray;
        undo.mNM = AppDmUtil.randomUUID(null);
        undo.mCreationDate = AppDmUtil.currentDateToDocumentDate();
        undo.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
        undo.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
        undo.mPageIndex = pageIndex;
        undo.mSubject = "Redact";
        undo.mFlags = Annot.e_FlagPrint;
        undo.mBorderColor = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
        undo.mDaFlags = DefaultAppearance.e_FlagFont
                | DefaultAppearance.e_FlagTextColor
                | DefaultAppearance.e_FlagFontSize;
        undo.mOverlayText = "";
        undo.mApplyFillColor = mRedactProperty.applyFillColor;
        undo.mTextColor = mRedactProperty.fontColor;
        undo.mFontSize = mRedactProperty.fontSize;
        undo.mFont = getSupportFont(mRedactProperty.fontName);
        return undo;
    }

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mBackItem != null) {
                mBackItem.setImageTintList(ThemeUtil.getItemIconColor(mContext));
            }
            if (mApplyAllItem != null) {
                mApplyAllItem.setTextColor(ThemeUtil.getPrimaryTextColor(mContext));
            }
            resetTopBar();
            resetBottomBar();
            if (mRedactPage != null) {
                if (!mRedactPage.isAdded()) {
                    return;
                }
                if (!mRedactPage.isHidden()) {
                    mRedactPage.dismiss();
                    mRedactPage = null;
                    mRedactPage = new RedactPageFragment();
                    mRedactPage.init(mPdfViewCtrl);
                }

            }
            mAnnotHandler.updateTheme();
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (mApplyAllItem != null) {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
                mApplyAllItem.setEnable(AppAnnotUtil.hasModuleLicenseRight(Constants.e_ModuleNameRedaction)
                        && !uiExtensionsManager.getDocumentManager().isSign()
                        && uiExtensionsManager.canModifyContents()
                        && uiExtensionsManager.isEnableModification()
                        && uiExtensionsManager.getDocumentManager().withAllPermission(null)
                        && uiExtensionsManager.getDocumentManager().withModifyPermission()
                        && uiExtensionsManager.getDocumentManager().withDeletePermission());
            }

            if (mUiExtensionsManager != null && ((UIExtensionsManager) mUiExtensionsManager).getState() == ReadStateConfig.STATE_REDACT) {
                ((UIExtensionsManager) mUiExtensionsManager).changeState(ReadStateConfig.STATE_NORMAL);
            }
        }

    };

   public void onActivity() {
       final Runnable doRunnable = new Runnable() {
           @Override
           public void run() {
               if (AppAnnotUtil.hasModuleLicenseRight(Constants.e_ModuleNameRedaction)) {
                   mToolHandler.setContinueAddAnnot(true);
                   ((UIExtensionsManager) mUiExtensionsManager).setCurrentToolHandler(mToolHandler);
               } else {
                   String message = AppUtil.getMessage(mContext, Constants.e_ErrNoRedactionModuleRight);
                   UIToast.getInstance(mContext).show(message);
               }
           }
       };
       if (((UIExtensionsManager) mUiExtensionsManager).getPermissionProvider() != null) {
               ((UIExtensionsManager) mUiExtensionsManager).getPermissionProvider().checkPermission(
                   IPermissionProvider.FUNCTION_REDACTION, new IPermissionProvider.IPermissionState() {
                       @Override
                       public void onState(int state) {
                           if (state == IPermissionProvider.PERMISSION_STATE_SHOW) {
                                   doRunnable.run();
                           }
                       }
                   }
           );
       } else {
           doRunnable.run();
       }
   }

    private final PropertyBar.CreatePropertyChangedListener mCreatePropertyChangedListener = new PropertyBar.CreatePropertyChangedListener() {
        @Override
        public void onCreateValueChanged(long property, Object value) {
            if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
                mRedactProperty.applyFillColor = (int) value;
                mToolHandler.onApplyFillColorChanged((int) value);
            } else if (property == PropertyBar.PROPERTY_FILL_COLOR) {
                mRedactProperty.fontColor = (int) value;
                mToolHandler.onFontColorChanged((int) value);
            } else if (property == PropertyBar.PROPERTY_FONTSIZE) {
                mRedactProperty.fontSize = (float) value;
                mToolHandler.onFontSizeChanged((float) value);
            } else if (property == PropertyBar.PROPERTY_FONTNAME) {
                mRedactProperty.fontName = (String) value;
                mToolHandler.onFontNameChanged((String) value);
            }
        }

        @Override
        public int getToolType() {
            return -1;
        }

        @Override
        public int getTabPosition() {
            return -1;
        }

        @Override
        public void resetToolPropertyBar() {
        }
    };

}
