/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.polygon;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.view.MotionEvent;
import android.widget.Toast;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.fxcrt.PointFArray;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Polygon;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;

import java.util.ArrayList;

public class PolygonToolHandler implements ToolHandler {

    private Context mContext;
    int mColor;
    int mFillColor;
    int mOpacity;
    float mThickness;
    private int mControlPtEx = 5;// Refresh the scope expansion width
    private float mCtlPtLineWidth = 2;
    private float mCtlPtRadius = 5;

    private boolean mTouchCaptured = false;
    private int mLastPageIndex = -1;
    private boolean mIsContinuousCreate = true;

    private Paint mPaint;
    private Paint mPathPaint;

    /**
     * toolbar
     */
    private PropertyBar mPropertyBar;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;
    private UIExtensionsManager mUiExtensionsManager;
    private PDFViewCtrl mPdfViewCtrl;

    private ArrayList<PointF> mVertexList = new ArrayList<PointF>();
    private ArrayList<PointF> mPdfVertexList = new ArrayList<PointF>();
    
    public PolygonToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mPropertyBar = mUiExtensionsManager.getMainFrame().getPropertyBar();
        
        mContext = context;
        mControlPtEx = AppDisplay.dp2px( mControlPtEx);

        mPaint = new Paint();
        mPaint.setStyle(Style.STROKE);
        mPaint.setAntiAlias(true);
        mPaint.setDither(true);

        mPathPaint = new Paint();
        mPathPaint.setStyle(Style.STROKE);
        mPathPaint.setAntiAlias(true);
        mPathPaint.setDither(true);
    }

    /**
     * init toolbar
     */
    protected void init() {
        mColor = mUiExtensionsManager.getConfig().uiSettings.annotations.polygon.color;
        mFillColor = mUiExtensionsManager.getConfig().uiSettings.annotations.polygon.fillColor;
        mOpacity =(int) (mUiExtensionsManager.getConfig().uiSettings.annotations.polygon.opacity * 100);
        mThickness = mUiExtensionsManager.getConfig().uiSettings.annotations.polygon.thickness;
    }

    protected void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    protected void removePropertyListener() {
        mPropertyChangeListener = null;
    }

    private void setPaint(int pageIndex) {
        mPaint.setColor(mColor);
        mPaint.setAlpha(AppDmUtil.opacity100To255(mOpacity));
        mPaint.setAntiAlias(true);
        PointF tranPt = new PointF(thicknessOnPageView(pageIndex, mThickness), thicknessOnPageView(pageIndex, mThickness));
        mPaint.setStrokeWidth(tranPt.x);
    }

    private RectF mPageViewThickness = new RectF(0, 0, 0, 0);

    private float thicknessOnPageView(int pageIndex, float thickness) {
        mPageViewThickness.set(0, 0, thickness, thickness);
        mPdfViewCtrl.convertPdfRectToPageViewRect(mPageViewThickness, mPageViewThickness, pageIndex);
        return Math.abs(mPageViewThickness.width());
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_POLYGON;
    }

    public int getColor() {
        return mColor;
    }

    public int getFillColor() {
        return mFillColor;
    }

    public int getOpacity() {
        return mOpacity;
    }

    public float getThickness() {
        return mThickness;
    }
    @Override
    public void onActivate() {
        mCtlPtRadius = 5;
        mCtlPtRadius = AppDisplay.dp2px( mCtlPtRadius);

        resetPropertyBar();
        mPdfViewCtrl.registerDoubleTapEventListener(mDoubleTapEventListener);
    }

    void resetPropertyBar() {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        mPropertyBar.setColors(colors);
        int[] fillColors = new int[PropertyBar.PB_FILLCOLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_FILLCOLORS_TOOL_DEFAULT, 0, fillColors, 0, fillColors.length);
        mPropertyBar.setFillColors(fillColors);

        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_FILL_COLOR, mFillColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_LINEWIDTH, mThickness);
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_LINEWIDTH, AppResource.getString(mContext, R.string.pb_border_thickness));
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_FILL_COLOR, AppResource.getString(mContext, R.string.pb_fill_color));
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_COLOR, AppResource.getString(mContext, R.string.pb_border_color));
        mPropertyBar.setArrowVisible(true);
        mPropertyBar.reset(getSupportedProperties());
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
    }

    private long getSupportedProperties() {
        return PropertyBar.PROPERTY_COLOR
                | PropertyBar.PROPERTY_FILL_COLOR
                | PropertyBar.PROPERTY_OPACITY
                | PropertyBar.PROPERTY_LINEWIDTH;
    }

    @Override
    public void onDeactivate() {
        mPdfViewCtrl.unregisterDoubleTapEventListener(mDoubleTapEventListener);
        checkToCreateAnnot();
    }

    private void checkToCreateAnnot() {
        int size = mVertexList.size();
        if (size < 3) {
            if (size != 0) {
                UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.add_polygon_failed_hints), Toast.LENGTH_LONG);
            }
            mTouchCaptured = false;
            mLastPageIndex = -1;
            if (mVertexList.size() > 0) {
                RectF rect = new RectF(mVertexList.get(0).x, mVertexList.get(0).y, mVertexList.get(0).x, mVertexList.get(0).y);
                for (int i = 1; i < mVertexList.size(); i ++) {
                    rect.union(mVertexList.get(i).x, mVertexList.get(i).y);
                }
                rect.inset(-mCtlPtRadius, -mCtlPtRadius);
                mPdfViewCtrl.invalidate();
                mVertexList.clear();
            }
            mPdfVertexList.clear();
            return;
        }
        createAnnot();
    }

    private void drawControls(Canvas canvas, ArrayList<PointF> vertexList, int color, int opacity) {
        if (vertexList.size() == 0) return;
        for (int i = 0; i < vertexList.size(); i ++) {
            PointF p = vertexList.get(i);
            mPaint.setColor(Color.WHITE);
            mPaint.setAlpha(255);
            mPaint.setStyle(Paint.Style.FILL);
            canvas.drawCircle(p.x, p.y, mCtlPtRadius, mPaint);
            mPaint.setColor(color);
            mPaint.setAlpha(opacity);
            mPaint.setStyle(Paint.Style.STROKE);
            canvas.drawCircle(p.x, p.y, mCtlPtRadius, mPaint);
        }
    }

    private void drawPath(Canvas canvas, int pageIndex, ArrayList<PointF> vertexList) {
        int size = vertexList.size();
        if (size == 0 || size == 1) return;
        mPathPaint.setColor(mColor);
        mPathPaint.setAlpha(AppDmUtil.opacity100To255(mOpacity));
        mPathPaint.setStrokeWidth(thicknessOnPageView(pageIndex, mThickness));
        for (int i = 0; i < size; i ++) {
            if (i == size - 1) return;
            PointF p1 = vertexList.get(i);
            PointF p2 = vertexList.get(i + 1);

            Path path = new Path();
            path.moveTo(p1.x, p1.y);
            path.lineTo(p2.x, p2.y);
            canvas.drawPath(path, mPathPaint);
        }
    }


    private RectF mInvalidateRect = new RectF(0, 0, 0, 0);

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultTouchEvent(pageIndex, motionEvent);
        if (!handled && motionEvent.getActionMasked() != MotionEvent.ACTION_DOWN){
            handled = onPolygonToolTouch(pageIndex, motionEvent);
        }
        return handled;
    }

    private boolean onPolygonToolTouch(int pageIndex, MotionEvent motionEvent) {
        PointF disPoint = new PointF(motionEvent.getX(), motionEvent.getY());
        PointF pvPoint = new PointF();
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(disPoint, pvPoint, pageIndex);
        float x = pvPoint.x;
        float y = pvPoint.y;

        int action = motionEvent.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (!mTouchCaptured && mLastPageIndex == -1 || mLastPageIndex == pageIndex) {
                    mTouchCaptured = true;
                    mVertexList.add(new PointF(x, y));
                    PointF temp = new PointF();
                    mPdfViewCtrl.convertPageViewPtToPdfPt(pvPoint, temp, pageIndex);
                    mPdfVertexList.add(temp);
                    if (mLastPageIndex == -1) {
                        mLastPageIndex = pageIndex;
                    }
                }
                return true;
            case MotionEvent.ACTION_MOVE:
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (!mTouchCaptured || mLastPageIndex != pageIndex || mVertexList.size() == 0)
                    break;
                if (motionEvent.getActionMasked() == MotionEvent.ACTION_MOVE){
                    if (mVertexList.size() == 1 && AppDmUtil.distanceOfTwoPoints(mVertexList.get(0),new PointF(x, y)) > 10){
                        mVertexList.add(new PointF(x, y));
                        PointF temp = new PointF();
                        mPdfViewCtrl.convertPageViewPtToPdfPt(pvPoint, temp, pageIndex);
                        mPdfVertexList.add(temp);
                    }
                    pvPoint.set(x,y);
                    adjustPointFOrNot(pageIndex, pvPoint, mCtlPtRadius);
                    PointF temp = new PointF();
                    mPdfViewCtrl.convertPageViewPtToPdfPt(pvPoint, temp, pageIndex);
                    mPdfVertexList.get(mPdfVertexList.size() - 1).set(temp);
                }
                float deltaXY = mCtlPtLineWidth + mCtlPtRadius * 2 + 2;
                if (mVertexList.size() == 1) {
                    mInvalidateRect.set(mVertexList.get(0).x, mVertexList.get(0).y, mVertexList.get(0).x, mVertexList.get(0).y);
                } else {
                    int size = mVertexList.size();
                    mInvalidateRect.union(mVertexList.get(size - 1).x, mVertexList.get(size - 1).y);
                }

                mInvalidateRect.inset(-deltaXY, -deltaXY);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));
                return true;
            default:
                return true;
        }

        return true;
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        onPolygonToolTouch(pageIndex, motionEvent);
        MotionEvent upEvent = MotionEvent.obtain(motionEvent);
        upEvent.setAction(MotionEvent.ACTION_UP);
        onPolygonToolTouch(pageIndex, upEvent);
        upEvent.recycle();
        mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
        return true;
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        onPolygonToolTouch(pageIndex, motionEvent);
        MotionEvent upEvent = MotionEvent.obtain(motionEvent);
        upEvent.setAction(MotionEvent.ACTION_UP);
        onPolygonToolTouch(pageIndex, upEvent);
        upEvent.recycle();
        return true;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinuousCreate;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinuousCreate = continueAddAnnot;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mPdfVertexList.size() == 0) {
            return;
        }
        if (mLastPageIndex == pageIndex) {
            canvas.save();
            ArrayList<PointF> vertexList = new ArrayList<PointF>();
            for (int i = 0; i < mPdfVertexList.size(); i ++) {
                PointF pointF = new PointF();
                pointF.set(mPdfVertexList.get(i));
                mPdfViewCtrl.convertPdfPtToPageViewPt(pointF, pointF, pageIndex);
                vertexList.add(pointF);
            }
            drawPath(canvas, pageIndex, vertexList);

            setPaint(pageIndex);
            drawControls(canvas, vertexList, mColor, AppDmUtil.opacity100To255(mOpacity));
            canvas.restore();
        }
    }

    private void createAnnot() {
        if (mLastPageIndex == -1) return;
        if (mPdfVertexList.size() == 0) return;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            try {
                final PDFPage page = mPdfViewCtrl.getDoc().getPage(mLastPageIndex);
                final Polygon newAnnot = (Polygon) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Polygon, new com.foxit.sdk.common.fxcrt.RectF(0, 0, 0 ,0)), Annot.e_Polygon);

                final PolygonAddUndoItem undoItem = new PolygonAddUndoItem(mPdfViewCtrl);
                undoItem.mPageIndex = mLastPageIndex;
                undoItem.mColor = mColor;
                undoItem.mFillColor = mFillColor;
                undoItem.mNM = AppDmUtil.randomUUID(null);
                undoItem.mOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
                undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
                undoItem.mBorderStyle = BorderInfo.e_Solid;
                undoItem.mLineWidth = mThickness;
                undoItem.mFlags = 4;
                undoItem.mSubject = "Polygon Dimension";
                undoItem.mIntent = "PolygonDimension";
                undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mVertexes = new PointFArray();
                for (int i = 0; i < mPdfVertexList.size(); i ++) {
                    undoItem.mVertexes.add(new com.foxit.sdk.common.fxcrt.PointF(mPdfVertexList.get(i).x, mPdfVertexList.get(i).y));
                }

                PolygonEvent event = new PolygonEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, newAnnot, mPdfViewCtrl);
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, newAnnot);
                            mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                            if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {

                                try {
                                    RectF viewRect = AppUtil.toRectF(newAnnot.getRect());
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, mLastPageIndex);
                                    Rect rect = new Rect();
                                    viewRect.roundOut(rect);
                                    rect.inset(-10, -10);
                                    mPdfViewCtrl.refresh(mLastPageIndex, rect);
                                } catch (PDFException e) {
                                    e.printStackTrace();
                                }

                                mTouchCaptured = false;
                                mLastPageIndex = -1;
                            }
                        }
                    }
                });
                mPdfViewCtrl.addTask(task);
                mVertexList.clear();
                mPdfVertexList.clear();
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    protected void changeCurrentColor(int color) {
        mColor = color;
        setProItemColor(color);
    }

    protected void changeCurrentFillColor(int currentFillColor) {
        mFillColor = currentFillColor;
        setProItemFillColor(currentFillColor);
    }

    protected void changeCurrentOpacity(int opacity) {
        mOpacity = opacity;
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
    }

    protected void changeCurrentThickness(float thickness) {
        mThickness = thickness;
        if (mCurToolItem == null) return;
        mCurToolItem.property.lineWidth = thickness;
    }

    private void setProItemColor(int color){
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
    }

    private void setProItemFillColor(int fillColor) {
        if (mCurToolItem == null) return;
        mCurToolItem.property.fillColor = fillColor;
    }

    private IToolSupply mToolSupply;
    private ToolItemBean mCurToolItem;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new PolygonToolSupply(mContext);
        return mToolSupply;
    }

    private class PolygonToolSupply extends ToolSupplyImpl {

        public PolygonToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.drawing_tool_polygon;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = ToolConstants.Polygon;
            property.color = mColor;
            property.fillColor = mFillColor;
            property.opacity = mOpacity;
            property.lineWidth = mThickness;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_POLYGON;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if(mUiExtensionsManager.getMainFrame().getCurrentTab()== ToolbarItemConfig.ITEM_DRAWING_TAB) {
                    mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_DrawingBar_Polygon);
                }
                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mColor = property.color;
                mFillColor = property.fillColor;
                mOpacity = property.opacity;
                mThickness = property.lineWidth;
                mUiExtensionsManager.setCurrentToolHandler(PolygonToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == PolygonToolHandler.this) {
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mColor = property.color;
            mFillColor = property.fillColor;
            mOpacity = property.opacity;
            mThickness = property.lineWidth;

            PolygonToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

    PDFViewCtrl.IDoubleTapEventListener mDoubleTapEventListener = new PDFViewCtrl.IDoubleTapEventListener() {
        @Override
        public boolean onSingleTapConfirmed(MotionEvent motionEvent) {
            return false;
        }

        @Override
        public boolean onDoubleTap(MotionEvent motionEvent) {
            return handleOnDoubleTap();
        }

        @Override
        public boolean onDoubleTapEvent(MotionEvent motionEvent) {
            return true;
        }
    };

    private boolean handleOnDoubleTap() {
        if (!(mUiExtensionsManager.getCurrentToolHandler() instanceof PolygonToolHandler))
            return false;
        if (mIsContinuousCreate) {
            checkToCreateAnnot();
        } else {
            mUiExtensionsManager.setCurrentToolHandler(null);
        }
        return true;
    }

    private void adjustPointFOrNot(int pageIndex, PointF pointF, float offset) {
        if (pointF.x < offset)
            pointF.x = offset;
        if (pointF.y < offset)
            pointF.y = offset;
        if (pointF.x > mPdfViewCtrl.getPageViewWidth(pageIndex) - offset)
            pointF.x = mPdfViewCtrl.getPageViewWidth(pageIndex) - offset;
        if (pointF.y > mPdfViewCtrl.getPageViewHeight(pageIndex) - offset)
            pointF.y = mPdfViewCtrl.getPageViewHeight(pageIndex) - offset;
    }
}
