/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.polygon;

import android.content.Context;
import android.graphics.Canvas;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Polygon;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.config.modules.annotations.AnnotationsConfig;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.controls.propertybar.imp.PropertyBarImpl;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;

public class PolygonModule implements Module, PropertyBar.PropertyChangeListener {

    private PolygonAnnotHandler mAnnotHandler;
    private PolygonToolHandler mToolHandler;

    private PolygonCloudToolHandler mCloudToolHandler;

    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public PolygonModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public void onValueChanged(long property, int value) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
        if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
            if (mToolHandler != null && uiExtensionsManager.getCurrentToolHandler() == mToolHandler) {
                mToolHandler.changeCurrentColor(value);
            } else if (mCloudToolHandler != null && uiExtensionsManager.getCurrentToolHandler() == mCloudToolHandler) { // polygon cloud
                mCloudToolHandler.changeCurrentColor(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onColorValueChanged(value);
            } else if (mToolHandler != null && mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.changeCurrentColor(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            } else if (mCloudToolHandler != null && mCloudToolHandler.getCustomPropertyListener() != null) {
                mCloudToolHandler.changeCurrentColor(value);
                mCloudToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        } else if(property == PropertyBar.PROPERTY_FILL_COLOR) {
            if (mToolHandler != null && uiExtensionsManager.getCurrentToolHandler() == mToolHandler) {
                mToolHandler.changeCurrentFillColor(value);
            } else if (mCloudToolHandler != null && uiExtensionsManager.getCurrentToolHandler() == mCloudToolHandler) { // polygon cloud
                mCloudToolHandler.changeCurrentFillColor(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onFillColorValueChanged(value);
            } else if (mToolHandler != null && mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.changeCurrentFillColor(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            } else if (mCloudToolHandler != null && mCloudToolHandler.getCustomPropertyListener() != null) {
                mCloudToolHandler.changeCurrentFillColor(value);
                mCloudToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }

        } else if (property == PropertyBar.PROPERTY_OPACITY) {
            if (mToolHandler != null && uiExtensionsManager.getCurrentToolHandler() == mToolHandler) {
                mToolHandler.changeCurrentOpacity(value);
            } else if (mCloudToolHandler != null && uiExtensionsManager.getCurrentToolHandler() == mCloudToolHandler) { // polygon cloud
                mCloudToolHandler.changeCurrentOpacity(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onOpacityValueChanged(value);
            } else if (mToolHandler != null && mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.changeCurrentOpacity(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            } else if (mCloudToolHandler != null && mCloudToolHandler.getCustomPropertyListener() != null) {
                mCloudToolHandler.changeCurrentOpacity(value);
                mCloudToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        }
    }

    @Override
    public void onValueChanged(long property, float value) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
        if (property == PropertyBar.PROPERTY_LINEWIDTH) {
            if (mToolHandler != null && uiExtensionsManager.getCurrentToolHandler() == mToolHandler) {
                mToolHandler.changeCurrentThickness(value);
            } else if (mCloudToolHandler != null && uiExtensionsManager.getCurrentToolHandler() == mCloudToolHandler) { // polygon cloud
                mCloudToolHandler.changeCurrentThickness(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onLineWidthValueChanged(value);
            } else if (mToolHandler != null && mToolHandler.getCustomPropertyListener() != null) {
                mToolHandler.changeCurrentThickness(value);
                mToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            } else if (mCloudToolHandler != null && mCloudToolHandler.getCustomPropertyListener() != null) {
                mCloudToolHandler.changeCurrentThickness(value);
                mCloudToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        }
    }

    @Override
    public void onValueChanged(long property, String value) {
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_POLYGON;
    }

    @Override
    public boolean loadModule() {
        mAnnotHandler = new PolygonAnnotHandler(mContext, mPdfViewCtrl);
        mAnnotHandler.setPropertyChangeListener(this);
        mAnnotHandler.setAnnotMenu(new AnnotMenuImpl(mContext, mPdfViewCtrl));
        mAnnotHandler.setPropertyBar(new PropertyBarImpl(mContext, mPdfViewCtrl));

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            Config config = ((UIExtensionsManager) mUiExtensionsManager).getConfig();
            AnnotationsConfig annotConfig = config.modules.annotations;

            //polygon
            if (annotConfig.isLoadPolygon) {
                mToolHandler = new PolygonToolHandler(mContext, mPdfViewCtrl);
                mToolHandler.setPropertyChangeListener(this);
                ((UIExtensionsManager) mUiExtensionsManager).registerToolHandler(mToolHandler);
                mToolHandler.init();
                ((UIExtensionsManager) mUiExtensionsManager).getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_DRAW, ToolConstants.Polygon, mToolHandler.getToolSupply());

                ((UIExtensionsManager) mUiExtensionsManager).addCreatePropertyChangedListener(mAnnotHandler.getType(), mCreatePropertyChangedListener);
            }

            //polygon cloud
            if (annotConfig.isLoadCloud) {
                mCloudToolHandler = new PolygonCloudToolHandler(mContext, mPdfViewCtrl);
                mCloudToolHandler.setPropertyChangeListener(this);
                ((UIExtensionsManager) mUiExtensionsManager).registerToolHandler(mCloudToolHandler);
                mCloudToolHandler.init();
                ((UIExtensionsManager) mUiExtensionsManager).getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_DRAW, ToolConstants.Cloud, mCloudToolHandler.getToolSupply());

                ((UIExtensionsManager) mUiExtensionsManager).addCreatePropertyChangedListener(mAnnotHandler.getType(), mCreatePropertyChangedListener);
            }

            ((UIExtensionsManager) mUiExtensionsManager).registerAnnotHandler(mAnnotHandler);
            ((UIExtensionsManager) mUiExtensionsManager).registerModule(this);
            ((UIExtensionsManager) mUiExtensionsManager).registerThemeEventListener(mThemeEventListener);
        }
        mPdfViewCtrl.registerRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        mAnnotHandler.removePropertyListener();

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {

            //polygon
            if (mToolHandler != null) {
                mToolHandler.removePropertyListener();
                ((UIExtensionsManager) mUiExtensionsManager).unregisterToolHandler(mToolHandler);
            }

            //polygon cloud
            if (mCloudToolHandler != null) {
                mCloudToolHandler.removePropertyListener();
                ((UIExtensionsManager) mUiExtensionsManager).unregisterToolHandler(mCloudToolHandler);
            }

            ((UIExtensionsManager) mUiExtensionsManager).unregisterAnnotHandler(mAnnotHandler);
            ((UIExtensionsManager) mUiExtensionsManager).unregisterThemeEventListener(mThemeEventListener);
            ((UIExtensionsManager) mUiExtensionsManager).removeCreatePropertyChangedListener(mAnnotHandler.getType());
        }

        mPdfViewCtrl.unregisterRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        return true;
    }

    public AnnotHandler getAnnotHandler() {
        return mAnnotHandler;
    }

    private PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    PDFViewCtrl.IRecoveryEventListener memoryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            if (mAnnotHandler.getAnnotMenu() != null && mAnnotHandler.getAnnotMenu().isShowing()) {
                mAnnotHandler.getAnnotMenu().dismiss();
            }

            if (mAnnotHandler.getPropertyBar() != null && mAnnotHandler.getPropertyBar().isShowing()) {
                mAnnotHandler.getPropertyBar().dismiss();
            }
        }

        @Override
        public void onRecovered() {
        }
    };

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            mAnnotHandler.updateTheme();
        }
    };

    private final PropertyBar.CreatePropertyChangedListener mCreatePropertyChangedListener = new PropertyBar.CreatePropertyChangedListener() {
        @Override
        public void onCreateValueChanged(long property, Object value) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            BorderInfo borderInfo = null;
            if (curAnnot instanceof Polygon) {
                try {
                    borderInfo = curAnnot.getBorderInfo();
                } catch (PDFException ignored) {
                }
            }
            if (borderInfo == null) return;

            if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
                if (borderInfo.getStyle() == BorderInfo.e_Cloudy) {
                    if (mCloudToolHandler != null)
                        mCloudToolHandler.mColor = (int) value;
                } else {
                    if (mToolHandler != null)
                        mToolHandler.mColor = (int) value;
                }
            } else if (property == PropertyBar.PROPERTY_FILL_COLOR) {
                if (borderInfo.getStyle() == BorderInfo.e_Cloudy) {
                    if (mCloudToolHandler != null)
                        mCloudToolHandler.mFillColor = (int) value;
                } else {
                    if (mToolHandler != null)
                        mToolHandler.mFillColor = (int) value;
                }
            } else if (property == PropertyBar.PROPERTY_OPACITY) {
                if (borderInfo.getStyle() == BorderInfo.e_Cloudy) {
                    if (mCloudToolHandler != null)
                        mCloudToolHandler.mOpacity = (int) value;
                } else {
                    if (mToolHandler != null)
                        mToolHandler.mOpacity = (int) value;
                }
            } else if (property == PropertyBar.PROPERTY_LINEWIDTH) {
                if (borderInfo.getStyle() == BorderInfo.e_Cloudy) {
                    if (mCloudToolHandler != null)
                        mCloudToolHandler.mThickness = (float) value;
                } else {
                    if (mToolHandler != null)
                        mToolHandler.mThickness = (float) value;
                }
            }
        }

        @Override
        public int getToolType() {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if (curAnnot instanceof Polygon) {
                try {
                    BorderInfo borderInfo = curAnnot.getBorderInfo();
                    if (borderInfo.getStyle() == BorderInfo.e_Cloudy) {
                        return ToolConstants.Cloud;
                    } else {
                        return ToolConstants.Polygon;
                    }
                } catch (PDFException ignored) {
                }
            }
            return -1;
        }

        @Override
        public int getTabPosition() {
            return ToolbarItemConfig.ITEM_DRAWING_TAB;
        }

        @Override
        public void resetToolPropertyBar() {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if (curAnnot instanceof Polygon) {
                try {
                    BorderInfo borderInfo = curAnnot.getBorderInfo();
                    if (borderInfo.getStyle() == BorderInfo.e_Cloudy) {
                        mCloudToolHandler.resetPropertyBar();
                    } else {
                        mToolHandler.resetPropertyBar();
                    }
                } catch (PDFException ignored) {
                }
            }
        }
    };

}
