/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.note;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.inputmethod.EditorInfo;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Markup;
import com.foxit.sdk.pdf.annots.Note;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;

//ToolHandler is for creation, and AnnotHandler is for edition.
public class NoteToolHandler implements ToolHandler {
    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;

    private EditText mET_Content;
    private TextView mDialog_title;
    private Button mCancel;
    private Button mSave;
    int mColor;
    int mOpacity;
    int mIconType;

    private boolean mIsContinuousCreate = true;
    private PropertyBar mPropertyBar;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;
    private  Dialog mAddDialog;

    private ToolItemBean mCurToolItem;

    public NoteToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mPropertyBar = mUiExtensionsManager.getMainFrame().getPropertyBar();
    }

    @Override
    public void onActivate() {
        resetPropertyBar();
    }

    protected void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    protected void removePropertyBarListener() {
        mPropertyChangeListener = null;
    }

    void resetPropertyBar() {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        mPropertyBar.setColors(colors);

        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_ANNOT_TYPE, mIconType);
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_ANNOT_TYPE, AppResource.getString(mContext, R.string.pb_shape_settings));
        mPropertyBar.setArrowVisible(true);
        mPropertyBar.reset(getSupportedProperties());
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
    }

    private long getSupportedProperties() {
        return PropertyBar.PROPERTY_COLOR
                | PropertyBar.PROPERTY_OPACITY
                | PropertyBar.PROPERTY_ANNOT_TYPE;
    }

    @Override
    public void onDeactivate() {
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
    }

    @SuppressLint("NewApi")
    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        return mUiExtensionsManager.defaultTouchEvent(pageIndex, motionEvent);
    }

    private boolean onNoteToolTouch(int pageIndex, MotionEvent motionEvent) {
        int action = motionEvent.getActionMasked();
        PointF pageViewPt = AppAnnotUtil.getPdfPoint(mPdfViewCtrl, pageIndex, motionEvent);
        if (action == MotionEvent.ACTION_DOWN) {
            initDialog(pageIndex, adjustPointF(pageIndex, pageViewPt));
            return true;
        }
        return false;
    }

    private PointF adjustPointF(int pageIndex, PointF pointF) {
        PDFPage page = mUiExtensionsManager.getDocumentManager().getPage(pageIndex, false);
        try {
            if (page != null) {
                RectF rectF = new RectF(0, 0, page.getWidth(), page.getHeight());
                if (pointF.x < rectF.left)
                    pointF.x = rectF.left;

                if (pointF.x > rectF.right - 40)
                    pointF.x = rectF.right - 40;

                if (pointF.y - 40 < rectF.top)
                    pointF.y = rectF.top + 40;

                if (pointF.y > rectF.bottom)
                    pointF.y = rectF.bottom;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return pointF;
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null){
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        return onNoteToolTouch(pageIndex, motionEvent);
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!handled){
            handled = onNoteToolTouch(pageIndex, motionEvent);
        }
        return handled;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinuousCreate;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinuousCreate = continueAddAnnot;
    }

    //show dialog to input note content
    protected void initDialog(final int pageIndex, final PointF point) {
        final Activity context = mUiExtensionsManager.getAttachedActivity();
        if (context == null) return;
        if (mAddDialog != null && mAddDialog.isShowing()) return;

        View view = View.inflate(context.getApplicationContext(), R.layout.rd_note_dialog_edit, null);
        mDialog_title = (TextView) view.findViewById(R.id.rd_note_dialog_edit_title);
        mET_Content = (EditText) view.findViewById(R.id.rd_note_dialog_edit);
        if (AppDisplay.isPad()) { // SDKRD-9313
            mET_Content.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }
        mCancel = (Button) view.findViewById(R.id.rd_note_dialog_edit_cancel);
        mSave = (Button) view.findViewById(R.id.rd_note_dialog_edit_ok);

        view.setLayoutParams(new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT));
        mAddDialog = new Dialog(context, R.style.rv_dialog_style);
        mAddDialog.setContentView(view, new ViewGroup.LayoutParams(
                AppDisplay.getUITextEditDialogWidth(), ViewGroup.LayoutParams.WRAP_CONTENT));
        mET_Content.setMaxLines(10);

        mAddDialog.getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
        mAddDialog.getWindow().setBackgroundDrawableResource(R.drawable.dlg_title_bg_4circle_corner_white);
        mAddDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                if (SystemUiHelper.getInstance().isFullScreen()){
                    if (AppDisplay.isPad())
                        SystemUiHelper.getInstance().hideSystemUI(context);
                    else
                        SystemUiHelper.getInstance().hideStatusBar(context);
                }
            }
        });
        mDialog_title.setText(mContext.getApplicationContext().getString(R.string.fx_string_note));

        mCancel.setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                mAddDialog.dismiss();
                AppUtil.dismissInputSoft(mET_Content);
            }
        });
        mSave.setOnClickListener(new View.OnClickListener() {

            @Override
            public void onClick(View v) {
                PointF pdfPoint = new PointF(point.x, point.y);
//                final RectF rect = new RectF(pdfPoint.x - 10, pdfPoint.y + 10, pdfPoint.x + 10, pdfPoint.y - 10);
                final RectF rect = new RectF(pdfPoint.x, pdfPoint.y, pdfPoint.x + 20, pdfPoint.y - 20);

                try {
                    final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
                    final Annot annot = AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Note, AppUtil.toFxRectF(rect)), Annot.e_Note);
                    if (annot == null) {
                        if (!mIsContinuousCreate) {
                            mUiExtensionsManager.setCurrentToolHandler(null);
                        }
                        mAddDialog.dismiss();
                        return;
                    }

                    final NoteAddUndoItem undoItem = new NoteAddUndoItem(mPdfViewCtrl);
                    undoItem.mPageIndex = pageIndex;
                    undoItem.mNM = AppDmUtil.randomUUID(null);
                    undoItem.mContents = mET_Content.getText().toString();
                    undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
                    undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
                    undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
                    undoItem.mFlags = Annot.e_FlagPrint | Annot.e_FlagNoZoom | Annot.e_FlagNoRotate;
                    undoItem.mColor = mColor;
                    undoItem.mOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
                    undoItem.mOpenStatus = false;
                    undoItem.mIconName = NoteUtil.getIconNameByType(mIconType);
                    undoItem.mBBox = new RectF(rect);
                    undoItem.mSubject = "Note";

                    addAnnot(pageIndex, annot, undoItem, true, null);

                    mAddDialog.dismiss();
                    AppUtil.dismissInputSoft(mET_Content);

                    if (!mIsContinuousCreate) {
                        mUiExtensionsManager.setCurrentToolHandler(null);
                    }
                } catch (PDFException e) {
                    e.printStackTrace();
                }
            }
        });
        mAddDialog.show();

        AppUtil.showSoftInput(mET_Content);
    }

    protected void addAnnot(int pageIndex, NoteAnnotContent content, boolean addUndo, Event.Callback result) {
        if (content.getFromType() != null && content.getFromType().equals(Module.MODULE_NAME_SELECTION)) {
            PointF point = new PointF(content.getBBox().left, content.getBBox().top);
//            mPdfViewCtrl.convertPdfPtToPageViewPt(point, point, pageIndex);
            initDialog(pageIndex, point);// use pdf point now.
        } else {
            try {
                NoteAddUndoItem undoItem = new NoteAddUndoItem(mPdfViewCtrl);
                undoItem.setCurrentValue(content);
                undoItem.mFlags = Annot.e_FlagPrint | Annot.e_FlagNoZoom | Annot.e_FlagNoRotate;
                undoItem.mOpenStatus = false;
                undoItem.mIconName = content.getIcon();
                if (content.getModifiedDate() != null) {
                    undoItem.mCreationDate = content.getModifiedDate();
                } else {
                    undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
                }
                final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
                if (content.getFromType().equals(Module.MODULE_NAME_REPLY)) {
                    undoItem.mIsFromReplyModule = true;
                    undoItem.mParentNM = content.getParentNM();

                    Annot parent = mUiExtensionsManager.getDocumentManager().getAnnot(page, undoItem.mParentNM);
                    Note reply = ((Markup) parent).addReply();
                    addAnnot(pageIndex, reply, undoItem, addUndo, result);
                } else {
                    final Annot annot = AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Note, AppUtil.toFxRectF(content.getBBox())), Annot.e_Note);
                    addAnnot(pageIndex, annot, undoItem, addUndo, result);
                }

            } catch (Exception e) {
                e.printStackTrace();
                if (result != null) {
                    result.result(null, false);
                }
            }
        }
    }

    protected void addAnnot(final int pageIndex, final Annot annot, final NoteAddUndoItem undoItem, final boolean addUndo, final Event.Callback result) {
        NoteEvent event = new NoteEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, (Note) annot, mPdfViewCtrl);

        final EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (success) {
                    try {
                        mUiExtensionsManager.getDocumentManager().onAnnotAdded(annot.getPage(), annot);
                        if (addUndo) {
                            mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                        }

                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            Matrix matrix = mPdfViewCtrl.getDisplayMatrix(pageIndex);
                            final RectF pageViewRect = AppUtil.toRectF(annot.getDeviceRect(AppUtil.toMatrix2D(matrix)));
                            Rect rectResult = new Rect();
                            pageViewRect.roundOut(rectResult);
                            rectResult.inset(-10, -10);
                            mPdfViewCtrl.refresh(pageIndex, rectResult);
                        }
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }

                if (result != null) {
                    result.result(event, success);
                }
            }
        });
        mPdfViewCtrl.addTask(task);
    }

    protected void setColor(int color) {
        mColor = color;
        setToolItemColor(color);
    }

    private void setToolItemColor(int color) {
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
    }

    protected void setOpacity(int opacity) {
        mOpacity = opacity;
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
    }

    protected void setIconType(int iconType) {
        mIconType = iconType;
        if (mCurToolItem == null) return;
        mCurToolItem.property.style = iconType;
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_NOTE;
    }

    public String getLastIconName() {
        return NoteUtil.getIconNameByType(mIconType);
    }

    public int getLastColor() {
        return mColor;
    }

    public int getLastOpacity() {
        return mOpacity;
    }

//    private int mLastColor;
//    private int mLastOpacity;
//    private int mLastStyle;
    private IToolSupply mToolSupply;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new NoteToolSupply(mContext);
        return mToolSupply;
    }

    private class NoteToolSupply extends ToolSupplyImpl {

        public NoteToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.comment_tool_note_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return R.drawable.comment_tool_note_src;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = ToolConstants.Note;
            property.color = mColor;
            property.opacity = mOpacity;
            property.style = mIconType;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_NOTE;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if(mUiExtensionsManager.getMainFrame().getCurrentTab()== ToolbarItemConfig.ITEM_COMMENT_TAB) {
                    mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_Note);
                }
//                mLastColor = mColor;
//                mLastOpacity = mOpacity;
//                mLastStyle = mIconType;

                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mColor = property.color;
                mOpacity = property.opacity;
                mIconType = property.style;
                mUiExtensionsManager.setCurrentToolHandler(NoteToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == NoteToolHandler.this) {
//                    mColor = mLastColor;
//                    mOpacity = mLastOpacity;
//                    mIconType = mLastStyle;
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, final PropertyBar.PropertyChangeListener changeListener) {
            mCustomPropertyListener = changeListener;
            mCurToolItem = itemBean;
//            mLastColor = mColor;
//            mLastOpacity = mOpacity;
//            mLastStyle = mIconType;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mColor = property.color;
            mOpacity = property.opacity;
            mIconType = property.style;

            NoteToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
//                    mColor = mLastColor;
//                    mOpacity = mLastOpacity;
//                    mIconType = mLastStyle;
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

}
