/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.multimedia.sound;


import com.foxit.sdk.PDFException;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.fxcrt.FileReaderCallback;
import com.foxit.sdk.pdf.FileSpec;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Sound;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.sdk.pdf.objects.PDFStream;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.Event;

import java.io.BufferedOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.RandomAccessFile;

public class OpenSoundTask extends Task {
    private static final int OPEN_FROM_SOUND = 0;
    private static final int OPEN_FROM_DICT = 1;

    private Annot mAnnot;
    private PDFObject mSoundObj;

    private final String mSavePath;
    private boolean mRet = false;
    private int mOpenType = 0;//0 : open from sound annot; 1: open from dict(current support widget action)

    public OpenSoundTask(Annot annot, String savePath, final Event.Callback callback) {
        super(new CallBack() {
            @Override
            public void result(Task task) {
                callback.result(null, ((OpenSoundTask) task).mRet);
            }
        });

        mSavePath = savePath;
        mAnnot = annot;
        mOpenType = OPEN_FROM_SOUND;
    }

    public OpenSoundTask(PDFObject soundObj, String savePath, final Event.Callback callback) {
        super(new CallBack() {
            @Override
            public void result(Task task) {
                callback.result(null, ((OpenSoundTask) task).mRet);
            }
        });

        mSavePath = savePath;
        mSoundObj = soundObj;
        mOpenType = OPEN_FROM_DICT;
    }

    @Override
    protected void execute() {
        FileOutputStream fileOutputStream = null;
        BufferedOutputStream bufferedOutputStream = null;
        try {
            if (mOpenType == OPEN_FROM_SOUND) {

                if (mAnnot == null || mAnnot.isEmpty() || mAnnot.getType() != Annot.e_Sound) {
                    mRet = false;
                    return;
                }

                Sound sound = new Sound(mAnnot);
                FileSpec fileSpec = sound.getFileSpec();
                if (fileSpec == null || fileSpec.isEmpty()) {
                    mRet = saveSoundToFile(sound, mSavePath);
                } else {
                    FileReaderCallback fileRead = fileSpec.getFileData();
                    if (fileRead == null || fileRead.getSize() == 0) {
                        mRet = false;
                        return;
                    }

                    fileOutputStream = new FileOutputStream(mSavePath);
                    bufferedOutputStream = new BufferedOutputStream(fileOutputStream);
                    int offset = 0;
                    int bufSize = 4 * 1024;
                    long fileSize = fileRead.getSize();
                    byte[] buf;
                    while (true) {
                        if (fileSize < bufSize + offset) {
                            buf = new byte[(int) (fileSize - offset)];
                            fileRead.readBlock(buf, offset, fileSize - offset);
                        } else {
                            buf = new byte[bufSize];
                            fileRead.readBlock(buf, offset, bufSize);
                        }
                        if (buf.length != bufSize) {
                            bufferedOutputStream.write(buf, 0, buf.length);
                            break;
                        } else {
                            bufferedOutputStream.write(buf, 0, bufSize);
                        }
                        offset += bufSize;
                    }
                    bufferedOutputStream.flush();
                    mRet = true;
                }

            } else {
                mRet = saveSoundToFile(mSoundObj, mSavePath);
            }
        } catch (Exception e) {
            e.printStackTrace();
            mRet = false;
        } finally {
            AppFileUtil.closeQuietly(bufferedOutputStream);
            AppFileUtil.closeQuietly(fileOutputStream);
        }
    }

    private boolean saveSoundToFile(Sound sound, String path) {
        try {
            PDFStream stream = sound.getSoundStream();
            int bits = sound.getBits();
            int channelCount = sound.getChannelCount();
            float samplingRate = sound.getSamplingRate();
            int encodingFormat = sound.getSampleEncodingFormat();
            return saveSoundToFile(stream, bits, channelCount, samplingRate, encodingFormat, path);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private boolean saveSoundToFile(PDFObject object, String path) {
        try {
            PDFStream stream = object.getStream();
            if (stream == null) return false;

            PDFDictionary dict = object.getDict();
            if (dict ==null) return  false;

            int bits = 8;
            PDFObject b_obj = dict.getElement("B");
            if (b_obj != null) {
                bits = b_obj.getInteger();
            }

            int channelCount = 1;
            PDFObject c_obj = dict.getElement("C");
            if (c_obj != null) {
                channelCount = c_obj.getInteger();
            }

            float samplingRate = -1.0f;
            PDFObject r_obj = dict.getElement("R");
            if (r_obj != null) {
                samplingRate = r_obj.getFloat();
            }

            int encodingFormat = Sound.e_SampleEncodingFormatRaw;
            PDFObject e_obj = dict.getElement("E");
            if (e_obj != null) {
                String encoding = e_obj.getWideString();
                if ("Signed".equals(encoding)) {
                    encodingFormat = Sound.e_SampleEncodingFormatSigned;
                } else if ("muLaw".equals(encoding)) {
                    encodingFormat = Sound.e_SampleEncodingFormatMuLaw;
                } else if ("ALaw".equals(encoding)) {
                    encodingFormat = Sound.e_SampleEncodingFormatALaw;
                } else {
                    encodingFormat = Sound.e_SampleEncodingFormatRaw;
                }
            }
            return saveSoundToFile(stream, bits, channelCount, samplingRate, encodingFormat, path);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private boolean saveSoundToFile(PDFStream stream, int bits, int channelCount, float samplingRate, int encodingFormat, String path) {
        if (stream == null) return false;

        RandomAccessFile randomAccessFile = null;
        try {
            randomAccessFile = new RandomAccessFile(path, "rw");
            int streamSize = stream.getDataSize(false);
            byte[] data = new byte[streamSize];
            stream.getData(false, streamSize, data);

            short bit = (short) bits;

            int riffSize = streamSize + 36;
            randomAccessFile.write("RIFF".getBytes());
            randomAccessFile.write(intToByteArray(riffSize));
            randomAccessFile.write("WAVEfmt ".getBytes());

            int chunkSize = 16;
            randomAccessFile.write(intToByteArray(chunkSize));

            short format = 1;
            randomAccessFile.write(shortToByteArray(format));

            randomAccessFile.write(shortToByteArray((short) channelCount));

            int rate = (int) samplingRate;
            randomAccessFile.write(intToByteArray(rate));

            int bytePerSec = rate * channelCount * bit / 8;
            randomAccessFile.write(intToByteArray(bytePerSec));

            short blockAlign = (short) (bit * channelCount / 8);
            randomAccessFile.write(shortToByteArray(blockAlign));
            randomAccessFile.write(shortToByteArray(bit));
            randomAccessFile.write("data".getBytes());
            randomAccessFile.write(intToByteArray(streamSize));

            boolean ret = false;
            switch (encodingFormat) {
                case Sound.e_SampleEncodingFormatALaw:
                    break;
                case Sound.e_SampleEncodingFormatMuLaw:
                    break;
                case Sound.e_SampleEncodingFormatSigned:
                    byte[] buffer = new byte[streamSize + 1];
                    int j = 0, k = 0;
                    for (int i = 0; i < streamSize; i += 2) {
                        byte low = data[j++];
                        byte high;
                        if (j == streamSize) {
                            high = 0;
                        } else {
                            high = data[j++];
                        }

                        buffer[k++] = high;
                        buffer[k++] = low;
                    }
                    randomAccessFile.write(buffer, 0, streamSize);
                    ret = true;
                    break;
                case Sound.e_SampleEncodingFormatRaw:
                default:
                    randomAccessFile.write(data, 0, streamSize);
                    ret = true;
                    break;
            }

            randomAccessFile.close();
            return ret;
        } catch (PDFException | IOException e) {
            e.printStackTrace();
        } finally {
            AppFileUtil.closeQuietly(randomAccessFile);
        }
        return false;
    }

    private static byte[] intToByteArray(int value) {
        return new byte[]{
                (byte) (value & 0xFF),
                (byte) ((value >> 8) & 0xFF),
                (byte) ((value >> 16) & 0xFF),
                (byte) ((value >> 24) & 0xFF)
        };
    }

    private static byte[] shortToByteArray(short value) {
        return new byte[]{
                (byte) (value & 0xFF),
                (byte) ((value >> 8) & 0xFF)
        };
    }

}
