/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.multimedia.screen.multimedia;


import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.net.Uri;
import android.os.Build;
import android.provider.MediaStore;
import android.text.TextUtils;
import android.view.MotionEvent;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Screen;
import com.foxit.uiextensions.ISystemPermissionProvider;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.multimedia.MultimediaUtil;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFileSelectDialog;
import com.foxit.uiextensions.controls.propertybar.AnnotMenu;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.pdfreader.config.ActRequestCode;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppIntentUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.List;

import androidx.core.content.ContextCompat;
import androidx.fragment.app.FragmentActivity;

public class MultimediaToolHandler implements ToolHandler {

    private final Context mContext;
    private final PDFViewCtrl mPdfViewCtrl;
    private final UIExtensionsManager mUiExtensionsManager;
    private final MultimediaUtil mMultimediaUtil;
    private List<String> mSupportMultimediaList;

    private boolean mIsContinue;
    private final String mIntent;

    private final AnnotMenu mAnnotMenu;
    private final ArrayList<Integer> mMenuText;
    private UIFileSelectDialog mFileSelectDialog;
    private String mCaptureVideoPath;

    public MultimediaToolHandler(Context context, PDFViewCtrl pdfViewCtrl, final String intent) {
        mContext = context;
        mIntent = intent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
        mMultimediaUtil = new MultimediaUtil(mContext);

        mMenuText = new ArrayList<>();
        mAnnotMenu = new AnnotMenuImpl(mContext, pdfViewCtrl);

        mCaptureVideoPath = AppFileUtil.getAppCacheDir(mContext) + "/Camera/screen_capture_video.mp4";
    }

    @Override
    public String getType() {
        return mIntent;
    }

    @Override
    public void onActivate() {
        resetAnnotMenu();
    }

    private void resetAnnotMenu() {
        mMenuText.clear();
        if (ToolHandler.TH_TYPE_SCREEN_AUDIO.equals(mIntent)) {
            mMenuText.add(AnnotMenu.AM_BT_CHOOSE_AUDIO);
            mMenuText.add(AnnotMenu.AM_BT_RECORD_AUDIO);
        } else {
            mMenuText.add(AnnotMenu.AM_BT_FILE);
            mMenuText.add(AnnotMenu.AM_BT_GALERY);
            mMenuText.add(AnnotMenu.AM_BT_RECORD_VIDEO);
        }
        mAnnotMenu.setMenuItems(mMenuText);
        mAnnotMenu.setListener(new AnnotMenu.ClickListener() {
            @Override
            public void onAMClick(int btType) {
                if (btType == AnnotMenu.AM_BT_CHOOSE_AUDIO) {
                    showSelectFileDialog();
                } else if (btType == AnnotMenu.AM_BT_RECORD_AUDIO) {
                    startRecordAudio();
                } else if (btType == AnnotMenu.AM_BT_FILE) {
                    showSelectFileDialog();
                } else if (btType == AnnotMenu.AM_BT_GALERY) {
                    Intent intent = new Intent(Intent.ACTION_PICK);
                    intent.setDataAndType(MediaStore.Video.Media.EXTERNAL_CONTENT_URI, "video/*");
                    mUiExtensionsManager.getAttachedActivity().startActivityForResult(intent, ActRequestCode.REQ_FILE_FROM_BROWSER);
                } else if (btType == AnnotMenu.AM_BT_RECORD_VIDEO) {
                    startRecordVideo();
                }
                mAnnotMenu.dismiss();
            }
        });
    }

    @Override
    public void onDeactivate() {
        if (mAnnotMenu.isShowing())
            mAnnotMenu.dismiss();
        mCreatePoint.set(0, 0);
        mCreatePageIndex = -1;
    }

    private void showSelectFileDialog() {
        mFileSelectDialog = getFileSelectDialog();
        mFileSelectDialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
                mFileSelectDialog.dismiss();
            }

            @Override
            public void onTitleRightButtonClick() {
                if (mFileSelectDialog.getSelectedFiles().size() > 0) {
                    createAnnot(mFileSelectDialog.getSelectedFiles().get(0).path);
                    mFileSelectDialog.clearCheckedItems();
                }
                mFileSelectDialog.dismiss();
            }
        });
        mFileSelectDialog.resetWH();
        mFileSelectDialog.showDialog();
    }

    private UIFileSelectDialog getFileSelectDialog() {
        if (mFileSelectDialog == null) {
            mFileSelectDialog = new UIFileSelectDialog(mUiExtensionsManager.getAttachedActivity());
            mFileSelectDialog.setCanceledOnTouchOutside(true);
        } else {
            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    mFileSelectDialog.notifyDataSetChanged();
                }
            });
        }
        mFileSelectDialog.init(new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                boolean isAcceptType;
                if (ToolHandler.TH_TYPE_SCREEN_AUDIO.equals(mIntent)) {
                    isAcceptType = mMultimediaUtil.isAudio(pathname);
                } else {
                    isAcceptType = mMultimediaUtil.isVideo(pathname);
                }
                return !(pathname.isHidden() || !AppFileUtil.canRead(pathname))
                        && !(pathname.isFile() && !isAcceptType);
            }
        }, true);
        return mFileSelectDialog;

    }

    private void startRecordAudio() {
        if (Build.VERSION.SDK_INT >= 23) {
            int permission = ContextCompat.checkSelfPermission(mContext, android.Manifest.permission.RECORD_AUDIO);
            if (permission != PackageManager.PERMISSION_GRANTED) {
                if (mUiExtensionsManager.getSystemPermissionProvider() == null) {
                    mUiExtensionsManager.getAttachedActivity().requestPermissions(new String[]{android.Manifest.permission.RECORD_AUDIO},
                            ActRequestCode.REQ_RECORD_AUDIO_PERMISSION);
                } else {
                    mUiExtensionsManager.getSystemPermissionProvider().requestPermission(
                            true,
                            AppResource.getString(mContext, R.string.record),
                            new String[]{android.Manifest.permission.RECORD_AUDIO},
                            new ISystemPermissionProvider.IPermissionResult() {
                                @Override
                                public void onResult(boolean allowed) {
                                    if (allowed)
                                        mUiExtensionsManager.getAttachedActivity().requestPermissions(new String[]{android.Manifest.permission.RECORD_AUDIO},
                                                ActRequestCode.REQ_RECORD_AUDIO_PERMISSION);
                                }
                            });
                }
                return;
            }
        }

        FragmentActivity act = (FragmentActivity) mUiExtensionsManager.getAttachedActivity();
        AudioRecordFragment recordFragment = (AudioRecordFragment) act.getSupportFragmentManager().findFragmentByTag("AudioRecordFragment");
        if (recordFragment == null) {
            recordFragment = new AudioRecordFragment();
            recordFragment.init(mPdfViewCtrl);
            recordFragment.setOnPickPicListener(new IPickResultListener() {
                @Override
                public void onResult(boolean isSuccess, String path) {
                    if (isSuccess) {
                        createAnnot(path);
                    }
                }
            });
        }
        AppDialogManager.getInstance().showAllowManager(recordFragment, act.getSupportFragmentManager(),
                "AudioRecordFragment", null);
    }

    private void startRecordVideo() {
        AppIntentUtil.selectVideoFromCamera(mUiExtensionsManager,
                mCaptureVideoPath, ActRequestCode.REQ_RECORD_VIDEO_PERMISSION, ActRequestCode.REQ_FILE_FROM_VIDEO);
    }

    private void createAnnot(String path) {
        if (!TextUtils.isEmpty(path)) {
            String mimeType = AppFileUtil.getMimeType(path);
            int lastIndex = path.lastIndexOf('.');
            String exp = lastIndex >= 0 ? path.substring(lastIndex).toLowerCase() : "";
            if (mimeType == null) {
                mimeType = AppIntentUtil.getMIMEType(exp);
            }

            if (ToolHandler.TH_TYPE_SCREEN_AUDIO.equals(mIntent)) {
                mSupportMultimediaList = mMultimediaUtil.getAudioSupportMimeList();
            } else {
                // In some phones, this type will return to video/ext-mpeg.
                if (exp.equals(".mpeg")) {
                    mimeType = "video/mpeg";
                }
                mSupportMultimediaList = mMultimediaUtil.getVideoSupportMimeList();
            }

            if (mSupportMultimediaList.contains(mimeType)) {
                PointF pdfPoint = new PointF(mCreatePoint.x, mCreatePoint.y);
                createAnnot(pdfPoint, mCreatePageIndex, path, mimeType);
            } else {
                UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.multimedia_type_not_support));
            }
        }
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        return mUiExtensionsManager.defaultTouchEvent(pageIndex, motionEvent);
    }

    public void dismissFileDialog() {
        if (mFileSelectDialog != null && mFileSelectDialog.isShowing()) {
            mFileSelectDialog.dismiss();
            mFileSelectDialog = null;
        }
    }

    private void createAnnot(PointF pdfPointF, int pageIndex, String path, String mimeType) {
        RectF viewRectF;
        Bitmap bmp;
        String tempPath = AppFileUtil.saveToScopedCache(path);
        if (tempPath == null) return;

        if (ToolHandler.TH_TYPE_SCREEN_AUDIO.equals(mIntent)) {
            bmp = AppResource.getBitmapFromDrawable(mContext, R.drawable.ic_audio);
            viewRectF = getImageRectOnPageView(pdfPointF, pageIndex, bmp.getWidth() / 4.0f, bmp.getHeight() / 4.0f);
        } else {
            bmp = mMultimediaUtil.getVideoThumbnail(mPdfViewCtrl, tempPath);
            viewRectF = getImageRectOnPageView(pdfPointF, pageIndex, bmp.getWidth() / 2f, bmp.getHeight() / 2f);
        }

        RectF pdfRect = new RectF();
        mPdfViewCtrl.convertPageViewRectToPdfRect(viewRectF, pdfRect, pageIndex);
        createAnnot(pageIndex, pdfRect, tempPath, mimeType, bmp, null);
    }

    public void createAnnot(final int pageIndex, final RectF annotRectF, String path, Bitmap thumbnail, Event.Callback callback) {
        String mimeType = AppFileUtil.getMimeType(path);
        int lastIndex = path.lastIndexOf('.');
        String exp = lastIndex >= 0 ? path.substring(lastIndex).toLowerCase() : "";
        if (mimeType == null) {
            mimeType = AppIntentUtil.getMIMEType(exp);
        }

        if (ToolHandler.TH_TYPE_SCREEN_AUDIO.equals(mIntent)) {
            mSupportMultimediaList = mMultimediaUtil.getAudioSupportMimeList();
        } else {
            // In some phones, this type will return to video/ext-mpeg.
            if (exp.equals(".mpeg")) {
                mimeType = "video/mpeg";
            }
            mSupportMultimediaList = mMultimediaUtil.getVideoSupportMimeList();
        }

        if (mSupportMultimediaList.contains(mimeType)) {
            createAnnot(pageIndex, annotRectF, path, mimeType, thumbnail, callback);
        } else {
            UIToast.getInstance(mContext).show(mContext.getApplicationContext().getString(R.string.multimedia_type_not_support));
        }
    }

    private void createAnnot(final int pageIndex, final RectF annotRectF, String path, String mimeType, Bitmap thumbnail, final Event.Callback callback) {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            final Screen annot = (Screen) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Screen, AppUtil.toFxRectF(annotRectF)), Annot.e_Screen);

            final MultimediaAddUndoItem undoItem = new MultimediaAddUndoItem(mPdfViewCtrl);
            undoItem.mPageIndex = pageIndex;
            undoItem.mNM = AppDmUtil.randomUUID(null);
            undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
            undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mFlags = Annot.e_FlagPrint;
            undoItem.mFileName = AppFileUtil.getFileName(path);
            undoItem.mFilePath = path;
            undoItem.mMediaClipContentType = mimeType;
            undoItem.mPreviewBitmap = thumbnail;
            undoItem.mContents = AppFileUtil.getFileName(path);
            undoItem.mBBox = new RectF(annotRectF);
            int rotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
            undoItem.mRotation = rotation == 0 ? rotation : 4 - rotation;

            MultimediaEvent event = new MultimediaEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, annot);
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(undoItem);
                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            RectF rectF = new RectF();
                            rectF.set(annotRectF);
                            mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
                            Rect rect = new Rect();
                            rectF.roundOut(rect);
                            mPdfViewCtrl.refresh(pageIndex, rect);
                        }
                        if (!mIsContinue)
                            mUiExtensionsManager.setCurrentToolHandler(null);

                        if (callback != null)
                            callback.result(event, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            if (callback != null)
                callback.result(null, false);
        }
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
        onMultimediaToolTouch(pageIndex, motionEvent);
        return true;
    }

    private PointF mCreatePoint = new PointF();
    private int mCreatePageIndex = -1;

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        int action = motionEvent.getActionMasked();
        boolean handled = mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!handled) {
            if (action == MotionEvent.ACTION_DOWN) {
                handled = true;
                onMultimediaToolTouch(pageIndex, motionEvent);
            }
        } else {
            if (mAnnotMenu.isShowing())
                mAnnotMenu.dismiss();
            mCreatePageIndex = -1;
            mCreatePoint.set(0, 0);
        }
        return handled;
    }

    private void onMultimediaToolTouch(int pageIndex, MotionEvent motionEvent) {
        PointF devPoint = new PointF(motionEvent.getX(), motionEvent.getY());
        RectF rectF = new RectF(devPoint.x, devPoint.y, devPoint.x, devPoint.y);
        rectF.inset(-5, -5);
        if (mAnnotMenu.isShowing())
            mAnnotMenu.update(rectF);
        else
            mAnnotMenu.show(rectF);

        mCreatePageIndex = pageIndex;

        mCreatePoint = AppAnnotUtil.getPdfPoint(mPdfViewCtrl, pageIndex, motionEvent);
    }

    public void AI_addMultimedia(int pageIndex, PointF devPoint) {
        RectF rectF = new RectF(devPoint.x, devPoint.y, devPoint.x, devPoint.y);
        rectF.inset(-5, -5);
        if (mAnnotMenu.isShowing())
            mAnnotMenu.update(rectF);
        else
            mAnnotMenu.show(rectF);

        mCreatePageIndex = pageIndex;
        PointF tmpPoint = new PointF(devPoint.x, devPoint.y);
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(tmpPoint, tmpPoint, pageIndex);
        mPdfViewCtrl.convertPageViewPtToPdfPt(tmpPoint, mCreatePoint, pageIndex);
    }


    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinue;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinue = continueAddAnnot;
    }

    private RectF getImageRectOnPageView(PointF pdfPoint, int pageIndex, float offsetX, float offsetY) {
        RectF docRect = new RectF(pdfPoint.x, pdfPoint.y, pdfPoint.x, pdfPoint.y);
        docRect.inset(-offsetX, -offsetY);
        RectF pageViewRect = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(docRect, pageViewRect, pageIndex);

        if (pageViewRect.left < 0) {
            pageViewRect.offset(-pageViewRect.left, 0);
        }
        if (pageViewRect.right > mPdfViewCtrl.getPageViewWidth(pageIndex)) {
            pageViewRect.offset(mPdfViewCtrl.getPageViewWidth(pageIndex) - pageViewRect.right, 0);
        }
        if (pageViewRect.top < 0) {
            pageViewRect.offset(0, -pageViewRect.top);
        }
        if (pageViewRect.bottom > mPdfViewCtrl.getPageViewHeight(pageIndex)) {
            pageViewRect.offset(0, mPdfViewCtrl.getPageViewHeight(pageIndex) - pageViewRect.bottom);
        }
        return pageViewRect;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
    }

    private IToolSupply mToolSupply;

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new MultimediaToolSupply(mContext);
        return mToolSupply;
    }

    private class MultimediaToolSupply extends ToolSupplyImpl {

        public MultimediaToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            int bgIcon = 0;
            if (ToolConstants.Audio == toolType) {
                bgIcon = R.drawable.edit_tool_audio;
            } else if (ToolConstants.Video == toolType) {
                bgIcon = R.drawable.edit_tool_video;
            }
            return bgIcon;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            return null;
        }

        @Override
        public String getToolName(int toolType) {
            return ToolHandler.TH_TYPE_SCREEN_AUDIO.equals(mIntent) ? JsonConstants.TYPE_AUDIO : JsonConstants.TYPE_VIDEO;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            if (itemBean.toolItem.isSelected()) {
                if (mUiExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_EDIT_TAB) {
                    if (itemBean.type == ToolConstants.Audio) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_EditBar_Audio);
                    } else if (itemBean.type == ToolConstants.Video) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_EditBar_Video);
                    }
                }

                showToastForSelectedItem(itemBean.type);
                mUiExtensionsManager.setCurrentToolHandler(MultimediaToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == MultimediaToolHandler.this) {
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        private void showToastForSelectedItem(int type) {
            if (type == ToolConstants.Audio || type == ToolConstants.Video) {
                setContinueAddAnnot(false);
            } else {
                setContinueAddAnnot(mUiExtensionsManager.getConfig().uiSettings.annotations.continuouslyAdd);
            }
            String content = AppResource.getString(mContext, type != ToolConstants.Video ? R.string.edit_add_audio_toast_content : R.string.edit_add_video_toast_content);
            UIToast.getInstance(mContext).show(content);
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
        }

        @Override
        public PropertyBar getPropertyBar() {
            return null;
        }

    }

    public void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
        if (resultCode == Activity.RESULT_OK) {
            if (requestCode == ActRequestCode.REQ_FILE_FROM_BROWSER) {
                Uri uri = data.getData();
                String path = AppFileUtil.getFilePathFromUri(mContext, uri);
                createAnnot(path);
            } else if (requestCode == ActRequestCode.REQ_FILE_FROM_VIDEO) {
                String path = mCaptureVideoPath;
                path = AppFileUtil.saveToScopedCache(path);

                if (!AppUtil.isEmpty(path)) {
                    //pdf is not support .mp4 ,so change the format of video
                    File file = new File(path);
                    int index = path.lastIndexOf('.');
                    String newPath = path.substring(0, index) + ".avi";
                    file.renameTo(new File(newPath));
                    createAnnot(newPath);
                }
            }
        }
    }

    public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        if (verifyPermissions(grantResults)) {
            if (requestCode == ActRequestCode.REQ_RECORD_AUDIO_PERMISSION) {
                startRecordAudio();
            } else if (requestCode == ActRequestCode.REQ_RECORD_VIDEO_PERMISSION) {
                startRecordVideo();
            }
        } else {
            if (mAnnotMenu.isShowing())
                mAnnotMenu.dismiss();
            mCreatePoint.set(0, 0);
            mCreatePageIndex = -1;
            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_permission_denied));
        }
    }

    private boolean verifyPermissions(int[] grantResults) {
        for (int grantResult : grantResults) {
            if (grantResult != PackageManager.PERMISSION_GRANTED) {
                return false;
            }
        }
        return true;
    }

    public void onConfigurationChanged(Configuration newConfig) {
        if (mFileSelectDialog != null && mFileSelectDialog.isShowing()) {
            mFileSelectDialog.resetWH();
            mFileSelectDialog.showDialog();
        }
    }

    void onDrawForControls(int pageIndex, Canvas canvas) {
        if (mCreatePageIndex != -1 && mPdfViewCtrl.isPageVisible(mCreatePageIndex)) {
            if (mCreatePoint.length() > 0) {
                PointF devPoint = new PointF();
                mPdfViewCtrl.convertPdfPtToPageViewPt(mCreatePoint, devPoint, mCreatePageIndex);
                mPdfViewCtrl.convertPageViewPtToDisplayViewPt(devPoint, devPoint, mCreatePageIndex);
                RectF rectF = new RectF(devPoint.x, devPoint.y, devPoint.x, devPoint.y);
                rectF.inset(-5, -5);
                mAnnotMenu.update(rectF);
            }
        }
    }

    interface IPickResultListener {
        void onResult(boolean isSuccess, String path);
    }

}
