/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.multimedia.screen.multimedia;


import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.res.Configuration;
import android.graphics.Canvas;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.uiextensions.IPolicyEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.config.modules.annotations.AnnotationsConfig;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppStorageManager;

import java.io.File;

public class MultimediaModule implements Module {

    private final Context mContext;
    private final PDFViewCtrl mPdfViewCtrl;
    private final PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    private MultimediaToolHandler mAudioToolHandler;
    private MultimediaToolHandler mVideoToolHandler;
    private MultimediaAnnotHandler mAnnotHandler;

    public MultimediaModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_MEDIA;
    }

    @Override
    public boolean loadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;

            Config config = uiExtensionsManager.getConfig();
            AnnotationsConfig annotConfig = config.modules.annotations;

            //audio
            if (annotConfig.isLoadAudio) {
                mAudioToolHandler = new MultimediaToolHandler(mContext, mPdfViewCtrl, ToolHandler.TH_TYPE_SCREEN_AUDIO);
                uiExtensionsManager.registerToolHandler(mAudioToolHandler);

                uiExtensionsManager.getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_EDIT, ToolConstants.Audio, mAudioToolHandler.getToolSupply());
            }

            //video
            if (annotConfig.isLoadVideo) {
                mVideoToolHandler = new MultimediaToolHandler(mContext, mPdfViewCtrl, ToolHandler.TH_TYPE_SCREEN_VIDEO);
                uiExtensionsManager.registerToolHandler(mVideoToolHandler);

                uiExtensionsManager.getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_EDIT, ToolConstants.Video, mVideoToolHandler.getToolSupply());
            }

            mAnnotHandler = new MultimediaAnnotHandler(mContext, mPdfViewCtrl);
            uiExtensionsManager.registerAnnotHandler(mAnnotHandler);
            uiExtensionsManager.registerLifecycleListener(mLifecycleEventListener);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
            uiExtensionsManager.registerPolicyEventListener(mPolicyEventListener);
            uiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
            uiExtensionsManager.registerModule(this);
        }

        mPdfViewCtrl.registerRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.registerDocEventListener(mDocEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;

            if (mAudioToolHandler != null) {
                uiExtensionsManager.unregisterToolHandler(mAudioToolHandler);
            }
            if (mVideoToolHandler != null) {
                uiExtensionsManager.unregisterToolHandler(mVideoToolHandler);
            }
            uiExtensionsManager.unregisterLifecycleListener(mLifecycleEventListener);
            uiExtensionsManager.unregisterAnnotHandler(mAnnotHandler);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
            uiExtensionsManager.unregisterPolicyEventListener(mPolicyEventListener);
            uiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
        }

        mPdfViewCtrl.unregisterRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        return true;
    }

    private final PDFViewCtrl.IRecoveryEventListener memoryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            if (mAnnotHandler.getAnnotMenu() != null && mAnnotHandler.getAnnotMenu().isShowing()) {
                mAnnotHandler.getAnnotMenu().dismiss();
            }
        }

        @Override
        public void onRecovered() {
        }
    };

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    private final PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            mAnnotHandler.release();
            if (AppFileUtil.needScopedStorageAdaptation()) {
                String tempPath = AppStorageManager.getInstance(mContext).getScopedCacheDir() + "/FoxitSDK/AttaTmp/multimedia/";
                File tempFile = new File(tempPath);
                if (tempFile.exists()) {
                    AppFileUtil.deleteFolder(tempFile, false);
                }
            }
        }

    };

    private final ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {

        @Override
        public void onDestroy(Activity act) {
            mAnnotHandler.release();
        }

        @Override
        public void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
            ToolHandler curToolHandler = ((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler();
            if (curToolHandler instanceof MultimediaToolHandler) {
                ((MultimediaToolHandler) curToolHandler).onActivityResult(act, requestCode, resultCode, data);
            }
        }
    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mAudioToolHandler != null) {
                mAudioToolHandler.dismissFileDialog();
            }
            if (mVideoToolHandler != null) {
                mVideoToolHandler.dismissFileDialog();
            }
        }
    };

    private final IPolicyEventListener mPolicyEventListener = new IPolicyEventListener() {
        @Override
        public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
            ToolHandler curToolHandler = ((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler();
            if (curToolHandler instanceof MultimediaToolHandler) {
                ((MultimediaToolHandler) curToolHandler).onRequestPermissionsResult(requestCode, permissions, grantResults);
            }
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            ToolHandler curToolHandler = ((UIExtensionsManager) mUiExtensionsManager).getCurrentToolHandler();
            if (curToolHandler instanceof MultimediaToolHandler) {
                ((MultimediaToolHandler) curToolHandler).onConfigurationChanged(newConfig);
            }
        }
    };

    public MultimediaToolHandler getAudioToolHandler() {
        return mAudioToolHandler;
    }

    public MultimediaToolHandler getVideoToolHandler() {
        return mVideoToolHandler;
    }

}
