/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.multimedia.screen.image;


import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.media.ExifInterface;
import android.view.MotionEvent;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Screen;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFileSelectDialog;
import com.foxit.uiextensions.controls.propertybar.AnnotMenu;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.pdfreader.config.ActRequestCode;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppIntentUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.ArrayList;

public class PDFImageToolHandler implements ToolHandler {
    private final PDFImageInfo mImageInfo;

    private final RectF mImagePdfRect;
    private int mLastPageIndex = -1;
    private boolean mTouchCaptured = false;
    private boolean mIsContinuousCreate = true;

    private final Context mContext;
    private final PDFViewCtrl mPdfViewCtrl;
    private final UIExtensionsManager mUiExtensionsManager;

    private final PropertyBar mPropertyBar;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;

    private AnnotMenu mAnnotMenu;
    private final ArrayList<Integer> mMenuText;
    private final String mCaptureImgPath;
    private UIFileSelectDialog mFileSelectDialog = null;

    public PDFImageToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mImageInfo = new PDFImageInfo();
        mPdfViewCtrl = pdfViewCtrl;
        mContext = context;
        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mPropertyBar = mUiExtensionsManager.getMainFrame().getPropertyBar();
        mDefaultBoundaryOffset = AppDisplay.dp2px(10);

        mImagePdfRect = new RectF();
        mMenuText = new ArrayList<>();
        mCaptureImgPath = AppFileUtil.getAppCacheDir(context) + "/camera_photos/form_capture_img";
    }

    @Override
    public String getType() {
        return TH_TYPE_PDFIMAGE;
    }

    @Override
    public void onActivate() {
        resetAnnotMenu();
        resetPropertyBar();
    }

    protected void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    protected void removePropertyBarListener() {
        mPropertyChangeListener = null;
    }

    private void resetAnnotMenu() {
        mMenuText.clear();
        mMenuText.add(AnnotMenu.AM_BT_FILE);
        mMenuText.add(AnnotMenu.AM_BT_GALERY);
        if (AppDevice.hasCamera(mUiExtensionsManager.getAttachedActivity())) {
            mMenuText.add(AnnotMenu.AM_BT_CAMERA);
        }
        mAnnotMenu.setMenuItems(mMenuText);
        mAnnotMenu.setListener(new AnnotMenu.ClickListener() {
            @Override
            public void onAMClick(int btType) {
                if (btType == AnnotMenu.AM_BT_GALERY) {
                    AppIntentUtil.selectImageFromGallery(mUiExtensionsManager.getAttachedActivity(), ActRequestCode.REQ_FILE_FROM_GALLERY);
                } else if (btType == AnnotMenu.AM_BT_CAMERA) {
                    AppIntentUtil.selectImageFromCamera(mUiExtensionsManager,
                            mCaptureImgPath, ActRequestCode.REQ_CAMERA_PERMISSION, ActRequestCode.REQ_FILE_FROM_CAMERA);
                } else if (btType == AnnotMenu.AM_BT_FILE) {
                    showSelectFileDialog();
                }
                mAnnotMenu.dismiss();
            }
        });
    }

    void resetPropertyBar() {
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, getImageInfo().getOpacity());
        int rotation = getImageInfo().getRotation(); //== 0 ? 0 : 4 - getImageInfo().getRotation();
        mPropertyBar.setProperty(PropertyBar.PROPERTY_ROTATION, rotation);
        mPropertyBar.setArrowVisible(true);
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.reset(getSupportedProperties());
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
    }

    private long getSupportedProperties() {
        return PropertyBar.PROPERTY_OPACITY
                | PropertyBar.PROPERTY_ROTATION;
    }

    @Override
    public void onDeactivate() {
        if (mAnnotMenu.isShowing()) {
            mAnnotMenu.dismiss();
        }

        setEmptyImageRect(mLastPageIndex);
        mLastDownPdfPoint.set(0, 0);
        mLastPageIndex = -1;
    }

    @Override
    public boolean onTouchEvent(final int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultTouchEvent(pageIndex, motionEvent);
        if (!handled && motionEvent.getActionMasked() != MotionEvent.ACTION_DOWN) {
            handled = onPDFImageToolTouch(pageIndex, motionEvent);
        }
        return handled;
    }

    private final PointF mDownPdfPoint = new PointF(0, 0);
    private final PointF mLastDownPdfPoint = new PointF(0, 0);
    private int mPageViewWidth;
    private int mPageViewHeight;
    private final int mDefaultBoundaryOffset;

    private boolean onPDFImageToolTouch(final int pageIndex, MotionEvent motionEvent) {
        PointF disPoint = new PointF(motionEvent.getX(), motionEvent.getY());
        final PointF pvPoint = new PointF();
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(disPoint, pvPoint, pageIndex);
        final PointF pdfPoint = new PointF();
        mPdfViewCtrl.convertPageViewPtToPdfPt(pvPoint, pdfPoint, pageIndex);

        int action = motionEvent.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (mAnnotMenu.isShowing())
                    mAnnotMenu.dismiss();

                mPageViewWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
                mPageViewHeight = mPdfViewCtrl.getPageViewHeight(pageIndex);
                mDownPdfPoint.set(pdfPoint.x, pdfPoint.y);
                if (!mTouchCaptured) {
                    mTouchCaptured = true;
                }
                mLastPageIndex = pageIndex;
                setEmptyImageRect(pageIndex);
                break;
            case MotionEvent.ACTION_MOVE:
                if (!mTouchCaptured) break;
                mImagePdfRect.set(mDownPdfPoint.x, mDownPdfPoint.y, pdfPoint.x, pdfPoint.y);
                AppUtil.normalizePDFRect(mImagePdfRect);
                adjustImageRect(pageIndex, mImagePdfRect);

                RectF invalidRect = new RectF();
                mPdfViewCtrl.convertPdfRectToPageViewRect(mImagePdfRect, invalidRect, pageIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(invalidRect, invalidRect, pageIndex);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(invalidRect));
                break;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (!mLastDownPdfPoint.equals(pdfPoint)) {
                    mLastDownPdfPoint.set(pdfPoint);
                }

                RectF viewRectF = new RectF();
                if (AppUtil.isEmptyPDFRect(mImagePdfRect)) {
                    PointF viewPoint = new PointF();
                    mPdfViewCtrl.convertPdfPtToPageViewPt(mLastDownPdfPoint, viewPoint, pageIndex);
                    viewRectF.set(viewPoint.x, viewPoint.y, viewPoint.x + 10, viewPoint.y + 10);
                } else {
                    mPdfViewCtrl.convertPdfRectToPageViewRect(mImagePdfRect, viewRectF, pageIndex);
                }
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(viewRectF, viewRectF, pageIndex);
                if (mAnnotMenu.isShowing()) {
                    mAnnotMenu.update(viewRectF);
                } else {
                    mAnnotMenu.show(viewRectF);
                }

                mDownPdfPoint.set(0, 0);
                break;
            default:
                break;
        }
        return true;
    }

    public void AI_addImage(RectF rect, int pageIndex) {
        RectF disRect = new RectF();
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rect, disRect, pageIndex);
        if (mAnnotMenu.isShowing()) {
            mAnnotMenu.update(disRect);
        } else {
            mAnnotMenu.show(disRect);
        }

        mLastPageIndex = pageIndex;
        mPdfViewCtrl.convertPageViewRectToPdfRect(rect, mImagePdfRect, pageIndex);
    }

    private void addImageFormPath(String path, int pageIndex) {
        if (!AppUtil.isEmpty(path)) {
            path = AppFileUtil.saveToScopedCache(path);
            setImageInfo(path, pageIndex);

            RectF docRect = new RectF();
            if (AppUtil.isEmptyPDFRect(mImagePdfRect)
                    || Math.max(Math.abs(mImagePdfRect.height()), Math.abs(mImagePdfRect.width())) < 8) {

                PointF viewPoint = new PointF();
                mPdfViewCtrl.convertPdfPtToPageViewPt(mLastDownPdfPoint, viewPoint, pageIndex);
                RectF viewRectF = getImageRectOnPageView(viewPoint, pageIndex);
                mPdfViewCtrl.convertPageViewRectToPdfRect(viewRectF, docRect, pageIndex);
            } else {
                docRect.set(mImagePdfRect);
            }
            createAnnot(docRect, pageIndex, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    mImagePdfRect.setEmpty();
                    mLastDownPdfPoint.set(0, 0);
                }
            });
        } else {
            setEmptyImageRect(pageIndex);
        }
    }

    private void setEmptyImageRect(int pageIndex) {
        if (!AppUtil.isEmptyPDFRect(mImagePdfRect)) {
            RectF viewRect = new RectF();
            mPdfViewCtrl.convertPdfRectToPageViewRect(mImagePdfRect, viewRect, pageIndex);
            RectF invalidateRect = new RectF();
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(viewRect, invalidateRect, pageIndex);
            mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(invalidateRect));

            mImagePdfRect.setEmpty();
        }
    }

    private void adjustImageRect(int pageIndex, RectF imageRect) {
        RectF viewRectF = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(imageRect, viewRectF, pageIndex);
        if (viewRectF.left < mDefaultBoundaryOffset) {
            viewRectF.left = mDefaultBoundaryOffset;
        }
        if (viewRectF.right > mPageViewWidth - mDefaultBoundaryOffset) {
            viewRectF.right = mPageViewWidth - mDefaultBoundaryOffset;
        }
        if (viewRectF.top < mDefaultBoundaryOffset) {
            viewRectF.top = mDefaultBoundaryOffset;
        }
        if (viewRectF.bottom > mPageViewHeight - mDefaultBoundaryOffset) {
            viewRectF.bottom = mPageViewHeight - mDefaultBoundaryOffset;
        }
        mPdfViewCtrl.convertPageViewRectToPdfRect(viewRectF, imageRect, pageIndex);
    }

    private RectF getImageRectOnPageView(PointF point, int pageIndex) {
        PointF pageViewPt = new PointF(point.x, point.y);
        float offsetX = mImageInfo.getWidth() * mImageInfo.getScale();
        float offsetY = mImageInfo.getHeight() * mImageInfo.getScale();

        RectF pageViewRect = new RectF(pageViewPt.x - offsetX, pageViewPt.y - offsetY, pageViewPt.x + offsetX, pageViewPt.y + offsetY);
        int borderWidth = AppDisplay.dp2px(5);
        if (pageViewRect.left - borderWidth < 0) {
            pageViewRect.offset(-pageViewRect.left + borderWidth, 0);
        }
        if (pageViewRect.right + borderWidth > mPdfViewCtrl.getPageViewWidth(pageIndex)) {
            pageViewRect.offset(mPdfViewCtrl.getPageViewWidth(pageIndex) - pageViewRect.right - borderWidth, 0);
        }
        if (pageViewRect.top - borderWidth < 0) {
            pageViewRect.offset(0, -pageViewRect.top + borderWidth);
        }
        if (pageViewRect.bottom + borderWidth > mPdfViewCtrl.getPageViewHeight(pageIndex)) {
            pageViewRect.offset(0, mPdfViewCtrl.getPageViewHeight(pageIndex) - pageViewRect.bottom - borderWidth);
        }
        return pageViewRect;
    }

    private void createAnnot(final RectF rectF, final int pageIndex, final Event.Callback callback) {
        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
            try {
                final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
                final Screen newAnnot = (Screen) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Screen, AppUtil.toFxRectF(rectF)), Annot.e_Screen);
                BorderInfo borderInfo =  newAnnot.getBorderInfo();
                borderInfo.setWidth(0);
                newAnnot.setBorderInfo(borderInfo);
                final PDFImageAddUndoItem undoItem = new PDFImageAddUndoItem(mPdfViewCtrl);
                undoItem.mPageIndex = pageIndex;
                undoItem.mNM = AppDmUtil.randomUUID(null);
                undoItem.mOpacity = AppDmUtil.opacity100To255(mImageInfo.getOpacity()) / 255f;
                undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
                int rotation = (mImageInfo.getRotation() + page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
                undoItem.mRotation = rotation == 0 ? rotation : 4 - rotation;
                undoItem.mImgPath = mImageInfo.getPath();
                undoItem.mFlags = Annot.e_FlagPrint;
                undoItem.mContents = AppFileUtil.getFileName(mImageInfo.getPath());
                undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mBBox = new RectF(rectF);

                PDFImageEvent event = new PDFImageEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, newAnnot, mPdfViewCtrl);
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, newAnnot);
                            mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                try {
                                    RectF viewRect = AppUtil.toRectF(newAnnot.getRect());
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                                    viewRect.inset(-10, -10);

                                    Rect rect = new Rect();
                                    viewRect.roundOut(rect);
                                    mPdfViewCtrl.refresh(pageIndex, rect);
                                    mTouchCaptured = false;
                                    mLastPageIndex = -1;

                                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRect));
                                    if (!mIsContinuousCreate)
                                        mUiExtensionsManager.setCurrentToolHandler(null);
                                } catch (PDFException e) {
                                    e.printStackTrace();
                                }
                            }
                        }

                        if (callback != null)
                            callback.result(event, success);
                    }
                });
                mPdfViewCtrl.addTask(task);

            } catch (PDFException e) {
                if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                    mPdfViewCtrl.recoverForOOM();
                }
            }
        }
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
        onPDFImageToolTouch(pageIndex, motionEvent);
        return true;
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!handled) {
            motionEvent.setAction(MotionEvent.ACTION_UP);
            if (mLastPageIndex == -1)
                mLastPageIndex = pageIndex;

            setEmptyImageRect(pageIndex);
            onPDFImageToolTouch(pageIndex, motionEvent);
        } else {
            if (mAnnotMenu.isShowing())
                mAnnotMenu.dismiss();
            if (mLastPageIndex != -1)
                setEmptyImageRect(mLastPageIndex);
        }
        return true;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinuousCreate;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinuousCreate = continueAddAnnot;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (!mTouchCaptured || mLastPageIndex != pageIndex)
            return;

        Paint paint = new Paint();
        paint.setColor((int) mUiExtensionsManager.getLinkHighlightColor());

        canvas.save();
        RectF viewRect = new RectF();
        mPdfViewCtrl.convertPdfRectToPageViewRect(mImagePdfRect, viewRect, pageIndex);
        canvas.drawRect(viewRect, paint);
        canvas.restore();
    }


    protected PDFImageInfo getImageInfo() {
        return mImageInfo;
    }

    private void setImageInfo(String picPath, int pageIndex) {
        BitmapFactory.Options options = new BitmapFactory.Options();
        options.inJustDecodeBounds = true;
        BitmapFactory.decodeFile(picPath, options);
        int picWidth = 0;
        int picHeight = 0;
        int degree = pictureDegree(picPath);

        switch (degree) {
            case 0:
                picWidth = options.outWidth;
                picHeight = options.outHeight;
                break;
            case 2:
                picWidth = options.outWidth;
                picHeight = options.outHeight;
                mImageInfo.setRotation(degree);
                break;
            case 1:
            case 3:
                picWidth = options.outHeight;
                picHeight = options.outWidth;
                mImageInfo.setRotation(4 - degree);
                break;
        }

        mImageInfo.setWidth(picWidth);
        mImageInfo.setHeight(picHeight);
        mImageInfo.setPath(picPath);
        mImageInfo.setScale(getImageScale(picWidth, picHeight, pageIndex));
        mImageInfo.setPageIndex(pageIndex);
    }

    private float getImageScale(int picWidth, int picHeight, int pageIndex) {
        int pageWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
        int pageHeight = mPdfViewCtrl.getPageViewHeight(pageIndex);

        float widthScale = (float) picWidth / pageWidth;
        float heightScale = (float) picHeight / pageHeight;
        float scale = widthScale > heightScale ? 1 / (5 * widthScale) : 1 / (5 * heightScale);
        scale = (float) (Math.round(scale * 100)) / 100;
        return scale;
    }

    private IToolSupply mToolSupply;
    private ToolItemBean mCurToolItem;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new ImageToolSupply(mContext);
        return mToolSupply;
    }

    private class ImageToolSupply extends ToolSupplyImpl {

        public ImageToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.comment_tool_image_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = ToolConstants.Image_Annotation;
            property.opacity = getImageInfo().getOpacity();
            property.rotation = getImageInfo().getRotation();
            property.color = AppResource.getColor(mContext, R.color.p3);
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_SCREEN_IMAGE;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if (mUiExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_COMMENT_TAB) {
                    mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_Image);
                }
                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mUiExtensionsManager.setCurrentToolHandler(PDFImageToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == PDFImageToolHandler.this) {
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;
            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }

            getImageInfo().setRotation(property.rotation);
            getImageInfo().setOpacity(property.opacity);
            PDFImageToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

    void setOpacity(int opacity) {
        getImageInfo().setOpacity(opacity);
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
    }

    void setRotation(int rotation) {
        getImageInfo().setRotation(rotation);
        if (mCurToolItem == null) return;
        mCurToolItem.property.rotation = rotation;
    }


    int pictureDegree(String path) {
        int degree = 0;
        try {
            ExifInterface exifInterface = new ExifInterface(path);
            int orientation = exifInterface.getAttributeInt(ExifInterface.TAG_ORIENTATION, ExifInterface.ORIENTATION_NORMAL);
            switch (orientation) {
                case ExifInterface.ORIENTATION_ROTATE_90:
                    degree = 1;
                    break;
                case ExifInterface.ORIENTATION_ROTATE_180:
                    degree = 2;
                    break;
                case ExifInterface.ORIENTATION_ROTATE_270:
                    degree = 3;
                    break;
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
        return degree;
    }

    void updateTheme() {
        if (mFileSelectDialog != null && mFileSelectDialog.isShowing()) {
            mFileSelectDialog.dismiss();
            mFileSelectDialog = null;
        }
    }

    void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        if (verifyPermissions(grantResults)) {
            if (requestCode == ActRequestCode.REQ_CAMERA_PERMISSION) {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                AppIntentUtil.selectImageFromCamera(uiExtensionsManager,
                        mCaptureImgPath, ActRequestCode.REQ_CAMERA_PERMISSION, ActRequestCode.REQ_FILE_FROM_CAMERA);
            }
        } else {
            setEmptyImageRect(mLastPageIndex);
            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_permission_denied));
        }
    }

    void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
        if (resultCode == Activity.RESULT_OK) {
            String path = null;
            if (requestCode == ActRequestCode.REQ_FILE_FROM_GALLERY && data != null) {
                path = AppFileUtil.getFilePathFromUri(mContext, data.getData());
            } else if (requestCode == ActRequestCode.REQ_FILE_FROM_CAMERA) {
                path = mCaptureImgPath;
            }

            addImageFormPath(path, mLastPageIndex);
        } else {
            setEmptyImageRect(mLastPageIndex);
        }
    }

    void onConfigurationChanged(Configuration newConfig) {
        if (mFileSelectDialog != null && mFileSelectDialog.isShowing()) {
            mFileSelectDialog.resetWH();
            mFileSelectDialog.showDialog();
        }
    }

    void onDrawForControls(int pageIndex, Canvas canvas) {
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            RectF displayRect = new RectF();
            if (AppUtil.isEmptyPDFRect(mImagePdfRect)) {
                if (mLastDownPdfPoint.x > 0 && mLastDownPdfPoint.y > 0) {
                    PointF viewPoint = new PointF();
                    mPdfViewCtrl.convertPdfPtToPageViewPt(mLastDownPdfPoint, viewPoint, mLastPageIndex);
                    RectF viewRectF = new RectF(viewPoint.x, viewPoint.y, viewPoint.x + 10, viewPoint.y + 10);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(viewRectF, displayRect, mLastPageIndex);
                    mAnnotMenu.update(displayRect);
                }
            } else {
                mPdfViewCtrl.convertPdfRectToPageViewRect(mImagePdfRect, displayRect, mLastPageIndex);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(displayRect, displayRect, mLastPageIndex);
                mAnnotMenu.update(displayRect);
            }
        }
    }

    private boolean verifyPermissions(int[] grantResults) {
        for (int grantResult : grantResults) {
            if (grantResult != PackageManager.PERMISSION_GRANTED) {
                return false;
            }
        }
        return true;
    }

    void setAnnotMenu(AnnotMenu annotMenu) {
        mAnnotMenu = annotMenu;
    }

    private void showSelectFileDialog() {
        mFileSelectDialog = getFileSelectDialog();
        mFileSelectDialog.init(new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                return !(pathname.isHidden() || !AppFileUtil.canRead(pathname))
                        && !(pathname.isFile() && !isPictures(pathname));
            }
        }, true);

        mFileSelectDialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
                mFileSelectDialog.dismiss();
            }

            @Override
            public void onTitleRightButtonClick() {
                if (mFileSelectDialog.getSelectedFiles().size() > 0) {
                    String filepath = mFileSelectDialog.getSelectedFiles().get(0).path;
                    addImageFormPath(filepath, mLastPageIndex);
                }
                mFileSelectDialog.dismiss();
            }
        });

        mFileSelectDialog.resetWH();
        mFileSelectDialog.showDialog();
    }

    private UIFileSelectDialog getFileSelectDialog() {
        if (mFileSelectDialog == null) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            mFileSelectDialog = new UIFileSelectDialog(uiExtensionsManager.getAttachedActivity());
            mFileSelectDialog.init(new FileFilter() {
                @Override
                public boolean accept(File pathname) {
                    return !(pathname.isHidden() || !AppFileUtil.canRead(pathname))
                            && !(pathname.isFile() && !isPictures(pathname));
                }
            }, true);
            mFileSelectDialog.setCanceledOnTouchOutside(true);
        } else {
            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    mFileSelectDialog.notifyDataSetChanged();
                }
            });
        }
        return mFileSelectDialog;
    }

    private boolean isPictures(File file) {
        String pathName = file.getName().toLowerCase();
        return pathName.endsWith(".bmp")
                || pathName.endsWith(".jpg")
                || pathName.endsWith(".png")
                || pathName.endsWith(".gif")
                || pathName.endsWith(".tif")
                || pathName.endsWith(".jpx")
                || pathName.endsWith(".jpeg");
    }

}
