/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.multimedia;


import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.media.AudioFormat;
import android.media.AudioRecord;
import android.media.MediaMetadataRetriever;
import android.media.MediaRecorder;
import android.net.Uri;
import android.os.Build;
import android.os.Environment;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.os.ParcelFileDescriptor;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;

public class MultimediaUtil {
    private static final int SAMPLE_RATE_IN_HZ = 44100;
    private static final int SAMPLE_RATE_IN_HZ_2 = 16000;
    private static final int AUDIO_CHANNEL = AudioFormat.CHANNEL_IN_STEREO;
    private static final int AUDIO_FORMAT = AudioFormat.ENCODING_PCM_16BIT;
    private static final int BUFFER_SIZE = AudioRecord.getMinBufferSize(SAMPLE_RATE_IN_HZ, AUDIO_CHANNEL, AUDIO_FORMAT);

    private final Context mContext;
    private AudioRecord mAudioRecord;

    private File mRawAudioFile;
    private File mNewAudioFile;

    private List<String> mAudioSupportList;
    private List<String> mVideoSupportList;

    private boolean mIsRecordAudio;

    public MultimediaUtil(Context context) {
        this.mContext = context;
    }

    public File getOutputMediaFile(String intent, String suffix) {
        if (!Environment.MEDIA_MOUNTED.equals(Environment.getExternalStorageState())) {
            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.the_sdcard_not_exist));
            return null;
        }

        String cachePath = getCachePath();
        File mediaStorageDir = new File(cachePath);
        if (!mediaStorageDir.exists()) {
            mediaStorageDir.mkdirs();
        }
        String timeStamp = new SimpleDateFormat("yyyyMMdd_HHmmss", Locale.getDefault()).format(new Date());
        String path;
        if (ToolHandler.TH_TYPE_PDFIMAGE.equals(intent)) {
            path = mediaStorageDir.getPath() + File.separator + "Image_" + timeStamp + suffix;
        } else if (ToolHandler.TH_TYPE_SCREEN_VIDEO.equals(intent)) {
            path = mediaStorageDir.getPath() + File.separator + "Video_" + timeStamp + suffix;
        } else {
            path = mediaStorageDir.getPath() + File.separator + "Audio_" + timeStamp + suffix;
        }
        return new File(path);
    }

    public List<String> getAudioSupportMimeList() {
        if (mAudioSupportList == null) {
            mAudioSupportList = new ArrayList<>();
            mAudioSupportList.add("audio/aiff");
            mAudioSupportList.add("audio/basic");
            mAudioSupportList.add("audio/mid");
            mAudioSupportList.add("audio/midi");
            mAudioSupportList.add("audio/mp3");
            mAudioSupportList.add("audio/mpeg");
            mAudioSupportList.add("audio/mpeg3");
            mAudioSupportList.add("audio/mpegurl");
            mAudioSupportList.add("audio/wav");
            mAudioSupportList.add("audio/x-aiff");
            mAudioSupportList.add("audio/x-midi");
            mAudioSupportList.add("audio/x-mp3");
            mAudioSupportList.add("audio/x-mpeg");
            mAudioSupportList.add("audio/x-mpeg3");
            mAudioSupportList.add("audio/x-mpegurl");
            mAudioSupportList.add("audio/x-wav");
            mAudioSupportList.add("audio/x-ms-wax");
            mAudioSupportList.add("audio/x-ms-wma");
        }
        return mAudioSupportList;
    }

    public List<String> getVideoSupportMimeList() {
        if (mVideoSupportList == null) {
            mVideoSupportList = new ArrayList<>();
            mVideoSupportList.add("application/x-shockwave-flash");
            mVideoSupportList.add("video/avi");
            mVideoSupportList.add("video/mpeg");
            mVideoSupportList.add("video/msvideo");
            mVideoSupportList.add("video/x-ivf");
            mVideoSupportList.add("video/x-mpeg");
            mVideoSupportList.add("video/x-ms-asf");
            mVideoSupportList.add("video/x-ms-asx");
            mVideoSupportList.add("video/x-ms-wm");
            mVideoSupportList.add("video/x-ms-wmp");
            mVideoSupportList.add("video/x-ms-wmv");
            mVideoSupportList.add("video/x-ms-wmx");
            mVideoSupportList.add("video/x-ms-wvx");
            mVideoSupportList.add("video/x-msvideo");
            mVideoSupportList.add("video/x-mpg");
            mVideoSupportList.add("video/mpg");
            mVideoSupportList.add("video/quicktime");
            mVideoSupportList.add("video/mp4");
            mVideoSupportList.add("video/3gpp");
        }
        return mVideoSupportList;
    }

    private float getImageScale(PDFViewCtrl pdfViewCtrl, int picWidth, int picHeight, int pageIndex) {
        int pageWidth = pdfViewCtrl.getPageViewWidth(pageIndex);
        int pageHeight = pdfViewCtrl.getPageViewHeight(pageIndex);

        float widthScale = (float) picWidth / pageWidth;
        float heightScale = (float) picHeight / pageHeight;
        float scale = widthScale > heightScale ? 1 / (5 * widthScale) : 1 / (5 * heightScale);
        scale = (float) (Math.round(scale * 100)) / 100;
        return scale;
    }

    private Bitmap getThumbnail(String filePath) {
        Bitmap thumbnail = null;
        MediaMetadataRetriever retriever = new MediaMetadataRetriever();
        try {
            retriever.setDataSource(filePath);
            thumbnail = retriever.getFrameAtTime();
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                    retriever.close();
                } else {
                    retriever.release();
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return thumbnail;
    }

    public Bitmap getVideoThumbnail(PDFViewCtrl pdfViewCtrl, String filePath) {
        Bitmap thumbnail = getThumbnail(filePath);
        if (thumbnail == null)
            thumbnail = BitmapFactory.decodeResource(mContext.getResources(), R.drawable.video_default_icon);
        float scale = getImageScale(pdfViewCtrl, thumbnail.getWidth(), thumbnail.getHeight(), pdfViewCtrl.getCurrentPage());
        Matrix matrix = new Matrix();
        matrix.postScale(scale, scale);

        Bitmap _thumbnail = Bitmap.createBitmap(thumbnail, 0, 0, thumbnail.getWidth(), thumbnail.getHeight(), matrix, true);
        thumbnail.recycle();
        return _thumbnail;
    }

    public void startRecordAudio() {
        if (!Environment.MEDIA_MOUNTED.equals(Environment.getExternalStorageState())) {
            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.the_sdcard_not_exist));
            return;
        }

        if (!mIsRecordAudio) {
            if (mAudioRecord == null) {
                mRawAudioFile = getOutputMediaFile(ToolHandler.TH_TYPE_SCREEN_AUDIO, ".raw");
                mNewAudioFile = getOutputMediaFile(ToolHandler.TH_TYPE_SCREEN_AUDIO, ".wav");

                if (Build.VERSION.SDK_INT > Build.VERSION_CODES.KITKAT) {
                    mAudioRecord = new AudioRecord(MediaRecorder.AudioSource.MIC, SAMPLE_RATE_IN_HZ,
                            AudioFormat.CHANNEL_IN_STEREO, AUDIO_FORMAT, BUFFER_SIZE);
                } else {
                    mAudioRecord = new AudioRecord(MediaRecorder.AudioSource.MIC, SAMPLE_RATE_IN_HZ_2,
                            AudioFormat.CHANNEL_IN_MONO, AUDIO_FORMAT, BUFFER_SIZE);
                }
                if (mAudioRecord.getRecordingState() != AudioRecord.RECORDSTATE_STOPPED) {
                    if (mRecordFinishCallback != null) {
                        mRecordFinishCallback.onFailed();
                    }
                    return;
                }
            }
            mIsRecordAudio = true;
            mAudioRecord.startRecording();

            if (mAudioRecord.getRecordingState() != AudioRecord.RECORDSTATE_RECORDING) {
                releaseAudioRecord();
                if (mRecordFinishCallback != null) {
                    mRecordFinishCallback.onFailed();
                }
                return;
            }
            AppThreadManager.getInstance().startThread(new AudioRecordThread());
        }
    }

    public void stopRecordAudio() {
        if (mIsRecordAudio) {
            AppThreadManager.getInstance().startThread(new AudioConvertThread());
        }
    }

    private class AudioRecordThread implements Runnable {
        @Override
        public void run() {
            writeDateTOFile();
        }
    }

    private class AudioConvertThread implements Runnable {
        @Override
        public void run() {
            convertRaw2Wave(mRawAudioFile, mNewAudioFile);
        }
    }

    private final Handler handler = new Handler(Looper.getMainLooper()) {
        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);

            //close AudioRecord
            switch (msg.what) {
                case 0:
                    releaseAudioRecord();
                    if (mRecordFinishCallback != null) {
                        mRecordFinishCallback.onSuccessed(mNewAudioFile);
                    }
                    break;
                case 1:
                    releaseAudioRecord();
                    if (mRecordFinishCallback != null) {
                        mRecordFinishCallback.onFailed();
                    }
                    break;
                default:
            }
        }

    };

    public void releaseAudioRecord() {
        if (mIsRecordAudio) {
            if (mRawAudioFile != null && mRawAudioFile.exists()) {
                mRawAudioFile.delete();
            }

            if (mAudioRecord != null) {
                mIsRecordAudio = false;
                mAudioRecord.stop();
                mAudioRecord.release();
                mAudioRecord = null;
            }
        }
    }

    private void writeDateTOFile() {
        byte[] audiodata = new byte[BUFFER_SIZE];
        FileOutputStream fos = null;
        int readsize = 0;
        try {
            if (mRawAudioFile.exists()) {
                mRawAudioFile.delete();
            }
            fos = new FileOutputStream(mRawAudioFile);

            while (mIsRecordAudio) {
                readsize = mAudioRecord.read(audiodata, 0, BUFFER_SIZE);
                if (AudioRecord.ERROR_INVALID_OPERATION != readsize) {
                    try {
                        fos.write(audiodata);
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (fos != null) {
                    fos.close();
                }
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    private void convertRaw2Wave(File inFile, File outFile) {
        FileInputStream in;
        FileOutputStream out;
        long totalAudioLen;
        long totalDataLen;
        long longSampleRate = SAMPLE_RATE_IN_HZ;
        int channels = 2;
        long byteRate = 16 * SAMPLE_RATE_IN_HZ * channels / 8;
        byte[] data = new byte[BUFFER_SIZE];
        try {
            String path = inFile.getAbsolutePath();
            File file = new File(path);
            if (file.canRead()) {
                in = new FileInputStream(inFile);
            } else {
                Uri uri = AppFileUtil.toDocumentUriFromPath(path);
                ParcelFileDescriptor parcelFileDescriptor = mContext.getContentResolver().openFileDescriptor(uri, "r");
                in = new FileInputStream(parcelFileDescriptor.getFileDescriptor());
            }
            out = new FileOutputStream(outFile);
            totalAudioLen = in.getChannel().size();
            totalDataLen = totalAudioLen + 36;
            writeWaveFileHeader(out, totalAudioLen, totalDataLen,
                    longSampleRate, channels, byteRate);
            while (in.read(data) != -1) {
                out.write(data);
            }
            in.close();
            out.close();

            handler.sendEmptyMessage(0);
        } catch (IOException e) {
            handler.sendEmptyMessage(1);
        }
    }

    private void writeWaveFileHeader(FileOutputStream out, long totalAudioLen,
                                     long totalDataLen, long longSampleRate, int channels, long byteRate)
            throws IOException {
        byte[] header = new byte[44];
        header[0] = 'R'; // RIFF/WAVE header
        header[1] = 'I';
        header[2] = 'F';
        header[3] = 'F';
        header[4] = (byte) (totalDataLen & 0xff);
        header[5] = (byte) ((totalDataLen >> 8) & 0xff);
        header[6] = (byte) ((totalDataLen >> 16) & 0xff);
        header[7] = (byte) ((totalDataLen >> 24) & 0xff);
        header[8] = 'W';
        header[9] = 'A';
        header[10] = 'V';
        header[11] = 'E';
        header[12] = 'f'; // 'fmt ' chunk
        header[13] = 'm';
        header[14] = 't';
        header[15] = ' ';
        header[16] = 16; // 4 bytes: size of 'fmt ' chunk
        header[17] = 0;
        header[18] = 0;
        header[19] = 0;
        header[20] = 1; // format = 1
        header[21] = 0;
        header[22] = (byte) channels;
        header[23] = 0;
        header[24] = (byte) (longSampleRate & 0xff);
        header[25] = (byte) ((longSampleRate >> 8) & 0xff);
        header[26] = (byte) ((longSampleRate >> 16) & 0xff);
        header[27] = (byte) ((longSampleRate >> 24) & 0xff);
        header[28] = (byte) (byteRate & 0xff);
        header[29] = (byte) ((byteRate >> 8) & 0xff);
        header[30] = (byte) ((byteRate >> 16) & 0xff);
        header[31] = (byte) ((byteRate >> 24) & 0xff);
        header[32] = (byte) (2 * 16 / 8); // block align
        header[33] = 0;
        header[34] = 16; // bits per sample
        header[35] = 0;
        header[36] = 'd';
        header[37] = 'a';
        header[38] = 't';
        header[39] = 'a';
        header[40] = (byte) (totalAudioLen & 0xff);
        header[41] = (byte) ((totalAudioLen >> 8) & 0xff);
        header[42] = (byte) ((totalAudioLen >> 16) & 0xff);
        header[43] = (byte) ((totalAudioLen >> 24) & 0xff);
        out.write(header, 0, 44);
    }

    public boolean canPlaySimpleAudio(String filePath) {
        int index = filePath.lastIndexOf('.');
        if (index < 0) return false;

        String ext = filePath.substring(index + 1);
        return support(ext) && isAudio(ext);
    }

    //call support(String) first, because {android os Version check};
    private boolean isAudio(String ext) {
        if (ext == null || ext.isEmpty()) return false;

        return ext.equals("m4a")
                || ext.equals("mp3")
                || ext.equals("wav")
                || ext.equals("mid")
                || ext.equals("aac")
                || ext.equals("flac");
    }

    private boolean support(String ext) {
        if (ext == null || ext.isEmpty()) return false;

        if (ext.equals("3gp")
                || ext.equals("mp4")
                || ext.equals("m4a")
                || ext.equals("mp3")
                || ext.equals("mid")
                || ext.equals("xmf")
                || ext.equals("mxmf")
                || ext.equals("rtttl")
                || ext.equals("rtx")
                || ext.equals("ota")
                || ext.equals("imy")
                || ext.equals("ogg")
                || ext.equals("wav")
                || ext.equals("jpg")
                || ext.equals("gif")
                || ext.equals("png")
                || ext.equals("bmp")
        ) {
            return true;
        } else if (ext.equals("aac")
                || ext.equals("flac")) {
            //3.1+
            if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB_MR1)
                return true;
        } else if (ext.equals("ts")) {
            //3.0+
            if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.HONEYCOMB)
                return true;
        } else if (ext.equals("mkv")
                || ext.equals("webp")) {
            //4.0+
            if (android.os.Build.VERSION.SDK_INT >= Build.VERSION_CODES.ICE_CREAM_SANDWICH)
                return true;
        }

        return false;
    }

    private IRecordFinishCallback mRecordFinishCallback;

    public void setRecordFinishCallback(IRecordFinishCallback callback) {
        this.mRecordFinishCallback = callback;
    }

    public interface IRecordFinishCallback {
        void onSuccessed(File file);

        void onFailed();
    }

    public String getCachePath() {
        return AppFileUtil.getDiskCachePath(mContext) + "/multimedia/";
    }

    public boolean isAudio(File file) {
        String pathName = file.getName().toLowerCase();
        return pathName.endsWith(".aiff") // audio/aiff
                || pathName.endsWith(".aif") // audio/aiff audio/x-aiff
                || pathName.endsWith(".aifc") // audio/x-aiff
                || pathName.endsWith(".au") // audio/basic
                || pathName.endsWith(".m3u") // audio/mpegurl
                || pathName.endsWith(".wav") // audio/wav audio/x-wav
                || pathName.endsWith(".wma") // audio/x-ms-wma
                || pathName.endsWith(".wax") // audio/x-ms-wax
                || pathName.endsWith(".mpa") // audio/mpeg
                || pathName.endsWith(".kar") // audio/midi
                || pathName.endsWith(".rmi") // audio/midi
                || pathName.endsWith(".midi") // application/x-midi audio/midi audio/x-mid audio/x-midi music/crescendo x-music/x-midi
                || pathName.endsWith(".mid")// audio/mid audio/midi application/x-midi audio/x-mid audio/x-midi music/crescendo x-music/x-midi
                || pathName.endsWith(".mp3");// audio/mp3 audio/mpeg audio/mpeg3 audio/mpg audio/x-mpeg audio/x-mpeg-3
    }

    public boolean isVideo(File file) {
        String pathName = file.getName().toLowerCase();
        return pathName.endsWith(".avi") // video/avi video/msvideo video/x-msvideo video/quicktime application/x-troff-msvideo
                || pathName.endsWith(".ivf") // video/x-ivf
                || pathName.endsWith(".wmp") // video/x-ms-wmp
                || pathName.endsWith(".wm") // video/x-ms-wm video/x-ms-asf
                || pathName.endsWith(".wvx") // video/x-ms-wvx
                || pathName.endsWith(".wmx") // video/x-ms-wmx
                || pathName.endsWith(".wmv") // video/x-ms-wmv
                || pathName.endsWith(".asf") // video/x-ms-asf
                || pathName.endsWith(".asx") // video/x-ms-asx
                || pathName.endsWith(".swf") // application/x-shockwave-flash
                || pathName.endsWith(".mp4")// video/mp4
                || pathName.endsWith(".mov")// video/quicktime
                || pathName.endsWith(".mpeg"); // video/mpeg
    }

}
