/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.line;

import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.RectF;
import android.text.TextPaint;
import android.view.MotionEvent;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AbstractToolHandler;
import com.foxit.uiextensions.annots.common.UIAnnotFrame;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.Event;

public class LineToolHandler extends AbstractToolHandler {

    protected LineRealAnnotHandler mAnnotHandler;
    protected LineUtil mUtil;
    protected String mIntent;

    protected boolean mTouchCaptured = false;
    protected int mCapturedPage = -1;
    protected PointF mStartPt = new PointF();
    protected PointF mStopPt = new PointF();
    protected Paint mPaint;
    private TextPaint mTextPaint;
    private int scaleFromUnitIndex;
    private int scaleToUnitIndex;
    private float scaleFromValue = 1.0f;
    private float scaleToValue = 1.0f;

    private PointF dStart, dEnd;

    private ToolItemBean mCurToolItem;
    private float mDistance = 5.0f;

    public LineToolHandler(Context context, PDFViewCtrl pdfViewCtrl, LineUtil util, String intent) {
        super(context, pdfViewCtrl, util.getToolName(intent), util.getToolPropertyKey(intent));
        if (intent.equals(LineConstants.INTENT_LINE_ARROW)) {
            mColor = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
        } else if (LineConstants.INTENT_LINE_DIMENSION.equals(intent)) {
            mColor = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
            mThickness = 3.0f;
            mTextPaint = new TextPaint();
            mTextPaint.setStyle(Style.FILL);
            mTextPaint.setTextSize(40f);
            dStart = new PointF();
            dEnd = new PointF();
        } else {
            mColor = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
        }

        mUtil = util;
        mIntent = intent;
        scaleFromUnitIndex = 0;//default:0=="pt"

        mPaint = new Paint();
        mPaint.setStyle(Style.STROKE);
        mPaint.setAntiAlias(true);
        mPaint.setDither(true);
        mDistance = mUtil.getControlExtent() + AppAnnotUtil.getAnnotBBoxSpace() + mThickness / 2;
        setColorChangeListener(new ColorChangeListener() {
            @Override
            public void onColorChange(int color) {
                if (mCurToolItem == null) return;
                mCurToolItem.property.color = color;
                ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
            }
        });
        mDefaultSizeDimension = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_48dp);
    }

    protected void initUiElements() {
    }

    protected void uninitUiElements() {
    }

    protected String getIntent() {
        return mIntent;
    }

    @Override
    public void onActivate() {
        mCapturedPage = -1;
        resetPropertyBar();
    }

    public int getColor() {
        return mColor;
    }

    public int getOpacity() {
        return mOpacity;
    }

    public float getThickness() {
        return mThickness;
    }
    @Override
    protected void resetPropertyBar() {
        if (mIntent.equals(LineConstants.INTENT_LINE_DIMENSION)) {
            mPropertyBar.scaleFromUnit(scaleFromUnitIndex);
            mPropertyBar.scaleToUnit(scaleToUnitIndex);
            mPropertyBar.scaleFromValue(scaleFromValue);
            mPropertyBar.scaleToValue(scaleToValue);
        }

        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        mPropertyBar.setColors(colors);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_LINEWIDTH, mThickness);
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.reset(getSupportedProperties());
        mPropertyBar.setPropertyChangeListener(this);
    }

    @Override
    protected void onCreateValueChanged(long property, Object value) {
        if (property == PropertyBar.PROPERTY_DISTANCE) {
            this.scaleFromUnitIndex = (int) value;
        } else if (property == PropertyBar.PROPERTY_DISTANCE_TIP) {
            this.scaleToUnitIndex = (int) value;
        } else if (property == PropertyBar.PROPERTY_DISTANCE_VALUE) {
            this.scaleFromValue = (float) value;
        } else if (property == PropertyBar.PROPERTY_DISTANCE_TIP_VALUE) {
            this.scaleToValue = (float) value;
        }
    }

    @Override
    public void onDeactivate() {
        if (mTouchCaptured) {
            if (mPdfViewCtrl.isPageVisible(mCapturedPage)) {
                addAnnot(mCapturedPage);
            }
            mTouchCaptured = false;
            mCapturedPage = -1;
        }
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent e) {
        boolean handled = mUiExtensionsManager.defaultTouchEvent(pageIndex, e);
        if (mTouchCaptured)
            handled = onLineToolTouch(pageIndex, e);
        return handled;
    }

    private boolean onLineToolTouch(int pageIndex, MotionEvent e) {
        PointF pt = new PointF(e.getX(), e.getY());
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(pt, pt, pageIndex);
        int action = e.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (!mTouchCaptured || mCapturedPage == pageIndex) {
                    mTouchCaptured = true;
                    mStartPt.set(pt);
                    mStopPt.set(pt);
                    if (mCapturedPage == -1) {
                        mCapturedPage = pageIndex;
                    }
                }
                break;
            case MotionEvent.ACTION_MOVE:
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (mTouchCaptured) {
                    PointF point = new PointF(pt.x, pt.y);
                    if (action == MotionEvent.ACTION_MOVE)
                        mUtil.correctPvPoint(mPdfViewCtrl, pageIndex, point, mThickness);
                    boolean keepSameOffset = false;
                    if (mCapturedPage == pageIndex && !point.equals(mStopPt.x, mStopPt.y)) {
                        float thickness = UIAnnotFrame.getPageViewThickness(mPdfViewCtrl, pageIndex, mThickness);
                        RectF rect1 = mUtil.getArrowBBox(mStartPt, mStopPt, thickness);
                        RectF rect2 = mUtil.getArrowBBox(mStartPt, point, thickness);
                        rect2.union(rect1);
                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rect2, rect2, pageIndex);
                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rect2));
                        mStopPt.set(point);
                    } else if (mCapturedPage == pageIndex && (mStartPt.equals(mStopPt.x, mStopPt.y) || Math.abs(mStopPt.length() - mStartPt.length()) <= 8)) {
                        if (!LineConstants.INTENT_LINE_DIMENSION.equals(mIntent)&&action == MotionEvent.ACTION_UP) {
                            int pageViewWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
                            int pageViewHeight = mPdfViewCtrl.getPageViewHeight(pageIndex);
                            float scale = Math.max(1f * pageViewWidth / mPdfViewCtrl.getDisplayViewWidth(), 1f * pageViewHeight / mPdfViewCtrl.getDisplayViewHeight());
                            int defaultSizeDimension = (int) (mDefaultSizeDimension * scale);
                            PointF startViewPointF = new PointF(mStartPt.x, mStartPt.y);
                            PointF endViewPointF = new PointF(mStartPt.x + defaultSizeDimension, mStartPt.y - defaultSizeDimension);
                            float maxY = pageViewHeight - mDistance;
                            float maxX = pageViewWidth - mDistance;
                            boolean isOutOfBounds = false;
                            float tempStart = 0.0f;
                            float tempEnd = 0.0f;
                            if (startViewPointF.x < mDistance) {
                                tempStart = tempEnd = mDistance - startViewPointF.x;
                                if (endViewPointF.y - tempEnd < mDistance) {
                                    tempEnd = endViewPointF.y - mDistance;
                                }
                                isOutOfBounds = true;
                            }
                            if (startViewPointF.y > maxY && mStartPt.x + mStartPt.y > maxY) {
                                tempStart = tempEnd = startViewPointF.y - maxY;
                                if (endViewPointF.x + tempEnd > pageViewWidth) {
                                    tempEnd = pageViewWidth - endViewPointF.x;
                                }
                                isOutOfBounds = true;
                            }
                            if (endViewPointF.x > maxX) {
                                tempStart = tempEnd = maxX - endViewPointF.x;
                                if (startViewPointF.y - tempEnd > maxY) {
                                    tempStart = startViewPointF.y - maxY;
                                }
                                isOutOfBounds = true;
                            }
                            if (endViewPointF.y < mDistance && mStartPt.x + mStartPt.y < maxX) {
                                tempStart = tempEnd = endViewPointF.y - mDistance;
                                if (startViewPointF.x + tempEnd < mDistance) {
                                    tempStart = mDistance - startViewPointF.x;
                                }
                                isOutOfBounds = true;
                            }
                            if (isOutOfBounds) {
                                startViewPointF.x += tempStart;
                                startViewPointF.y -= tempStart;
                                endViewPointF.x += tempEnd;
                                endViewPointF.y -= tempEnd;
                                mStartPt.set(startViewPointF.x, startViewPointF.y);
                            }
                            mStopPt.set(endViewPointF);
                            keepSameOffset = true;
                        }
                    }
                    if (action == MotionEvent.ACTION_UP || action == MotionEvent.ACTION_CANCEL) {
                        adjustPoint(mStartPt, mStopPt, pageIndex, keepSameOffset);
                        addAnnot(pageIndex);
                        mTouchCaptured = false;
                        if (!mIsContinuousCreate) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
                        }
                    }
                }
                break;
            default:
                break;
        }
        return true;
    }

    private void adjustPoint(PointF start, PointF stop, int pageIndex, boolean keepSameOffset) {
        PointF adjustStartPoint = new PointF();
        adjustStartPoint.set(start);
        PointF adjustStopPoint = new PointF();
        adjustStopPoint.set(stop);
        mUtil.correctPvPoint(mPdfViewCtrl, pageIndex, adjustStartPoint, mThickness);
        mUtil.correctPvPoint(mPdfViewCtrl, pageIndex, adjustStopPoint, mThickness);
        float offsetX = 0;
        float offsetY = 0;
        if (!adjustStartPoint.equals(start)){
            offsetX = adjustStartPoint.x - start.x;
            offsetY = adjustStartPoint.y - start.y;
        }else if (!adjustStopPoint.equals(stop)){
            offsetX = adjustStopPoint.x - stop.x;
            offsetY = adjustStopPoint.y - stop.y;
        }
        if (!keepSameOffset)
            start.offset(adjustStartPoint.x - start.x,adjustStartPoint.y - start.y);
        stop.offset(offsetX,offsetY);
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null){
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
        return onLineToolTouch(pageIndex, motionEvent);
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!handled && !LineConstants.INTENT_LINE_DIMENSION.equals(getIntent())){
            addLineByDefault(pageIndex, motionEvent);
        }
        return true;
    }

    private int mDefaultSizeDimension;

    private void addLineByDefault(int pageIndex, MotionEvent motionEvent) {
        int pageViewWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
        int pageViewHeight = mPdfViewCtrl.getPageViewHeight(pageIndex);
        float scale = Math.max(1f * pageViewWidth / mPdfViewCtrl.getDisplayViewWidth(), 1f * pageViewHeight / mPdfViewCtrl.getDisplayViewHeight());
        int defaultSizeDimension = (int) (mDefaultSizeDimension * scale);
        PointF startViewPointF = new PointF(motionEvent.getX(), motionEvent.getY());
        PointF endViewPointF = new PointF(motionEvent.getX() + defaultSizeDimension, motionEvent.getY() - defaultSizeDimension);
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(startViewPointF, startViewPointF, pageIndex);
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(endViewPointF, endViewPointF, pageIndex);
        float maxY = pageViewHeight - mDistance;
        float maxX = pageViewWidth - mDistance;
        boolean isOutOfBounds = false;
        float tempStart = 0.0f;
        float tempEnd = 0.0f;
        if (startViewPointF.x < mDistance) {
            tempStart = tempEnd = mDistance - startViewPointF.x;
            if (endViewPointF.y - tempEnd < mDistance) {
                tempEnd = endViewPointF.y - mDistance;
            }
            isOutOfBounds = true;
        }
        if (startViewPointF.y > maxY && mStartPt.x + mStartPt.y > maxY) {
            tempStart = tempEnd = startViewPointF.y - maxY;
            if (endViewPointF.x + tempEnd > pageViewWidth) {
                tempEnd = pageViewWidth - endViewPointF.x;
            }
            isOutOfBounds = true;
        }
        if (endViewPointF.x > maxX) {
            tempStart = tempEnd = maxX - endViewPointF.x;
            if (startViewPointF.y - tempEnd > maxY) {
                tempStart = startViewPointF.y - maxY;
            }
            isOutOfBounds = true;
        }
        if (endViewPointF.y < mDistance && mStartPt.x + mStartPt.y < maxX) {
            tempStart = tempEnd = endViewPointF.y - mDistance;
            if (startViewPointF.x + tempEnd < mDistance) {
                tempStart = mDistance - startViewPointF.x;
            }
            isOutOfBounds = true;
        }
        if (isOutOfBounds) {
            startViewPointF.x += tempStart;
            startViewPointF.y -= tempStart;
            endViewPointF.x += tempEnd;
            endViewPointF.y -= tempEnd;
            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(startViewPointF, startViewPointF, pageIndex);
            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(endViewPointF, endViewPointF, pageIndex);
            motionEvent.setLocation(startViewPointF.x, startViewPointF.y);
        }

        onLineToolTouch(pageIndex, motionEvent);
        MotionEvent upEvent = MotionEvent.obtain(motionEvent);
        upEvent.setAction(MotionEvent.ACTION_UP);
        if (isOutOfBounds) {
            upEvent.setLocation(endViewPointF.x, endViewPointF.y);
        } else {
            upEvent.offsetLocation(defaultSizeDimension, -defaultSizeDimension);
        }
        onLineToolTouch(pageIndex, upEvent);
        upEvent.recycle();
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {

        if (mCapturedPage == pageIndex) {
            float distance = AppDmUtil.distanceOfTwoPoints(mStartPt, mStopPt);
            float thickness = mThickness;
            thickness = thickness < 1.0f ? 1.0f : thickness;
//			thickness = (thickness + 3)*15.0f/8.0f;
            thickness = UIAnnotFrame.getPageViewThickness(mPdfViewCtrl, pageIndex, thickness);
            if (distance > thickness * LineUtil.ARROW_WIDTH_SCALE / 2) {
                canvas.save();
                setPaintProperty(mPdfViewCtrl, pageIndex, mPaint);
                Path path = mUtil.getLinePath(getIntent(), mStartPt, mStopPt, thickness);
                canvas.drawPath(path, mPaint);
                if (getIntent() != null && getIntent().equals(LineConstants.INTENT_LINE_DIMENSION)) {
                    mTextPaint.setTextAlign(Paint.Align.CENTER);
                    mTextPaint.setSubpixelText(true);
                    mPdfViewCtrl.convertPageViewPtToPdfPt(mStartPt, dStart, pageIndex);
                    mPdfViewCtrl.convertPageViewPtToPdfPt(mStopPt, dEnd, pageIndex);
                    distance = AppDmUtil.distanceOfTwoPoints(dStart, dEnd);
                    float factor = DistanceMeasurement.valueOf(scaleFromUnitIndex).getScaleWithDefault() * scaleToValue / scaleFromValue;
                    if (scaleFromValue == 0) {
                        factor = 0;
                    }
                    String text = String.valueOf((float) (Math.round(distance * 100 * factor)) / 100) + DistanceMeasurement.valueOf(scaleToUnitIndex).getName();
                    float x = (Math.min(mStartPt.x, mStopPt.x)) + Math.abs(mStopPt.x - mStartPt.x) / 2;
                    float y = (Math.min(mStartPt.y, mStopPt.y)) + Math.abs(mStopPt.y - mStartPt.y) / 2;

                    int rotate = getRotationBetweenLines(mStartPt.x, mStartPt.y, mStopPt.x, mStopPt.y);
                    int  alpha,red,green,blue=0;
                    alpha = red = green = blue = 255;
                    red= Color.red(mColor ^ 0x00A5A5A5);
                    green=Color.green(mColor ^ 0x00A5A5A5);
                    blue=Color.blue(mColor ^ 0x00A5A5A5);
                    int color = Color.argb(alpha, (255-red), (255-green), (255-blue));
                    mTextPaint.setColor(color);
                    canvas.rotate(rotate - 90, x, y);
                    canvas.drawText(text, x, y - 10, mTextPaint);
                }
                canvas.restore();
            }
        }
    }


    public int getRotationBetweenLines(float centerX, float centerY, float xInView, float yInView) {
        double rotation = 0;

        double k1 = (double) (centerY - centerY) / (centerX * 2 - centerX);
        double k2 = (double) (yInView - centerY) / (xInView - centerX);
        double tmpDegree = Math.atan((Math.abs(k1 - k2)) / (1 + k1 * k2)) / Math.PI * 180;

        if (xInView > centerX && yInView < centerY) {
            rotation = 90 - tmpDegree;
        } else if (xInView > centerX && yInView > centerY) {
            rotation = 90 + tmpDegree;
        } else if (xInView < centerX && yInView > centerY) {
            rotation = 270 - tmpDegree;
        } else if (xInView < centerX && yInView < centerY) {
            rotation = 270 + tmpDegree;
        } else if (xInView == centerX && yInView < centerY) {
            rotation = 0;
        } else if (xInView == centerX && yInView > centerY) {
            rotation = 180;
        } else {
            if(xInView > centerX) {
                rotation = 90;
            }else{
                rotation = 270;
            }
        }

        return (int) rotation;
    }

    @Override
    protected void setPaintProperty(PDFViewCtrl pdfViewCtrl, int pageIndex, Paint paint) {
        paint.setColor(mColor);
        paint.setAlpha(AppDmUtil.opacity100To255(mOpacity));
        paint.setStrokeWidth(UIAnnotFrame.getPageViewThickness(pdfViewCtrl, pageIndex, mThickness));
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
    }

    @Override
    public void onValueChanged(long property, int value) {
        super.onValueChanged(property, value);
        if (property == PropertyBar.PROPERTY_DISTANCE) {
            setScaleFromUnitIndex(value);
        } else if (property == PropertyBar.PROPERTY_DISTANCE_TIP) {
            setScaleToUnitIndex(value);
        }
    }

    @Override
    public void onValueChanged(long property, float value) {
        super.onValueChanged(property, value);
        if (property == PropertyBar.PROPERTY_DISTANCE_VALUE) {
            setScaleFromValue(value);
        } else if (property == PropertyBar.PROPERTY_DISTANCE_TIP_VALUE) {
            setScaleToValue(value);
        }
    }

    @Override
    public long getSupportedProperties() {
        if (LineConstants.INTENT_LINE_DIMENSION.equals(getIntent())) {
            return mUtil.getDistanceSupportedProperties();
        }
        return mUtil.getSupportedProperties();
    }

    @Override
    protected void setPropertyBarProperties(PropertyBar propertyBar) {
        if (mIntent.equals(LineConstants.INTENT_LINE_ARROW) || mIntent.equals(LineConstants.INTENT_LINE_DIMENSION)) {
            int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
            System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
            colors[0] = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
            propertyBar.setColors(colors);
            if (mIntent.equals(LineConstants.INTENT_LINE_DIMENSION)) {
                propertyBar.scaleFromUnit(scaleFromUnitIndex);
                propertyBar.scaleToUnit(scaleToUnitIndex);
                propertyBar.scaleFromValue(scaleFromValue);
                propertyBar.scaleToValue(scaleToValue);
            }
        } else {
            int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
            System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
            colors[0] = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
            propertyBar.setColors(colors);
        }
        super.setPropertyBarProperties(propertyBar);
    }

    private void addAnnot(int pageIndex) {
        if (mTouchCaptured && mCapturedPage >= 0) {
            float distance = AppDmUtil.distanceOfTwoPoints(mStartPt, mStopPt);
            float thickness = UIAnnotFrame.getPageViewThickness(mPdfViewCtrl, pageIndex, mThickness);
            if (distance > thickness * LineUtil.ARROW_WIDTH_SCALE / 2) {
                RectF bbox = mUtil.getArrowBBox(mStartPt, mStopPt, thickness);
                PointF startPt = new PointF(mStartPt.x, mStartPt.y);
                PointF stopPt = new PointF(mStopPt.x, mStopPt.y);
                mPdfViewCtrl.convertPageViewRectToPdfRect(bbox, bbox, pageIndex);
                mPdfViewCtrl.convertPageViewPtToPdfPt(startPt, startPt, pageIndex);
                mPdfViewCtrl.convertPageViewPtToPdfPt(stopPt, stopPt, pageIndex);

                LineAddUndoItem undoItem = new LineAddUndoItem(mAnnotHandler, mPdfViewCtrl);
                if (getIntent().equals(LineConstants.INTENT_LINE_DIMENSION)) {
                    undoItem.mHasCaption = true;

                    float factor = DistanceMeasurement.valueOf(scaleFromUnitIndex).getScaleWithDefault() * scaleToValue / scaleFromValue;
                    if (scaleFromValue == 0) {
                        factor = 0;
                    }
                    String ratio = "" + scaleFromValue + " " + DistanceMeasurement.valueOf(scaleFromUnitIndex).getName() +
                            " = " + scaleToValue + " " + DistanceMeasurement.valueOf(scaleToUnitIndex).getName();
                    String unit = "" + DistanceMeasurement.valueOf(scaleToUnitIndex).getName();
                    mAnnotHandler.addAnnot(pageIndex,
                            undoItem,
                            new RectF(bbox),
                            mColor,
                            AppDmUtil.opacity100To255(mOpacity),
                            mThickness, startPt, stopPt, getIntent(),
                            factor, unit, ratio,
                            new Event.Callback() {
                                @Override
                                public void result(Event event, boolean success) {
                                    mCapturedPage = -1;
                                }
                            });
                } else {
                    undoItem.mHasCaption = false;
                    mAnnotHandler.addAnnot(pageIndex,
                            undoItem,
                            new RectF(bbox),
                            mColor,
                            AppDmUtil.opacity100To255(mOpacity),
                            mThickness, startPt, stopPt, getIntent(),
                            new Event.Callback() {
                                @Override
                                public void result(Event event, boolean success) {
                                    mCapturedPage = -1;
                                }
                            });
                }

            } else {
                RectF bbox = mUtil.getArrowBBox(mStartPt, mStopPt, thickness);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bbox, bbox, pageIndex);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(bbox));
            }
        }
    }

    public int getScaleFromUnitIndex() {
        return scaleFromUnitIndex;
    }

    public void setScaleFromUnitIndex(int index) {
        this.scaleFromUnitIndex = index;
        if (mCurToolItem == null) return;
        mCurToolItem.property.scaleFromUnitIndex = index;
    }

    public int getScaleToUnitIndex() {
        return scaleToUnitIndex;
    }

    public void setScaleToUnitIndex(int index) {
        this.scaleToUnitIndex = index;
        if (mCurToolItem == null) return;
        mCurToolItem.property.scaleToUnitIndex = index;
    }

    public float getScaleFromValue() {
        return scaleFromValue;
    }

    public void setScaleFromValue(float value) {
        this.scaleFromValue = value;
        if (mCurToolItem == null) return;
        mCurToolItem.property.scaleFromValue = value;
    }

    public float getScaleToValue() {
        return scaleToValue;
    }

    public void setScaleToValue(float value) {
        this.scaleToValue = value;
        if (mCurToolItem == null) return;
        mCurToolItem.property.scaleToValue = value;
    }

    private int mLastScaleFromUnitIndex;
    private int mLastScaleToUnitIndex;
    private float mLastScaleFromValue;
    private float mLastScaleToValue;
    private IToolSupply mToolSupply;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new LineToolSupply(mContext);
        return mToolSupply;
    }

    private class LineToolSupply extends ToolSupplyImpl {

        public LineToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            if (mIntent.equals(LineConstants.INTENT_LINE_DEFAULT))
                return R.drawable.drawing_tool_line;
            else if (mIntent.equals(LineConstants.INTENT_LINE_ARROW))
                return R.drawable.drawing_tool_line_arrow;
            else
                return R.drawable.comment_tool_line_measure_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            if (mIntent.equals(LineConstants.INTENT_LINE_DEFAULT))
                property.type = ToolConstants.Line;
            else if (mIntent.equals(LineConstants.INTENT_LINE_ARROW))
                property.type = ToolConstants.Arrow;
            else {
                property.type = ToolConstants.Measure;
                property.scaleFromUnitIndex = scaleFromUnitIndex;
                property.scaleToUnitIndex = scaleToUnitIndex;
                property.scaleFromValue = scaleFromValue;
                property.scaleToValue = scaleToValue;
            }
            property.color = mColor;
            property.opacity = mOpacity;
            property.lineWidth = mThickness;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            if (mIntent.equals(LineConstants.INTENT_LINE_DEFAULT))
                return JsonConstants.TYPE_LINE;
            else if (mIntent.equals(LineConstants.INTENT_LINE_ARROW))
                return JsonConstants.TYPE_LINEARROW;
            else
                return JsonConstants.TYPE_LINEDIMENSION;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if(mUiExtensionsManager.getMainFrame().getCurrentTab()== ToolbarItemConfig.ITEM_DRAWING_TAB) {
                    if (itemBean.property.type == ToolConstants.Line) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_DrawingBar_Line);
                    } else if (itemBean.property.type == ToolConstants.Arrow) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_DrawingBar_Arrow);
                    }
                }else if(mUiExtensionsManager.getMainFrame().getCurrentTab()== ToolbarItemConfig.ITEM_COMMENT_TAB){
                    if (itemBean.property.type == ToolConstants.Measure) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_Distance);
                    }

                }

                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                if (itemBean.type == ToolConstants.Measure) {
                    mLastScaleFromUnitIndex = scaleFromUnitIndex;
                    mLastScaleToUnitIndex = scaleToUnitIndex;
                    mLastScaleFromValue = scaleFromValue;
                    mLastScaleToValue = scaleToValue;

                    scaleFromUnitIndex = property.scaleFromUnitIndex;
                    scaleToUnitIndex = property.scaleToUnitIndex;
                    scaleFromValue = property.scaleFromValue;
                    scaleToValue = property.scaleToValue;
                }
                mColor = property.color;
                mOpacity = property.opacity;
                mThickness = property.lineWidth;
                mUiExtensionsManager.setCurrentToolHandler(LineToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == LineToolHandler.this) {
                    if (itemBean.type ==ToolConstants.Measure){
                        scaleFromUnitIndex = mLastScaleFromUnitIndex;
                        scaleToUnitIndex = mLastScaleToUnitIndex;
                        scaleFromValue = mLastScaleFromValue;
                        scaleToValue = mLastScaleToValue;
                    }
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(final ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            if (itemBean.type == ToolConstants.Measure) {
                mLastScaleFromUnitIndex = scaleFromUnitIndex;
                mLastScaleToUnitIndex = scaleToUnitIndex;
                mLastScaleFromValue = scaleFromValue;
                mLastScaleToValue = scaleToValue;

                scaleFromUnitIndex = property.scaleFromUnitIndex;
                scaleToUnitIndex = property.scaleToUnitIndex;
                scaleFromValue = property.scaleFromValue;
                scaleToValue = property.scaleToValue;
            }
            mColor = property.color;
            mOpacity = property.opacity;
            mThickness = property.lineWidth;

            LineToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    if (itemBean.type ==ToolConstants.Measure){
                        scaleFromUnitIndex = mLastScaleFromUnitIndex;
                        scaleToUnitIndex = mLastScaleToUnitIndex;
                        scaleFromValue = mLastScaleFromValue;
                        scaleToValue = mLastScaleToValue;
                    }

                    mPropertyBar.setDismissListener(null);
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

    @Override
    public void setThickness(float thickness) {
        super.setThickness(thickness);
        if (mCurToolItem == null) return;
        mCurToolItem.property.lineWidth = thickness;
        if (mCustomPropertyListener != null)
            mCustomPropertyListener.onValueChanged(PropertyBar.PROPERTY_LINEWIDTH, thickness);
    }

    @Override
    public void setOpacity(int opacity) {
        super.setOpacity(opacity);
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
        if (mCustomPropertyListener != null)
            mCustomPropertyListener.onValueChanged(PropertyBar.PROPERTY_OPACITY, opacity);
    }

    @Override
    public void setColor(int color) {
        super.setColor(color);
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
        if (mCustomPropertyListener != null)
            mCustomPropertyListener.onValueChanged(PropertyBar.PROPERTY_COLOR, color);
    }
}
