/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink.ocr;

import android.util.Log;

import com.google.android.gms.tasks.SuccessContinuation;
import com.google.android.gms.tasks.Task;
import com.google.android.gms.tasks.Tasks;
import com.google.mlkit.vision.digitalink.recognition.DigitalInkRecognizer;
import com.google.mlkit.vision.digitalink.recognition.Ink;
import com.google.mlkit.vision.digitalink.common.RecognitionResult;

import java.util.concurrent.atomic.AtomicBoolean;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/** Task to run asynchronously to obtain recognition results. */
public class RecognitionTask {

  private static final String TAG = "MLKD.RecognitionTask";
  private final DigitalInkRecognizer recognizer;
  private final Ink ink;
  @Nullable private RecognizedInk currentResult;
  private final AtomicBoolean cancelled;
  private final AtomicBoolean done;

  public RecognitionTask(DigitalInkRecognizer recognizer, Ink ink) {
    this.recognizer = recognizer;
    this.ink = ink;
    this.currentResult = null;
    cancelled = new AtomicBoolean(false);
    done = new AtomicBoolean(false);
  }

  public void cancel() {
    cancelled.set(true);
  }

  public boolean done() {
    return done.get();
  }

  @Nullable
  public RecognizedInk result() {
    return this.currentResult;
  }

  /** Helper class that stores an ink along with the corresponding recognized text. */
  public static class RecognizedInk {
    public final Ink ink;
    public final String text;

    RecognizedInk(Ink ink, String text) {
      this.ink = ink;
      this.text = text;
    }
  }

  public Task<String> run() {
    Log.i(TAG, "RecoTask.run");
    return recognizer
            .recognize(this.ink)
            .onSuccessTask(new SuccessContinuation<RecognitionResult, String>() {
              @NonNull
              @Override
              public Task<String> then(RecognitionResult result) throws Exception {
                if (cancelled.get() || result.getCandidates().isEmpty()) {
                  return Tasks.forResult(null);
                }
                currentResult = new RecognizedInk(ink, result.getCandidates().get(0).getText());
                Log.i(TAG, "result: " + currentResult.text);
                done.set(true);
                return Tasks.forResult(currentResult.text);
              }
            });
  }
}
