/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink.ocr;


import android.app.Activity;
import android.content.Context;
import android.graphics.PointF;
import android.view.View;
import android.view.ViewGroup;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Ink;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.ink.InkAnnotUtil;
import com.foxit.uiextensions.controls.dialog.FxProgressDialog;
import com.foxit.uiextensions.controls.dialog.UIMatchDialog;
import com.foxit.uiextensions.controls.toolbar.BaseBar;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppSharedPreferences;
import com.foxit.uiextensions.utils.UIToast;
import com.google.android.gms.tasks.OnFailureListener;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.mlkit.vision.digitalink.recognition.DigitalInkRecognitionModelIdentifier;

import java.util.ArrayList;
import java.util.Set;

import androidx.annotation.NonNull;
import androidx.annotation.VisibleForTesting;


public class InkRecognizeTextDialog extends UIMatchDialog implements StrokeManager.DownloadedModelsChangedListener,
        StrokeManager.StatusChangedListener, StrokeManager.ContentChangedListener {
    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUIExtensionsManager;
    private ViewGroup mContentView;
    private EditText mEditText;

    private TextView mTextView;
    @VisibleForTesting
    final StrokeManager mStrokeManager = new StrokeManager();
    ModelLanguageContainer mItem;
    private int mSelectPosition = 0;
    private ArrayList<Annot> mAnnots;
    private RecognitionInkCallBack mRecognitionInkCallBack;

    public InkRecognizeTextDialog(Context context,
                                  UIExtensionsManager uIExtensionsManager
    ) {
        super(context);
        mContext = context.getApplicationContext();
        mUIExtensionsManager = uIExtensionsManager;
        mPdfViewCtrl = uIExtensionsManager.getPDFViewCtrl();
        initStyle();
        initView();
        setOnDLDismissListener(new DismissListener() {
            @Override
            public void onDismiss() {
                mUIExtensionsManager.unregisterThemeEventListener(mThemeColorChangedListener);
            }
        });
    }

    private void initStyle() {
        setBackButtonStyle(TEXT_BACK);
        setBackButtonText(AppResource.getString(mContext, R.string.fx_string_cancel));
        setTitle(AppResource.getString(mContext, R.string.ocr_ink_recognize_text));
        setTitlePosition(BaseBar.TB_Position.Position_CENTER);
        setRightButtonVisible(View.VISIBLE);
        setRightButtonText(AppResource.getString(mContext, R.string.fx_string_ok));
        setStyle(DLG_TITLE_STYLE_BG_DEFAULT);
        setListener(new DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {

            }

            @Override
            public void onTitleRightButtonClick() {
                dismiss();
                mRecognitionInkCallBack.callBack(mAnnots,mEditText.getText().toString());

            }
        });
    }


    private void initView() {
        mContentView = (ViewGroup) View.inflate(mContext, R.layout.ink_recognize_text_layout, null);
        setContentView(mContentView);
        LinearLayout languagesLl = mContentView.findViewById(R.id.ink_recognize_text_languages_ll);
        mEditText = mContentView.findViewById(R.id.ink_recognize_text_more);
        mTextView = mContentView.findViewById(R.id.ink_recognize_text_languages_tv);
        languagesLl.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                showLanguagesDialog();
            }
        });
    }

    public void loadData() {
        mUIExtensionsManager.registerThemeEventListener(mThemeColorChangedListener);
        mStrokeManager.setDownloadedModelsChangedListener(this);
        mStrokeManager.setContentChangedListener(this);
        mStrokeManager.setStatusChangedListener(this);
        mStrokeManager.setClearCurrentInkAfterRecognition(true);
        mStrokeManager.setTriggerRecognitionAfterInput(false);
        String languages = ModelLanguageContainer.getCurrentRegionalLanguage();
        String inkLanguages = AppSharedPreferences.getInstance(mContext).getString(mContext.getPackageName(),"InkLanguages","");
        if(!inkLanguages.isEmpty()){
            languages = inkLanguages;
        }

        for (DigitalInkRecognitionModelIdentifier modelIdentifier :
                DigitalInkRecognitionModelIdentifier.allModelIdentifiers()) {
            if (languages.equals(modelIdentifier.getLanguageTag())) {
                mStrokeManager.setActiveModel(languages);
                mTextView.setText(AppResource.getString(mContext,ModelLanguageContainer.getLanguagesId(modelIdentifier.getLanguageTag())));
                showProgressDialog();
                mStrokeManager.modelManager.checkIsModelDownloaded()
                        .addOnSuccessListener(new OnSuccessListener<Boolean>() {
                            @Override
                            public void onSuccess(Boolean aBoolean) {
                                if (aBoolean) {
                                    recognitionInk();
                                } else {
                                    mStrokeManager.download();
                                }
                            }
                        })
                        .addOnFailureListener(new OnFailureListener() {
                            @Override
                            public void onFailure(@NonNull Exception e) {
                                mStrokeManager.download();
                            }
                        });
                break;
            }
        }

    }

    public void setAnnots(ArrayList<Annot> annots) {
        mAnnots = annots;
    }

    private void recognitionInk() {
        mStrokeManager.reset();
        ArrayList<ArrayList<ArrayList<PointF>>> inkListss = new ArrayList<>();
        int index = 0;
        for (Annot annot : mAnnots) {
            try {
                ArrayList<ArrayList<PointF>> inkLists = InkAnnotUtil.generateInkList(((Ink) annot).getInkList());
                inkListss.add(inkLists);
                index = annot.getPage().getIndex();
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
        if (inkListss.size() > 0) {
            mStrokeManager.addPoint(mPdfViewCtrl, inkListss, index);
            mStrokeManager.recognize();
        }
    }
    public void setRecognitionInkCallBack(RecognitionInkCallBack callback) {
        mRecognitionInkCallBack = callback;
    }

    public interface RecognitionInkCallBack {
        void callBack(ArrayList<Annot> annots,String text);
    }
    private void showLanguagesDialog() {
        Activity activity = mUIExtensionsManager.getAttachedActivity();
        final String inkLanguages = AppSharedPreferences.getInstance(mContext).getString(mContext.getPackageName(),"InkLanguages","");
        InkLanguagesDialog inkLanguagesDialog = new InkLanguagesDialog(activity, mUIExtensionsManager, mStrokeManager);
        inkLanguagesDialog.setPosition(mSelectPosition);
        inkLanguagesDialog.loadData();
        inkLanguagesDialog.showDialog();
        inkLanguagesDialog.setLanguagesCallback(new InkLanguagesDialog.LanguagesCallback() {
            @Override
            public void onDismiss(int position, ModelLanguageContainer item) {
                mSelectPosition = position;
                String language = "";
                if (mItem != null) {
                    language = mItem.getLanguageTag();
                }else{
                    if(!inkLanguages.isEmpty()){
                        language = inkLanguages;
                    }else{
                        language = ModelLanguageContainer.getCurrentRegionalLanguage();
                    }
                }

                if (!language.equals(item.getLanguageTag())) {
                    showProgressDialog();
                    recognitionInk();
                }
                mItem = item;
                mTextView.setText(AppResource.getString(mContext,ModelLanguageContainer.getLanguagesId(item.getLanguageTag())));
            }
        });
    }

    private IThemeEventListener mThemeColorChangedListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            dismiss();
        }
    };

    @Override
    public void onDownloadedModelsChanged(Set<String> downloadedLanguageTags) {
        recognitionInk();
    }

    @Override
    public void onStatusChanged() {

    }

    @Override
    public void onInkContentChanged(boolean isConvert) {
        if(!isConvert){
            UIToast.getInstance(mContext).show(AppResource.getString(mContext,R.string.ocr_ink_recognition_failed));
            dismissProgressDialog();
//            dismiss();
            return;
        }
        for (RecognitionTask.RecognizedInk recognizedInk : mStrokeManager.getContent()) {
            mEditText.setText(recognizedInk.text);
            mEditText.setSelection(recognizedInk.text.length());
            dismissProgressDialog();
        }
    }

    private FxProgressDialog mProgressDialog;

    public void showProgressDialog() {
        if (mUIExtensionsManager.getAttachedActivity() == null) {
            UIToast.getInstance(mContext).show(AppResource.getString(mContext.getApplicationContext(), R.string.rv_unknown_error));
            return;
        }
        dismissProgressDialog();
        mProgressDialog = new FxProgressDialog(mUIExtensionsManager.getAttachedActivity(),"");
        mProgressDialog.setTipsHide(true);
        mProgressDialog.show();
    }

    public void dismissProgressDialog() {
        if (mProgressDialog != null && mProgressDialog.isShowing()) {
            mProgressDialog.dismiss();
            mProgressDialog = null;
        }
    }
}



