/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink;

import android.content.Context;
import android.content.res.ColorStateList;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.graphics.PaintFlagsDrawFilter;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Handler;
import android.os.Looper;
import android.util.SparseArray;
import android.view.MotionEvent;
import android.view.View;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.PSI;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Ink;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.annots.AbstractToolHandler;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.UIAnnotFrame;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.config.uisettings.annotations.annots.PencilConfig;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.ResultInfo;

import java.util.ArrayList;


public class InkToolHandler extends AbstractToolHandler {
    public static final float IA_MIN_DIST = 0.5f;
    protected static final String PROPERTY_KEY = "INK";

    protected InkAnnotHandler mAnnotHandler;
    private Ink curInk;

    private boolean mTouchCaptured = false;
    private int mCapturedPage = -1;

    private ArrayList<PointF> line;
    private ArrayList<PointF> lastLine;

    private ArrayList<PSIData> curPSIData;
    private final ArrayList<ArrayList<PSIData>> psiDataArray = new ArrayList<>();

    private Path mPath;
    private final SparseArray<Path> pendingPaths = new SparseArray<>();
    private int pendingPathIdCounter = 0;

    private final PointF downPoint = new PointF(0, 0);
    private final PointF mLastPt = new PointF(0, 0);
    private final Paint mPaint;
    private final PaintFlagsDrawFilter mPaintFilter;
    int mPencilType;

    private final PsiBitmap mPsiBmp;
    private PSI mPsi = null;

    private boolean mbHighlighter = false;
    private boolean mShowPenOnlySwitch = true;

    private long mInkTimeout = -1;

    public InkToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        super(context, pdfViewCtrl, Module.MODULE_NAME_INK, PROPERTY_KEY);
        mPsiBmp = new PsiBitmap();
        mColor = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];

        mPaint = new Paint();
        mPaint.setStyle(Style.STROKE);
        mPaint.setAntiAlias(true);
        mPaint.setDither(true);
        mPaintFilter = new PaintFlagsDrawFilter(0, Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG);

        mPencilType = InkConstants.PENCIL_WITH_PEN;
        if (mUiExtensionsManager.getConfig().modules.annotations.isLoadPencil) {
            PencilConfig pencilConfig = mUiExtensionsManager.getConfig().uiSettings.annotations.pencil;
            mColor = pencilConfig.color;
            mOpacity = (int) (pencilConfig.opacity * 100);
            mThickness = pencilConfig.thickness;
            mShowPenOnlySwitch = mUiExtensionsManager.getConfig().uiSettings.showPenOnlySwitch;
            setColorChangeListener(new ColorChangeListener() {
                @Override
                public void onColorChange(int color) {
                    if (mCurToolItem == null) return;
                    mCurToolItem.property.color = color;
                    ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
                }
            });
        }
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_INK;
    }

    @Override
    public void setColor(final int color) {
        if (mColor == color) return;
        this.reset();

        mColor = color;
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        if (mCustomPropertyListener != null)
            mCustomPropertyListener.onValueChanged(PropertyBar.PROPERTY_COLOR, color);
    }

    @Override
    public void setOpacity(final int opacity) {
        if (mOpacity == opacity) return;
        this.reset();

        mOpacity = opacity;
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
        if (mCustomPropertyListener != null)
            mCustomPropertyListener.onValueChanged(PropertyBar.PROPERTY_OPACITY, opacity);
    }

    @Override
    public void setThickness(final float thickness) {
        if (mThickness == thickness) return;
        this.reset();

        mThickness = thickness;
        if (mCurToolItem == null) return;
        mCurToolItem.property.lineWidth = thickness;
        if (mCustomPropertyListener != null)
            mCustomPropertyListener.onValueChanged(PropertyBar.PROPERTY_LINEWIDTH, thickness);
    }

    @Override
    public void onValueChanged(long property, final int value) {
        super.onValueChanged(property, value);
        if (property == PropertyBar.PROPERTY_PENCIL_TYPE) {
            if (mPencilType == value) return;
            this.reset();

            mPencilType = value;
            if (mCurToolItem == null) return;
            mCurToolItem.property.style = value;
            if (mCustomPropertyListener != null)
                mCustomPropertyListener.onValueChanged(PropertyBar.PROPERTY_PENCIL_TYPE, mPencilType);
        }
    }

    @Override
    public void onActivate() {
        mInkTimeout = mUiExtensionsManager.getInkDrawingTimeout();
        this.drawMode = mUiExtensionsManager.getInkDrawToolType();
        mCapturedPage = -1;

        resetPropertyBar();
    }

    @Override
    protected void resetPropertyBar() {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        mPropertyBar.setColors(colors);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_LINEWIDTH, mThickness);
        if (!mbHighlighter) {
            mPropertyBar.setProperty(PropertyBar.PROPERTY_PENCIL_TYPE, mPencilType);
            mPropertyBar.setProperty(PropertyBar.PROPERTY_MAX_LINEWIDTH, 21);
        } else {
            mPropertyBar.setProperty(PropertyBar.PROPERTY_MAX_LINEWIDTH, 45);
        }

        mPropertyBar.clearPropertyTitle();
        mPropertyBar.reset(getSupportedProperties());
        if (mShowPenOnlySwitch)
            mPropertyBar.addCustomItem(PropertyBar.PROPERTY_PEN_ONLY, getPenOnlyView(), 0, -1);
        mPropertyBar.setPropertyChangeListener(this);
    }

    @Override
    protected void onCreateValueChanged(long property, Object value) {
        if (property == PropertyBar.PROPERTY_PENCIL_TYPE) {
            if (mPencilType == (int) value) return;
            mPencilType = (int) value;
        }
    }

    private View mPenOnlyView;
    private RelativeLayout mPenOnlySwitchRela;
    private ImageView mPenOnlySwitchIv;
    private TextView mPenOnlyTitleTv;

    private View getPenOnlyView() {
        mPenOnlyView = View.inflate(mContext, R.layout.pb_ink_draw_type, null);
        mPenOnlyTitleTv = mPenOnlyView.findViewById(R.id.ink_pen_only_switch_tv);
        mPenOnlySwitchRela = mPenOnlyView.findViewById(R.id.ink_pen_only_switch_rl);
        mPenOnlySwitchIv = mPenOnlyView.findViewById(R.id.ink_pen_only_switch_iv);
        ThemeUtil.setBackgroundTintList(mPenOnlySwitchRela, getSelectedButtonColorStateList());
        setSelectedButtonState(mPenOnlySwitchRela, mPenOnlySwitchIv, mUiExtensionsManager.getInkDrawToolType() == InkDrawToolType.STYLUS);

        mPenOnlySwitchRela.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mPenOnlySwitchRela.isSelected()) {
                    drawMode = InkDrawToolType.STYLUS_OR_FINGER;
                    setSelectedButtonState(mPenOnlySwitchRela, mPenOnlySwitchIv, false);
                    if (mCurToolItem != null) {
                        mUiExtensionsManager.setInkDrawToolType(InkDrawToolType.STYLUS_OR_FINGER);
                    }
                } else {
                    drawMode = InkDrawToolType.STYLUS;
                    setSelectedButtonState(mPenOnlySwitchRela, mPenOnlySwitchIv, true);
                    if (mCurToolItem != null) {
                        mUiExtensionsManager.setInkDrawToolType(InkDrawToolType.STYLUS);
                    }
                }
            }
        });
        return mPenOnlyView;
    }

    private void setSelectedButtonState(RelativeLayout checkLayout, ImageView checkView, boolean isSelected) {
        checkLayout.setSelected(isSelected);
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) checkView.getLayoutParams();
        if (params == null) return;
        params.removeRule(isSelected ? RelativeLayout.ALIGN_PARENT_LEFT : RelativeLayout.ALIGN_PARENT_RIGHT);
        params.addRule(isSelected ? RelativeLayout.ALIGN_PARENT_RIGHT : RelativeLayout.ALIGN_PARENT_LEFT, RelativeLayout.TRUE);
        checkView.setLayoutParams(params);
    }

    void updateTheme() {
        if (mPenOnlyView != null) {
            mPenOnlyTitleTv.setTextColor(AppResource.getColor(mContext, R.color.t2));
            ThemeUtil.setBackgroundTintList(mPenOnlySwitchRela, getSelectedButtonColorStateList());
            setSelectedButtonState(mPenOnlySwitchRela, mPenOnlySwitchIv, mUiExtensionsManager.getInkDrawToolType() == InkDrawToolType.STYLUS);
        }
    }

    private ColorStateList getSelectedButtonColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.p1);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.p1);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    @Override
    public void onDeactivate() {
        this.mTouchCaptured = false;
        this.mLastPt.set(0, 0);
        this.drawMode = InkDrawToolType.STYLUS_OR_FINGER;
        this.reset();
    }

    private final PointF history_pt = new PointF();
    private final ArrayList<PSIData> historicalPSIData = new ArrayList<>();
    private final RectF mPSIRectF = new RectF();
    private int mPageWidth;
    private int mPageHeight;
    private int drawMode = InkDrawToolType.STYLUS_OR_FINGER;
    private final PointF[] vertices = new PointF[4];

    @Override
    public boolean onTouchEvent(int pageIndex, final MotionEvent e) {
        int tool = e.getToolType(0);
        if (this.drawMode == InkDrawToolType.STYLUS_OR_FINGER && e.getActionMasked() == MotionEvent.ACTION_DOWN) {
            if (tool == MotionEvent.TOOL_TYPE_STYLUS) {
                this.drawMode = InkDrawToolType.STYLUS;
            } else if (tool == MotionEvent.TOOL_TYPE_FINGER) {
                this.drawMode = InkDrawToolType.FINGER;
            }
        }

        if ((this.drawMode == InkDrawToolType.STYLUS && tool != MotionEvent.TOOL_TYPE_STYLUS) ||
                (this.drawMode == InkDrawToolType.FINGER && tool != MotionEvent.TOOL_TYPE_FINGER)) {
            this.mTouchCaptured = false;
            return mUiExtensionsManager.defaultTouchEvent(pageIndex, e);
        }

        boolean handled = true;
        int action = e.getActionMasked();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                handled = onInkToolTouch(pageIndex, e);
                break;
            case MotionEvent.ACTION_POINTER_DOWN:
                this.reset();

                MotionEvent downEvent = MotionEvent.obtain(e);
                downEvent.setAction(MotionEvent.ACTION_DOWN);
                handled = mPdfViewCtrl.defaultTouchEvent(downEvent);
                downEvent.recycle();
                break;
            case MotionEvent.ACTION_MOVE:
                if (e.getPointerCount() > 1) {
                    handled = mPdfViewCtrl.defaultTouchEvent(e);
                    break;
                }
            case MotionEvent.ACTION_CANCEL:
            case MotionEvent.ACTION_UP:
                if (!mTouchCaptured) {
                    handled = mPdfViewCtrl.defaultTouchEvent(e);
                } else {
                    handled = onInkToolTouch(pageIndex, e);
                }
                break;
            default:
                break;
        }
        return handled;
    }

    private boolean onInkToolTouch(final int pageIndex, MotionEvent e) {
        PointF point = new PointF(e.getX(), e.getY());
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
        float pressure = calibrationPressure(e.getPressure());

        int action = e.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (!mTouchCaptured) {
                    this.cancelInkTimeOut();

                    if (mCapturedPage == -1) {
                        mTouchCaptured = true;
                        mCapturedPage = pageIndex;
                    } else if (pageIndex == mCapturedPage) {
                        mTouchCaptured = true;
                    } else {
                        // support add ink to multiple pages
                        this.reset();

                        mTouchCaptured = true;
                        mCapturedPage = pageIndex;
                    }

                    this.line = new ArrayList<>();
                    this.line.add(new PointF(point.x, point.y));
                    this.mLastPt.set(point);
                    this.downPoint.set(point);

                    this.vertices[0] = point;
                    mPageWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
                    mPageHeight = mPdfViewCtrl.getPageViewHeight(pageIndex);

                    if (mPencilType == InkConstants.PENCIL_WITH_PEN) {
                        mPath = new Path();
                        mPath.moveTo(point.x, point.y);
                    } else {
                        if (this.mPsi == null) {
                            this.initPsiCanvas();
                        }

                        if (mPsiBmp.mLeftTop == null || mPsiBmp.mPage != mCapturedPage) {
                            mPsiBmp.mPage = pageIndex;
                            mPsiBmp.mLeftTop = new PointF(0, 0);
                            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(mPsiBmp.mLeftTop, mPsiBmp.mLeftTop, pageIndex);
                            mPsiBmp.mPvSize = new PointF(mPageWidth, mPageHeight);
                        }

                        PointF psiPointF = new PointF();
                        mPdfViewCtrl.convertPageViewPtToDisplayViewPt(point, psiPointF, pageIndex);
                        PSIData psiData = new PSIData(psiPointF, pressure, com.foxit.sdk.common.Path.e_TypeMoveTo);

                        this.curPSIData = new ArrayList<>();
                        this.curPSIData.add(psiData);
                        addPsiPoint(psiData);

                        if (mPSIRectF.isEmpty()) {
                            mPSIRectF.left = psiPointF.x;
                            mPSIRectF.top = psiPointF.y;
                        }
                    }
                }
                return true;
            case MotionEvent.ACTION_MOVE:
                if (!this.mTouchCaptured || this.line == null ||
                        point.x < 0 || point.x > this.mPageWidth || point.y < 0 || point.y > this.mPageHeight) {
                    break;
                }

                if (point.equals(this.mLastPt)) {
                    break;
                }
                InkAnnotUtil.correctPvPoint(point, this.mPageWidth, this.mPageHeight);

                if (mPencilType == InkConstants.PENCIL_WITH_PEN) {
                    this.line.add(point);

                    if (this.line.size() == 2) {
                        this.vertices[1] = this.vertices[0];
                        this.vertices[2] = this.line.get(0);
                        this.vertices[3] = point;
                    } else if (this.line.size() >= 3) {
                        this.vertices[1] = this.line.get(this.line.size() - 3);
                        this.vertices[2] = this.line.get(this.line.size() - 2);
                        this.vertices[3] = point;
                    }

                    PointF[] cp = InkAnnotUtil.calcBesselControlPoints(this.vertices, 0.8f);
                    if (cp != null) {
                        this.mPath.cubicTo(cp[0].x, cp[0].y, cp[1].x, cp[1].y, this.vertices[2].x, this.vertices[2].y);
                        this.vertices[0] = this.vertices[1];
                        this.mPdfViewCtrl.invalidate();
                    }
                } else {
                    boolean dist = Math.abs(point.x - downPoint.x) >= IA_MIN_DIST || Math.abs(point.y - downPoint.y) >= IA_MIN_DIST;
                    if (dist) {
//                        if (mUiExtensionsManager.getConfig().uiSettings.annotations.pencil.addHistoricalPoints) {
//                            this.historicalPSIData.clear();
//                            int historySize = e.getHistorySize();
//                            for (int i = 0; i < historySize; i++) {
//                                history_pt.set(e.getHistoricalX(i), e.getHistoricalY(i));
//                                mPdfViewCtrl.convertDisplayViewPtToPageViewPt(history_pt, history_pt, pageIndex);
//                                InkAnnotUtil.correctPvPoint(history_pt, this.mPageWidth, this.mPageHeight);
//                                if (history_pt.x - mLastPt.x >= IA_MIN_DIST || history_pt.y - mLastPt.y >= IA_MIN_DIST) {
//                                    this.line.add(new PointF(history_pt.x, history_pt.y));
//                                    mLastPt.set(this.line.get(this.line.size() - 1));
//
//                                    PointF psiPointF = new PointF(e.getHistoricalX(i), e.getHistoricalY(i));
//                                    float historicalPressure = calibrationPressure(e.getHistoricalPressure(i));
//                                    PSIData psiData = new PSIData(psiPointF, historicalPressure, com.foxit.sdk.common.Path.e_TypeLineTo);
//                                    this.curPSIData.add(psiData);
//
//                                    this.historicalPSIData.add(psiData);
//                                }
//                            }
//                            addPsiPoint(this.historicalPSIData);
//                        }

                        if (!point.equals(this.mLastPt)) {
                            this.line.add(point);

                            PointF psiPointF = new PointF();
                            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(point, psiPointF, pageIndex);
                            PSIData psiData = new PSIData(psiPointF, pressure, com.foxit.sdk.common.Path.e_TypeLineTo);

                            this.curPSIData.add(psiData);
                            addPsiPoint(psiData);
                        }
                    }
                }
                this.mLastPt.set(point.x, point.y);
                break;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (!this.mTouchCaptured || this.line == null) {
                    break;
                }
                InkAnnotUtil.correctPvPoint(point, this.mPageWidth, this.mPageHeight);
                if (mPencilType == InkConstants.PENCIL_WITH_PEN) {
                    if (this.line.size() == 1) {
                        PointF _point = new PointF(point.x + 0.001f, point.y + 0.001f);

                        this.line.add(_point);
                        this.mPath.lineTo(_point.x, _point.y);
                    } else {
                        this.line.add(point);

                        if (this.line.size() == 2) {
                            this.vertices[1] = this.vertices[0];
                        } else if (this.line.size() >= 3) {
                            this.vertices[0] = this.line.get(this.line.size() - 3);
                            this.vertices[1] = this.line.get(this.line.size() - 2);
                        }
                        this.vertices[2] = point;
                        this.vertices[3] = point;

                        PointF[] cp = InkAnnotUtil.calcBesselControlPoints(this.vertices, 0.8f);
                        if (cp != null) {
                            this.mPath.cubicTo(cp[0].x, cp[0].y, cp[1].x, cp[1].y, this.vertices[2].x, this.vertices[2].y);
                            this.mPdfViewCtrl.invalidate();
                        }
                    }

                    ++this.pendingPathIdCounter;
                    this.pendingPaths.set(this.pendingPathIdCounter, this.mPath);
                    this.mPath = null;
                } else {
                    if (this.line.size() == 1) {
                        float offset = InkAnnotUtil.widthOnPageView(mPdfViewCtrl, pageIndex, 1.5f);
                        PointF offsetPoint = new PointF(point.x, point.y);
                        offsetPoint.offset(offset, 0);
                        this.line.add(new PointF(offsetPoint.x, offsetPoint.y));

                        PointF psiPointF = new PointF();
                        mPdfViewCtrl.convertPageViewPtToDisplayViewPt(offsetPoint, psiPointF, pageIndex);
                        PSIData psiData = new PSIData(psiPointF, pressure, com.foxit.sdk.common.Path.e_TypeLineTo);
                        this.curPSIData.add(psiData);
                        addPsiPoint(psiData);
                    } else {
                        this.line.add(point);
                    }
                    PointF psiPointF = new PointF();
                    mPdfViewCtrl.convertPageViewPtToDisplayViewPt(point, psiPointF, pageIndex);
                    PSIData psiData = new PSIData(psiPointF, pressure, com.foxit.sdk.common.Path.e_TypeLineToCloseFigure);

                    this.curPSIData.add(psiData);
                    addPsiPoint(psiData);

                    this.psiDataArray.add(this.curPSIData);
                    this.curPSIData = null;
                }

                this.lastLine = this.line;
                this.line = null;

                if (this.curInk == null) {
                    this.addAnnot(pageIndex);
                } else {
                    this.modifyAnnot(pageIndex);
                }

                if (mInkTimeout > 0) {
                    handleInkTimeOut();
                }
                this.removePendingPath(this.pendingPathIdCounter);
                mTouchCaptured = false;
                mLastPt.set(0, 0);
                break;
            default:
                return true;
        }
        return true;
    }

    private void addPsiPoint(PSIData psiData) {
        try {
            if (mPsi == null || mPsi.isEmpty() || psiData.pointF == null) return;

            mPsi.addPoint(AppUtil.toFxPointF(psiData.pointF), psiData.path_type, Math.min(psiData.pressure, 1.0f));
            mPSIRectF.union(psiData.pointF.x, psiData.pointF.y);
            mPsiBmp.mBmp = mPsi.getBitmap();
            if (mPdfViewCtrl.isPageVisible(mCapturedPage)) {
                RectF rectF = new RectF(mPSIRectF);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rectF));
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void addPsiPoint(ArrayList<PSIData> psiData) {
        try {
            if (mPsi == null || mPsi.isEmpty() || psiData.isEmpty()) return;

            for (PSIData data : psiData) {
                mPsi.addPoint(AppUtil.toFxPointF(data.pointF), data.path_type, Math.min(data.pressure, 1.0f));
                mPSIRectF.union(data.pointF.x, data.pointF.y);
            }
            mPsiBmp.mBmp = mPsi.getBitmap();
            if (mPdfViewCtrl.isPageVisible(mCapturedPage)) {
                RectF rectF = new RectF(mPSIRectF);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rectF));
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void removePendingPath(final int pathId) {
        class RemovePendingPathTask extends Task {
            RemovePendingPathTask(Task.CallBack callBack) {
                super(callBack);
                this.setPriority(2);
            }

            @Override
            protected void execute() {
                pendingPaths.remove(pathId);
            }
        }
        this.mPdfViewCtrl.addTask(new RemovePendingPathTask(new Task.CallBack() {
            @Override
            public void result(Task task) {
                mPdfViewCtrl.invalidate();
            }
        }));
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }

        mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
        return onTouchEvent(pageIndex, motionEvent);
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        return mUiExtensionsManager.getCurrentToolHandler() == this;
    }

    private final Rect tv_srcrect = new Rect();
    private final Rect tv_dstrect = new Rect();

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mCapturedPage == pageIndex) {
            // draw current creating annotation
            setPaintProperty(mPdfViewCtrl, pageIndex, mPaint);
            canvas.setDrawFilter(mPaintFilter);

            if (mPencilType == InkConstants.PENCIL_WITH_PEN) {
                mPaint.setColorFilter(null);

                for (int i = 0; i < this.pendingPaths.size(); i++) {
                    Path path = this.pendingPaths.valueAt(i);
                    if (path != null) {
                        canvas.drawPath(path, mPaint);
                    }
                }

                if (mPath != null) {
                    canvas.drawPath(mPath, mPaint);
                }
            } else {
                if (mPsiBmp.mBmp != null && mPsiBmp.mPage == pageIndex) {
                    mPaint.setColorFilter(new PorterDuffColorFilter(mColor, PorterDuff.Mode.SRC_IN));
                    if (mPsiBmp.mPvSize.x == mPageWidth && mPsiBmp.mPvSize.y == mPageHeight) {
                        canvas.getClipBounds(tv_dstrect);
                        tv_srcrect.set(tv_dstrect);
                        tv_srcrect.offset((int) mPsiBmp.mLeftTop.x, (int) mPsiBmp.mLeftTop.y);
                        tv_srcrect.intersect(0, 0, mPsiBmp.mBmp.getWidth(), mPsiBmp.mBmp.getHeight());
                        tv_dstrect.set(tv_srcrect);
                        tv_dstrect.offset((int) -mPsiBmp.mLeftTop.x, (int) -mPsiBmp.mLeftTop.y);
                        canvas.drawBitmap(mPsiBmp.mBmp, tv_srcrect, tv_dstrect, mPaint);
                    } else {
                        float scale = (float) mPsiBmp.mPvSize.x / mPageWidth;
                        canvas.getClipBounds(tv_dstrect);
                        tv_srcrect.set(tv_dstrect);
                        tv_srcrect.left *= scale;
                        tv_srcrect.top *= scale;
                        tv_srcrect.right *= scale;
                        tv_srcrect.bottom *= scale;
                        tv_srcrect.offset((int) mPsiBmp.mLeftTop.x, (int) mPsiBmp.mLeftTop.y);
                        tv_srcrect.intersect(0, 0, mPsiBmp.mBmp.getWidth(), mPsiBmp.mBmp.getHeight());
                        tv_dstrect.set(tv_srcrect);
                        tv_dstrect.offset((int) -mPsiBmp.mLeftTop.x, (int) -mPsiBmp.mLeftTop.y);
                        tv_dstrect.left /= scale;
                        tv_dstrect.top /= scale;
                        tv_dstrect.right /= scale;
                        tv_dstrect.bottom /= scale;
                        canvas.drawBitmap(mPsiBmp.mBmp, tv_srcrect, tv_dstrect, mPaint);
                    }
                }
            }
        }
    }

    @Override
    protected void setPaintProperty(PDFViewCtrl pdfViewCtrl, int pageIndex, Paint paint) {
        paint.setColor(mColor);
        paint.setAlpha(AppDmUtil.opacity100To255(mOpacity));
        paint.setStrokeCap(Paint.Cap.ROUND);
        paint.setStrokeWidth(UIAnnotFrame.getPageViewThickness(pdfViewCtrl, pageIndex, mThickness));
    }

    @Override
    public long getSupportedProperties() {
        return InkAnnotUtil.getSupportedProperties(true, mbHighlighter);
    }

    @Override
    protected void setPropertyBarProperties(PropertyBar propertyBar) {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        colors[0] = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
        propertyBar.setColors(colors);
        propertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        super.setPropertyBarProperties(propertyBar);
    }

    @Override
    public void onConfigurationChanged(Configuration newConfig) {
        this.reset();
    }

    private void addAnnot(final int pageIndex) {
        if (pageIndex == -1) {
            return;
        }

        if (this.lastLine == null || this.lastLine.isEmpty()) {
            return;
        }

        float thickness = mThickness;
        ArrayList<Float> pressures = null;
        if (mPencilType == InkConstants.PENCIL_WITH_BRUSH) {
            pressures = new ArrayList<>();

            for (ArrayList<PSIData> psiDataArrayList : this.psiDataArray) {
                for (PSIData psiData : psiDataArrayList) {
                    pressures.add(psiData.pressure);
                }
            }

            int diameter = thicknessToPsiDiameter(pageIndex, thickness);
            thickness = diameter < 1 ? 0.3f : thickness;
        }

        ResultInfo<ArrayList<PointF>, RectF, Void> resultInfo = InkAnnotUtil.convertPageLineToPdfLine(mPdfViewCtrl, pageIndex, this.lastLine);
        ArrayList<PointF> points = resultInfo.param1;
        RectF bbox = resultInfo.param2;
        bbox.inset(-thickness, -thickness);

        final InkAddUndoItem undoItem = new InkAddUndoItem(mAnnotHandler, mPdfViewCtrl);
        undoItem.mPageIndex = pageIndex;
        undoItem.mNM = AppDmUtil.randomUUID(null);
        undoItem.mBBox = new RectF(bbox);
        undoItem.mAuthor = this.mUiExtensionsManager.getAnnotAuthor();
        undoItem.mFlags = Annot.e_FlagPrint;
        undoItem.mSubject = InkAnnotHandler.SUBJECT;
        undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
        undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
        undoItem.mColor = mColor;
        undoItem.mOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
        undoItem.mLineWidth = mThickness;
        undoItem.mPencilType = mPencilType;
        undoItem.mPath = InkAnnotUtil.pointsToPath(points);
        undoItem.mPSIPressures = pressures;

        try {
            PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            Ink annot = (Ink) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Ink, AppUtil.toFxRectF(bbox)), Annot.e_Ink);

            InkEvent event = new InkEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, annot, mPdfViewCtrl);
            event.add();

            this.mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, annot);
            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                RectF annotRect = AppUtil.toRectF(annot.getRect());
                mPdfViewCtrl.convertPdfRectToPageViewRect(annotRect, annotRect, pageIndex);
                Rect tv_rect1 = new Rect();
                annotRect.roundOut(tv_rect1);
                mPdfViewCtrl.refresh(pageIndex, tv_rect1);
            }

            this.mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
            this.mUiExtensionsManager.getDocumentManager().setDocModified(true);
            this.curInk = annot;
        } catch (PDFException ignored) {
        }
    }

    private void modifyAnnot(int pageIndex) {
        if (pageIndex == -1 || this.lastLine == null || this.lastLine.isEmpty()) {
            return;
        }

        ArrayList<Float> pressures = null;
        if (mPencilType == InkConstants.PENCIL_WITH_BRUSH) {
            pressures = new ArrayList<>();
            for (ArrayList<PSIData> psiDataArrayList : this.psiDataArray) {
                for (PSIData psiData : psiDataArrayList) {
                    pressures.add(psiData.pressure);
                }
            }
        }

        try {
            RectF oldRect = AppUtil.toRectF(this.curInk.getRect());
            com.foxit.sdk.common.Path oldPath = this.curInk.getInkList();

            ResultInfo<ArrayList<PointF>, RectF, Void> resultInfo = InkAnnotUtil.convertPageLineToPdfLine(mPdfViewCtrl, pageIndex, this.lastLine);
            com.foxit.sdk.common.Path path = this.curInk.getInkList();
            InkAnnotUtil.appendPointsToPath(path, resultInfo.param1);
            this.curInk.setInkList(path);
            this.curInk.setModifiedDateTime(AppDmUtil.currentDateToDocumentDate());
            this.curInk.resetAppearanceStream();

            if (mPencilType == InkConstants.PENCIL_WITH_BRUSH) {
                PDFDictionary dict = this.curInk.getDict();
                dict.setAtString(InkConstants.PSINK_DICT_KEY, InkConstants.PSINK_DICT_VALUE);
            }

            InkModifyUndoItem undoItem = new InkModifyUndoItem(mAnnotHandler, mPdfViewCtrl);
            undoItem.mNM = AppAnnotUtil.getAnnotUniqueID(this.curInk);
            undoItem.mPageIndex = pageIndex;
            undoItem.mColor = mColor;
            undoItem.mBBox = AppUtil.toRectF(this.curInk.getRect());
            undoItem.mOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
            undoItem.mLineWidth = mThickness;
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mPencilType = mPencilType;
            undoItem.mPath = this.curInk.getInkList();

            undoItem.mOldModifiedDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mOldColor = mColor;
            undoItem.mOldBBox = new RectF(oldRect);
            undoItem.mOldOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
            undoItem.mOldLineWidth = mThickness;
            undoItem.mOldPath = oldPath;

            this.mUiExtensionsManager.getDocumentManager().onAnnotModified(this.curInk.getPage(), this.curInk);
            this.mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
            this.mUiExtensionsManager.getDocumentManager().setDocModified(true);

            if (this.mPdfViewCtrl.isPageVisible(pageIndex)) {
                mPdfViewCtrl.convertPdfRectToPageViewRect(oldRect, oldRect, pageIndex);
                RectF annotRect = AppUtil.toRectF(this.curInk.getRect());
                mPdfViewCtrl.convertPdfRectToPageViewRect(annotRect, annotRect, pageIndex);

                Rect tv_rect1 = new Rect();
                annotRect.roundOut(tv_rect1);
                mPdfViewCtrl.refresh(pageIndex, tv_rect1);
            }
        } catch (PDFException ignored) {
        }
    }

    private IToolSupply mToolSupply;
    private ToolItemBean mCurToolItem;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    public ToolItemBean getCurToolItem() {
        return mCurToolItem;
    }

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new InkToolSupply(mContext);
        return mToolSupply;
    }

    private class InkToolSupply extends ToolSupplyImpl {

        public InkToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            if (toolType == ToolConstants.Pencil)
                return R.drawable.drawing_tool_pencil_bg;
            else
                return R.drawable.drawing_tool_highlighter_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            if (toolType == ToolConstants.Pencil)
                return R.drawable.drawing_tool_pencil_src;
            else
                return R.drawable.drawing_tool_highlighter_src;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            if (toolType == ToolConstants.Pencil) {
                PencilConfig config = mUiExtensionsManager.getConfig().uiSettings.annotations.pencil;
                property.type = ToolConstants.Pencil;
                property.color = config.color;
                property.opacity = (int) (config.opacity * 100);
                property.lineWidth = config.thickness;
                property.style = InkConstants.PENCIL_WITH_PEN;
            } else {
                property.type = ToolConstants.Highlighter;
                property.color = PropertyBar.PB_COLORS_TOOL_DEFAULT[1];
                property.opacity = 50;
                property.lineWidth = 12f;
                property.style = InkConstants.PENCIL_WITH_PEN;
            }
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_INK;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                mbHighlighter = itemBean.type == ToolConstants.Highlighter;
                if (mUiExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_DRAWING_TAB) {
                    if (mbHighlighter) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_DrawingBar_Highlight);
                    } else {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_DrawingBar_Pencil);
                    }
                }
                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mPencilType = property.style;
                mColor = property.color;
                mOpacity = property.opacity;
                mThickness = property.lineWidth;
                mUiExtensionsManager.setCurrentToolHandler(InkToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == InkToolHandler.this) {
                    mbHighlighter = false;
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mbHighlighter = itemBean.type == ToolConstants.Highlighter;
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mColor = property.color;
            mOpacity = property.opacity;
            mThickness = property.lineWidth;
            mPencilType = property.style;

            InkToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

    void initPsiCanvas() {
        try {
            Rect outRect = new Rect();
            mUiExtensionsManager.getRootView().getWindowVisibleDisplayFrame(outRect);
            mPsi = new PSI(outRect.width(), outRect.height(), true);
            mPsi.setColor(mColor);
            mPsi.setOpacity((float) mOpacity / 100);
            int pageIndex = mCapturedPage == -1 ? mPdfViewCtrl.getCurrentPage() : mCapturedPage;
            mPsi.setDiameter(getPsiDiameter(pageIndex, mThickness));
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private int getPsiDiameter(int pageIndex, float thickness) {
        return Math.max(1, thicknessToPsiDiameter(pageIndex, thickness));
    }

    private int thicknessToPsiDiameter(int pageIndex, float thickness) {
        return (int) (UIAnnotFrame.getPageViewThickness(mPdfViewCtrl, pageIndex, thickness) * 1.2419);
    }

    private void releasePsiCanvas() {
        mPsiBmp.release();
        if (mPsi != null) {
            mPsi.delete();
            mPsi = null;
        }
    }

    private void setPSIColor(int color) {
        try {
            if (mPsi != null) {
                mPsi.setColor(color);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void setPSIOpacity(int opacity) {
        try {
            if (mPsi != null) {
                mPsi.setOpacity((float) opacity / 100);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void setPSIThickness(float thickness) {
        try {
            if (mPsi != null) {
                int pageIndex = mCapturedPage == -1 ? mPdfViewCtrl.getCurrentPage() : mCapturedPage;
                mPsi.setDiameter(getPsiDiameter(pageIndex, thickness));
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    static class PsiBitmap {
        Bitmap mBmp;
        int mPage;
        PointF mLeftTop;
        PointF mPvSize;
        long mToken;

        void release() {
            if (mBmp != null && !mBmp.isRecycled()) {
                mBmp.recycle();
                mBmp = null;
            }
            mPage = -1;
            mLeftTop = null;
            mPvSize = null;
        }
    }

    private float calibrationPressure(float pressure) {
        if (pressure <= 1)
            return pressure;

        int powerNumber = String.valueOf((int) Math.floor(pressure)).length();
        if (pressure % 10 == 0) {
            powerNumber -= 1;
        }
        return (float) (pressure / Math.pow(10, powerNumber));
    }

    private Handler handler;
    private Runnable timeoutRunnable;

    private void handleInkTimeOut() {
        cancelInkTimeOut();

        timeoutRunnable = new Runnable() {
            @Override
            public void run() {
                reset();
            }
        };
        if (handler == null) {
            handler = new Handler(Looper.getMainLooper());
        }
        handler.postDelayed(timeoutRunnable, mInkTimeout);
    }

    private void cancelInkTimeOut() {
        if (timeoutRunnable != null) {
            handler.removeCallbacks(timeoutRunnable);
        }
    }

    public void reset() {
        if (this.lastLine != null) {
            this.lastLine.clear();
            this.lastLine = null;
        }
        this.releasePsiCanvas();
        this.pendingPaths.clear();
        this.psiDataArray.clear();
        this.mCapturedPage = -1;
        this.mTouchCaptured = false;
        this.curInk = null;
    }
}
