/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.ink;

import android.app.Activity;
import android.content.Context;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.graphics.Path;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.view.MotionEvent;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.Ink;
import com.foxit.sdk.pdf.objects.PDFArray;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IPermissionProvider;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AbstractAnnotHandler;
import com.foxit.uiextensions.annots.AbstractToolHandler;
import com.foxit.uiextensions.annots.AnnotContent;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.common.IAnnotTaskResult;
import com.foxit.uiextensions.annots.common.UIAnnotFlatten;
import com.foxit.uiextensions.annots.common.UIAnnotReply;
import com.foxit.uiextensions.annots.freetext.typewriter.TypewriterModule;
import com.foxit.uiextensions.annots.freetext.typewriter.TypewriterToolHandler;
import com.foxit.uiextensions.annots.ink.ocr.InkRecognizeTextDialog;
import com.foxit.uiextensions.annots.multiselect.GroupManager;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.config.modules.annotations.AnnotationsConfig;
import com.foxit.uiextensions.controls.propertybar.AnnotMenu;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.utils.AnnotPermissionUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;

import java.util.ArrayList;


public class InkAnnotHandler extends AbstractAnnotHandler {
    protected InkToolHandler mToolHandler;
    protected ArrayList<Integer> mMenuText;
    protected static final String SUBJECT = "Pencil";

    protected float mBackOpacity;
    protected int mBackColor;
    protected com.foxit.sdk.common.Path mOldPath;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public InkAnnotHandler(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager, InkToolHandler toolHandler) {
        super(context, pdfViewCtrl, Annot.e_Ink);
        mToolHandler = toolHandler;
        mColor = mToolHandler.getColor();
        mOpacity = mToolHandler.getOpacity();
        mThickness = mToolHandler.getThickness();
        mUiExtensionsManager = uiExtensionsManager;
        mMenuText = new ArrayList<Integer>();
        mPaint.setStrokeJoin(Paint.Join.ROUND);
        mPaint.setStrokeCap(Paint.Cap.ROUND);
    }

    @Override
    protected AbstractToolHandler getToolHandler() {
        return mToolHandler;
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent e, Annot annot) {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        if (uiExtensionsManager == null) return false;

        Config config = ((UIExtensionsManager) mUiExtensionsManager).getConfig();
        AnnotationsConfig annotConfig = config.modules.annotations;
        if (!annotConfig.isLoadPencil) return false;
        return super.onTouchEvent(pageIndex, e, annot);
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent, Annot annot) {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        if (uiExtensionsManager == null) return false;

        Config config = ((UIExtensionsManager) mUiExtensionsManager).getConfig();
        AnnotationsConfig annotConfig = config.modules.annotations;
        if (!annotConfig.isLoadPencil) return false;
        return super.onSingleTapConfirmed(pageIndex, motionEvent, annot);
    }

    @Override
    public boolean shouldViewCtrlDraw(Annot annot) {
        Annot curAnnot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
        return !AppAnnotUtil.isSameAnnot(curAnnot, annot);
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent, Annot annot) {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
        if (uiExtensionsManager == null) return false;

        Config config = ((UIExtensionsManager) mUiExtensionsManager).getConfig();
        AnnotationsConfig annotConfig = config.modules.annotations;
        if (!annotConfig.isLoadPencil) return false;
        return super.onLongPress(pageIndex, motionEvent, annot);
    }

    @Override
    public void onAnnotSelected(final Annot annot, boolean reRender) {
        try {
            mColor = (int) annot.getBorderColor();
            mOpacity = AppDmUtil.opacity255To100((int) (((Ink) annot).getOpacity() * 255f + 0.5f));
            mThickness = annot.getBorderInfo().getWidth();

            mBackColor = mColor;
            mBackOpacity = ((Ink) annot).getOpacity();
            mOldPath = ((Ink) annot).getInkList();
            super.onAnnotSelected(annot, reRender);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void onAnnotDeselected(final Annot annot, boolean needInvalid) {
        if (mIsModified) {
            try {
                if (needInvalid) {
                    int borderColor = annot.getBorderColor();
                    float opacity = ((Ink) annot).getOpacity();
                    float thickness = annot.getBorderInfo().getWidth();
                    RectF annotRectF = AppUtil.toRectF(annot.getRect());

                    if (mBackColor != borderColor
                            || mBackOpacity != opacity
                            || mBackThickness != thickness
                            || !mBackRect.equals(annotRectF)) {

                        InkModifyUndoItem undoItem = new InkModifyUndoItem(this, mPdfViewCtrl);
                        undoItem.setCurrentValue(annot);

                        undoItem.mPath = ((Ink) annot).getInkList();
                        undoItem.mOldColor = mBackColor;
                        undoItem.mOldOpacity = mBackOpacity;
                        undoItem.mOldBBox = new RectF(mBackRect);
                        undoItem.mOldLineWidth = mBackThickness;
                        undoItem.mOldPath = this.mOldPath;

                        modifyAnnot(annot, undoItem, false, true, needInvalid, new Event.Callback() {
                            @Override
                            public void result(Event event, boolean success) {
                                if (annot != ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot()) {
                                    resetStatus();
                                }
                            }
                        });
                    }
                } else {
                    ((Ink) annot).setInkList(this.mOldPath);
                    ((Ink) annot).setOpacity(mBackOpacity);
                    BorderInfo borderInfo = annot.getBorderInfo();
                    borderInfo.setWidth(mBackThickness);
                    annot.setBorderInfo(borderInfo);
                    annot.setBorderColor(mBackColor);
                    annot.resetAppearanceStream();
                }
                dismissPopupMenu();
                hidePropertyBar();
            } catch (PDFException e) {
                e.printStackTrace();
            }
        } else {
            super.onAnnotDeselected(annot, needInvalid);
        }
    }

    @Override
    public void addAnnot(int pageIndex, final AnnotContent content, boolean addUndo, final Event.Callback result) {
        try {
            PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            final InkAnnotContent inkAnnotContent = (InkAnnotContent) content;
            final Ink annot = (Ink) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Ink, AppUtil.toFxRectF(inkAnnotContent.getBBox())), Annot.e_Ink);
            InkAddUndoItem undoItem = new InkAddUndoItem(this, mPdfViewCtrl);
            undoItem.setCurrentValue(inkAnnotContent);
            undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mAuthor = content.getAuthor() != null ? content.getAuthor() : ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();

            ArrayList<ArrayList<PointF>> lines = ((InkAnnotContent) content).getInkLisk();
            if (lines != null)
                undoItem.mPath = InkAnnotUtil.linesToPath(lines);
            addAnnot(pageIndex, annot, undoItem, addUndo, true, result);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    protected void modifyAnnot(Annot annot,
                               final int pencilType,
                               final int pageIndex,
                               final RectF bbox,
                               final int color,
                               final int opacity,
                               final float thickness,
                               final ArrayList<ArrayList<PointF>> oldLines,
                               final ArrayList<ArrayList<PointF>> newLines,
                               final boolean addUndo,
                               final Event.Callback result) {
        try {
            InkModifyUndoItem undoItem = new InkModifyUndoItem(InkAnnotHandler.this, mPdfViewCtrl);
            undoItem.mPageIndex = pageIndex;
            undoItem.mColor = color;
            undoItem.mBBox = new RectF(bbox);
            undoItem.mNM = AppAnnotUtil.getAnnotUniqueID(annot);
            undoItem.mOpacity = opacity / 255f;
            undoItem.mLineWidth = thickness;
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
//            undoItem.mInkLists = newLines;
            undoItem.mPencilType = pencilType;
            undoItem.mPath = InkAnnotUtil.linesToPath(newLines);

            undoItem.mOldModifiedDate = annot.getModifiedDateTime();
            undoItem.mOldBBox = new RectF(bbox);
            undoItem.mOldColor = color;
            undoItem.mOldOpacity = opacity / 255f;
            undoItem.mOldLineWidth = thickness;
//            undoItem.mOldInkLists = oldLines;
            undoItem.mOldPath = InkAnnotUtil.linesToPath(oldLines);

            modifyAnnot(annot, undoItem, false, addUndo, true, result);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    protected Annot addAnnot(String nm,
                             int pencilType,
                             int pageIndex,
                             RectF bbox,
                             int color,
                             int opacity,
                             float thickness,
                             ArrayList<ArrayList<PointF>> lines,
                             ArrayList<Float> pressures,
                             boolean reRender,
                             boolean addUndo,
                             Event.Callback result) {
        try {
            PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            final Ink annot = (Ink) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Ink, AppUtil.toFxRectF(bbox)), Annot.e_Ink);

            final InkAddUndoItem undoItem = new InkAddUndoItem(this, mPdfViewCtrl);
            undoItem.mPageIndex = pageIndex;
            undoItem.mNM = AppUtil.isEmpty(nm) ? AppDmUtil.randomUUID(null) : nm;
            undoItem.mBBox = new RectF(bbox);
            undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
            undoItem.mFlags = Annot.e_FlagPrint;
            undoItem.mSubject = SUBJECT;
            undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mColor = color;
            undoItem.mOpacity = opacity / 255f;
            undoItem.mLineWidth = thickness;
            undoItem.mPencilType = pencilType;
            undoItem.mPath = InkAnnotUtil.linesToPath(lines);
//            undoItem.mInkLists = InkAnnotUtil.cloneInkList(lines);
            undoItem.mPSIPressures = pressures;

            addAnnot(pageIndex, annot, undoItem, addUndo, reRender, result);
            return annot;
        } catch (PDFException e) {
            if (result != null) {
                result.result(null ,false);
            }
        }
        return null;
    }

    protected void addAnnot(int pageIndex,
                            final Annot annot,
                            final InkAddUndoItem undoItem,
                            boolean addUndo,
                            final boolean reRender,
                            final Event.Callback result) {
        final InkEvent event = new InkEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, (Ink) annot, mPdfViewCtrl);
        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().isMultipleSelectAnnots()
                || undoItem.isFromEraser) {
            if (result != null) {
                result.result(event, true);
            }
            return;
        }
        handleAddAnnot(pageIndex, annot, event, addUndo, reRender, new IAnnotTaskResult<PDFPage, Annot, Void>() {
            @Override
            public void onResult(boolean success, PDFPage p1, Annot p2, Void p3) {
                if (result != null) {
                    result.result(event, true);
                }
            }
        });
    }

    @Override
    public Annot handleAddAnnot(final int pageIndex,
                                final Annot annot,
                                final EditAnnotEvent addEvent,
                                final boolean addUndo,
                                final boolean reRender,
                                final IAnnotTaskResult<PDFPage, Annot, Void> result) {
        try {
            final PDFPage page = annot.getPage();

            EditAnnotTask task = new EditAnnotTask(addEvent, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, annot);
                        if (addUndo) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(addEvent.mUndoItem);
                        }
                        if (reRender && mPdfViewCtrl.isPageVisible(pageIndex)) {
                            RectF pvRect = getBBox(mPdfViewCtrl, annot);
                            final Rect tv_rect1 = new Rect();
                            pvRect.roundOut(tv_rect1);
                            mPdfViewCtrl.refresh(pageIndex, tv_rect1);
                        }

                    }

                    if (result != null) {
                        result.onResult(success, page, annot, null);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            if (result != null) {
                result.onResult(false, null, null, null);
            }
        }
        return annot;
    }

    @Override
    public void modifyAnnot(final Annot annot, final AnnotContent content, boolean addUndo, final Event.Callback result) {
        try {
            InkModifyUndoItem undoItem = new InkModifyUndoItem(this, mPdfViewCtrl);
            undoItem.setOldValue(annot);
            undoItem.mOldPath = ((Ink) annot).getInkList();
            undoItem.setCurrentValue(content);
            if (content instanceof InkAnnotContent) {
                ArrayList<ArrayList<PointF>> lines = ((InkAnnotContent) content).getInkLisk();
                if (lines != null) {
                    undoItem.mPath = InkAnnotUtil.linesToPath(lines);
                }
            }

            modifyAnnot(annot, undoItem, false, addUndo, true, result);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    protected void modifyAnnot(Annot annot, InkUndoItem undoItem, boolean useOldValue,
                               boolean addUndo, boolean reRender, final Event.Callback result) {
        InkEvent event = new InkEvent(EditAnnotEvent.EVENTTYPE_MODIFY, undoItem, (Ink) annot, mPdfViewCtrl);
        event.useOldValue = useOldValue;
        if (undoItem.isFromEraser) {
            if (result != null) {
                result.result(event, true);
            }
            return;
        }
        handleModifyAnnot(annot, event, addUndo, reRender,
                new IAnnotTaskResult<PDFPage, Annot, Void>() {
                    @Override
                    public void onResult(boolean success, PDFPage p1, Annot p2, Void p3) {
                        if (result != null) {
                            result.result(null, success);
                        }
                    }
                });
    }

    @Override
    public void removeAnnot(Annot annot, boolean addUndo, final Event.Callback result) {
        final DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        if (documentManager.getCurrentAnnot() != null
                && AppAnnotUtil.isSameAnnot(annot, documentManager.getCurrentAnnot())) {
            documentManager.setCurrentAnnot(null, false);
        }

        InkDeleteUndoItem undoItem = new InkDeleteUndoItem(this, mPdfViewCtrl);
        undoItem.setCurrentValue(annot);
        try {
            undoItem.mPath = ((Ink) annot).getInkList();
            undoItem.mGroupNMList = GroupManager.getInstance().getGroupUniqueIDs(mPdfViewCtrl, annot);
            undoItem.mPencilType = InkAnnotUtil.getInkType((Ink) annot);
            PDFObject pdfObject = annot.getDict().getElement("FxPList");
            if (pdfObject != null) {
                PDFArray array = pdfObject.getArray();
                if (array != null) {
                    ArrayList<Float> pressures = new ArrayList<>();
                    int size = array.getElementCount();
                    for (int i= 0; i < size; i++){
                        pressures.add(array.getElement(i).getFloat());
                    }
                    undoItem.mPSIPressures = pressures;
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        removeAnnot(annot, undoItem, addUndo, result);
    }

    protected void removeAnnot(Annot annot, final InkDeleteUndoItem undoItem, boolean addUndo, final Event.Callback result) {
        InkEvent event = new InkEvent(EditAnnotEvent.EVENTTYPE_DELETE, undoItem, (Ink) annot, mPdfViewCtrl);
        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().isMultipleSelectAnnots()
                || undoItem.isFromEraser) {
            try {
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotWillDelete(annot.getPage(), annot);
                if (result != null) {
                    result.result(event, true);
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return;
        }

        handleRemoveAnnot(annot, event, addUndo,
                new IAnnotTaskResult<PDFPage, Void, Void>() {
                    @Override
                    public void onResult(boolean success, PDFPage page, Void p2, Void p3) {
                        if (result != null) {
                            result.result(null, success);
                        }

                        if (success) {

                            if (undoItem.mGroupNMList.size() >= 2) {
                                ArrayList<String> newGroupList = new ArrayList<>(undoItem.mGroupNMList);
                                newGroupList.remove(undoItem.mNM);
                                if (newGroupList.size() >= 2)
                                    GroupManager.getInstance().setAnnotGroup(mPdfViewCtrl, page, newGroupList);
                                else
                                    GroupManager.getInstance().unGroup(page, newGroupList.get(0));
                            }
                        }
                    }
                });
    }

    @Override
    protected ArrayList<Path> generatePathData(PDFViewCtrl pdfViewCtrl, int pageIndex, Annot annot) {
        return InkAnnotUtil.generatePathData(mPdfViewCtrl, pageIndex, (Ink) annot);
    }

    @Override
    protected void transformAnnot(PDFViewCtrl pdfViewCtrl, int pageIndex, Annot annot, Matrix matrix) {
        RectF bbox = getBBox(pdfViewCtrl, annot);
        matrix.mapRect(bbox);
        pdfViewCtrl.convertPageViewRectToPdfRect(bbox, bbox, pageIndex);

        transformLines(pdfViewCtrl, pageIndex, (Ink) annot, matrix);

        try {
            annot.move(AppUtil.toFxRectF(bbox));
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void resetStatus() {
        mBackRect = null;
        mBackThickness = 0.0f;
        mSelectedAnnot = null;
        mIsModified = false;
    }

    @Override
    protected void showPopupMenu(final Annot annot) {
        try {
            final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if (curAnnot == null || curAnnot.isEmpty() || curAnnot.getType() != Annot.e_Ink) return;

            reloadPopupMenuString((Ink) curAnnot);
            mAnnotMenu.setMenuItems(mMenuText);
            RectF bbox = AppUtil.toRectF(curAnnot.getRect());
            int pageIndex = curAnnot.getPage().getIndex();
            mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, pageIndex);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bbox, bbox, pageIndex);
            mAnnotMenu.show(bbox);
            mAnnotMenu.setListener(new AnnotMenu.ClickListener() {
                @Override
                public void onAMClick(int flag) {
                    if (annot == null) return;
                    if (flag == AnnotMenu.AM_BT_COMMENT) { // comment
                        uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                        UIAnnotReply.showComments(mPdfViewCtrl, uiExtensionsManager.getRootView(), annot);
                    } else if (flag == AnnotMenu.AM_BT_REPLY) { // reply
                        uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                        UIAnnotReply.replyToAnnot(mPdfViewCtrl, uiExtensionsManager.getRootView(), annot);
                    } else if (flag == AnnotMenu.AM_BT_DELETE) { // delete
                        if (annot == uiExtensionsManager.getDocumentManager().getCurrentAnnot()) {
                            removeAnnot(annot, true, null);
                        }
                    } else if (flag == AnnotMenu.AM_BT_STYLE) { // line color
                        dismissPopupMenu();
                        showPropertyBar(PropertyBar.PROPERTY_COLOR);
                    } else if (flag == AnnotMenu.AM_BT_FLATTEN) {
                        uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                        UIAnnotFlatten.flattenAnnot(mPdfViewCtrl, annot);
                    } else if (flag == AnnotMenu.AM_BT_INK_RECOGNIZE_TEXT) {
                        if (uiExtensionsManager.getPermissionProvider() != null) {
                            uiExtensionsManager.getPermissionProvider().checkPermission(IPermissionProvider.FUNCTION_INK_RECOGNIZE,
                                    new IPermissionProvider.IPermissionState() {
                                        @Override
                                        public void onState(int state) {
                                            if(state == IPermissionProvider.PERMISSION_STATE_SHOW){
                                                showInkDialog(annot);
                                            }
                                        }
                                    });
                        } else {
                            showInkDialog(annot);
                        }
                    }
                }
            });
        } catch (PDFException e) {
            e.printStackTrace();
        }

    }

    private void showInkDialog(Annot annot){
        ((UIExtensionsManager)mUiExtensionsManager).onUIInteractElementClicked(IUIInteractionEventListener.Reading_Ink_Recognition_Text);
        Activity activity =  ((UIExtensionsManager)mUiExtensionsManager).getAttachedActivity();
        ArrayList<Annot> list =new ArrayList<>();
        list.add(annot);
        InkRecognizeTextDialog recognizeTextDialog = new InkRecognizeTextDialog(activity, ((UIExtensionsManager)mUiExtensionsManager));
        recognizeTextDialog.setAnnots(list);
        recognizeTextDialog.showDialog();
        recognizeTextDialog.loadData();
        recognizeTextDialog.setRecognitionInkCallBack(new InkRecognizeTextDialog.RecognitionInkCallBack() {
            @Override
            public void callBack(ArrayList<Annot> annots, String text) {
                if(annots.size() >0) {
                    removeAnnot(annots.get(0), true, null);
                    TypewriterModule typewriterModule = (TypewriterModule) ((UIExtensionsManager) mUiExtensionsManager).getModuleByName(Module.MODULE_NAME_TYPEWRITER);
                    TypewriterToolHandler toolHandler = (TypewriterToolHandler) typewriterModule.getToolHandler();
                    toolHandler.addInkAnnotToTypewriter(annots,text);
                }
            }
        });
    }

    @Override
    protected void dismissPopupMenu() {
        mAnnotMenu.setListener(null);
        mAnnotMenu.dismiss();
    }

    @Override
    protected void showPropertyBar(long curProperty) {
        UIExtensionsManager extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        Annot annot = extensionsManager.getDocumentManager().getCurrentAnnot();
        if (annot == null || annot.isEmpty() || !(annot instanceof Ink)) return;
        long properties = getSupportedProperties();

        mPropertyBar.setEditable(AnnotPermissionUtil.canEditabled(extensionsManager.getDocumentManager(), annot));
        mPropertyBar.setPropertyChangeListener(this);
        setPropertyBarProperties(mPropertyBar);
        mPropertyBar.reset(properties);

        try {
            RectF bbox = AppUtil.toRectF(annot.getRect());
            int pageIndex = annot.getPage().getIndex();
            mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, pageIndex);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(bbox, bbox, pageIndex);
            if (SystemUiHelper.getInstance().isStatusBarShown(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity())) {
                RectF rectF = AppUtil.toGlobalVisibleRectF(extensionsManager.getRootView(), bbox);
                mPropertyBar.show(rectF, false);
            } else {
                mPropertyBar.show(bbox, false);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void setPaintProperty(PDFViewCtrl pdfViewCtrl, int pageIndex, Paint paint, Annot annot) {
        super.setPaintProperty(pdfViewCtrl, pageIndex, paint, annot);
        paint.setStrokeCap(Paint.Cap.ROUND);
        if (InkAnnotUtil.getInkType((Ink) annot) == InkConstants.PENCIL_WITH_BRUSH)
            paint.setStyle(Style.FILL);
        else
            paint.setStyle(Style.STROKE);
    }

    @Override
    protected long getSupportedProperties() {
        return InkAnnotUtil.getSupportedProperties(false, false);
    }

    @Override
    protected void setPropertyBarProperties(PropertyBar propertyBar) {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        colors[0] = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
        propertyBar.setColors(colors);
        super.setPropertyBarProperties(propertyBar);
    }

    protected void reloadPopupMenuString(Ink ink) {
        mMenuText.clear();
        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        if (documentManager.canAddAnnot()&& ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()) {
            mMenuText.add(AnnotMenu.AM_BT_STYLE);
            mMenuText.add(AnnotMenu.AM_BT_COMMENT);
            if (AnnotPermissionUtil.canReplyAnnot(documentManager, ink)) {
                mMenuText.add(AnnotMenu.AM_BT_REPLY);
            }

            if (((UIExtensionsManager)mUiExtensionsManager).getConfig().uiSettings.handwritingRecognition) {
                if (hasDigitalInkRecognition()) {
                    mMenuText.add(AnnotMenu.AM_BT_INK_RECOGNIZE_TEXT);
                }
            }
            if (AnnotPermissionUtil.canFlattenAnnot(documentManager, ink)) {
                mMenuText.add(AnnotMenu.AM_BT_FLATTEN);
            }
            if (!(AppAnnotUtil.isLocked(ink) || AppAnnotUtil.isReadOnly(ink))) {
                if (AnnotPermissionUtil.canDeleteAnnot(documentManager, ink))
                    mMenuText.add(AnnotMenu.AM_BT_DELETE);
            }
        } else {
            mMenuText.add(AnnotMenu.AM_BT_COMMENT);
        }
    }

    private static boolean hasDigitalInkRecognition() {
        String[] candidates = {
                "com.google.mlkit.vision.digitalink.DigitalInkRecognition",
                "com.google.mlkit.vision.digitalink.recognition.DigitalInkRecognition"
        };

        for (String className : candidates) {
            try {
                Class.forName(className);
                return true;
            } catch (ClassNotFoundException ignored) {
            }
        }
        return false;
    }
    private void transformLines(PDFViewCtrl pdfViewCtrl, int pageIndex, Ink annot, Matrix matrix) {
        try {
            float[] tmp = {0, 0};
            com.foxit.sdk.common.Path path = annot.getInkList();
            for (int i = 0; i < path.getPointCount(); i++) {
                PointF pt = AppUtil.toPointF(path.getPoint(i));
                pdfViewCtrl.convertPdfPtToPageViewPt(pt, pt, pageIndex);
                tmp[0] = pt.x;
                tmp[1] = pt.y;
                matrix.mapPoints(tmp);
                pt.set(tmp[0], tmp[1]);
                pdfViewCtrl.convertPageViewPtToPdfPt(pt, pt, pageIndex);

                path.setPoint(i, AppUtil.toFxPointF(pt), path.getPointType(i));
            }

            annot.setInkList(path);
            annot.resetAppearanceStream();
        } catch (PDFException ignored) {
        }

    }

    void updateTheme() {
        if (mPropertyBar != null)
            mPropertyBar.updateTheme();
    }

}