/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.freetext.typewriter;

import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Build;
import android.os.Handler;
import android.text.Editable;
import android.text.TextWatcher;
import android.view.MotionEvent;
import android.view.ViewGroup.LayoutParams;
import android.view.inputmethod.EditorInfo;
import android.widget.EditText;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.DefaultAppearance;
import com.foxit.sdk.pdf.annots.FreeText;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.IUndoItem;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.freetext.FtTextUtil;
import com.foxit.uiextensions.annots.freetext.FtUtil;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.modules.UndoModule;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppKeyboardUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;

public class TypewriterToolHandler implements ToolHandler {

    private Context mContext;
    int mColor;
    int mOpacity;
    float mFontSize;
    private int mFontId;
    private EditText mEditView;
    private boolean mCreating;
    private int mCreateIndex = -1;
    private PointF mStartPoint = new PointF(0, 0);
    private PointF mStartPdfPoint = new PointF(0, 0);
    private PointF mEditPoint = new PointF(0, 0);
    public int mLastPageIndex = -1;
    private String mAnnotText;
    private FtTextUtil mTextUtil;
    private float mPageViewWidth;
    private float mPageViewHeigh;
    private float mBBoxWidth;
    private float mBBoxHeight;

    private PropertyBar mPropertyBar;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;

    private CreateAnnotResult mListener;
    private PDFViewCtrl mPdfViewCtrl;
    private UIExtensionsManager mUiExtensionsManager;

    private boolean mCreateAlive = true;
    private boolean mIsSelcetEndText = false;
    private boolean mIsContinuousCreate = true;

    private ToolItemBean mCurToolItem;

    public interface CreateAnnotResult {
        public void callBack();
    }

    public TypewriterToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        mPropertyBar = mUiExtensionsManager.getMainFrame().getPropertyBar();

        mTextUtil = new FtTextUtil(mContext, mPdfViewCtrl);

        pdfViewCtrl.registerDocEventListener(new DocEventListener() {

            @Override
            public void onDocClosed(PDFDoc document, int errCode) {
                dismissEditView();
            }

        });
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_TYPEWRITER;
    }

    public int getColor() {
        return mColor;
    }
    public int getOpacity() {
        return mOpacity;
    }

    public String getFontName() {
        return mTextUtil.getSupportFontName(mFontId);
    }
    @Override
    public void onActivate() {
        setUndoItemCallback(mUndoItemCallback);
        mLastPageIndex = -1;
        mCreateAlive = true;
        AppKeyboardUtil.setKeyboardListener(mUiExtensionsManager.getRootView(),
                mUiExtensionsManager.getRootView(), new AppKeyboardUtil.IKeyboardListener() {
                    @Override
                    public void onKeyboardOpened(int keyboardHeight) {
                        if (SystemUiHelper.getInstance().isFullScreen())
                            SystemUiHelper.getInstance().hideStatusBar(mUiExtensionsManager.getAttachedActivity());
                    }

                    @Override
                    public void onKeyboardClosed() {
                        mTextUtil.setKeyboardOffset(0);
                        if (SystemUiHelper.getInstance().isFullScreen())
                            SystemUiHelper.getInstance().hideStatusBar(mUiExtensionsManager.getAttachedActivity());
                        if (mContext.getResources().getConfiguration().keyboard != Configuration.KEYBOARDHIDDEN_YES) {
                            mCreateAlive = false;
                            if (mEditView != null) {
                                createFTAnnot(false);
                                AppUtil.dismissInputSoft(mEditView);
                                mUiExtensionsManager.getRootView().removeView(mEditView);
                                mEditView = null;

                                endCreating();
                            }
                        }
                    }
                });

        resetPropertyBar();
    }

    void resetPropertyBar() {
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        mPropertyBar.setColors(colors);

        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_FONTNAME, mTextUtil.getSupportFontName(mFontId));
        mPropertyBar.setProperty(PropertyBar.PROPERTY_FONTSIZE, mFontSize);
        mPropertyBar.clearPropertyTitle();
        mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_FONTNAME, AppResource.getString(mContext, R.string.pb_font_settings));
        mPropertyBar.setArrowVisible(true);
        mPropertyBar.reset(getSupportedProperties());
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
    }

    private long getSupportedProperties() {
        return PropertyBar.PROPERTY_COLOR
                | PropertyBar.PROPERTY_OPACITY
                | PropertyBar.PROPERTY_FONTSIZE
                | PropertyBar.PROPERTY_FONTNAME;
    }

    @Override
    public void onDeactivate() {
        setUndoItemCallback(null);
        if (mEditView != null) {
            if (mCreateAlive) {
                mCreateAlive = false;
                createFTAnnot(false);
            }
        }
        if (SystemUiHelper.getInstance().isFullScreen())
            SystemUiHelper.getInstance().hideSystemUI(mUiExtensionsManager.getAttachedActivity());
        AppKeyboardUtil.removeKeyboardListener(mUiExtensionsManager.getRootView());
    }

    protected void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    protected void removeProbarListener() {
        mPropertyChangeListener = null;
    }

    String mText = "";
    public void addInkAnnotToTypewriter(ArrayList<Annot> annots,String text){
        Annot annot = annots.get(0);
        mText = text;
        try {
            RectF rectF = new RectF();
            int pageIndex = annot.getPage().getIndex();
            mPdfViewCtrl.convertPdfRectToPageViewRect(AppUtil.toRectF(annot.getRect()),rectF,pageIndex);
            PointF pointF =new PointF();
            pointF.x = rectF.left;
            pointF.y = rectF.top;
            mCreateIndex = pageIndex;
            addAnnotToPosition(pageIndex,pointF);
            new Handler().postDelayed(new Runnable() {
                @Override
                public void run() {
                    onDeactivate();
                }
            },510);


        } catch (PDFException e) {
            e.printStackTrace();
        }

    }


    @Override
    public boolean onTouchEvent(final int pageIndex, MotionEvent e) {
        return mUiExtensionsManager.defaultTouchEvent(pageIndex, e);
    }

    private boolean onTypewriterToolTouch(int pageIndex, MotionEvent e) {
        PointF point = new PointF(e.getX(), e.getY());
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
        PointF pdfPoint = new PointF(point.x, point.y);
        mPdfViewCtrl.convertPageViewPtToPdfPt(pdfPoint, pdfPoint, pageIndex);

        float x = point.x;
        float y = point.y;

        int action = e.getAction();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                return onTouchDown(pageIndex, x, y);
            case MotionEvent.ACTION_MOVE:
                return true;
            case MotionEvent.ACTION_UP:
                onTouchUp(pageIndex);
                return false;
            case MotionEvent.ACTION_CANCEL:
                mStartPoint.set(0, 0);
                mEditPoint.set(0, 0);
                mCreateAlive = true;
                return true;
            default:
                break;
        }
        return false;
    }

    boolean mAddingToBlank = false;

    public void addAnnotToPosition(int pageIndex, PointF point) {
        mUiExtensionsManager.setCurrentToolHandler(this);

        onTouchDown(pageIndex, point.x, point.y);
        onTouchUp(pageIndex);

        mAddingToBlank = true;
    }

    void endCreating() {
        mCreating = false;
        if (mAddingToBlank) {
            if (mUiExtensionsManager.getCurrentToolHandler() == this)
                mUiExtensionsManager.setCurrentToolHandler(null);
            mAddingToBlank = false;
        }
    }

    private boolean onTouchDown(int pageIndex, float x, float y) {
        if (mUiExtensionsManager.getCurrentToolHandler() == this) {
            if (!mCreating) {
                mPageViewWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
                mPageViewHeigh = mPdfViewCtrl.getPageViewHeight(pageIndex);
                mStartPoint.set(x, y);
                adjustStartPt(mPdfViewCtrl, pageIndex, mStartPoint);
                mStartPdfPoint.set(mStartPoint.x, mStartPoint.y);
                mPdfViewCtrl.convertPageViewPtToPdfPt(mStartPdfPoint, mStartPdfPoint, pageIndex);
                if (mLastPageIndex == -1) {
                    mLastPageIndex = pageIndex;
                }
                mCreateIndex = pageIndex;
                return true;
            }
        }
        return false;
    }

    private void onTouchUp(final int pageIndex) {
        if (mUiExtensionsManager.getCurrentToolHandler() == this && mEditView == null) {
            mEditView = new EditText(mContext);
            if(!AppUtil.isEmpty(mText)){
                mAnnotText = mText;
                mEditView.setText(mText);
            }
            if (AppDisplay.isPad()) { // SDKRD-9313
                mEditView.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            }
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
                mEditView.setForceDarkAllowed(false);
            }
            mEditView.setLayoutParams(new LayoutParams(1, 1));
            mEditView.addTextChangedListener(new TextWatcher() {

                @Override
                public void onTextChanged(CharSequence s, int start, int before, int count) {
                    mAnnotText = FtTextUtil.filterEmoji(String.valueOf(s));
                    mPdfViewCtrl.invalidate();
                }

                @Override
                public void beforeTextChanged(CharSequence s, int start, int count,
                                              int after) {
                }

                @Override
                public void afterTextChanged(Editable s) {

                }
            });

            mTextUtil.setOnWidthChanged(new FtTextUtil.OnTextValuesChangedListener() {

                @Override
                public void onMaxWidthChanged(float maxWidth) {
                    if (mBBoxWidth != maxWidth)
                        mBBoxWidth = maxWidth;
                }

                @Override
                public void onMaxHeightChanged(float maxHeight) {
                    if (mBBoxHeight != maxHeight)
                        mBBoxHeight = maxHeight;
                }

                @Override
                public void onCurrentSelectIndex(int selectIndex) {
                    if (selectIndex >= mEditView.getText().length()) {
                        selectIndex = mEditView.getText().length();
                        mIsSelcetEndText = true;
                    } else {
                        mIsSelcetEndText = false;
                    }
                    mEditView.setSelection(selectIndex);
                }

                @Override
                public void onEditPointChanged(float editPointX,
                                               float editPointY) {
                    PointF point = new PointF(editPointX, editPointY);
                    mPdfViewCtrl.convertPageViewPtToPdfPt(point, point, pageIndex);
                    mEditPoint.set(point.x, point.y);
                }
            });
            mUiExtensionsManager.getRootView().addView(mEditView);
            mPdfViewCtrl.invalidate();

            if(AppUtil.isEmpty(mText)){
                AppUtil.showSoftInput(mEditView);
            }
            mText = "";
            mTextUtil.getBlink().postDelayed((Runnable) mTextUtil.getBlink(), 500);
            mCreating = true;
        }
        mCreateAlive = true;
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiExtensionsManager.getDocumentManager().getCurrentAnnot() != null) {
            return mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        editAnnotationOrNot(pageIndex, motionEvent);
        return true;
    }

    private void editAnnotationOrNot(int pageIndex, MotionEvent motionEvent) {
        if (mEditView == null) {
            onTypewriterToolTouch(pageIndex, motionEvent);
            onTouchUp(pageIndex);
        } else {
            PointF point = new PointF(motionEvent.getX(), motionEvent.getY());
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
            onSingleTapOrLongPress(pageIndex, point);
        }
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiExtensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (handled) {
            createFTAnnot(false);
        } else {
            editAnnotationOrNot(pageIndex, motionEvent);
        }
        return true;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinuousCreate;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinuousCreate = continueAddAnnot;
    }

    private boolean onSingleTapOrLongPress(final int pageIndex, final PointF point) {
        float x = point.x;
        float y = point.y;
        if (mUiExtensionsManager.getCurrentToolHandler() == this && mEditView != null) {
            RectF rectF = new RectF(mStartPoint.x, mStartPoint.y,
                    mStartPoint.x + mBBoxWidth, mStartPoint.y + mBBoxHeight);
            if (rectF.contains(x, y)) {
                PointF pointF = new PointF(x, y);
                mPdfViewCtrl.convertPageViewPtToPdfPt(pointF, pointF, pageIndex);
                mEditPoint.set(pointF.x, pointF.y);
                mTextUtil.resetEditState();

                RectF _rect = new RectF(rectF);
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(_rect, _rect, pageIndex);
                mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(_rect));
                AppUtil.showSoftInput(mEditView);
                return true;
            } else {
                if (!mIsContinuousCreate) {
                    mUiExtensionsManager.setCurrentToolHandler(null);
                    return true;
                }

                if (mCreateAlive) {
                    mCreateAlive = false;
                    if (mUiExtensionsManager.getCurrentToolHandler() == TypewriterToolHandler.this) {
                        createFTAnnot(false);
                    }
                    return true;
                }

                mCreateAlive = true;
                setCreateAnnotListener(new CreateAnnotResult() {
                    @Override
                    public void callBack() {
                        mStartPoint.set(point.x, point.y);
                        adjustStartPt(mPdfViewCtrl, pageIndex, mStartPoint);
                        mStartPdfPoint.set(mStartPoint.x, mStartPoint.y);
                        mPdfViewCtrl.convertPageViewPtToPdfPt(mStartPdfPoint, mStartPdfPoint, pageIndex);
                        if (mLastPageIndex == -1) {
                            mLastPageIndex = pageIndex;
                        }
                        mCreateIndex = pageIndex;
                        if (mEditView != null) {
                            AppUtil.showSoftInput(mEditView);
                        }
                    }
                });
                createFTAnnot(false);
                return true;
            }
        }
        return false;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mCreateIndex == pageIndex && mEditView != null) {
            canvas.save();
            PointF startPoint = new PointF(mStartPdfPoint.x, mStartPdfPoint.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, pageIndex);
            RectF frameRectF = new RectF();
            frameRectF.set(startPoint.x, startPoint.y,
                    startPoint.x + mBBoxWidth, startPoint.y + mBBoxHeight);
            float deltaXY = FtUtil.widthOnPageView(mPdfViewCtrl, pageIndex, 6);
            float adjustY = 0;
            if (frameRectF.bottom > mPdfViewCtrl.getPageViewHeight(pageIndex) - deltaXY) {
                adjustY = mPdfViewCtrl.getPageViewHeight(pageIndex) - frameRectF.bottom - deltaXY;
            }
            startPoint.offset(0, adjustY);

            PointF editPoint = new PointF(mEditPoint.x, mEditPoint.y);
            if (editPoint.x != 0 || editPoint.y != 0) {
                mPdfViewCtrl.convertPdfPtToPageViewPt(editPoint, editPoint, pageIndex);
            }
            mTextUtil.setTextString(pageIndex, mAnnotText, true);
            mTextUtil.setStartPoint(startPoint);
            mTextUtil.setEditPoint(editPoint);
            mTextUtil.setMaxRect(mPdfViewCtrl.getPageViewWidth(pageIndex) - startPoint.x, mPdfViewCtrl.getPageViewHeight(pageIndex) - startPoint.y);
            mTextUtil.setTextColor(mColor, AppDmUtil.opacity100To255(mOpacity));
            mTextUtil.setFont(mTextUtil.getSupportFontName(mFontId), mFontSize);
            if (mIsSelcetEndText) {
                mTextUtil.setEndSelection(mEditView.getSelectionEnd() + 1);
            } else {
                mTextUtil.setEndSelection(mEditView.getSelectionEnd());
            }
            mTextUtil.loadText();
            mTextUtil.drawText(canvas);
            canvas.restore();
        }
    }

    private void createFTAnnot(final boolean disableAutoGoToPage) {
        createFTAnnot(false, disableAutoGoToPage);
    }

    private void createFTAnnot(boolean fromUndoClick, final boolean disableAutoGoToPage) {
        if (mAnnotText != null && mAnnotText.length() > 0) {
            PointF pdfPointF = new PointF(mStartPdfPoint.x, mStartPdfPoint.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mCreateIndex);
            final RectF rect = new RectF(pdfPointF.x, pdfPointF.y, pdfPointF.x + mBBoxWidth, pdfPointF.y
                    + mBBoxHeight);
            float deltaXY = FtUtil.widthOnPageView(mPdfViewCtrl, mCreateIndex, 6);
            float adjustY = 0;
            if (rect.bottom > mPdfViewCtrl.getPageViewHeight(mCreateIndex) - deltaXY) {
                adjustY = mPdfViewCtrl.getPageViewHeight(mCreateIndex) - rect.bottom - deltaXY;
            }
            rect.offset(0, adjustY);

            RectF pdfRectF = new RectF(rect);
            RectF _rect = new RectF(pdfRectF);// page view rect
            mPdfViewCtrl.convertPageViewRectToPdfRect(pdfRectF, pdfRectF, mCreateIndex);
            String content = "";
            try {
                content = new String(mAnnotText.getBytes(), "UTF-8");
            } catch (UnsupportedEncodingException e) {
                e.printStackTrace();
            }
//            ArrayList<String> composeText = mTextUtil.getComposedText(mPdfViewCtrl, mCreateIndex, _rect, content, mTextUtil.getSupportFontName(mFontId), mFontSize);
//
//            StringBuffer sb = new StringBuffer();
//            for (int i = 0; i < composeText.size(); i++) {
//                sb.append(composeText.get(i));
//                if (i != composeText.size() - 1 && sb.charAt(sb.length() - 1) != '\n') {
//                    sb.append("\r");
//                }
//            }
//            String annotContent = sb.toString(); // SDKRD-10359

            try {

                final PDFPage page = mPdfViewCtrl.getDoc().getPage(mCreateIndex);
                final TypewriterAddUndoItem undoItem = createUndoItem(page, pdfRectF, content);
                if (fromUndoClick) {
                    ArrayList<IUndoItem> redoItems = new ArrayList<>();
                    redoItems.add(undoItem);
                    mUiExtensionsManager.getDocumentManager().addRedoItems(redoItems);
                    UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
                    if (undoModule != null)
                        undoModule.changeButtonStatus();

                    invalidatePageByCreateAnnot(disableAutoGoToPage);
                    return;
                }

                final Annot annot = AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_FreeText, AppUtil.toFxRectF(pdfRectF)), Annot.e_FreeText);
                TypewriterEvent event = new TypewriterEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, (FreeText) annot, mPdfViewCtrl);
                event.mDisallowTextOverflow = true;
                mUiExtensionsManager.getDocumentManager().setHasModifyTask(true);
                mUiExtensionsManager.getDocumentManager().setDocModified(true);
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, annot);
                            mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);
                            mUiExtensionsManager.getDocumentManager().setHasModifyTask(false);
                            if (mPdfViewCtrl.isPageVisible(mCreateIndex)) {
                                mPdfViewCtrl.refresh(mCreateIndex, AppDmUtil.rectFToRect(rect));
                                if (mIsContinuousCreate && mCreateAlive) {
                                    if (mEditView != null)
                                        mEditView.setText("");

                                } else {
                                    AppUtil.dismissInputSoft(mEditView);
                                    mUiExtensionsManager.getRootView().removeView(mEditView);
                                    mEditView = null;
                                    endCreating();
                                    mTextUtil.getBlink().removeCallbacks((Runnable) mTextUtil.getBlink());
                                    mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
                                    if ((mCreateIndex == mPdfViewCtrl.getPageCount() - 1 || isSignleMode())
                                            && mCreateIndex == mPdfViewCtrl.getCurrentPage()) {
                                        int pageWidth = mPdfViewCtrl.getPageViewWidth(mCreateIndex);
                                        int pageHeight = mPdfViewCtrl.getPageViewHeight(mCreateIndex);
                                        PointF endPoint = new PointF(pageWidth, pageHeight);
                                        mPdfViewCtrl.convertPageViewPtToDisplayViewPt(endPoint, endPoint, mCreateIndex);

                                        if (AppDisplay.getRawScreenHeight() - (endPoint.y - mTextUtil.getKeyboardOffset()) > 0) {
                                            mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
                                            mTextUtil.setKeyboardOffset(0);
                                            PointF startPoint = new PointF(mStartPdfPoint.x, mStartPdfPoint.y);

                                            mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mCreateIndex);
                                            if (!disableAutoGoToPage) {
                                                PointF pointF = mTextUtil.getPageViewOrigin(mPdfViewCtrl, mCreateIndex,
                                                        startPoint.x, startPoint.y);
                                                mPdfViewCtrl.gotoPage(mCreateIndex, pointF.x, pointF.y);
                                            }
                                        }
                                    }
                                }

                                mCreateIndex = -1;
                                mAnnotText = "";
                                mStartPoint.set(0, 0);
                                mEditPoint.set(0, 0);
                                mTextUtil.resetEditState();
                                mLastPageIndex = -1;
                                if (mIsContinuousCreate && mListener != null) {
                                    mListener.callBack();
                                }
                            } else {
                                dismissEditView();
                            }
                        } else {
                            dismissEditView();
                        }
                    }
                });
                mPdfViewCtrl.addTask(task);
            } catch (PDFException e) {
                if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                    mPdfViewCtrl.recoverForOOM();
                }
            }
        } else {
            invalidatePageByCreateAnnot(disableAutoGoToPage);
        }
    }

    private void invalidatePageByCreateAnnot(boolean disableAutoGoToPage) {
        if (mIsContinuousCreate && mCreateAlive && mListener != null) {
            mLastPageIndex = -1;
            mListener.callBack();
        } else {
            AppUtil.dismissInputSoft(mEditView);
            mUiExtensionsManager.getRootView().removeView(mEditView);
            mEditView = null;
            endCreating();
            mTextUtil.getBlink().removeCallbacks((Runnable) mTextUtil.getBlink());
        }

        mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
        if (mPdfViewCtrl.isPageVisible(mCreateIndex)) {
            PointF startPoint = new PointF(mStartPdfPoint.x, mStartPdfPoint.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(startPoint, startPoint, mCreateIndex);

            float fontWidth = mBBoxWidth == 0 ? FtUtil.widthOnPageView(mPdfViewCtrl, mCreateIndex, 10) : mBBoxWidth;
            float fontHeight = mBBoxHeight == 0 ? mTextUtil.getFontHeight(mPdfViewCtrl, mCreateIndex,
                    mTextUtil.getSupportFontName(mFontId), mFontSize) : mBBoxHeight;
            RectF rect = new RectF(startPoint.x, startPoint.y,
                    startPoint.x + fontWidth, startPoint.y + fontHeight);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rect, rect, mCreateIndex);
            mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(rect));

            if (mCreateIndex == mPdfViewCtrl.getPageCount() - 1 || isSignleMode()) {
                int pageWidth = mPdfViewCtrl.getPageViewWidth(mCreateIndex);
                int pageHeight = mPdfViewCtrl.getPageViewHeight(mCreateIndex);
                PointF endPoint = new PointF(pageWidth, pageHeight);
                mPdfViewCtrl.convertPageViewPtToDisplayViewPt(endPoint, endPoint, mCreateIndex);
                if (AppDisplay.getRawScreenHeight() - (endPoint.y - mTextUtil.getKeyboardOffset()) > 0) {
                    mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
                    mTextUtil.setKeyboardOffset(0);

                    if (!disableAutoGoToPage) {
                        PointF pointF = mTextUtil.getPageViewOrigin(mPdfViewCtrl, mCreateIndex, startPoint.x, startPoint.y);
                        mPdfViewCtrl.gotoPage(mCreateIndex, pointF.x, pointF.y);
                    }
                }
            }
        }

        mAnnotText = "";
        mStartPoint.set(0, 0);
        mEditPoint.set(0, 0);
        mTextUtil.resetEditState();
        mLastPageIndex = -1;
        mBBoxHeight = 0;
        mBBoxWidth = 0;
        mCreateIndex = -1;
    }

    private boolean isSignleMode(){
        return !mPdfViewCtrl.isContinuous() && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_SINGLE;
    }

    private TypewriterAddUndoItem createUndoItem(PDFPage page, RectF rectF, String content) {
        try {
            final TypewriterAddUndoItem undoItem = new TypewriterAddUndoItem(mPdfViewCtrl);
            undoItem.mNM = AppDmUtil.randomUUID(null);
            undoItem.mPageIndex = mCreateIndex;
            undoItem.mColor = mColor;
            undoItem.mOpacity = AppDmUtil.opacity100To255(mOpacity) / 255f;
            undoItem.mContents = content;
            undoItem.mFontId = mFontId;
            undoItem.mFontSize = mFontSize;
            undoItem.mTextColor = mColor;
            undoItem.mDaFlags = DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagTextColor | DefaultAppearance.e_FlagFontSize;
            undoItem.mAuthor = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotAuthor();
            undoItem.mBBox = new RectF(rectF);
            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mCreationDate = AppDmUtil.currentDateToDocumentDate();
            undoItem.mFlags = Annot.e_FlagPrint;
            undoItem.mIntent = "FreeTextTypewriter";
            undoItem.mSubject = "Typewriter";
            int rotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
            undoItem.mRotation = rotation == 0 ? rotation : 4 - rotation;
            return undoItem;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    private void dismissEditView() {
        mAnnotText = "";
        mStartPoint.set(0, 0);
        mEditPoint.set(0, 0);
        mLastPageIndex = -1;
        mTextUtil.resetEditState();
        mTextUtil.setKeyboardOffset(0);
        if (mEditView != null) {
            AppUtil.dismissInputSoft(mEditView);
            mUiExtensionsManager.getRootView().removeView(mEditView);
            mEditView = null;
        }
        mBBoxHeight = 0;
        mBBoxWidth = 0;
        endCreating();
        mPdfViewCtrl.layout(0, 0, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight());
        mTextUtil.getBlink().removeCallbacks((Runnable) mTextUtil.getBlink());
    }

    protected void onColorValueChanged(int color) {
        mColor = color;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            PointF pdfPointF = new PointF(mStartPdfPoint.x, mStartPdfPoint.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mLastPageIndex);
            RectF rectF = new RectF(pdfPointF.x, pdfPointF.y,
                    pdfPointF.x + mBBoxWidth, pdfPointF.y + mBBoxHeight);
            Rect rect = new Rect((int) rectF.left, (int) rectF.top, (int) rectF.right, (int) rectF.bottom);
            mPdfViewCtrl.refresh(mLastPageIndex, rect);
        }

        setProItemColor(color);
    }

    private void setProItemColor(int color) {
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
    }

    protected void onOpacityValueChanged(int opacity) {
        mOpacity = opacity;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            PointF pdfPointF = new PointF(mStartPdfPoint.x, mStartPdfPoint.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mLastPageIndex);
            RectF rectF = new RectF(pdfPointF.x, pdfPointF.y,
                    pdfPointF.x + mBBoxWidth, pdfPointF.y + mBBoxHeight);
            Rect rect = new Rect((int) rectF.left, (int) rectF.top, (int) rectF.right, (int) rectF.bottom);
            mPdfViewCtrl.refresh(mLastPageIndex, rect);
        }

        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
    }

    protected void changeFontDefaultValue(String fontName) {
        mFontId = mTextUtil.getSupportFontID(fontName);
    }

    protected void onFontValueChanged(String fontName) {
        mFontId = mTextUtil.getSupportFontID(fontName);
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            PointF pdfPointF = new PointF(mStartPdfPoint.x, mStartPdfPoint.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mLastPageIndex);
            RectF rectF = new RectF(pdfPointF.x, pdfPointF.y,
                    pdfPointF.x + mBBoxWidth, pdfPointF.y + mBBoxHeight);
            Rect rect = new Rect((int) rectF.left, (int) rectF.top, (int) rectF.right, (int) rectF.bottom);
            mPdfViewCtrl.refresh(mLastPageIndex, rect);
        }

        if (mCurToolItem == null) return;
        mCurToolItem.property.fontName = fontName;
    }

    protected void onFontSizeValueChanged(float fontSize) {
        mFontSize = fontSize;
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            PointF pdfPointF = new PointF(mStartPdfPoint.x, mStartPdfPoint.y);
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPointF, pdfPointF, mLastPageIndex);
            adjustStartPt(mPdfViewCtrl, mLastPageIndex, pdfPointF);
            PointF pdfPtChanged = new PointF(pdfPointF.x, pdfPointF.y);
            mPdfViewCtrl.convertPageViewPtToPdfPt(pdfPtChanged, pdfPtChanged, mLastPageIndex);
            mStartPdfPoint.x = pdfPtChanged.x;
            mStartPdfPoint.y = pdfPtChanged.y;
            RectF rectF = new RectF(pdfPointF.x, pdfPointF.y,
                    pdfPointF.x + mBBoxWidth, pdfPointF.y + mBBoxHeight);
            Rect rect = new Rect((int) rectF.left, (int) rectF.top, (int) rectF.right, (int) rectF.bottom);
            mPdfViewCtrl.refresh(mLastPageIndex, rect);
        }

        if (mCurToolItem == null) return;
        mCurToolItem.property.fontSize = fontSize;
    }

    private void adjustStartPt(PDFViewCtrl pdfViewCtrl, int pageIndex, PointF point) {
        float fontWidth = mTextUtil.getFontWidth(pdfViewCtrl, pageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize);
        if (pdfViewCtrl.getPageViewWidth(pageIndex) - point.x < fontWidth) {
            point.x = mPageViewWidth - fontWidth;
        }
        float fontHeight = mTextUtil.getFontHeight(pdfViewCtrl, pageIndex, mTextUtil.getSupportFontName(mFontId), mFontSize);
        if (pdfViewCtrl.getPageViewHeight(pageIndex) - point.y < fontHeight) {
            point.y = mPageViewHeigh - fontHeight;
        }
    }

    private void setCreateAnnotListener(CreateAnnotResult listener) {
        mListener = listener;
    }

    private IToolSupply mToolSupply;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener() {
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new TypewriterToolSupply(mContext);
        return mToolSupply;
    }

    private class TypewriterToolSupply extends ToolSupplyImpl {

        public TypewriterToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return R.drawable.comment_tool_typewriter_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return R.drawable.comment_tool_typewriter_src;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = ToolConstants.Typewriter;
            property.color = mColor;
            property.opacity = mOpacity;
            property.fontName = mTextUtil.getSupportFontName(mFontId);
            property.fontSize = mFontSize;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_TYPEWRITER;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if (mUiExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_COMMENT_TAB) {
                    mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_TypeWriter);
                }
                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mColor = property.color;
                mOpacity = property.opacity;
                mFontId = mTextUtil.getSupportFontID(property.fontName);
                mFontSize = property.fontSize;
                mUiExtensionsManager.setCurrentToolHandler(TypewriterToolHandler.this);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == TypewriterToolHandler.this) {
                    mCurToolItem = null;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mColor = property.color;
            mOpacity = property.opacity;
            mFontId = mTextUtil.getSupportFontID(property.fontName);
            mFontSize = property.fontSize;
            TypewriterToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

    void handlePageChanged() {
        if (mEditView == null) {
            return;
        }

        // MOBRD-7200
        if (mLastPageIndex == mPdfViewCtrl.getPageCount() - 1) {
            Rect rect = mPdfViewCtrl.getPageViewRect(mLastPageIndex);
            int h = mPdfViewCtrl.getHeight();
            if (rect.bottom <= mPdfViewCtrl.getHeight())
                return;
        }

        mCreateAlive = false;
        createFTAnnot(true);
    }

    private void setUndoItemCallback(UndoModule.IUndoItemCallback callback) {
        UndoModule undoModule = (UndoModule) mUiExtensionsManager.getModuleByName(Module.MODULE_NAME_UNDO);
        if (undoModule != null) {
            undoModule.setUndoItemCallback(callback);
        }
    }

    private final UndoModule.IUndoItemCallback mUndoItemCallback = new UndoModule.IUndoItemCallback() {
        @Override
        public boolean undo() {
            if (mCurToolItem != null && mUiExtensionsManager.getCurrentAnnotHandler() instanceof TypewriterAnnotHandler) {
                ((TypewriterAnnotHandler) mUiExtensionsManager.getCurrentAnnotHandler()).handleKeyboardClosed();
                return true;
            }

            if (mEditView != null && mCurToolItem != null) {
//                mCurToolItem.toolItem.performClick();
                mCreateAlive = false;
                createFTAnnot(true, false);
                endCreating();
                return true;
            }
            return false;
        }

        @Override
        public boolean canUndo() {
            return mUiExtensionsManager.getDocumentManager().canUndo();
        }

        @Override
        public boolean redo() {
            if (mEditView != null && mCurToolItem != null) {
                mCurToolItem.toolItem.performClick();
                return true;
            }
            return false;
        }

        @Override
        public boolean canRedo() {
            return mUiExtensionsManager.getDocumentManager().canRedo();
        }
    };

}
