/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.form;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.text.TextUtils;
import android.util.SparseArray;
import android.view.MotionEvent;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Font;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.Signature;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Widget;
import com.foxit.sdk.pdf.interform.Control;
import com.foxit.sdk.pdf.interform.Field;
import com.foxit.sdk.pdf.interform.Form;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.form.undo.FormFillerAddUndoItem;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.config.uisettings.form.field.BaseFieldConfig;
import com.foxit.uiextensions.config.uisettings.form.field.CheckBoxConfig;
import com.foxit.uiextensions.config.uisettings.form.field.ComboBoxConfig;
import com.foxit.uiextensions.config.uisettings.form.field.ListBoxConfig;
import com.foxit.uiextensions.config.uisettings.form.field.RadioButtonConfig;
import com.foxit.uiextensions.config.uisettings.form.field.TextFieldConfig;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;


public class FormFillerToolHandler implements ToolHandler {
    private PDFViewCtrl mPdfViewCtrl;
    private FormFillerModule mFormFillerModule;
    private UIExtensionsManager mUiExtensionsManager;
    private Context mContext;
    private Form mForm;

    private PointF mStartPoint = new PointF(0, 0);
    private PointF mStopPoint = new PointF(0, 0);
    private PointF mDownPoint = new PointF(0, 0);// whether moving point

    private RectF mDrawRect = new RectF(0, 0, 0, 0);
    private Paint mPaint;

    private float mThickness;
    private float mCtlPtLineWidth = 2;
    private float mCtlPtRadius = 5;
    private boolean mTouchCaptured = false;
    private int mLastPageIndex = -1;
    private int mControlPtEx = 5;// Refresh the scope expansion width

    private int mCreateMode;

    private int mDefaultFontId;
    private float mDefaultFontSize;
    private int mDefaultFontColor;
    private final int mfingerArea;
    private int mDefaultFieldFlags = 0;

    private final SparseArray<BaseFieldConfig> mFieldUpdateConfigs = new SparseArray<>();

    public FormFillerToolHandler(Context context, PDFViewCtrl pdfViewCtrl, FormFillerModule module) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
        mFormFillerModule = module;
        mfingerArea = (int) (AppDisplay.getFingerArea() / 4);

        mPaint = new Paint();
        mPaint.setStyle(Paint.Style.STROKE);
        mPaint.setAntiAlias(true);
        mPaint.setDither(true);
        // FormHighlightColor
        int color;
        if (mUiExtensionsManager.isFormHighlightEnable())
            color = (int) mUiExtensionsManager.getFormHighlightColor();
        else
            color = Color.parseColor("#0066cc");
        mPaint.setColor(color);

        mThickness = 2.0f;
        mDefaultFontSize = 0;
        mDefaultFontColor = Color.BLACK;
        mDefaultFontId = Font.e_StdIDCourier;
    }

    @Override
    public String getType() {
        return ToolHandler.TH_TYPE_FORMFILLER;
    }

    @Override
    public void onActivate() {
        mTouchCaptured = false;
        mLastPageIndex = -1;
        mCtlPtRadius = 5;
        mCtlPtRadius = AppDisplay.dp2px(mCtlPtRadius);
    }

    @Override
    public void onDeactivate() {
    }

    private Rect mTempRectInTouch = new Rect(0, 0, 0, 0);
    private Rect mInvalidateRect = new Rect(0, 0, 0, 0);

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        PointF disPoint = new PointF(motionEvent.getX(), motionEvent.getY());
        PointF pvPoint = new PointF();
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(disPoint, pvPoint, pageIndex);
        float x = pvPoint.x;
        float y = pvPoint.y;
        int action = motionEvent.getActionMasked();
        switch (action) {
            case MotionEvent.ACTION_DOWN:
                if (!mTouchCaptured && mLastPageIndex == -1 || mLastPageIndex == pageIndex) {
                    mTouchCaptured = true;
                    mStartPoint.x = x;
                    mStartPoint.y = y;
                    mStopPoint.x = x;
                    mStopPoint.y = y;
                    mDownPoint.set(x, y);
                    mTempRectInTouch.setEmpty();
                    if (mLastPageIndex == -1) {
                        mLastPageIndex = pageIndex;
                    }
                }
                return true;
            case MotionEvent.ACTION_POINTER_DOWN:
                mStartPoint.set(0, 0);
                mStopPoint.set(0, 0);
                mDrawRect.setEmpty();
                mDownPoint.set(0, 0);

                mTouchCaptured = false;
                mLastPageIndex = -1;
                mDownPoint.set(0, 0);
                MotionEvent downEvent = MotionEvent.obtain(motionEvent);
                downEvent.setAction(MotionEvent.ACTION_DOWN);
                boolean handled = mPdfViewCtrl.defaultTouchEvent(downEvent);
                downEvent.recycle();
                return handled;
            case MotionEvent.ACTION_MOVE:
                if (motionEvent.getPointerCount() > 1) {
                    return mPdfViewCtrl.defaultTouchEvent(motionEvent);
                }
                if (!mTouchCaptured || mLastPageIndex != pageIndex)
                    break;
                if (!mDownPoint.equals(x, y)) {
                    mStopPoint.x = x;
                    mStopPoint.y = y;
                    float thickness = thicknessOnPageView(pageIndex, mThickness);
                    float deltaXY = thickness / 2 + mCtlPtLineWidth + mCtlPtRadius * 2 + 2;// Judging border value
                    float line_k = (y - mStartPoint.y) / (x - mStartPoint.x);
                    float line_b = mStartPoint.y - line_k * mStartPoint.x;
                    if (y <= deltaXY && line_k != 0) {
                        // whether created annot beyond a PDF page(pageView)
                        mStopPoint.y = deltaXY;
                        mStopPoint.x = (mStopPoint.y - line_b) / line_k;
                    } else if (y >= (mPdfViewCtrl.getPageViewHeight(pageIndex) - deltaXY) && line_k != 0) {
                        mStopPoint.y = (mPdfViewCtrl.getPageViewHeight(pageIndex) - deltaXY);
                        mStopPoint.x = (mStopPoint.y - line_b) / line_k;
                    }
                    if (mStopPoint.x <= deltaXY) {
                        mStopPoint.x = deltaXY;
                    } else if (mStopPoint.x >= mPdfViewCtrl.getPageViewWidth(pageIndex) - deltaXY) {
                        mStopPoint.x = mPdfViewCtrl.getPageViewWidth(pageIndex) - deltaXY;
                    }

                    getDrawRect(mStartPoint.x, mStartPoint.y, mStopPoint.x, mStopPoint.y);

                    mInvalidateRect.set((int) mDrawRect.left, (int) mDrawRect.top, (int) mDrawRect.right, (int) mDrawRect.bottom);
                    mInvalidateRect.inset((int) (-mThickness * 12f - mControlPtEx), (int) (-mThickness * 12f - mControlPtEx));
                    if (!mTempRectInTouch.isEmpty()) {
                        mInvalidateRect.union(mTempRectInTouch);
                    }
                    mTempRectInTouch.set(mInvalidateRect);

                    mDownPoint.set(x, y);
                    if (Math.abs(mStartPoint.x - mStopPoint.x) > mfingerArea
                            || Math.abs(mStartPoint.y - mStopPoint.y) > mfingerArea) {
                        RectF _rect = AppDmUtil.rectToRectF(mInvalidateRect);
                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(_rect, _rect, pageIndex);
                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(_rect));
                    }
                }
                return true;
            case MotionEvent.ACTION_UP:
            case MotionEvent.ACTION_CANCEL:
                if (!mTouchCaptured || mLastPageIndex != pageIndex)
                    return mPdfViewCtrl.defaultTouchEvent(motionEvent);
                if (Math.abs(mStartPoint.x - mStopPoint.x) > mfingerArea
                        || Math.abs(mStartPoint.y - mStopPoint.y) > mfingerArea) {
                    createFormField();
                } else {
                    mStartPoint.set(0, 0);
                    mStopPoint.set(0, 0);
                    mDrawRect.setEmpty();
                    mDownPoint.set(0, 0);

                    mTouchCaptured = false;
                    mLastPageIndex = -1;
                    mDownPoint.set(0, 0);
                }
                return true;
            default:
                return true;
        }
        return false;
    }

    private RectF mPageViewThickness = new RectF(0, 0, 0, 0);

    private float thicknessOnPageView(int pageIndex, float thickness) {
        mPageViewThickness.set(0, 0, thickness, thickness);
        mPdfViewCtrl.convertPdfRectToPageViewRect(mPageViewThickness, mPageViewThickness, pageIndex);
        return Math.abs(mPageViewThickness.width());
    }

    private void getDrawRect(float x1, float y1, float x2, float y2) {
        float minx = Math.min(x1, x2);
        float miny = Math.min(y1, y2);
        float maxx = Math.max(x1, x2);
        float maxy = Math.max(y1, y2);

        mDrawRect.left = minx;
        mDrawRect.top = miny;
        mDrawRect.right = maxx;
        mDrawRect.bottom = maxy;
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        return false;
    }

    private RectF getBBox(int pageIndex) {
        RectF bboxRect = new RectF();
        bboxRect.set(mDrawRect);
        bboxRect.inset(-thicknessOnPageView(pageIndex, mThickness) / 2f, -thicknessOnPageView(pageIndex, mThickness) / 2f);
        mPdfViewCtrl.convertPageViewRectToPdfRect(bboxRect, bboxRect, pageIndex);
        return bboxRect;
    }

    private boolean mHasForm = false;

    private void createFormField() {
        if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
            RectF bboxRect = getBBox(mLastPageIndex);
            try {
                final PDFDoc doc = mPdfViewCtrl.getDoc();
                mHasForm = doc.hasForm();
                if (mForm == null)
                    mForm = new Form(doc);

                Widget widget = null;
                int fieldType = Field.e_TypeUnknown;
                String fieldName;
                final PDFPage page = doc.getPage(mLastPageIndex);
                final FormFillerAddUndoItem undoItem = new FormFillerAddUndoItem(mPdfViewCtrl);

                if (FormFillerConstants.CREATE_SIGNATURE_FILED == mCreateMode) {
                    final Signature signature = page.addSignature(AppUtil.toFxRectF(bboxRect));
                    fieldType = Field.e_TypeSignature;
                    fieldName = signature.getName();

                    widget = signature.getControl(0).getWidget();
                } else {
                    fieldName = FormFillerUtil.createFieldName(mForm, mCreateMode);
                    if (FormFillerConstants.CREATE_TEXT_FILED == mCreateMode) {
                        fieldType = Field.e_TypeTextField;
                    } else if (FormFillerConstants.CREATE_CHECKBOX == mCreateMode) {
                        fieldType = Field.e_TypeCheckBox;
                    } else if (FormFillerConstants.CREATE_RADIO_BUTTON == mCreateMode) {
                        fieldType = Field.e_TypeRadioButton;
                    } else if (FormFillerConstants.CREATE_COMBOBOX == mCreateMode) {
                        fieldType = Field.e_TypeComboBox;
                    } else if (FormFillerConstants.CREATE_LISTBOX == mCreateMode) {
                        fieldType = Field.e_TypeListBox;
                    } else if (FormFillerConstants.CREATE_IMAGE_FILED == mCreateMode) {
                        fieldType = Field.e_TypePushButton;
                    }

                    final Control control = mForm.addControl(page, fieldName, fieldType, AppUtil.toFxRectF(bboxRect));
                    widget = control.getWidget();
                }
                if (widget.isEmpty()) {
                    RectF invalidateRect = AppDmUtil.rectToRectF(mInvalidateRect);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(invalidateRect, invalidateRect, mLastPageIndex);
                    mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(invalidateRect));

                    mTouchCaptured = false;
                    mLastPageIndex = -1;
                    mDownPoint.set(0, 0);
                    return;
                }

                undoItem.mNM = FormFillerModule.ID_TAG + "_" + AppDmUtil.randomUUID(null);
                undoItem.mFieldName = fieldName;
                undoItem.mFieldType = fieldType;
                undoItem.mPageIndex = mLastPageIndex;
                undoItem.mFlags = Annot.e_FlagPrint;
                undoItem.mBBox = new RectF(bboxRect);
                undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
                undoItem.mRotation = (page.getRotation() + mPdfViewCtrl.getViewRotation()) % 4;
                undoItem.mFontId = mDefaultFontId;
                undoItem.mFontSize = mDefaultFontSize;
                undoItem.mFontColor = mDefaultFontColor;
                undoItem.mFieldFlags = mDefaultFieldFlags;

                FormFillerEvent event = new FormFillerEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, widget, mPdfViewCtrl);
                final Widget finalWidget = widget;
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success) {
                            mUiExtensionsManager.getDocumentManager().onAnnotAdded(page, finalWidget);
                            mUiExtensionsManager.getDocumentManager().addUndoItem(undoItem);

                            if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
                                try {
                                    RectF viewRect = AppUtil.toRectF(finalWidget.getRect());
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, mLastPageIndex);
                                    Rect rect = new Rect();
                                    viewRect.roundOut(rect);
                                    rect.inset(-10, -10);
                                    mPdfViewCtrl.refresh(mLastPageIndex, rect);
                                } catch (PDFException e) {
                                    e.printStackTrace();
                                }
                                mTouchCaptured = false;
                                mLastPageIndex = -1;
                                mDownPoint.set(0, 0);
                            }

                            if (!mHasForm) {
                                mHasForm = true;
                                mFormFillerModule.initForm(doc);
                            }
                        }
                    }
                });
                mPdfViewCtrl.addTask(task);
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        return false;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return false;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mLastPageIndex == pageIndex) {
            canvas.save();
            setPaint(pageIndex);

            mPaint.setStyle(Paint.Style.STROKE);
            mPaint.setAlpha(200);
            canvas.drawRect(mDrawRect, mPaint);

            mDrawRect.inset(thicknessOnPageView(pageIndex, mThickness) / 2, thicknessOnPageView(pageIndex, mThickness) / 2);
            mPaint.setStyle(Paint.Style.FILL);
            mPaint.setAlpha(32);
            canvas.drawRect(mDrawRect, mPaint);
            canvas.restore();
        }
    }

    private void setPaint(int pageIndex) {
        mPaint.setAntiAlias(true);
        PointF tranPt = new PointF(thicknessOnPageView(pageIndex, mThickness), thicknessOnPageView(pageIndex, mThickness));
        mPaint.setStrokeWidth(tranPt.x);
    }

    protected void changeCreateMode(int mode) {
        this.mCreateMode = mode;

        if (mCreateMode == FormFillerConstants.CREATE_TEXT_FILED) {
            BaseFieldConfig config = mFieldUpdateConfigs.get(Field.e_TypeTextField);
            if (config == null)
                config = mUiExtensionsManager.getConfig().uiSettings.form.textField;
            mDefaultFontId = FormFillerUtil.getSupportFontId(config.textFace);
            mDefaultFontColor = config.textColor;
            mDefaultFontSize = config.textSize;
        } else if (mCreateMode == FormFillerConstants.CREATE_CHECKBOX) {
            BaseFieldConfig config = mFieldUpdateConfigs.get(Field.e_TypeCheckBox);
            if (config == null)
                config = mUiExtensionsManager.getConfig().uiSettings.form.checkBox;
            mDefaultFontColor = config.textColor;
        } else if (mCreateMode == FormFillerConstants.CREATE_RADIO_BUTTON) {
            BaseFieldConfig config = mFieldUpdateConfigs.get(Field.e_TypeRadioButton);
            if (config == null)
                config = mUiExtensionsManager.getConfig().uiSettings.form.radioButton;
            mDefaultFontColor = config.textColor;
        } else if (mCreateMode == FormFillerConstants.CREATE_COMBOBOX) {
            ComboBoxConfig config = (ComboBoxConfig) mFieldUpdateConfigs.get(Field.e_TypeComboBox);
            if (config == null)
                config = mUiExtensionsManager.getConfig().uiSettings.form.comboBox;
            mDefaultFontId = FormFillerUtil.getSupportFontId(config.textFace);
            mDefaultFontColor = config.textColor;
            mDefaultFontSize = config.textSize;
            if (config.customText)
                mDefaultFieldFlags = Field.e_FlagComboEdit;
            else
                mDefaultFieldFlags = 0;
        } else if (mCreateMode == FormFillerConstants.CREATE_LISTBOX) {
            ListBoxConfig config = (ListBoxConfig) mFieldUpdateConfigs.get(Field.e_TypeListBox);
            if (config == null)
                config = mUiExtensionsManager.getConfig().uiSettings.form.listBox;
            mDefaultFontId = FormFillerUtil.getSupportFontId(config.textFace);
            mDefaultFontColor = config.textColor;
            mDefaultFontSize = config.textSize;
            if (config.multipleSelection)
                mDefaultFieldFlags = Field.e_FlagChoiceMultiSelect;
            else
                mDefaultFieldFlags = 0;
        }
    }

    protected boolean onKeyBack() {
        if (mUiExtensionsManager.getCurrentToolHandler() == this) {
            mUiExtensionsManager.setCurrentToolHandler(null);
            return true;
        }
        return false;
    }

    protected void release() {
        mHasForm = false;

        if (mForm != null) {
            mForm = null;
        }
    }

    void onFontColorChanged(int fieldType, int color) {
        BaseFieldConfig config = getConfig(fieldType);
        config.textColor = color;
        mFieldUpdateConfigs.put(fieldType, config);
    }

    void onFontNameChanged(int fieldType, String name) {
        BaseFieldConfig config = getConfig(fieldType);
        config.textFace = name;
        mFieldUpdateConfigs.put(fieldType, config);
    }

    void onFontSizeChanged(int fieldType, float size) {
        BaseFieldConfig config = getConfig(fieldType);
        config.textSize = (int) size;
        mFieldUpdateConfigs.put(fieldType, config);
    }

    void onFieldFlagsChanged(int fieldType, int flags) {
        if (Field.e_TypeComboBox == fieldType) {
            ComboBoxConfig config = (ComboBoxConfig) getConfig(Field.e_TypeComboBox);
            config.customText = (flags & Field.e_FlagComboEdit) != 0;
            mFieldUpdateConfigs.put(Field.e_TypeComboBox, config);
        } else if (Field.e_TypeListBox == fieldType) {
            ListBoxConfig config = (ListBoxConfig) getConfig(Field.e_TypeListBox);
            config.multipleSelection = (flags & Field.e_FlagChoiceMultiSelect) != 0;
            mFieldUpdateConfigs.put(Field.e_TypeListBox, config);
        }
    }

    private BaseFieldConfig getConfig(int fieldType) {
        BaseFieldConfig config = null;
        if (Field.e_TypeTextField == fieldType) {
            config = mFieldUpdateConfigs.get(Field.e_TypeTextField);
            if (config == null)
                config = new TextFieldConfig();
        } else if (Field.e_TypeCheckBox == fieldType) {
            config = mFieldUpdateConfigs.get(Field.e_TypeCheckBox);
            if (config == null)
                config = new CheckBoxConfig();
        } else if (Field.e_TypeRadioButton == fieldType) {
            config = mFieldUpdateConfigs.get(Field.e_TypeRadioButton);
            if (config == null)
                config = new RadioButtonConfig();
        } else if (Field.e_TypeComboBox == fieldType) {
            config = mFieldUpdateConfigs.get(Field.e_TypeComboBox);
            if (config == null)
                config = new ComboBoxConfig();
        } else if (Field.e_TypeListBox == fieldType) {
            config = mFieldUpdateConfigs.get(Field.e_TypeListBox);
            if (config == null)
                config = new ListBoxConfig();
        }
        return config;
    }

    private int mLastCreateMode;
    private IToolSupply mToolSupply;

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new FormFillerToolSupply(mContext);
        return mToolSupply;
    }

    private class FormFillerToolSupply extends ToolSupplyImpl {

        public FormFillerToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            int bgIcon = 0;
            if (ToolConstants.TextField == toolType) {
                bgIcon = R.drawable.form_tool_textfield;
            } else if (ToolConstants.CheckBox == toolType) {
                bgIcon = R.drawable.form_tool_checkbox;
            } else if (ToolConstants.RadioButton == toolType) {
                bgIcon = R.drawable.form_tool_radiobutton;
            } else if (ToolConstants.ComboBox == toolType) {
                bgIcon = R.drawable.form_tool_combobox;
            } else if (ToolConstants.ListBox == toolType) {
                bgIcon = R.drawable.form_tool_listbox;
            } else if (ToolConstants.SignatureField == toolType) {
                bgIcon = R.drawable.form_tool_sign;
            } else if (ToolConstants.ImageField == toolType) {
                bgIcon = R.drawable.form_tool_image;
            }
            return bgIcon;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return 0;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            return null;
        }

        @Override
        public String getToolName(int toolType) {
            return JsonConstants.TYPE_TEXTFIELD;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            if (itemBean.toolItem.isSelected()) {
                if (mUiExtensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_FORM_TAB) {
                    if (ToolConstants.TextField == itemBean.type) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FormBar_TextField);
                    } else if (ToolConstants.CheckBox == itemBean.type) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FormBar_CheckBox);
                    } else if (ToolConstants.RadioButton == itemBean.type) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FormBar_RadioButton);
                    } else if (ToolConstants.ComboBox == itemBean.type) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FormBar_ComboBox);
                    } else if (ToolConstants.ListBox == itemBean.type) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FormBar_ListBox);
                    } else if (ToolConstants.SignatureField == itemBean.type) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FormBar_SignatureField);
                    } else if (ToolConstants.ImageField == itemBean.type) {
                        mUiExtensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_FormBar_ImageField);
                    }
                }
                int mode = getCreateMode(itemBean.type);
                if (mode == FormFillerConstants.CREATE_NONE) return;

                mLastCreateMode = mCreateMode;
                mUiExtensionsManager.setCurrentToolHandler(FormFillerToolHandler.this);
                changeCreateMode(mode);
            } else {
                if (mUiExtensionsManager.getCurrentToolHandler() == FormFillerToolHandler.this) {
                    mCreateMode = mLastCreateMode;
                    mUiExtensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, PropertyBar.PropertyChangeListener propertyChangeListener) {
        }

        @Override
        public PropertyBar getPropertyBar() {
            return null;
        }

        private int getCreateMode(int type) {
            int mode = FormFillerConstants.CREATE_NONE;
            if (ToolConstants.TextField == type) {
                mode = FormFillerConstants.CREATE_TEXT_FILED;
            } else if (ToolConstants.CheckBox == type) {
                mode = FormFillerConstants.CREATE_CHECKBOX;
            } else if (ToolConstants.RadioButton == type) {
                mode = FormFillerConstants.CREATE_RADIO_BUTTON;
            } else if (ToolConstants.ComboBox == type) {
                mode = FormFillerConstants.CREATE_COMBOBOX;
            } else if (ToolConstants.ListBox == type) {
                mode = FormFillerConstants.CREATE_LISTBOX;
            } else if (ToolConstants.SignatureField == type) {
                mode = FormFillerConstants.CREATE_SIGNATURE_FILED;
            } else if (ToolConstants.ImageField == type) {
                mode = FormFillerConstants.CREATE_IMAGE_FILED;
            }
            return mode;
        }
    }

}
