/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.form;


import android.app.Activity;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.Intent;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Rect;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.view.ScaleGestureDetector;
import android.view.View;
import android.view.ViewGroup;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.interform.Form;
import com.foxit.uiextensions.IPolicyEventListener;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.IFlattenEventListener;
import com.foxit.uiextensions.annots.IRedactionEventListener;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.PropertyBarImpl;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemsManager;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.modules.dynamicxfa.XFADocProvider;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.impl.LifecycleEventListener;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.util.ArrayList;
import java.util.List;


public class FormFillerModule implements Module, PropertyBar.PropertyChangeListener {
    public static final String ID_TAG = "FoxitPDFSDK";

    private FormFillerToolHandler mToolHandler;
    private FormFillerAnnotHandler mAnnotHandler;
    private final Context mContext;
    private final PDFViewCtrl mPdfViewCtrl;
    private ViewGroup mParent;
    private final PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;
    private XFADocProvider mDocProvider = null;
    private boolean mbReloadForm = false;

    private static final int PAGING_SIZE = 20;
    private final List<Integer> mFixedPageds;

    protected void initForm(PDFDoc document) {
        try {
            if (mAnnotHandler.hasInitialized()) return;
            if (document != null && !mPdfViewCtrl.isDynamicXFA()) {
                boolean hasForm = document.hasForm();
                if (!hasForm) return;

                if (mHandler == null)
                    initHandler();
                mHandler.sendEmptyMessage(0);

                mAnnotHandler.init(document);
                if (mPdfViewCtrl.getXFADoc() != null)
                    mPdfViewCtrl.getXFADoc().setDocProviderCallback(mDocProvider = new XFADocProvider(mContext, mPdfViewCtrl));

                fixedPageFields(0);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        } finally {
            if (mHandler != null)
                mHandler.sendEmptyMessage(1);
        }
    }

    private Handler mHandler;

    private void initHandler() {
        mHandler = new Handler(Looper.getMainLooper()) {
            @Override
            public void handleMessage(Message msg) {
                switch (msg.what) {
                    case 0:
                        showProgressDlg();
                        break;
                    case 1:
                        dismissProgressDlg();
                        break;
                    default:
                        break;
                }
            }
        };
    }

    private ProgressDialog mProgressDlg;

    private void showProgressDlg() {
        if (mProgressDlg == null) {
            mProgressDlg = new ProgressDialog(((UIExtensionsManager) mUiExtensionsManager).getAttachedActivity());
            mProgressDlg.setProgressStyle(ProgressDialog.STYLE_SPINNER);
            mProgressDlg.setCancelable(false);
            mProgressDlg.setIndeterminate(false);
            mProgressDlg.setMessage(AppResource.getString(mContext, R.string.fx_string_opening));
        }
        AppDialogManager.getInstance().showAllowManager(mProgressDlg, null);
    }

    private void dismissProgressDlg() {
        if (mProgressDlg != null && mProgressDlg.isShowing()) {
            AppDialogManager.getInstance().dismiss(mProgressDlg);
            mProgressDlg = null;
        }
    }

    private final PDFViewCtrl.IDocEventListener mDocumentEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            if (errCode != Constants.e_ErrSuccess) return;

            release();
            if (mDocProvider != null)
                mDocProvider.setWillClose(true);
            initForm(document);
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
            release();
            if (mDocProvider != null) {
                mDocProvider.setWillClose(true);
            }
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
        }

        @Override
        public void onDocWillSave(PDFDoc document) {
            mAnnotHandler.startSave();
        }

        @Override
        public void onDocSaved(PDFDoc document, int errCode) {
            mAnnotHandler.stopSave();
        }
    };

    private final PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {
        @Override
        public void onPagesInserted(boolean success, int dstIndex, int[] range) {
            if (!success || mAnnotHandler.hasInitialized()) return;
            AppThreadManager.getInstance().runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    initForm(mPdfViewCtrl.getDoc());
                }
            });
        }

        @Override
        public void onPageChanged(int oldPageIndex, int curPageIndex) {
            if (mAnnotHandler.hasInitialized()) {
                int pageNum = (int) Math.floor((float) curPageIndex / PAGING_SIZE);
                if (!mFixedPageds.contains(pageNum)) {
                    fixedPageFields(pageNum);
                }
            }

            if (oldPageIndex == -1 && curPageIndex == 0) return;
            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
            AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if (curAnnot != null && currentAnnotHandler == mAnnotHandler && !mPdfViewCtrl.isContinuous()) {
                if (mAnnotHandler.mLastPageIndex != curPageIndex) {
                    uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                }
            }
        }
    };

    private void fixedPageFields(int pageNum) {
        mFixedPageds.add(pageNum);
        FixedPageFieldsTask fixedTask = new FixedPageFieldsTask(mAnnotHandler.getForm(), pageNum);
        mPdfViewCtrl.addTask(fixedTask);
    }

    private class FixedPageFieldsTask extends Task {

        private boolean mNeedRefresh = false;
        private final int mPageNum;
        private final Form mForm;
        private final List<Integer> mFixedPages;

        private FixedPageFieldsTask(Form form, int pageNum) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    FixedPageFieldsTask fixedTask = (FixedPageFieldsTask) task;
                    if (fixedTask.mNeedRefresh) {
                        for (int pageIndex : fixedTask.mFixedPages) {
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                int pageWidth = mPdfViewCtrl.getPageViewWidth(pageIndex);
                                int pageHeight = mPdfViewCtrl.getPageViewHeight(pageIndex);
                                mPdfViewCtrl.refresh(pageIndex, new Rect(0, 0, pageWidth, pageHeight));
                            }
                        }
                    }
                }
            });

            mPageNum = pageNum;
            mFixedPages = new ArrayList<>();
            mForm = form;
        }

        @Override
        protected void execute() {
            try {
                PDFDoc doc = mPdfViewCtrl.getDoc();
                if (doc == null) return;

                // MOBRD-6394, foxit::pdf::interform::Form::IsEmpty() const+16
                if (mAnnotHandler.getForm() == null)
                    return;

                int startIndex = mPageNum * PAGING_SIZE;
                int endIndex = Math.min(mPdfViewCtrl.getPageCount(), (mPageNum + 1) * PAGING_SIZE);
                for (int i = startIndex; i < endIndex; i++) {
                    PDFPage page = doc.getPage(i);
                    int annotCount = page.getAnnotCount();

                    int widgetCount = 0;
                    for (int j = 0; j < annotCount; j++) {
                        Annot annot = page.getAnnot(j);
                        if (!annot.isEmpty() && annot.getType() == Annot.e_Widget) {
                            widgetCount++;
                        }
                        annot.delete();
                    }

                    // MOBRD-6394, foxit::pdf::interform::Form::IsEmpty() const+16
                    if (mAnnotHandler.getForm() == null) {
                        mNeedRefresh = false;
                        return;
                    }

                    // MOBRD-7904: foxit::pdf::interform::Form::IsEmpty() const+16
                    if (mForm != null && Form.getCPtr(mForm) != 0 && !mForm.isEmpty()) {
                        int actCount = mForm.getControlCount(page);
                        if (widgetCount != actCount) {
                            mNeedRefresh = true;
                            mForm.fixPageFields(i);
                            mFixedPages.add(i);
                        }
                    }
                    page.delete();
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    }

    private final PDFViewCtrl.IScaleGestureEventListener mScaleGestureEventListener = new PDFViewCtrl.IScaleGestureEventListener() {

        @Override
        public boolean onScale(ScaleGestureDetector detector) {
            return false;
        }

        @Override
        public boolean onScaleBegin(ScaleGestureDetector detector) {
            if (mAnnotHandler.getFormFillerAssist() != null) {
                mAnnotHandler.getFormFillerAssist().setScaling(true);
            }

            if (mDocProvider != null) {
                mDocProvider.setScaleState(true);
            }
            return false;
        }

        @Override
        public void onScaleEnd(ScaleGestureDetector detector) {
            if (mAnnotHandler.getFormFillerAssist() != null) {
                mAnnotHandler.getFormFillerAssist().setScaling(false);
            }

            if (mDocProvider != null) {
                mDocProvider.setScaleState(false);
            }
        }
    };

    public FormFillerModule(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mParent = parent;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
        mFixedPageds = new ArrayList<>();
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_FORMFILLER;
    }

    public ToolHandler getToolHandler() {
        return mToolHandler;
    }

    public AnnotHandler getAnnotHandler() {
        return mAnnotHandler;
    }

    public void enableFormHighlight(boolean enable) {
        mAnnotHandler.enableFormHighlight(enable);
    }

    public void setFormHighlightColor(long color) {
        mAnnotHandler.setFormHighlightColor(color);
    }

    public void resetForm(Event.Callback callback) {
        EditFormTask editFormTask = new EditFormTask(FormFillerConstants.RESET_FORM_FIELDS, callback);
        mPdfViewCtrl.addTask(editFormTask);
    }

    public void exportFormToXML(String path, Event.Callback callback) {
        EditFormTask editFormTask = new EditFormTask(FormFillerConstants.EXPORT_FORM_DATA, path, callback);
        mPdfViewCtrl.addTask(editFormTask);
    }

    public void importFormFromXML(final String path, Event.Callback callback) {
        EditFormTask editFormTask = new EditFormTask(FormFillerConstants.IMPORT_FORM_DATA, path, callback);
        mPdfViewCtrl.addTask(editFormTask);
    }

    class EditFormTask extends Task {

        private boolean ret;
        private final int mType;
        private String mPath;

        private EditFormTask(int type, String path, final Event.Callback callBack) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    callBack.result(null, ((EditFormTask) task).ret);
                }
            });
            this.mPath = path;
            mType = type;
        }

        private EditFormTask(int type, final Event.Callback callBack) {
            super(new CallBack() {
                @Override
                public void result(Task task) {
                    callBack.result(null, ((EditFormTask) task).ret);
                }
            });
            mType = type;
        }

        @Override
        protected void execute() {
            switch (mType) {
                case FormFillerConstants.RESET_FORM_FIELDS:
                    try {
                        PDFViewCtrl.lock();
                        if (mAnnotHandler.getForm() != null)
                            ret = mAnnotHandler.getForm().reset();
                    } catch (PDFException e) {
                        e.printStackTrace();
                        ret = false;
                    } finally {
                        PDFViewCtrl.unlock();
                    }
                    break;
                case FormFillerConstants.IMPORT_FORM_DATA:
                    try {
                        PDFViewCtrl.lock();
                        if (mAnnotHandler.getForm() != null)
                            ret = mAnnotHandler.getForm().importFromXML(mPath);
                    } catch (PDFException e) {
                        ret = false;
                        e.printStackTrace();
                    } finally {
                        PDFViewCtrl.unlock();
                    }
                    break;
                case FormFillerConstants.EXPORT_FORM_DATA:
                    try {
                        PDFViewCtrl.lock();
                        if (mbReloadForm) {
                            if (mAnnotHandler.getForm() != null)
                                mAnnotHandler.getForm().reload();
                            mbReloadForm = false;
                        }
                        if (mAnnotHandler.getForm() != null)
                            ret = mAnnotHandler.getForm().exportToXML(mPath);
                        AppStorageManager storageManager = AppStorageManager.getInstance(mContext);
                        String target = storageManager.toExternalPathFromScopedCache(mPath);
                        if (target != null) {
                            ret = storageManager.copyDocument(mPath, target, true);
                        }
                    } catch (PDFException e) {
                        ret = false;
                        e.printStackTrace();
                    } finally {
                        PDFViewCtrl.unlock();
                    }
                    break;
                default:
                    break;
            }
        }
    }

    @Override
    public boolean loadModule() {
        mToolHandler = new FormFillerToolHandler(mContext, mPdfViewCtrl, this);
        mAnnotHandler = new FormFillerAnnotHandler(mContext, mParent, mPdfViewCtrl, this);
        mAnnotHandler.setPropertyBar(new PropertyBarImpl(mContext, mPdfViewCtrl));
        mAnnotHandler.setPropertyChangeListener(this);
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.registerToolHandler(mToolHandler);
            uiExtensionsManager.registerAnnotHandler(mAnnotHandler);
            uiExtensionsManager.registerLayoutChangeListener(mLayoutChangeListener);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerLifecycleListener(mLifecycleEventListener);
            uiExtensionsManager.registerConfigurationChangedListener(mConfigurationChangedListener);
            uiExtensionsManager.registerPolicyEventListener(mPolicyEventListener);
            uiExtensionsManager.getDocumentManager().registerFlattenEventListener(mFlattenEventListener);
            uiExtensionsManager.getDocumentManager().registerRedactionEventListener(mRedactionEventListener);
            uiExtensionsManager.addCreatePropertyChangedListener(mAnnotHandler.getType(), mCreatePropertyChangedListener);

            ToolItemsManager toolItemsManager = uiExtensionsManager.getToolsManager();
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_FORM, ToolConstants.TextField, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_FORM, ToolConstants.CheckBox, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_FORM, ToolConstants.RadioButton, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_FORM, ToolConstants.ComboBox, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_FORM, ToolConstants.ListBox, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_FORM, ToolConstants.SignatureField, mToolHandler.getToolSupply());
            toolItemsManager.addToolItem(UIToolBaseBar.USAGE_TYPE_FORM, ToolConstants.ImageField, mToolHandler.getToolSupply());
        }
        mPdfViewCtrl.registerDocEventListener(mDocumentEventListener);
        mPdfViewCtrl.registerPageEventListener(mPageEventListener);
        mPdfViewCtrl.registerScaleGestureEventListener(mScaleGestureEventListener);
        mPdfViewCtrl.registerRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            uiExtensionsManager.unregisterToolHandler(mToolHandler);
            uiExtensionsManager.unregisterLayoutChangeListener(mLayoutChangeListener);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
            uiExtensionsManager.unregisterAnnotHandler(mAnnotHandler);
            uiExtensionsManager.unregisterLifecycleListener(mLifecycleEventListener);
            uiExtensionsManager.unregisterPolicyEventListener(mPolicyEventListener);
            uiExtensionsManager.unregisterConfigurationChangedListener(mConfigurationChangedListener);
            uiExtensionsManager.getDocumentManager().unregisterFlattenEventListener(mFlattenEventListener);
            uiExtensionsManager.getDocumentManager().unregisterRedactionEventListener(mRedactionEventListener);
            uiExtensionsManager.removeCreatePropertyChangedListener(mAnnotHandler.getType());
        }
        mPdfViewCtrl.unregisterDocEventListener(mDocumentEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);
        mPdfViewCtrl.unregisterScaleGestureEventListener(mScaleGestureEventListener);
        mPdfViewCtrl.unregisterRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        if (mHandler != null)
            mHandler.removeCallbacksAndMessages(null);
        if (mAnnotHandler != null)
            mAnnotHandler.release();
        mParent = null;
        return true;
    }

    private final PDFViewCtrl.IRecoveryEventListener memoryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            release();
        }

        @Override
        public void onRecovered() {
        }
    };

    private void release() {
        mFixedPageds.clear();
        if (mAnnotHandler != null && mAnnotHandler.hasInitialized())
            mAnnotHandler.clear();
        if (mToolHandler != null)
            mToolHandler.release();
    }

    @Override
    public void onValueChanged(long property, int value) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        if (mAnnotHandler == uiExtensionsManager.getCurrentAnnotHandler()) {
            if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
                mAnnotHandler.onFontColorChanged(value);
            }
        }
    }

    @Override
    public void onValueChanged(long property, float value) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
        if (mAnnotHandler == currentAnnotHandler) {
            if (property == PropertyBar.PROPERTY_AUTO_FONTSIZE) {
                mAnnotHandler.onFontSizeChanged(value);
            }
        }
    }

    @Override
    public void onValueChanged(long property, String value) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        if (mAnnotHandler == uiExtensionsManager.getCurrentAnnotHandler()) {
            if (property == PropertyBar.PROPERTY_FONTNAME) {
                mAnnotHandler.onFontNameChanged(value);
            } else if (property == PropertyBar.PROPERTY_RADIO_NAME) {
                mAnnotHandler.onFieldNameChanged(value);
            }
        }
    }

    void onFieldFlagsChanged(int fieldType, int flags) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        if (uiExtensionsManager.canUpdateAnnotDefaultProperties()) {
            mToolHandler.onFieldFlagsChanged(fieldType, flags);
        }
    }

    public boolean onKeyBack() {
        return mToolHandler.onKeyBack() || mAnnotHandler.onKeyBack();
    }

    private final ILifecycleEventListener mLifecycleEventListener = new LifecycleEventListener() {

        @Override
        public void onResume(Activity act) {
            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
            AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if (curAnnot != null && currentAnnotHandler == mAnnotHandler) {
                if (mAnnotHandler.shouldShowInputSoft(curAnnot)) {
                    mAnnotHandler.showSoftInput();
                }
            }
        }

        @Override
        public void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
            AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if (curAnnot != null && currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onActivityResult(act, requestCode, resultCode, data);
            }
        }
    };

    private final PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {


        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    private final ILayoutChangeListener mLayoutChangeListener = new ILayoutChangeListener() {
        @Override
        public void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
            if (newWidth != oldWidth || newHeight != oldHeight) {
                UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
                AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
                if (currentAnnotHandler == mAnnotHandler) {
                    mAnnotHandler.onLayoutChange(v, newWidth, newHeight, oldWidth, oldHeight);
                }
            }
        }
    };

    private final IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            if (mAnnotHandler != null)
                mAnnotHandler.onThemeColorChanged();
        }
    };

    private final IFlattenEventListener mFlattenEventListener = new IFlattenEventListener() {
        @Override
        public void onAnnotsFlattened(PDFDoc doc) {
            release();
        }
    };

    private final IRedactionEventListener mRedactionEventListener = new IRedactionEventListener() {
        @Override
        public void onAnnotsApplied(PDFDoc doc) {
            mbReloadForm = true;
        }

        @Override
        public void onAnnotApplied(PDFPage page, Annot annot) {
            mbReloadForm = true;
        }
    };

    private final UIExtensionsManager.ConfigurationChangedListener mConfigurationChangedListener = new UIExtensionsManager.ConfigurationChangedListener() {
        @Override
        public void onConfigurationChanged(Configuration newConfig) {
            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
            AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
            if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onConfigurationChanged(newConfig);
            }
        }
    };

    private final IPolicyEventListener mPolicyEventListener = new IPolicyEventListener() {
        @Override
        public void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
            AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
            if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onRequestPermissionsResult(requestCode, permissions, grantResults);
            }
        }
    };

    private final PropertyBar.CreatePropertyChangedListener mCreatePropertyChangedListener = new PropertyBar.CreatePropertyChangedListener() {
        @Override
        public void onCreateValueChanged(long property, Object value) {
            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            int fieldType = FormFillerUtil.getAnnotFieldType(curAnnot);
            if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
                mToolHandler.onFontColorChanged(fieldType, (int) value);
            } else if (property == PropertyBar.PROPERTY_AUTO_FONTSIZE) {
                mToolHandler.onFontSizeChanged(fieldType, (float) value);
            } else if (property == PropertyBar.PROPERTY_FONTNAME) {
                mToolHandler.onFontNameChanged(fieldType, (String) value);
            }
        }

        @Override
        public int getToolType() {
            return -1;
        }

        @Override
        public int getTabPosition() {
            return ToolbarItemConfig.ITEM_FORM_TAB;
        }

        @Override
        public void resetToolPropertyBar() {
        }
    };

}
