/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.form;


import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Matrix;
import android.text.TextUtils;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Font;
import com.foxit.sdk.common.Image;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.Signature;
import com.foxit.sdk.pdf.actions.Action;
import com.foxit.sdk.pdf.actions.JavaScriptAction;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.BorderInfo;
import com.foxit.sdk.pdf.annots.DefaultAppearance;
import com.foxit.sdk.pdf.annots.IconFit;
import com.foxit.sdk.pdf.annots.Widget;
import com.foxit.sdk.pdf.interform.ChoiceOptionArray;
import com.foxit.sdk.pdf.interform.Control;
import com.foxit.sdk.pdf.interform.Field;
import com.foxit.sdk.pdf.interform.Form;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.form.undo.FormFillerAddUndoItem;
import com.foxit.uiextensions.annots.form.undo.FormFillerDeleteUndoItem;
import com.foxit.uiextensions.annots.form.undo.FormFillerModifyUndoItem;
import com.foxit.uiextensions.annots.form.undo.FormFillerUndoItem;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppUtil;

import java.util.ArrayList;

public class FormFillerEvent extends EditAnnotEvent {

    private Widget mWidget;

    public FormFillerEvent(int eventType, FormFillerUndoItem undoItem, Widget widget, PDFViewCtrl pdfViewCtrl) {
        mType = eventType;
        mUndoItem = undoItem;
        mWidget = widget;
        mPdfViewCtrl = pdfViewCtrl;
    }

    @Override
    public boolean add() {
        FormFillerAddUndoItem addUndoItem = (FormFillerAddUndoItem) mUndoItem;

        if (mWidget == null || mWidget.isEmpty())
            return false;
        try {
//            if (addUndoItem.mModifiedDate != null && AppDmUtil.isValidDateTime(addUndoItem.mModifiedDate)) {
//                mWidget.setModifiedDateTime(addUndoItem.mModifiedDate);
//            }

            if (addUndoItem.mPDFDict != null) {
                boolean reset = AppAnnotUtil.resetPDFDict(mWidget, addUndoItem.mPDFDict, true);
                if (reset) {
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
                    return true;
                }
            }

            if (addUndoItem.mValue != null) {
                mWidget.getField().setValue(addUndoItem.mValue);
            }
            int fieldType = FormFillerUtil.getAnnotFieldType(mWidget);
            if (addUndoItem.mFontId != -1) {
                DefaultAppearance da = new DefaultAppearance();
                int flags = DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagFontSize | DefaultAppearance.e_FlagTextColor;
                Font font = new Font(addUndoItem.mFontId);
                da.set(flags, font, addUndoItem.mFontSize, addUndoItem.mFontColor);
                if (fieldType == Field.e_TypeRadioButton)
                    mWidget.getControl().setDefaultAppearance(da);
                else
                    mWidget.getField().setDefaultAppearance(da);
            }

            if (fieldType == Field.e_TypeComboBox || fieldType == Field.e_TypeListBox) {
                mWidget.getField().setFlags(addUndoItem.mFieldFlags);
                if (addUndoItem.mOptions != null) {
                    ChoiceOptionArray choiceOptionArray = FormFillerUtil.options2Native(addUndoItem.mOptions);
                    mWidget.getField().setOptions(choiceOptionArray);
                }
            } else if (fieldType == Field.e_TypeRadioButton) {
                if (mWidget.getControl().isChecked() != addUndoItem.mIsChecked)
                    mWidget.getControl().setChecked(addUndoItem.mIsChecked);
            } else if (fieldType == Field.e_TypePushButton) {
                mWidget.setMKBackgroundColor(0xFFFFFF);
                mWidget.setMKBorderColor(0x000000);

                IconFit iconfit = new IconFit();
                iconfit.setFit_bounds(false);
                iconfit.setHorizontal_fraction(0.5f);
                iconfit.setVertical_fraction(0.5f);
                iconfit.setIs_proportional_scaling(true);
                iconfit.setScale_way_type(IconFit.e_ScaleWayAlways);
                mWidget.setMKIconFit(iconfit);
                mWidget.setMKIconCaptionRelation(Annot.e_MKRelationNoCaption);

                BorderInfo borderInfo = new BorderInfo();
                borderInfo.setStyle(BorderInfo.e_Solid);
                borderInfo.setWidth(0.5f);
                mWidget.setBorderInfo(borderInfo);

                Action action = Action.create(mPdfViewCtrl.getDoc(), Action.e_TypeJavaScript);
                JavaScriptAction javaScriptAction = new JavaScriptAction(action);
                javaScriptAction.setScript("event.target.buttonImportIcon()");
                mWidget.setAction(javaScriptAction);
                Bitmap logoBitmp = BitmapFactory.decodeResource(mPdfViewCtrl.getContext().getResources(), R.drawable.form_image_logo);
                if (logoBitmp != null) {
                    mWidget.setMKNormalIconBitmap(logoBitmp);
                    if (!logoBitmp.isRecycled()) {
                        logoBitmp.recycle();
                    }
                }
            }

            mWidget.setMKRotation(addUndoItem.mRotation);
            mWidget.setFlags(addUndoItem.mFlags);
            mWidget.setUniqueID(addUndoItem.mNM);
            mWidget.resetAppearanceStream();
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            return true;
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
        return false;
    }

    @Override
    public boolean modify() {
        if (mWidget == null || mWidget.isEmpty()) {
            return false;
        }

        try {
            FormFillerModifyUndoItem undoItem = (FormFillerModifyUndoItem) mUndoItem;

            int fieldType = FormFillerUtil.getAnnotFieldType(mWidget);
            if (fieldType == Field.e_TypeRadioButton) {
                setDefaultAppearance(mWidget, fieldType, undoItem);

                if (!TextUtils.isEmpty(undoItem.mFieldName) && !undoItem.mFieldName.equals(mWidget.getField().getName())) {
                    Form form = new Form(mPdfViewCtrl.getDoc());
                    FormFillerUtil.renameField(form, mWidget.getControl(), undoItem.mFieldName);
                    form.delete();
                }
                if (undoItem.mNeedResetChecked) {
                    Field field = mWidget.getField();
                    if (undoItem.mCheckedIndex != -1) {
                        Control control = field.getControl(undoItem.mCheckedIndex);
                        if (!control.isEmpty() && !control.isChecked()) {
                            control.setChecked(true);
                        }
                    } else {
                        int controlCount = field.getControlCount();
                        for (int i = 0; i < controlCount; i++) {
                            Control control = field.getControl(i);
                            if (!control.isEmpty()) {
                                control.setChecked(false);
                            }
                        }
                    }
                }
            } else if (fieldType == Field.e_TypePushButton) {
                if (undoItem.mPDFDict != null) {
                    AppAnnotUtil.resetPDFDict(mWidget, undoItem.mPDFDict, true);
                } else if (!AppUtil.isEmpty(undoItem.mImagePath)) {
                    if (undoItem.mImagePath.contains(FormFillerConstants.CAPTURE_IMAGE_SUFFIX_PATH)
                            || undoItem.mImagePath.endsWith(".jpg")
                            || undoItem.mImagePath.endsWith(".jpeg")) {
                        Bitmap bitmap = getBitmap(undoItem.mImagePath);
                        if (bitmap != null) {
                            mWidget.setMKNormalIconBitmap(bitmap);
                            mWidget.resetAppearanceStream();
                            bitmap.recycle();
                        }
                    } else {
                        Image image = new Image(undoItem.mImagePath);
                        mWidget.setMKNormalIconImage(image, 0);
                        mWidget.resetAppearanceStream();
                    }
                } else if (undoItem.mClearImage) {
                    mWidget.removeMKEntry(Annot.e_MKEntryNormalIcon);
                    mWidget.resetAppearanceStream();
                } else {
                    if (!undoItem.mBBox.equals(AppUtil.toRectF(mWidget.getRect()))) {
                        mWidget.move(AppUtil.toFxRectF(undoItem.mBBox));
                    }
                    mWidget.resetAppearanceStream();
                }
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
                return true;
            } else {
                Field field = mWidget.getField();

                if (undoItem.mValue != null
                        && !field.getValue().equals(undoItem.mValue)) {
                    field.setValue(undoItem.mValue);
                }

                if (fieldType != Field.e_TypeSignature) {
                    setDefaultAppearance(mWidget, fieldType, undoItem);

                    if (fieldType == Field.e_TypeComboBox || fieldType == Field.e_TypeListBox) {
                        if (field.getFlags() != undoItem.mFieldFlags) {
                            field.setFlags(undoItem.mFieldFlags);
                        }

                        ArrayList<ChoiceItemInfo> options = FormFillerUtil.getOptions(field);
                        if (undoItem.mOptions != null && FormFillerUtil.optionsIsChanged(undoItem.mOptions, options)) {
                            ChoiceOptionArray choiceOptionArray = FormFillerUtil.options2Native(undoItem.mOptions);
                            field.setOptions(choiceOptionArray);
                        }
                    }
                }
            }

            if (!undoItem.mBBox.equals(AppUtil.toRectF(mWidget.getRect()))) {
                mWidget.move(AppUtil.toFxRectF(undoItem.mBBox));
            }

            mWidget.resetAppearanceStream();
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            return true;
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
        return false;
    }

    private Bitmap getBitmap(String path) {
        try {
            BitmapFactory.Options options = new BitmapFactory.Options();
            options.inSampleSize = 2;
            Bitmap bitmap = BitmapFactory.decodeFile(path, options);
            int pictureRotate = AppFileUtil.readPictureDegree(path);
            if (pictureRotate != 0) {
                Matrix matrix = new Matrix();
                matrix.postRotate(pictureRotate);
                bitmap = Bitmap.createBitmap(bitmap, 0, 0,
                        bitmap.getWidth(), bitmap.getHeight(), matrix, true);
            }
            return bitmap;
        } catch (Exception ignored) {
        }
        return null;
    }

    private void setDefaultAppearance(Widget widget, int type, FormFillerModifyUndoItem undoItem) {
        try {
            DefaultAppearance da;
            if (type == Field.e_TypeRadioButton) {
                da = widget.getControl().getDefaultAppearance();
            } else {
                da = widget.getField().getDefaultAppearance();
            }

            Font font = da.getFont();
            if ((!font.isEmpty() && undoItem.mFontId != FormFillerUtil.getStandard14Font(da, mPdfViewCtrl.getDoc()))
                    || (da.getText_size() != undoItem.mFontSize)
                    || (da.getText_color() != undoItem.mFontColor)) {
                int flags = DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagFontSize | DefaultAppearance.e_FlagTextColor;
                da.setFlags(flags);
                da.setFont(new Font(undoItem.mFontId));
                da.setText_color(undoItem.mFontColor);
                da.setText_size(undoItem.mFontSize);

                if (type == Field.e_TypeRadioButton) {
                    widget.getControl().setDefaultAppearance(da);
                } else {
                    widget.getField().setDefaultAppearance(da);
                }
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    @Override
    public boolean delete() {
        if (mWidget == null || mWidget.isEmpty()) {
            return false;
        }
        try {
            FormFillerDeleteUndoItem deleteUndoItem = (FormFillerDeleteUndoItem) mUndoItem;
            if (Field.e_TypeSignature == deleteUndoItem.mFieldType) {
                PDFDoc doc = mPdfViewCtrl.getDoc();
                Field field = mWidget.getField();
                Signature signature = new Signature(field);
                doc.removeSignature(signature);
            } else {
                Field field = mWidget.getField();
                Form form = new Form(mPdfViewCtrl.getDoc());
                if (field.getControlCount() <= 1) {
                    form.removeField(field);
                } else {
                    form.removeControl(mWidget.getControl());
                }
                form.delete();
            }
            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            return true;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

}
