/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.form;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.DatePickerDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.ColorStateList;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.DashPathEffect;
import android.graphics.DrawFilter;
import android.graphics.Paint;
import android.graphics.PaintFlagsDrawFilter;
import android.graphics.Path;
import android.graphics.PathEffect;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.RectF;
import android.os.Handler;
import android.os.SystemClock;
import android.text.ClipboardManager;
import android.text.Editable;
import android.text.TextUtils;
import android.text.TextWatcher;
import android.util.SparseBooleanArray;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnKeyListener;
import android.view.ViewGroup;
import android.view.inputmethod.EditorInfo;
import android.widget.AdapterView;
import android.widget.DatePicker;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Font;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.actions.Action;
import com.foxit.sdk.pdf.actions.AdditionalAction;
import com.foxit.sdk.pdf.actions.JavaScriptAction;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.DefaultAppearance;
import com.foxit.sdk.pdf.annots.Widget;
import com.foxit.sdk.pdf.interform.Control;
import com.foxit.sdk.pdf.interform.Field;
import com.foxit.sdk.pdf.interform.Filler;
import com.foxit.sdk.pdf.interform.Form;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.uiextensions.DocumentManager;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotActionHandler;
import com.foxit.uiextensions.annots.AnnotContent;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.form.undo.FormFillerDeleteUndoItem;
import com.foxit.uiextensions.annots.form.undo.FormFillerModifyUndoItem;
import com.foxit.uiextensions.annots.form.undo.FormFillerUndoItem;
import com.foxit.uiextensions.annots.freetext.FtTextUtil;
import com.foxit.uiextensions.annots.multimedia.IAudioPlayView;
import com.foxit.uiextensions.annots.multimedia.sound.OpenSoundTask;
import com.foxit.uiextensions.annots.multimedia.sound.SoundModule;
import com.foxit.uiextensions.controls.dialog.FxProgressDialog;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFileSelectDialog;
import com.foxit.uiextensions.controls.propertybar.AnnotMenu;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.data.ISuggestionsType;
import com.foxit.uiextensions.data.SuggestionsHelper;
import com.foxit.uiextensions.modules.panel.annot.AnnotPanelModule;
import com.foxit.uiextensions.pdfreader.config.ActRequestCode;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AnnotPermissionUtil;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDevice;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppIntentUtil;
import com.foxit.uiextensions.utils.AppKeyboardUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.IResult;
import com.foxit.uiextensions.utils.SystemUiHelper;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.io.FileFilter;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.CountDownLatch;

public class FormFillerAnnotHandler implements AnnotHandler {
    private static final int DEFAULT_COLOR = PropertyBar.PB_COLORS_TOOL_DEFAULT[0];
    private static final int DEFAULT_THICKNESS = 4;

    /*
     *  LT     T     RT
     *   1-----2-----3
     *   |	         |
     *   |	         |
     * L 8           4 R
     *   |           |
     *   |           |
     *   7-----6-----5
     *   LB    B     RB
     *   */
    public static final int CTR_NONE = -1;
    public static final int CTR_LT = 1;
    public static final int CTR_T = 2;
    public static final int CTR_RT = 3;
    public static final int CTR_R = 4;
    public static final int CTR_RB = 5;
    public static final int CTR_B = 6;
    public static final int CTR_LB = 7;
    public static final int CTR_L = 8;
    private int mCurrentCtr = CTR_NONE;

    public static final int OPER_DEFAULT = -1;
    public static final int OPER_SCALE_LT = 1;// old:start at 0
    public static final int OPER_SCALE_T = 2;
    public static final int OPER_SCALE_RT = 3;
    public static final int OPER_SCALE_R = 4;
    public static final int OPER_SCALE_RB = 5;
    public static final int OPER_SCALE_B = 6;
    public static final int OPER_SCALE_LB = 7;
    public static final int OPER_SCALE_L = 8;
    public static final int OPER_TRANSLATE = 9;
    private int mLastOper = OPER_DEFAULT;

    private float mCtlPtLineWidth = 2;
    private float mCtlPtRadius = 5;
    private float mCtlPtTouchExt = 20;
    private float mCtlPtDeltyXY = 20;// Additional refresh range

    private Paint mFrmPaint;// outline
    private Paint mCtlPtPaint;

    private PointF mDownPoint;
    private PointF mLastPoint;
    private PointF mLastTouchPoint = new PointF(0, 0);

    private PropertyBar mPropertyBar;
    private PropertyBar.PropertyChangeListener mPropertyChangeListener;
    private FtTextUtil mTextUtil;

    private Blink mBlink = null;
    private AnnotMenu mAnnotationMenu;
    private Annot mLastAnnot;
    private Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private ViewGroup mParent;
    private Filler mFormFiller;
    private FormFillerAssistImpl mAssist;
    private Form mForm;
    private EditText mEditView = null;
    private FormNavigationModule mFNModule = null;
    private FormFillerModule mFormFillerModule;

    private ArrayList<Integer> mMenuText;
    private String mBeforeText = null;
    private String mLastFieldName = "";
    private Paint mPathPaint;
    private int mOffset;
    private int mPageOffset;
    private int mLastOrientation;
    int mLastPageIndex;

    private boolean mTouchCaptured = false;
    //    private boolean mIsBackBtnPush = false; //for some input method, double backspace click
    private boolean mAdjustPosition = false;
    private boolean mIsShowEditText = false;
    private boolean bInitialize = false;
    private boolean mIsModify = false;
    private boolean mIsLongPressTouchEvent = false;
    private boolean mIsEdittextOffset = false;
    private boolean mAnnotIsSelected = false;
    private boolean mShouldRefreshAnnotPanel;
    private boolean mbSupportMultiline;
    private boolean mShouldAddSuggestions = false;
    private boolean mShouldInterceptTextChanged = false;
    private boolean mIsTouchComboPopup = false;
    private boolean isLButtonDownHandled = false;

    private SparseBooleanArray mMenuItemArray;
    private PointF mLastPopupPdfPoint;

    private JSActionCallback mJSActionCallback;
    private final String mCaptureImgPath;
    private UIFileSelectDialog mFileSelectDialog = null;

    private boolean mIsDateField;
    private String mDatePattern;
    private Bitmap mDateFieldIcon;
    private Date mDateFieldValue;

    public FormFillerAnnotHandler(Context context, ViewGroup parent, PDFViewCtrl pdfViewCtrl, FormFillerModule formFillerModule) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mParent = parent;
        mFormFillerModule = formFillerModule;

        mCaptureImgPath = AppFileUtil.getDiskCachePath(context) + FormFillerConstants.CAPTURE_IMAGE_SUFFIX_PATH;
    }

    public void init(final PDFDoc doc) {
        mLastOrientation = mContext.getResources().getConfiguration().orientation;

        mTextUtil = new FtTextUtil(mContext, mPdfViewCtrl);
        mMenuItemArray = new SparseBooleanArray();
        mAssist = new FormFillerAssistImpl(mPdfViewCtrl, new FillerFocusEventListener() {
            @Override
            public void focusGotOnControl(Control control, String filedValue) {
                if (mAnnotIsSelected) return;
                if (control == null || control.isEmpty())
                    return;

                try {
                    Widget widget = control.getWidget();
                    if (widget == null || widget.isEmpty())
                        return;
                    Annot curAnnot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
                    if (!AppAnnotUtil.equals(widget, curAnnot)) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(widget);
                    }
                } catch (PDFException e) {
                    e.printStackTrace();
                }
            }

            @Override
            public void focusLostFromControl(Control control, String filedValue) {
                if (!mAnnotIsSelected) return;
                if (control == null || control.isEmpty())
                    return;

                try {
                    Widget widget = control.getWidget();
                    if (widget == null || widget.isEmpty())
                        return;

                    Annot curAnnot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
                    if (AppAnnotUtil.equals(widget, curAnnot)) {
                        if (shouldShowInputSoft(widget)) {
                            if (mBlink != null) {
                                mBlink.removeCallbacks(mBlink);
                                mBlink = null;
                            }
                            dismissSoftInput();
                            mParent.removeView(mEditView);
                        }
                        if (shouldShowNavigation(widget)) {
                            if (mFNModule != null) {
                                mFNModule.hideFormNavigationBar();
                            }
                            resetDocViewerOffset();
                        }
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null);
                    }
                } catch (PDFException e) {
                    e.printStackTrace();
                }
            }
        });
        mAssist.setPopupMenuListener(new FillerPopupMenuListener() {
            @Override
            void appendPopupMenuItem(int itemId, String defaultString) {
                mMenuItemArray.put(itemId + AnnotMenu.PM_BT_INDEX_OFFSET, true);
            }

            @Override
            void enablePopupMenuItem(int itemId, boolean isEnabled) {
                int key = itemId + AnnotMenu.PM_BT_INDEX_OFFSET;
                if (!isEnabled) {
                    mMenuItemArray.delete(key);
                } else {
                    mMenuItemArray.put(key, true);
                }
            }

            @Override
            void showPopupMenu(com.foxit.sdk.common.fxcrt.PointF point) {
                PointF pointF = AppUtil.toPointF(mLastLongPressFxPoint != null ? mLastLongPressFxPoint : point);
                mLastPopupPdfPoint = pointF;
                prepareFillerPopupMenu();
                if (mAnnotationMenu.isShowing()) {
                    mAnnotationMenu.dismiss();
                }
                mAnnotationMenu.show(getPopupMenuRect(pointF));
            }

            @Override
            void setClipboardText(String text) {
                ClipboardManager clipboard = (ClipboardManager) mContext.getSystemService(Context.CLIPBOARD_SERVICE);
                clipboard.setText(text);
                AppAnnotUtil.toastAnnotCopy(mContext);
            }

            @Override
            String getClipboardText() {
                ClipboardManager clipboard = (ClipboardManager) mContext.getSystemService(Context.CLIPBOARD_SERVICE);
                if (clipboard != null && clipboard.hasText()) {
                    return clipboard.getText().toString();
                }
                return super.getClipboardText();
            }
        });
        mAssist.bWillClose = false;
        mPathPaint = new Paint(Paint.ANTI_ALIAS_FLAG);
        mPathPaint.setStyle(Paint.Style.STROKE);
        mPathPaint.setAntiAlias(true);
        mPathPaint.setDither(true);
        PathEffect effects = new DashPathEffect(new float[]{1, 2, 4, 8}, 1);
        mPathPaint.setPathEffect(effects);

        PathEffect effect = AppAnnotUtil.getAnnotBBoxPathEffect();
        mFrmPaint = new Paint();
        mFrmPaint.setPathEffect(effect);
        mFrmPaint.setStyle(Paint.Style.STROKE);
        mFrmPaint.setAntiAlias(true);
        mFrmPaint.setColor(DEFAULT_COLOR | 0xFF000000);

        mCtlPtPaint = new Paint();
        mDownPoint = new PointF();
        mLastPoint = new PointF();

        mMenuText = new ArrayList<>();
        mAnnotationMenu = new AnnotMenuImpl(mContext, mPdfViewCtrl);

        try {
            mForm = new Form(doc);
            mFormFiller = new Filler(mForm, mAssist);

            boolean enableFormHighlight = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isFormHighlightEnable();
            mFormFiller.highlightFormFields(enableFormHighlight);
            if (enableFormHighlight) {
                mFormFiller.setHighlightColor((int) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getFormHighlightColor());
            }
        } catch (PDFException e) {
            e.printStackTrace();
            return;
        }

        initFormNavigation();
        bInitialize = true;
    }

    private RectF getPopupMenuRect(PointF anchor) {
        PointF pointF = new PointF();
        pointF.set(anchor);
        mPdfViewCtrl.convertPdfPtToPageViewPt(pointF, pointF, mLastPageIndex);
        mPdfViewCtrl.convertPageViewPtToDisplayViewPt(pointF, pointF, mLastPageIndex);
        Rect rect = new Rect();
        mPdfViewCtrl.getGlobalVisibleRect(rect);
        pointF.y = pointF.y + rect.height() - mPdfViewCtrl.getHeight();
        if (mDownPdfPoint != null && mUpPdfPoint != null) {
            PointF downPoint = new PointF();
            PointF upPoint = new PointF();
            mPdfViewCtrl.convertPdfPtToPageViewPt(mDownPdfPoint, downPoint, mLastPageIndex);
            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(downPoint, downPoint, mLastPageIndex);
            mPdfViewCtrl.convertPdfPtToPageViewPt(mUpPdfPoint, upPoint, mLastPageIndex);
            mPdfViewCtrl.convertPageViewPtToDisplayViewPt(upPoint, upPoint, mLastPageIndex);
            int offsetX = (int) (downPoint.x - upPoint.x);
            if (Math.abs(offsetX) > 1 && pointF.x > 0) {
                if (offsetX > 0) {
                    return new RectF(pointF.x, pointF.y - 10, pointF.x + offsetX, pointF.y + 10);
                } else {
                    return new RectF(pointF.x + offsetX, pointF.y - 10, pointF.x, pointF.y + 10);
                }
            }
        }
        return new RectF(pointF.x - 10, pointF.y - 10, pointF.x + 10, pointF.y + 10);
    }

    void enableFormHighlight(boolean enable) {
        try {
            if (mFormFiller != null)
                mFormFiller.highlightFormFields(enable);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void setFormHighlightColor(long color) {
        try {
            if (mFormFiller != null)
                mFormFiller.setHighlightColor((int) color);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void preparePropertyBar(Annot annot) {
        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        mPropertyBar.setEditable(documentManager.canAddAnnot() && documentManager.canModifyForm()
                && ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()
                && AnnotPermissionUtil.canEditabled(documentManager, annot));
        int[] colors = new int[PropertyBar.PB_COLORS_TOOL_DEFAULT.length];
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, colors, 0, colors.length);
        mPropertyBar.setColors(colors);
        mPropertyBar.setArrowVisible(false);
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);

        try {
            int fieldType = FormFillerUtil.getAnnotFieldType(annot);
            if (fieldType == Field.e_TypeCheckBox) {
                DefaultAppearance da = ((Widget) annot).getField().getDefaultAppearance();
                mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, da.getText_color());
                mPropertyBar.reset(getProperties(Field.e_TypeCheckBox));
            } else if (fieldType == Field.e_TypeTextField) {
                float[] fontSizes = new float[PropertyBar.PB_FONTSIZES.length + 1];
                fontSizes[0] = 0.0f;
                System.arraycopy(PropertyBar.PB_FONTSIZES, 0, fontSizes, 1, fontSizes.length - 1);
                mPropertyBar.setProperty(PropertyBar.PROPERTY_AUTO_FONTSIZE, fontSizes);

                DefaultAppearance da = ((Widget) annot).getField().getDefaultAppearance();
                mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, da.getText_color());
                mPropertyBar.setProperty(PropertyBar.PROPERTY_FONTNAME, mTextUtil.getSupportFontName(mTextUtil.getSupportFontID(da, mPdfViewCtrl.getDoc())));
                mPropertyBar.setProperty(PropertyBar.PROPERTY_AUTO_FONTSIZE, da.getText_size());
                mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_FONTNAME, AppResource.getString(mContext, R.string.pb_font_settings));
                mPropertyBar.reset(getProperties(Field.e_TypeTextField));
            } else if (fieldType == Field.e_TypeRadioButton) {
                DefaultAppearance da = ((Widget) annot).getControl().getDefaultAppearance();
                mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, da.getText_color());
                mPropertyBar.setProperty(PropertyBar.PROPERTY_RADIO_NAME, mTempFieldName);
                mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_RADIO_NAME, AppResource.getString(mContext, R.string.pb_font_settings));
                mPropertyBar.reset(getProperties(Field.e_TypeRadioButton));
            } else if (fieldType == Field.e_TypeComboBox) {
                float[] fontSizes = new float[PropertyBar.PB_FONTSIZES.length + 1];
                fontSizes[0] = 0.0f;
                System.arraycopy(PropertyBar.PB_FONTSIZES, 0, fontSizes, 1, fontSizes.length - 1);
                mPropertyBar.setProperty(PropertyBar.PROPERTY_AUTO_FONTSIZE, fontSizes);

                DefaultAppearance da = ((Widget) annot).getField().getDefaultAppearance();
                mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, da.getText_color());
                mPropertyBar.setProperty(PropertyBar.PROPERTY_FONTNAME, mTextUtil.getSupportFontName(mTextUtil.getSupportFontID(da, mPdfViewCtrl.getDoc())));
                mPropertyBar.setProperty(PropertyBar.PROPERTY_AUTO_FONTSIZE, da.getText_size());
                mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_FONTNAME, AppResource.getString(mContext, R.string.pb_font_settings));
                mPropertyBar.reset(getProperties(Field.e_TypeComboBox));

                mPropertyBar.addCustomItem(PropertyBar.PROPERTY_OPTIONS, getOptionsView(Field.e_TypeComboBox, annot), 0, -1);
            } else if (fieldType == Field.e_TypeListBox) {
                float[] fontSizes = new float[PropertyBar.PB_FONTSIZES.length + 1];
                fontSizes[0] = 0.0f;
                System.arraycopy(PropertyBar.PB_FONTSIZES, 0, fontSizes, 1, fontSizes.length - 1);
                mPropertyBar.setProperty(PropertyBar.PROPERTY_AUTO_FONTSIZE, fontSizes);

                DefaultAppearance da = ((Widget) annot).getField().getDefaultAppearance();
                mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, da.getText_color());
                mPropertyBar.setProperty(PropertyBar.PROPERTY_FONTNAME, mTextUtil.getSupportFontName(mTextUtil.getSupportFontID(da, mPdfViewCtrl.getDoc())));
                mPropertyBar.setProperty(PropertyBar.PROPERTY_AUTO_FONTSIZE, da.getText_size());
                mPropertyBar.setPropertyTitle(PropertyBar.PROPERTY_FONTNAME, AppResource.getString(mContext, R.string.pb_font_settings));
                mPropertyBar.reset(getProperties(Field.e_TypeListBox));

                mPropertyBar.addCustomItem(PropertyBar.PROPERTY_OPTIONS, getOptionsView(Field.e_TypeListBox, annot), 0, -1);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private ChoiceItemAdapter mChoiceItemAdapter;
    private FormOptionsListView mLvChoiceOptions;
    private TextView mTvAddChoiceOptions;
    private ArrayList<ChoiceItemInfo> mCurOptions;

    private View getOptionsView(final int type, final Annot annot) {
        View optionsItem = View.inflate(mContext, R.layout.pb_shape, null);
        TextView optionsTitleTv = optionsItem.findViewById(R.id.pb_shape_type_tv);
        optionsTitleTv.setText(AppResource.getString(mContext, R.string.pb_options_tab));
        TextView subTitleTv = optionsItem.findViewById(R.id.pb_subtitle_tv);
        subTitleTv.setVisibility(View.VISIBLE);
        subTitleTv.setText(AppResource.getString(mContext, R.string.pb_options_tab));
        optionsItem.findViewById(R.id.pb_ll_shape).setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mPropertyBar.setSubTitleVisible(false);
                LinearLayout pbContentView = mPropertyBar.getContentView();
                int contentWidth = pbContentView.getMeasuredWidth();
                for (int i = 0; i < pbContentView.getChildCount(); i++) {
                    LinearLayout tabContent = (LinearLayout) pbContentView.getChildAt(i);
                    tabContent.removeAllViews();
                }
                pbContentView.removeAllViews();
                View contentView = getOptionsListView(type, annot);
                pbContentView.addView(contentView);

                contentView.measure(0, 0);
                int contentHeight = contentView.getMeasuredHeight();
                ViewGroup.LayoutParams layoutParams = pbContentView.getLayoutParams();
                layoutParams.height = contentHeight;
                pbContentView.setLayoutParams(layoutParams);

                try {
                    int pageIndex = annot.getPage().getIndex();
                    RectF annotRectF = AppUtil.toRectF(annot.getRect());
                    mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, pageIndex);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(annotRectF, annotRectF, pageIndex);

                    UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                    RectF rectF = new RectF();
                    if (SystemUiHelper.getInstance().isStatusBarShown(uiExtensionsManager.getAttachedActivity())) {
                        rectF.set(AppUtil.toGlobalVisibleRectF(uiExtensionsManager.getRootView(), annotRectF));
                    } else {
                        rectF.set(annotRectF);
                    }
                    mPropertyBar.show(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getRootView(),
                            rectF,
                            contentWidth,
                            contentHeight,
                            UIPopoverWin.ARROW_AUTO);
                } catch (PDFException e) {
                    e.printStackTrace();
                }
            }
        });
        return optionsItem;
    }

    private RelativeLayout mPropCheckTrack;
    private ImageView mPropCheckThumb;

    @SuppressLint("ClickableViewAccessibility")
    private View getOptionsListView(final int type, final Annot annot) {
        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        final boolean canEdit = documentManager.canAddAnnot() && documentManager.canModifyForm() && !(AppAnnotUtil.isLocked(annot)
                && ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()
                || AppAnnotUtil.isReadOnly(annot));
        final View view = View.inflate(mContext, R.layout.pb_form_itemlist, null);

        LinearLayout titleView = view.findViewById(R.id.pb_ll_back_view);
        TextView titleTv = titleView.findViewById(R.id.pb_title_text_view);
        titleTv.setText(AppResource.getString(mContext, R.string.pb_options_tab));
        titleView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                mPropertyBar.setSubTitleVisible(true);
                mPropertyBar.reset(getProperties(type), false);
                mPropertyBar.update();
            }
        });

        TextView customTypeTv = view.findViewById(R.id.pb_custom_type_tv);
        mPropCheckTrack = view.findViewById(R.id.pb_custom_type_check_track);
        mPropCheckThumb = view.findViewById(R.id.pb_custom_type_check_thumb);
        ThemeUtil.setBackgroundTintList(mPropCheckTrack, getPropTrackColorStateList());
        mLvChoiceOptions = view.findViewById(R.id.rd_form_item_list);
        mTvAddChoiceOptions = view.findViewById(R.id.pb_tv_no_item_tips);

        try {
            if (type == Field.e_TypeComboBox) {
                customTypeTv.setText(AppResource.getString(mContext, R.string.fx_combox_custom_text));
                setPropChekced((((Widget) annot).getField().getFlags() & Field.e_FlagComboEdit) != 0);
            } else if (type == Field.e_TypeListBox) {
                customTypeTv.setText(AppResource.getString(mContext, R.string.fx_listbox_multiple_text));
                setPropChekced((((Widget) annot).getField().getFlags() & Field.e_FlagChoiceMultiSelect) != 0);
            }

            if (mCurOptions == null) {
                mCurOptions = new ArrayList<>();
                mCurOptions = FormFillerUtil.cloneChoiceOptions(mTempOptions);
            }
            long size = mCurOptions.size();
            if (size > 0) {
                mLvChoiceOptions.setVisibility(View.VISIBLE);
                mTvAddChoiceOptions.setText("");
            } else {
                mLvChoiceOptions.setVisibility(View.GONE);
                mTvAddChoiceOptions.setText(AppResource.getString(mContext, R.string.fx_no_item_tips));
            }

            mChoiceItemAdapter = new ChoiceItemAdapter(mContext, mCurOptions, mLvChoiceOptions);
            mLvChoiceOptions.setAdapter(mChoiceItemAdapter);
            if (canEdit) {
                mPropCheckTrack.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        setPropChekced(!mPropCheckTrack.isSelected());
                        boolean isChecked = mPropCheckTrack.isSelected();

                        int flags = 0;
                        try {
                            flags = ((Widget) annot).getField().getFlags();
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }

                        if (type == Field.e_TypeComboBox) {
                            if (isChecked)
                                flags = flags | Field.e_FlagComboEdit;
                            else
                                flags = flags & (~Field.e_FlagComboEdit);
                        } else if (type == Field.e_TypeListBox) {
                            if (isChecked) {
                                flags = flags | Field.e_FlagChoiceMultiSelect;
                            } else {
                                flags = flags & (~Field.e_FlagChoiceMultiSelect);

                                boolean hasSelected = false;
                                for (ChoiceItemInfo itemInfo : mCurOptions) {
                                    if (!hasSelected && itemInfo.selected) {
                                        hasSelected = true;
                                        continue;
                                    }
                                    itemInfo.selected = false;
                                }
                                mChoiceItemAdapter.notifyDataSetChanged();
                            }
                        }
                        mFormFillerModule.onFieldFlagsChanged(type, flags);

                        FormFillerContent fillerContent = new FormFillerContent(mPdfViewCtrl, (Widget) mLastAnnot);
                        fillerContent.mFieldFlags = flags;
                        fillerContent.mOptions = mCurOptions;
                        modifyAnnot(fillerContent, false, false, null);
                    }
                });

                mLvChoiceOptions.setOnItemClickListener(new AdapterView.OnItemClickListener() {
                    @Override
                    public void onItemClick(AdapterView<?> parent, View view, int position, long id) {
                        ChoiceOptionsAdapter.SelectMode selectMode = ChoiceOptionsAdapter.SelectMode.SINGLE_SELECT;
                        if (type == Field.e_TypeListBox && mPropCheckTrack.isSelected()) {
                            selectMode = ChoiceOptionsAdapter.SelectMode.MULTI_SELECT;
                        }

                        showOptionsDialog(annot, selectMode, mCurOptions, mPickOptionsCallback);
                    }
                });
                mLvChoiceOptions.setOnBlankClickListener(new FormOptionsListView.OnBlankClickListener() {
                    @Override
                    public void onBlankClicked() {
                        ChoiceOptionsAdapter.SelectMode selectMode = ChoiceOptionsAdapter.SelectMode.SINGLE_SELECT;
                        if (type == Field.e_TypeListBox && mPropCheckTrack.isSelected()) {
                            selectMode = ChoiceOptionsAdapter.SelectMode.MULTI_SELECT;
                        }

                        showOptionsDialog(annot, selectMode, mCurOptions, mPickOptionsCallback);
                    }
                });
                mLvChoiceOptions.setOnTouchListener(new View.OnTouchListener() {

                    @Override
                    public boolean onTouch(View v, MotionEvent event) {
                        mLvChoiceOptions.requestDisallowInterceptTouchEvent(mLvChoiceOptions.canScrollVertically(-1));
                        return false;
                    }
                });

                mTvAddChoiceOptions.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        ChoiceOptionsAdapter.SelectMode selectMode = ChoiceOptionsAdapter.SelectMode.SINGLE_SELECT;
                        if (type == Field.e_TypeListBox && mPropCheckTrack.isSelected()) {
                            selectMode = ChoiceOptionsAdapter.SelectMode.MULTI_SELECT;
                        }

                        showOptionsDialog(annot, selectMode, mCurOptions, mPickOptionsCallback);
                    }
                });
            } else {
                view.setAlpha(PropertyBar.PB_ALPHA);
            }

            mPropCheckTrack.setEnabled(canEdit);
            mPropCheckThumb.setEnabled(canEdit);
            mLvChoiceOptions.setEnabled(canEdit);
            mTvAddChoiceOptions.setEnabled(canEdit);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return view;
    }

    private final IResult<ArrayList<ChoiceItemInfo>, Object, Object> mPickOptionsCallback = new IResult<ArrayList<ChoiceItemInfo>, Object, Object>() {
        @Override
        public void onResult(boolean success, final ArrayList<ChoiceItemInfo> itemInfos, Object o1, Object o2) {
            if (success) {
                ArrayList<ChoiceItemInfo> lastInfos = mChoiceItemAdapter.getChoiceInfos();
                if (FormFillerUtil.optionsIsChanged(lastInfos, itemInfos)) {
                    mCurOptions = FormFillerUtil.cloneChoiceOptions(itemInfos);
                    mChoiceItemAdapter.setChoiceInfos(mCurOptions);

                    updateOptionsUI();
                    FormFillerContent fillerContent = new FormFillerContent(mPdfViewCtrl, (Widget) mLastAnnot);
                    fillerContent.mOptions = mCurOptions;
                    modifyAnnot(fillerContent, false, false, null);
                }
            } else {
                updateOptionsUI();
            }
        }
    };

    private void updateOptionsUI() {
        AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
            @Override
            public void run() {
                if (mCurOptions.size() > 0) {
                    mLvChoiceOptions.setVisibility(View.VISIBLE);
                    mTvAddChoiceOptions.setText("");
                } else {
                    mLvChoiceOptions.setVisibility(View.GONE);
                    mTvAddChoiceOptions.setText(AppResource.getString(mContext, R.string.fx_no_item_tips));
                }
                mChoiceItemAdapter.notifyDataSetChanged();

                if (AppDisplay.isPad()) {
                    if (mLastAnnot == null || mLastAnnot.isEmpty()) return;
                    try {
                        int pageIndex = mLastAnnot.getPage().getIndex();
                        RectF annotRectF = AppUtil.toRectF(mLastAnnot.getRect());
                        mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, pageIndex);
                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(annotRectF, annotRectF, pageIndex);

                        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                        RectF rectF = new RectF();
                        if (SystemUiHelper.getInstance().isStatusBarShown(uiExtensionsManager.getAttachedActivity())) {
                            rectF.set(AppUtil.toGlobalVisibleRectF(uiExtensionsManager.getRootView(), annotRectF));
                        } else {
                            rectF.set(annotRectF);
                        }
                        mPropertyBar.setDismissWithResetProperties(true);
                        mPropertyBar.show(rectF, false);
                    } catch (PDFException ignored) {
                    }
                }
            }
        });
    }

    private long getProperties(int type) {
        long properties;
        if (type == Field.e_TypeCheckBox) {
            properties = PropertyBar.PROPERTY_COLOR;
        } else if (type == Field.e_TypeTextField) {
            properties = PropertyBar.PROPERTY_COLOR | PropertyBar.PROPERTY_AUTO_FONTSIZE | PropertyBar.PROPERTY_FONTNAME;
        } else if (type == Field.e_TypeRadioButton) {
            properties = PropertyBar.PROPERTY_COLOR | PropertyBar.PROPERTY_RADIO_NAME;
        } else if (type == Field.e_TypeComboBox) {
            properties = PropertyBar.PROPERTY_COLOR | PropertyBar.PROPERTY_AUTO_FONTSIZE | PropertyBar.PROPERTY_FONTNAME;
        } else if (type == Field.e_TypeListBox) {
            properties = PropertyBar.PROPERTY_COLOR | PropertyBar.PROPERTY_AUTO_FONTSIZE | PropertyBar.PROPERTY_FONTNAME;
        } else {
            properties = PropertyBar.PROPERTY_COLOR;
        }
        return properties;
    }

    private void initFormNavigation() {
        mFNModule = (FormNavigationModule) ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getModuleByName(Module.MODULE_NAME_FORM_NAVIGATION);
        if (mFNModule != null) {
            mFNModule.initView();
            mFNModule.hide();
            mFNModule.getPreView().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    AppThreadManager.getInstance().startThread(preNavigation);
                }
            });

            mFNModule.getNextView().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    AppThreadManager.getInstance().startThread(nextNavigation);
                }
            });

            mFNModule.getClearView().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_FormBar_Reset);
                    final Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
                    if (annot instanceof Widget) {
                        resetField((Widget) annot, false, true);
                    }
                }
            });

            mFNModule.getFinishView().setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot() != null) {
                        if (shouldShowInputSoft(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot())) {
                            if (mBlink != null) {
                                mBlink.removeCallbacks(mBlink);
                                mBlink = null;
                            }
                            dismissSoftInput();
                            mParent.removeView(mEditView);
                        }
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null);
                    }
                    mFNModule.hide();
                    resetDocViewerOffset();
                }
            });
            mFNModule.setClearEnable(false);
        }
    }

    private void resetField(Widget annot, boolean clear, boolean needRefresh) {
        try {
            PDFViewCtrl.lock();
            Control formControl = annot.getControl();
            mAnnotIsSelected = false;
            mFormFiller.killFocus();
            Field field = formControl.getField();
            if (clear) {
                field.setValue("");
            } else {
                field.reset();
            }
            mAnnotIsSelected = true;
            mFormFiller.setFocus(formControl);
            mIsModify = true;

            if (isEditingText()) {
                if (AppUtil.isEmpty(field.getValue())) {
                    mShouldAddSuggestions = true;
                    mShouldInterceptTextChanged = true;
                    mEditView.setText("");
                }
            }

            if (mFNModule != null)
                mFNModule.hideSuggestionsbar();

            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setDocModified(true);
            if (needRefresh)
                refreshField(field);
        } catch (PDFException e) {
            e.printStackTrace();
        } finally {
            PDFViewCtrl.unlock();
        }
    }

    private void hideStatusBar(Activity activity) {
        SystemUiHelper.getInstance().hideStatusBar(activity);
    }

    protected boolean hasInitialized() {
        return bInitialize;
    }

    private long mDismissSoftTime;

    protected void showSoftInput() {
        AppThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
            @Override
            public void run() {
                mDismissSoftTime = 0;
                AppUtil.showSoftInput(mEditView);
            }
        }, System.currentTimeMillis() - mDismissSoftTime > 100 ? 0 : 300);
    }

    protected void dismissSoftInput() {
        mDismissSoftTime = System.currentTimeMillis();
        AppUtil.dismissInputSoft(mEditView);
    }

    private void postDismissNavigation() {
        DismissNavigation dn = new DismissNavigation();
        dn.postDelayed(dn, 500);
    }

    public void release() {
        if (mAssist != null) {
            mAssist.destroy();
        }
        AppUtil.removeViewFromParent(mEditView);
        mParent = null;
        mAnnotationMenu = null;
        mEditView = null;
        mPropertyBar = null;
        mFNModule = null;
        mFormFillerModule = null;
    }

    private class DismissNavigation extends Handler implements Runnable {

        @Override
        public void run() {
            if (mPdfViewCtrl == null || mPdfViewCtrl.getDoc() == null) return;
            Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
            if (!(annot instanceof Widget)) {
                if (mFNModule != null) {
                    mFNModule.hide();
                }
                resetDocViewerOffset();
            }
        }
    }

    private void modifyAnnot(IFormFillerContent formContent, final boolean isModifyJni, final boolean addUndo, final Event.Callback result) {
        try {
            final int pageIndex = formContent.getPageIndex();
            final Annot annot = formContent.getAnnot();

            if (isModifyJni) {
                final FormFillerModifyUndoItem undoItem = (FormFillerModifyUndoItem) createModifyUndoItem(formContent);
                final PDFPage page = annot.getPage();

                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setHasModifyTask(addUndo);
                FormFillerEvent event = new FormFillerEvent(EditAnnotEvent.EVENTTYPE_MODIFY, undoItem, (Widget) annot, mPdfViewCtrl);
                EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                    @Override
                    public void result(Event event, boolean success) {
                        if (success && !mAssist.bWillClose) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotModified(page, annot);

                            if (addUndo) {
                                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(undoItem);
                                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setHasModifyTask(false);
                            }

                            RectF tempRectF = mTempLastBBox;
                            if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                try {
                                    RectF annotRectF = AppUtil.toRectF(annot.getRect());
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, pageIndex);
                                    mPdfViewCtrl.convertPdfRectToPageViewRect(tempRectF, tempRectF, pageIndex);
                                    annotRectF.union(tempRectF);
                                    annotRectF.inset(-AppAnnotUtil.getAnnotBBoxSpace() - 10, -AppAnnotUtil.getAnnotBBoxSpace() - 10);
                                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(annotRectF));
                                } catch (PDFException e) {
                                    e.printStackTrace();
                                }
                            }
                        }

                        if (result != null) {
                            result.result(null, success);
                        }
                    }
                });
                mPdfViewCtrl.addTask(task);
            }

            mIsModify = true;
            if (!isModifyJni) {
                boolean isModify = false;

                RectF oldRect = AppUtil.toRectF(annot.getRect());
                Field field = ((Widget) annot).getField();
                int fieldType = field.getType();
                DefaultAppearance da;
                if (field.getType() == Field.e_TypeRadioButton)
                    da = ((Widget) annot).getControl().getDefaultAppearance();
                else
                    da = field.getDefaultAppearance();

                int flags = DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagTextColor | DefaultAppearance.e_FlagFontSize;
                da.setFlags(flags);
                if (da.getText_color() != formContent.getFontColor()) {
                    isModify = true;
                    da.setText_color(formContent.getFontColor());
                }

                if (formContent.getFontId() != mTextUtil.getSupportFontID(da, mPdfViewCtrl.getDoc())) {
                    isModify = true;
                    Font font = mTextUtil.getStandardFont(formContent.getFontId());
                    da.setFont(font);
                }
                if (formContent.getFontSize() != da.getText_size()) {
                    isModify = true;
                    da.setText_size(formContent.getFontSize());
                }

                if (fieldType == Field.e_TypeRadioButton) {
                    if (isModify)
                        ((Widget) annot).getControl().setDefaultAppearance(da);

                    String name = field.getName();
                    if (!name.equals(formContent.getFieldName()) && mForm != null) {
                        isModify = true;
                        FormFillerUtil.renameField(mForm, ((Widget) annot).getControl(), formContent.getFieldName());
                        field = ((Widget) annot).getField();
                    }

                    int checkedIndex = FormFillerUtil.getCheckedIndex(field);
                    if ((checkedIndex != formContent.getCheckedIndex())) {
                        isModify = true;
                        field.getControl(formContent.getCheckedIndex()).setChecked(true);
                    }
                } else if (fieldType == Field.e_TypeListBox || fieldType == Field.e_TypeComboBox) {
                    if (isModify)
                        field.setDefaultAppearance(da);

                    if (formContent.getFieldFlags() != field.getFlags()) {
                        isModify = true;
                        field.setFlags(formContent.getFieldFlags());
                    }
                    ArrayList<ChoiceItemInfo> choiceOptions = FormFillerUtil.getOptions(field);
                    if (FormFillerUtil.optionsIsChanged(formContent.getOptions(), choiceOptions)) {
                        isModify = true;
                        field.setOptions(FormFillerUtil.options2Native(formContent.getOptions()));
                    }
                } else {
                    if (isModify) {
                        field.setDefaultAppearance(da);
                    }
                }

                if (isModify) {
                    annot.resetAppearanceStream();
                }

                if (!AppUtil.toRectF(annot.getRect()).equals(formContent.getBBox())) {
                    isModify = true;
                    annot.move(AppUtil.toFxRectF(formContent.getBBox()));
                    if (Field.e_TypeSignature != fieldType) {
                        annot.resetAppearanceStream(fieldType == Field.e_TypeRadioButton);
                    }
                }
                if (isModify)
                    annot.setModifiedDateTime(AppDmUtil.currentDateToDocumentDate());

                RectF annotRectF = AppUtil.toRectF(annot.getRect());
                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                    float thickness = thicknessOnPageView(pageIndex, annot.getBorderInfo().getWidth());

                    mPdfViewCtrl.convertPdfRectToPageViewRect(oldRect, oldRect, pageIndex);
                    mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, pageIndex);
                    annotRectF.union(oldRect);
                    annotRectF.inset(-thickness - mCtlPtRadius - mCtlPtDeltyXY, -thickness - mCtlPtRadius - mCtlPtDeltyXY);
                    mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(annotRectF));
                }
            }
        } catch (PDFException e) {
            if (e.getLastError() == Constants.e_ErrOutOfMemory) {
                mPdfViewCtrl.recoverForOOM();
            }
        }
    }

    private void modifyImageField(boolean clearImage, String path, final Annot annot) {
        final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        try {
            final PDFPage page = annot.getPage();
            final int pageIndex = page.getIndex();
            final FormFillerModifyUndoItem undoItem = new FormFillerModifyUndoItem(mPdfViewCtrl);
            undoItem.setCurrentValue(annot);

            undoItem.mClearImage = clearImage;
            undoItem.mImagePath = path;
            undoItem.mPageIndex = pageIndex;

//            undoItem.mBBox = AppUtil.toRectF(annot.getRect());
//            undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
//
//            undoItem.mRedoBbox = AppUtil.toRectF(annot.getRect());
//            undoItem.mUndoBbox = new RectF(mTempLastBBox);
//            undoItem.mUndoPDFDict = AppAnnotUtil.clonePDFDict(annot.getDict());

            uiExtensionsManager.getDocumentManager().setHasModifyTask(true);
            FormFillerEvent event = new FormFillerEvent(EditAnnotEvent.EVENTTYPE_MODIFY, undoItem, (Widget) annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
//                        try {
//                            undoItem.mRedoPDFDict = AppAnnotUtil.clonePDFDict(annot.getDict());
//                            uiExtensionsManager.getDocumentManager().addUndoItem(undoItem);

                        uiExtensionsManager.getDocumentManager().onAnnotModified(page, annot);
                        uiExtensionsManager.getDocumentManager().setHasModifyTask(false);
                        uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                        uiExtensionsManager.setSaveOptions(new UIExtensionsManager.SaveOptions(
                                PDFDoc.e_SaveFlagRemoveRedundantObjects));

                        Task refreshTask = new Task(null) {
                            @Override
                            protected void execute() {
                                if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                                    try {
                                        RectF annotRectF = AppUtil.toRectF(annot.getRect());
                                        mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, pageIndex);
                                        annotRectF.inset(-AppAnnotUtil.getAnnotBBoxSpace() - 10, -AppAnnotUtil.getAnnotBBoxSpace() - 10);
                                        mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(annotRectF));
                                    } catch (PDFException e) {
                                        e.printStackTrace();
                                    }
                                }
                            }
                        };
                        mPdfViewCtrl.addTask(refreshTask);
//                        } catch (PDFException e) {
//                            e.printStackTrace();
//                        }
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void deleteAnnot(final Annot annot, final boolean addUndo, final Event.Callback result) {
        try {
            final RectF viewRect = AppUtil.toRectF(annot.getRect());
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            final DocumentManager documentManager = uiExtensionsManager.getDocumentManager();
            if (documentManager.getCurrentAnnot() != null && AppAnnotUtil.isSameAnnot(annot, documentManager.getCurrentAnnot())) {
                documentManager.setCurrentAnnot(null, false);
            }

            final PDFPage page = annot.getPage();
            Field field = ((Widget) annot).getField();
            final int pageIndex = page.getIndex();
            int fieldType = FormFillerUtil.getAnnotFieldType(annot);

            final FormFillerDeleteUndoItem undoItem = new FormFillerDeleteUndoItem(mPdfViewCtrl);
            undoItem.setCurrentValue(annot);
            undoItem.mPageIndex = pageIndex;
            undoItem.mFieldType = fieldType;
            if (fieldType == Field.e_TypeCheckBox && field.getControlCount() > 0)
                undoItem.mFieldName = AppDmUtil.randomUUID(null);
            else
                undoItem.mFieldName = field.getName();
            undoItem.mFieldFlags = field.getFlags();
            undoItem.mOptions = FormFillerUtil.getOptions(field);
            undoItem.mValue = field.getValue();
            undoItem.mRotation = ((Widget) annot).getMKRotation();
            if (fieldType == Field.e_TypePushButton) {
                Action action = ((Widget) annot).getAction();
                if (!action.isEmpty() && action.getType() == Action.e_TypeJavaScript) {
                    JavaScriptAction javaScriptAction = new JavaScriptAction(action);
                    if (!javaScriptAction.isEmpty()) {
                        String script = javaScriptAction.getScript();
                        if (script.contains("buttonImportIcon")) {
//                            if (((Widget) annot).getMKNormalIconBitmap() != null) { // SDKRD-9718
//                                undoItem.mPDFDict = AppAnnotUtil.clonePDFDict(annot.getDict());
//                            }
                            uiExtensionsManager.setSaveOptions(new UIExtensionsManager.SaveOptions(
                                    PDFDoc.e_SaveFlagRemoveRedundantObjects));
                        }
                    }
                }
            }
            DefaultAppearance da;
            if (fieldType == Field.e_TypeRadioButton)
                da = ((Widget) annot).getControl().getDefaultAppearance();
            else
                da = field.getDefaultAppearance();
            undoItem.mFontId = mTextUtil.getSupportFontID(da, mPdfViewCtrl.getDoc());
            undoItem.mFontColor = da.getText_color();
            undoItem.mFontSize = da.getText_size();
            undoItem.mIsChecked = ((Widget) annot).getControl().isChecked();

            documentManager.onAnnotWillDelete(page, annot);
            FormFillerEvent event = new FormFillerEvent(EditAnnotEvent.EVENTTYPE_DELETE, undoItem, (Widget) annot, mPdfViewCtrl);
            EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (success) {
                        documentManager.onAnnotDeleted(page, annot);
                        if (addUndo) {
                            documentManager.addUndoItem(undoItem);
                        }

                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                            mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(viewRect));
                        }
                    }

                    if (result != null) {
                        result.result(null, success);
                    }
                }
            });
            mPdfViewCtrl.addTask(task);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }


    private boolean shouldShowNavigation(Annot annot) {
        if (annot == null) return false;
        if (!(annot instanceof Widget)) return false;
        return FormFillerUtil.getAnnotFieldType(annot) != Field.e_TypePushButton;
    }

    public void navigationDismiss() {
        if (mFNModule != null) {
            mFNModule.hideFormNavigationBar();
        }
        if (mBlink != null) {
            mBlink.removeCallbacks(mBlink);
            mBlink = null;
        }
        if (mEditView != null) {
            mParent.removeView(mEditView);
        }
        resetDocViewerOffset();
        dismissSoftInput();
    }

    private boolean isFind = false;
    private boolean isDocFinish = false;
    private PDFPage curPage = null;
    private int prePageIdx;
    private int preAnnotIdx;
    private int nextPageIdx;
    private int nextAnnotIdx;
    private CountDownLatch mCountDownLatch;
    private final Runnable preNavigation = new Runnable() {

        @Override
        public void run() {

            Annot curAnnot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
            try {
                if (curAnnot instanceof Widget) {
                    curPage = curAnnot.getPage();
                    final int curPageIdx = curPage.getIndex();
                    prePageIdx = curPageIdx;
                    final int curAnnotIdx = curAnnot.getIndex();
                    preAnnotIdx = curAnnotIdx;
                    isFind = false;
                    isDocFinish = false;
                    while (prePageIdx >= 0) {
                        mCountDownLatch = new CountDownLatch(1);
                        curPage = mPdfViewCtrl.getDoc().getPage(prePageIdx);
                        if (prePageIdx == curPageIdx && !isDocFinish) {
                            preAnnotIdx = curAnnotIdx - 1;
                        } else {
                            preAnnotIdx = curPage.getAnnotCount() - 1;
                        }

                        while (curPage != null && preAnnotIdx >= 0) {
                            final Annot preAnnot = AppAnnotUtil.createAnnot(curPage.getAnnot(preAnnotIdx));
                            final int preAnnotType = FormFillerUtil.getAnnotFieldType(preAnnot);
                            if ((preAnnot instanceof Widget)
                                    && (!FormFillerUtil.isReadOnly(preAnnot))
                                    && FormFillerUtil.isVisible(preAnnot)
                                    && (preAnnotType != Field.e_TypePushButton)
                                    && (preAnnotType != Field.e_TypeSignature)) {
                                isFind = true;
                                AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {

                                    @Override
                                    public void run() {
                                        try {
                                            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
                                            if (preAnnotType == Field.e_TypeComboBox) {
                                                RectF rect = AppUtil.toRectF(preAnnot.getRect());
                                                rect.left += 5;
                                                rect.top -= 5;
                                                mLastTouchPoint.set(rect.left, rect.top);
                                            }
                                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null);
                                            if (!preAnnot.isEmpty()) {
                                                if (uiExtensionsManager.getCurrentToolHandler() != null) {
                                                    uiExtensionsManager.setCurrentToolHandler(null);
                                                }
                                                RectF bbox = AppUtil.toRectF(preAnnot.getRect());
                                                RectF rect = new RectF(bbox);

                                                if (mPdfViewCtrl.convertPdfRectToPageViewRect(rect, rect, prePageIdx)) {
                                                    float devX = rect.left - (mPdfViewCtrl.getWidth() - rect.width()) / 2;
                                                    float devY = rect.top - (mPdfViewCtrl.getHeight() - rect.height()) / 2;
                                                    mPdfViewCtrl.gotoPage(prePageIdx, devX, devY);
                                                } else {
                                                    mPdfViewCtrl.gotoPage(prePageIdx, new PointF(bbox.left, bbox.top));
                                                }

                                                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(preAnnot);
                                                mFormFiller.setFocus(((Widget) preAnnot).getControl());
                                                if (mFNModule != null) {
                                                    mFNModule.hideSuggestionsbar();
                                                }
                                            }
                                        } catch (PDFException e) {
                                            e.printStackTrace();
                                        }
                                    }
                                });

                                break;
                            } else {
                                preAnnotIdx--;
                            }
                        }
                        mCountDownLatch.countDown();

                        try {
                            if (mCountDownLatch.getCount() > 0)
                                mCountDownLatch.await();
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                        }

                        if (isFind) break;
                        prePageIdx--;
                        if (prePageIdx < 0) {
                            prePageIdx = mPdfViewCtrl.getDoc().getPageCount() - 1;
                            isDocFinish = true;
                        }
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    };

    private void refreshField(Field field) {
        int[] visiblePages = mPdfViewCtrl.getVisiblePages();
        for (int pageIndex : visiblePages) {
            RectF rectF = getRefreshRect(field, pageIndex);
            if (rectF == null) continue;
            mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
            mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(rectF));
        }
    }

    private RectF getRefreshRect(Field field, int pageIndex) {
        RectF rectF = null;
        try {
            PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            int nControlCount = field.getControlCount(page);
            for (int i = 0; i < nControlCount; i++) {
                Control formControl = field.getControl(page, i);
                RectF annotRect = AppUtil.toRectF(formControl.getWidget().getRect());
                if (rectF == null) {
                    rectF = annotRect;
                } else {
                    AppUtil.unionFxRectF(rectF, annotRect);
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return rectF;
    }

    private final Runnable nextNavigation = new Runnable() {

        @Override
        public void run() {
            Annot curAnnot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
            try {
                if (curAnnot instanceof Widget) {
                    curPage = curAnnot.getPage();

                    final int curPageIdx = curPage.getIndex();
                    nextPageIdx = curPageIdx;
                    final int curAnnotIdx = curAnnot.getIndex();
                    nextAnnotIdx = curAnnotIdx;
                    isFind = false;
                    isDocFinish = false;

                    while (nextPageIdx < mPdfViewCtrl.getDoc().getPageCount()) {

                        mCountDownLatch = new CountDownLatch(1);
                        curPage = mPdfViewCtrl.getDoc().getPage(nextPageIdx);
                        if (nextPageIdx == curPageIdx && !isDocFinish) {
                            nextAnnotIdx = curAnnotIdx + 1;
                        } else {
                            nextAnnotIdx = 0;
                        }

                        while (curPage != null && nextAnnotIdx < curPage.getAnnotCount()) {
                            final Annot nextAnnot = AppAnnotUtil.createAnnot(curPage.getAnnot(nextAnnotIdx));
                            final int nextAnnotType = FormFillerUtil.getAnnotFieldType(nextAnnot);
                            if (nextAnnot instanceof Widget
                                    && !FormFillerUtil.isReadOnly(nextAnnot)
                                    && FormFillerUtil.isVisible(nextAnnot)
                                    && nextAnnotType != Field.e_TypePushButton
                                    && nextAnnotType != Field.e_TypeSignature) {
                                isFind = true;

                                AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {

                                    @Override
                                    public void run() {
                                        try {
                                            UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
                                            if (nextAnnotType == Field.e_TypeComboBox) {
                                                RectF rect = AppUtil.toRectF(nextAnnot.getRect());
                                                rect.left += 5;
                                                rect.top -= 5;
                                                mLastTouchPoint.set(rect.left, rect.top);
                                            }
                                            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                                            if (!nextAnnot.isEmpty()) {
                                                if (uiExtensionsManager.getCurrentToolHandler() != null) {
                                                    uiExtensionsManager.setCurrentToolHandler(null);
                                                }
                                                RectF bbox = AppUtil.toRectF(nextAnnot.getRect());
                                                RectF rect = new RectF(bbox);

                                                if (mPdfViewCtrl.convertPdfRectToPageViewRect(rect, rect, nextPageIdx)) {
                                                    float devX = rect.left - (mPdfViewCtrl.getWidth() - rect.width()) / 2;
                                                    float devY = rect.top - (mPdfViewCtrl.getHeight() - rect.height()) / 2;
                                                    mPdfViewCtrl.gotoPage(nextPageIdx, devX, devY);
                                                } else {
                                                    mPdfViewCtrl.gotoPage(nextPageIdx, new PointF(bbox.left, bbox.top));
                                                }

                                                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(nextAnnot);
                                                mFormFiller.setFocus(((Widget) nextAnnot).getControl());
                                                if (mFNModule != null) {
                                                    mFNModule.hideSuggestionsbar();
                                                }
                                            }
                                        } catch (PDFException e) {
                                            e.printStackTrace();
                                        }
                                    }

                                });

                                break;
                            } else {
                                nextAnnotIdx++;
                            }
                        }
                        mCountDownLatch.countDown();

                        try {
                            if (mCountDownLatch.getCount() > 0)
                                mCountDownLatch.await();
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                        }

                        if (isFind) break;
                        nextPageIdx++;
                        if (nextPageIdx >= mPdfViewCtrl.getDoc().getPageCount()) {
                            nextPageIdx = 0;
                            isDocFinish = true;
                        }
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
        }
    };

    void stopSave() {
        if (mAssist != null) {
            mAssist.bWillSave = false;
        }
    }

    void startSave() {
        if (mAssist != null) {
            mAssist.bWillSave = true;
        }
    }

    protected void clear() {
        bInitialize = false;

        if (mAssist != null) {
            mAssist.bWillClose = true;
            mAssist.reset();
        }

        if (mForm != null) {
            mForm.delete();
            mForm = null;
        }

        if (mFormFiller != null) {
            mFormFiller.delete();
            mFormFiller = null;
        }
    }

    Form getForm() {
        return mForm;
    }

    FormFillerAssistImpl getFormFillerAssist() {
        return mAssist;
    }

    @Override
    public int getType() {
        return Annot.e_Widget;
    }


    @Override
    public boolean annotCanAnswer(Annot annot) {
        return true;
    }

    @Override
    public RectF getAnnotBBox(Annot annot) {

        try {
            return AppUtil.toRectF(annot.getRect());
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public boolean isHitAnnot(Annot annot, PointF point) {
        try {
            RectF r = AppUtil.toRectF(annot.getRect());
            RectF rf = new RectF(r.left, r.top, r.right, r.bottom);
            PointF p = new PointF(point.x, point.y);
            int pageIndex = annot.getPage().getIndex();
            Control control = AppAnnotUtil.getControlAtPos(annot.getPage(), p, 1);

            mPdfViewCtrl.convertPdfRectToPageViewRect(rf, rf, pageIndex);
            mPdfViewCtrl.convertPdfPtToPageViewPt(p, p, pageIndex);

            if (rf.contains(p.x, p.y)) {
                return true;
            } else {
                if (AppAnnotUtil.isSameAnnot(annot, control != null ? control.getWidget() : null))
                    return true;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return false;
    }


    public void onBackspaceBtnDown() {
        try {
            mFormFiller.onChar((char) 8, 0);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private String mTempValue;
    private String mTempFieldName;
    private int mTempFieldFlags;
    private ArrayList<ChoiceItemInfo> mTempOptions = new ArrayList<>();

    private RectF mTempLastBBox = new RectF();
    private int mTempLastFontColor;
    private int mTempLastFontID;
    private float mTempLastFontSize;
    private int mTempLastCheckedIndex;

    private RectF mTempLastDisplayBBox = new RectF();
    private RectF mPageViewRect = new RectF(0, 0, 0, 0);

    @Override
    public void onAnnotSelected(final Annot annot, boolean needInvalid) {
        mAnnotIsSelected = true;
        try {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();

            RectF rectF = AppUtil.toRectF(annot.getRect());
            mLastPageIndex = annot.getPage().getIndex();
            mTempLastBBox = new RectF(rectF);

            RectF pageViewRect = new RectF(mTempLastBBox);
            mPdfViewCtrl.convertPdfRectToPageViewRect(pageViewRect, pageViewRect, mLastPageIndex);
            mTempLastDisplayBBox = new RectF(pageViewRect);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mTempLastDisplayBBox, mTempLastDisplayBBox, mLastPageIndex);

            Field field = ((Widget) annot).getField();
            int fieldType = FormFillerUtil.getAnnotFieldType(annot);

            DefaultAppearance da;
            if (fieldType == Field.e_TypeComboBox || fieldType == Field.e_TypeListBox) {
                mTempFieldFlags = field.getFlags();
                mTempOptions = FormFillerUtil.getOptions(field);
                da = field.getDefaultAppearance();
            } else if (fieldType == Field.e_TypeRadioButton) {
                mTempLastCheckedIndex = FormFillerUtil.getCheckedIndex(field);
                da = ((Widget) annot).getControl().getDefaultAppearance();
            } else if (fieldType == Field.e_TypeTextField) {
                mbSupportMultiline = (field.getFlags() & Field.e_FlagTextMultiline) == Field.e_FlagTextMultiline;
                da = field.getDefaultAppearance();
            } else {
                da = field.getDefaultAppearance();
            }

            mTempLastFontColor = da.getText_color();
            mTempLastFontID = mTextUtil.getSupportFontID(da, mPdfViewCtrl.getDoc());
            mTempLastFontSize = da.getText_size();
            mTempFieldName = field.getName();
            mTempValue = field.getValue();

            mPageViewRect.set(rectF);
            mPdfViewCtrl.convertPdfRectToPageViewRect(mPageViewRect, mPageViewRect, mLastPageIndex);
            if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
                if (annot == uiExtensionsManager.getDocumentManager().getCurrentAnnot()) {
                    mLastAnnot = annot;
                }
            } else {
                mLastAnnot = annot;
            }

            if (mIsLongPressTouchEvent) {
                onAnnotSeletedByLongPress(annot, needInvalid);
            } else {
                onAnnotSeletedBySingleTap(annot, needInvalid);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void onAnnotSeletedBySingleTap(final Annot annot, boolean needInvalid) {
        final UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();

        try {
            int fieldType = FormFillerUtil.getAnnotFieldType(annot);
            if (fieldType == Field.e_TypePushButton) {
                Action action = ((Widget) annot).getAction();
                if (!action.isEmpty()) {
                    if (action.getType() == Action.e_TypeJavaScript) {
                        JavaScriptAction javaScriptAction = new JavaScriptAction(action);
                        if (!javaScriptAction.isEmpty()) {
                            String script = javaScriptAction.getScript();
                            if (script.contains("buttonImportIcon")) {
                                if (uiExtensionsManager.getDocumentManager().getActionCallback() instanceof AnnotActionHandler) {
                                    AnnotActionHandler actionHandler = (AnnotActionHandler) uiExtensionsManager.getDocumentManager().getActionCallback();
                                    mJSActionCallback = new JSActionCallback(mLastPageIndex, (Widget) annot);
                                    actionHandler.setJSActionCallback(mJSActionCallback);
                                }
                            }
                        }
                    }
                } else {
                    AdditionalAction additionalAction = new AdditionalAction(annot);
                    action = additionalAction.getAction(AdditionalAction.e_TriggerAnnotMouseButtonPressed);
                    if (!action.isEmpty() && action.getType() == Action.e_TypeSound) {
                        PDFDictionary dict = action.getDict();
                        if (dict != null) {
                            PDFObject soundObj = dict.getElement("Sound");
                            if (soundObj != null) {
                                openSound(annot, soundObj);
                            }
                        }
                    }
                }
            } else {
                if (mFNModule != null) {
                    mFNModule.setClearEnable(!FormFillerUtil.isReadOnly(annot));
                    mFNModule.show();

                    mFNModule.setSuggestionApplyCallback(new FormNavigationModule.ISuggestionApplyCallback() {
                        @Override
                        public void onSuggestionApply(String content) {
                            if (isEditingText()) {
                                try {
                                    resetField((Widget) mLastAnnot, true, false);
                                    char[] chars = content.toCharArray();
                                    for (char c : chars) {
                                        if (c == '\n' && mbSupportMultiline)
                                            mFormFiller.onChar('\r', 0);
                                        mFormFiller.onChar(c, 0);
                                    }

                                    mShouldInterceptTextChanged = true;
                                    mEditView.setText(content);
                                    mEditView.setSelection(content.length());
                                } catch (Exception ignored) {
                                }
                            }
                        }
                    });
                }

                if (fieldType == Field.e_TypeTextField) {
                    Field textField = ((Widget) annot).getField();
                    if (textField.isEmpty()) return;

                    AdditionalAction aa = new AdditionalAction(textField);
                    if (!aa.isEmpty()) {
                        Action action = aa.getAction(AdditionalAction.e_TriggerFieldWillFormat);
                        if (!action.isEmpty() && action.getType() == Action.e_TypeJavaScript) {
                            JavaScriptAction javaScriptAction = new JavaScriptAction(action);
                            String format = javaScriptAction.getScript();
                            if (format != null && format.contains("AFDate_FormatEx")) {
                                mDateFieldIcon = BitmapFactory.decodeResource(mContext.getResources(), R.drawable.form_date_field_icon);
                                mIsDateField = true;

                                int beginIndex = format.indexOf("\"");
                                int endIndex = format.lastIndexOf("\"");
                                if (beginIndex > 0 && endIndex > beginIndex) {
                                    mDatePattern = format.substring(beginIndex + 1, endIndex);
                                    mDatePattern = FormFillerUtil.toJavaDatePattern(mDatePattern);

                                    try {
                                        DateFormat dateFormat = new SimpleDateFormat(mDatePattern, Locale.ENGLISH);
                                        if (!AppUtil.isEmpty(textField.getValue()))
                                            mDateFieldValue = dateFormat.parse(textField.getValue());
                                    } catch (ParseException e) {
                                        e.printStackTrace();
                                    }
                                }
                                showDatePickerDialog();
                            }
                        }
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }

        if (shouldShowInputSoft(annot)) {
            try {
                mShouldAddSuggestions = AppUtil.isEmpty(((Widget) annot).getField().getValue());
            } catch (PDFException e) {
                e.printStackTrace();
            }

            SystemUiHelper.getInstance().showNavigationBar(uiExtensionsManager.getAttachedActivity());

            mIsShowEditText = true;
            mAdjustPosition = true;

            if (mEditView != null) {
                AppUtil.removeViewFromParent(mEditView);
            }
            mEditView = new EditText(mContext);
            if (AppDisplay.isPad()) { // SDKRD-9313
                mEditView.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
            }
            mEditView.setText("");

            Rect rect = new Rect();
            mParent.getGlobalVisibleRect(rect);
            mEditView.setLayoutParams(new ViewGroup.LayoutParams(1, 1));
            if (AppDisplay.isPad() && rect.top != 0 && rect.top != AppDisplay.getStatusBarHeight()) {
                mParent.addView(mEditView, 0);
                mIsEdittextOffset = true;
            } else {
                mEditView.setSingleLine(false);
                mParent.addView(mEditView);
                mIsEdittextOffset = false;
            }

            mEditView.setOnKeyListener(new OnKeyListener() {
                @Override
                public boolean onKey(View v, int keyCode, KeyEvent event) {
                    if (keyCode == KeyEvent.KEYCODE_DEL && event.getAction() == KeyEvent.ACTION_DOWN) {
                        if (AppUtil.isEmpty(mEditView.getText().toString())) {
                            onBackspaceBtnDown();
//                            mIsBackBtnPush = true;
                        }
                    }
                    return false;
                }
            });

            mEditView.addTextChangedListener(mTextWatcher);

            if (!mIsDateField && !mIsTouchComboPopup)
                showSoftInput();

            if (mBlink == null) {
                mBlink = new Blink(annot);
                mBlink.postDelayed(mBlink, 300);
            } else {
                mBlink.setAnnot(annot);
            }

            ViewGroup parent = uiExtensionsManager.getRootView();
            AppKeyboardUtil.setKeyboardListener(parent, parent, new AppKeyboardUtil.IKeyboardListener() {
                @Override
                public void onKeyboardOpened(int keyboardHeight) {
                    if (SystemUiHelper.getInstance().isFullScreen())
                        hideStatusBar(uiExtensionsManager.getAttachedActivity());
                    setEdittextLocation(mLastAnnot);
                    if (mFNModule != null)
                        mFNModule.showFormNavigationBarWithKeyBoard(getFNBottomPadding());

                    try {
                        if (mLastAnnot != null) {
                            int pageIndex = mLastAnnot.getPage().getIndex();
                            RectF annotRectF = AppUtil.toRectF(mLastAnnot.getRect());
                            RectF displayRectF = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, displayRectF, pageIndex);
                            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(displayRectF, displayRectF, pageIndex);
                            mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(displayRectF));
                        }
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }

                @Override
                public void onKeyboardClosed() {
                    if (uiExtensionsManager.getCurrentAnnotHandler() != FormFillerAnnotHandler.this) {
                        AppKeyboardUtil.removeKeyboardListener(uiExtensionsManager.getRootView());
                        if (mFNModule != null)
                            mFNModule.hideFormNavigationBar();
                    } else {
                        if (mFNModule != null)
                            mFNModule.showFormNavigationBar();
                    }

                    if (AppDisplay.isPad() && !mPdfViewCtrl.isContinuous())
                        mPdfViewCtrl.setTranslationY(0);

                    if (SystemUiHelper.getInstance().isFullScreen())
                        hideStatusBar(uiExtensionsManager.getAttachedActivity());
                }
            });
        }
    }

    private final TextWatcher mTextWatcher = new TextWatcher() {

        @Override
        public void beforeTextChanged(CharSequence s, int start, int count, int after) {
            if (mShouldInterceptTextChanged) return;

            if (count > 0) {
                mBeforeText = s.subSequence(start, start + count).toString();
            } else {
                mBeforeText = "";
            }
        }

        @Override
        public void onTextChanged(CharSequence s, int start, int before, int count) {
            if (mShouldInterceptTextChanged) return;

            try {
                String charInput = "";

                if (mBeforeText.length() > 0
                        && count >= before
                        && s.subSequence(start, start + before).toString().equals(mBeforeText)) {
                    charInput = s.subSequence(start + before, start + count).toString();
                } else {
                    for (int i = 0; i < mBeforeText.length(); i++) {
                        charInput += (char) 8;
                    }

                    if (count > 0) {
                        String afterTxt = s.subSequence(start, start + count).toString();
                        charInput += afterTxt;
                    }
                }

                if (charInput.length() > 0) {
                    for (int i = 0; i < charInput.length(); i++) {
                        char c = charInput.charAt(i);
                        if (FormFillerUtil.isEmojiCharacter(c))
                            break;

                        if (c == '\n' && mbSupportMultiline)
                            mFormFiller.onChar('\r', 0);
                        mFormFiller.onChar(c, 0);
                    }
                }
            } catch (PDFException ignored) {
            }
        }

        @Override
        public void afterTextChanged(Editable s) {
            if (mShouldInterceptTextChanged) {
                mShouldInterceptTextChanged = false;
                return;
            }

            if (mFNModule != null) {
                mFNModule.updateSuggestionsbar(s.toString());
            }
        }
    };


    private void onAnnotSeletedByLongPress(final Annot annot, boolean needInvalid) {
        mCtlPtRadius = AppDisplay.dp2px(mCtlPtRadius);
        mCtlPtDeltyXY = AppDisplay.dp2px(mCtlPtDeltyXY);
        try {
            int pageIndex = annot.getPage().getIndex();
            prepareAnnotMenu(annot);
            RectF menuRect = new RectF(mPageViewRect);
            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(menuRect, menuRect, pageIndex);
            mAnnotationMenu.show(menuRect);
            preparePropertyBar(annot);

            if (mPdfViewCtrl.isPageVisible(pageIndex))
                mPdfViewCtrl.refresh(pageIndex, AppDmUtil.rectFToRect(mPageViewRect));
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private AnnotMenu.ClickListener mMenuClickListener;

    private void prepareAnnotMenu(final Annot annot) {
        mMenuText.clear();
        String uid = AppAnnotUtil.getAnnotUniqueID(annot);
        if (!TextUtils.isEmpty(uid) && uid.contains(FormFillerModule.ID_TAG)) {
            DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
            int fieldType = FormFillerUtil.getAnnotFieldType(annot);
            if (fieldType == Field.e_TypeCheckBox
                    || fieldType == Field.e_TypeTextField
                    || fieldType == Field.e_TypeRadioButton
                    || fieldType == Field.e_TypeListBox
                    || fieldType == Field.e_TypeComboBox) {
                mMenuText.add(AnnotMenu.AM_BT_STYLE);
            }

            if (documentManager.canAddAnnot()
                    && documentManager.canModifyForm()
                    && ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()
                    && !(AppAnnotUtil.isLocked(annot) || FormFillerUtil.isReadOnly(annot))) {
                if (AnnotPermissionUtil.canDeleteAnnot(documentManager, annot))
                    mMenuText.add(AnnotMenu.AM_BT_DELETE);
            }

            mAnnotationMenu.setMenuItems(mMenuText);
            initAnnotationMenuListener();
        }
    }

    private void prepareImageMenu(Widget widget) {
        mMenuText.clear();
        mMenuText.add(AnnotMenu.AM_BT_FILE);
        mMenuText.add(AnnotMenu.AM_BT_GALERY);
        if (AppDevice.hasCamera(((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity())) {
            mMenuText.add(AnnotMenu.AM_BT_CAMERA);
        }
        try {
            if (widget.hasMKEntry(Annot.e_MKEntryNormalIcon)) {
                mMenuText.add(AnnotMenu.AM_BT_CLEAR);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        mAnnotationMenu.setMenuItems(mMenuText);
        initAnnotationMenuListener();
    }

    @Override
    public void onAnnotDeselected(final Annot annot, boolean needInvalid) {
        if (mDateFieldIcon != null && !mDateFieldIcon.isRecycled()) {
            mDateFieldIcon.recycle();
            mDateFieldIcon = null;
        }
        mIsDateField = false;
        mDatePattern = "";
        mDateFieldValue = null;

        mIsTouchComboPopup = false;
        mShouldInterceptTextChanged = false;
        mbSupportMultiline = false;
        mAnnotIsSelected = false;
        mIsLongPressTouchEvent = false;
        mLastPopupPdfPoint = null;
        mCtlPtRadius = 5;
        mCtlPtDeltyXY = 20;
        mPropertyBar.setSubTitleVisible(true);
        try {
            if (this.isLButtonDownHandled) {
                mFormFiller.killFocus();
            }
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            SystemUiHelper.getInstance().setAllowedHideSystemUI(uiExtensionsManager.getAttachedActivity(), true);

            postDismissNavigation();
            if (mBlink != null) {
                mBlink.removeCallbacks(mBlink);
                mBlink = null;
            }
            if (mFNModule != null) {
                mFNModule.setSuggestionApplyCallback(null);
                mFNModule.hideFormNavigationBar();
            }
            if (mJSActionCallback != null) {
                mJSActionCallback = null;
                if (uiExtensionsManager.getDocumentManager().getActionCallback() instanceof AnnotActionHandler) {
                    AnnotActionHandler actionHandler = (AnnotActionHandler) uiExtensionsManager.getDocumentManager().getActionCallback();
                    actionHandler.setJSActionCallback(null);
                }
            }

            Field field = ((Widget) annot).getField();
            DefaultAppearance da;
            if (field.getType() == Field.e_TypeRadioButton)
                da = ((Widget) annot).getControl().getDefaultAppearance();
            else
                da = field.getDefaultAppearance();

            if (needInvalid && (mIsModify || !mTempValue.equals(field.getValue()))) {
                int pageIndex = annot.getPage().getIndex();
                RectF rectF = AppUtil.toRectF(annot.getRect());
                String value = field.getValue();
                String lastFieldName = mLastFieldName.trim();

                if (lastFieldName.startsWith(".")
                        || lastFieldName.endsWith(".")
                        || lastFieldName.contains("..")) {
                    showInputIncorrectTips();

                    while (lastFieldName.contains("..")) {
                        lastFieldName = lastFieldName.replaceAll("\\.\\.", ".");
                    }
                    if (lastFieldName.startsWith("."))
                        lastFieldName = lastFieldName.replaceFirst("\\.", "");
                    if (lastFieldName.endsWith("."))
                        lastFieldName = lastFieldName.substring(0, lastFieldName.lastIndexOf("."));
                }
                String name = TextUtils.isEmpty(lastFieldName) ? field.getName() : lastFieldName;
                int fontId = mTextUtil.getSupportFontID(da, mPdfViewCtrl.getDoc());
                int fontColor = da.getText_color();
                float fontSize = da.getText_size();
                int fieldFlags = field.getFlags();
                int checkedIndex = ((Widget) annot).getControl().getIndex();
                ArrayList<ChoiceItemInfo> options = FormFillerUtil.getOptions(field);

                boolean isModify = !mTempValue.equals(value)
                        || mTempLastFontColor != fontColor
                        || mTempLastFontID != fontId
                        || mTempLastFontSize != fontSize
                        || !mTempLastBBox.equals(rectF);

                int fieldType = field.getType();
                if (fieldType == Field.e_TypeRadioButton) {
                    if (!mTempFieldName.equals(name) && mForm != null) {
                        isModify = true;
                        FormFillerUtil.renameField(mForm, ((Widget) annot).getControl(), name);
                        field = ((Widget) annot).getField();
                    }

                    checkedIndex = FormFillerUtil.getCheckedIndex(field);
                    isModify = isModify || mTempLastCheckedIndex != checkedIndex;
                } else if (fieldType == Field.e_TypeComboBox) {
                    isModify = isModify || mTempFieldFlags != fieldFlags || FormFillerUtil.optionsIsChanged(mTempOptions, options);
                    if ((field.getFlags() & Field.e_FlagComboEdit) != 0) {
                        addSuggestion(value);
                    }
                } else if (fieldType == Field.e_TypeListBox) {
                    isModify = isModify || mTempFieldFlags != fieldFlags || FormFillerUtil.optionsIsChanged(mTempOptions, options);
                } else if (fieldType == Field.e_TypeTextField) {
                    addSuggestion(value);
                }

                FormFillerContent fillerContent = new FormFillerContent(mPdfViewCtrl, (Widget) annot);
                fillerContent.mPageIndex = pageIndex;
                fillerContent.mRectF = rectF;
                fillerContent.mFieldValue = value;
                fillerContent.mFontId = fontId;
                fillerContent.mFontColor = fontColor;
                fillerContent.mFontSize = fontSize;
                fillerContent.mFieldName = name;
                fillerContent.mOptions = options;
                fillerContent.mFieldFlags = fieldFlags;
                fillerContent.mCheckedIndex = checkedIndex;
                modifyAnnot(fillerContent, isModify, isModify, null);
            } else if (mIsModify) {
                int fieldType = field.getType();
                int flags = DefaultAppearance.e_FlagFont | DefaultAppearance.e_FlagTextColor | DefaultAppearance.e_FlagFontSize;
                da.setFlags(flags);
                if (da.getText_color() != mTempLastFontColor)
                    da.setText_color(mTempLastFontColor);
                if (mTempLastFontID != mTextUtil.getSupportFontID(da, mPdfViewCtrl.getDoc())) {
                    Font font = mTextUtil.getStandardFont(mTempLastFontID);
                    da.setFont(font);
                }
                if (mTempLastFontSize != da.getText_size())
                    da.setText_size(mTempLastFontSize);

                if (field.getType() == Field.e_TypeRadioButton)
                    ((Widget) annot).getControl().setDefaultAppearance(da);
                else
                    field.setDefaultAppearance(da);

                if (fieldType == Field.e_TypeRadioButton) {
                    String lastFieldName = mLastFieldName.trim();

                    while (lastFieldName.contains("..")) {
                        lastFieldName = lastFieldName.replaceAll("\\.\\.", ".");
                    }
                    if (lastFieldName.startsWith(".")) {
                        lastFieldName = lastFieldName.replaceFirst("\\.", "");
                    } else if (lastFieldName.endsWith(".")) {
                        lastFieldName = lastFieldName.substring(0, lastFieldName.lastIndexOf("."));
                    }
                    String name = TextUtils.isEmpty(lastFieldName) ? field.getName() : lastFieldName;
                    if (!name.equals(mTempFieldName) && mForm != null) {
                        FormFillerUtil.renameField(mForm, ((Widget) annot).getControl(), mTempFieldName);
                        field = ((Widget) annot).getField();
                    }
                } else if (fieldType == Field.e_TypeListBox || fieldType == Field.e_TypeComboBox) {
                    if (mTempFieldFlags != field.getFlags())
                        field.setFlags(mTempFieldFlags);

                    ArrayList<ChoiceItemInfo> choiceOptions = FormFillerUtil.getOptions(field);
                    if (FormFillerUtil.optionsIsChanged(mTempOptions, choiceOptions))
                        field.setOptions(FormFillerUtil.options2Native(mTempOptions));
                }
                if (!AppUtil.toRectF(annot.getRect()).equals(mTempLastBBox))
                    annot.move(AppUtil.toFxRectF(mTempLastBBox));
                if (!field.getValue().equals(mTempValue))
                    field.setValue(mTempValue);
                annot.setModifiedDateTime(AppDmUtil.currentDateToDocumentDate());
                annot.resetAppearanceStream();
            }

            if (mPdfViewCtrl.isPageVisible(annot.getPage().getIndex()) && needInvalid) {
                refreshField(((Widget) annot).getField());
            }

            if (mShouldRefreshAnnotPanel) {
                AnnotPanelModule annotPanelModule = (AnnotPanelModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_ANNOTPANEL);
                annotPanelModule.updatePageAnnots(annot.getPage().getIndex());
                mShouldRefreshAnnotPanel = false;
            }
            mTouchBeforeAnnotCount = 0;
            mTouchAfterAnnotCount = 0;

            if (mIsShowEditText) {
                dismissSoftInput();
                mParent.removeView(mEditView);
                mIsShowEditText = false;
            }
            if (mAnnotationMenu.isShowing())
                mAnnotationMenu.dismiss();
            if (mPropertyBar.isShowing())
                mPropertyBar.dismiss();
            mLastAnnot = null;
            mIsModify = false;
            mIsEdittextOffset = false;
            mLastPageIndex = -1;
            mLastFieldName = "";
            mTempOptions.clear();
            mLastTouchPoint.set(0, 0);
            if (mCurOptions != null) {
                mCurOptions.clear();
                mCurOptions = null;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void addSuggestion(String value) {
        if (SuggestionsHelper.getInstance().isShowSuggestions(ISuggestionsType.Form_Filling)) {
            if (mShouldAddSuggestions && !AppUtil.isEmpty(value)) {
                mShouldAddSuggestions = false;
                SuggestionsHelper.getInstance().addSuggestion(ISuggestionsType.Form_Filling, value);
            }
        }
    }

    private void showInputIncorrectTips() {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        final UITextEditDialog dialog = new UITextEditDialog(uiExtensionsManager.getAttachedActivity(), UIDialog.NO_INPUT);
        dialog.getCancelButton().setVisibility(View.GONE);
        dialog.setTitle(AppResource.getString(mContext, R.string.fx_string_error));
        dialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.fx_form_rename_toast));
        dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dialog.dismiss();
            }
        });
        dialog.show();
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent, Annot annot) {
        if (!hasInitialized() || mFormFiller == null) return false;
        if (!((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().canFillForm() || !((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification())
            return false;
        if (AppAnnotUtil.isLocked(annot) || FormFillerUtil.isReadOnly(annot))
            return false;

        mBeforeText = null;
        if (mEditView != null) {
            mEditView.removeTextChangedListener(mTextWatcher);
            mEditView.setText("");
            mEditView.setSelection(mEditView.getText().length());
            mEditView.addTextChangedListener(mTextWatcher);
        }

        if (mIsLongPressTouchEvent) {
            return onTouchEventByLongPress(pageIndex, motionEvent, annot);
        } else {
            return onTouchEventBySingleTap(pageIndex, motionEvent, annot);
        }
    }

    private boolean isDown = false;
    private PointF oldPoint = new PointF();
    private int mTouchBeforeAnnotCount;
    private int mTouchAfterAnnotCount;
    private long mFormFillerActionDownTime;
    private PointF mDownPdfPoint;
    private PointF mUpPdfPoint;

    private boolean onTouchEventBySingleTap(int pageIndex, MotionEvent motionEvent, Annot annot) {
        try {
            final PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);

            PointF devPt = new PointF(motionEvent.getX(), motionEvent.getY());
            PointF pageViewPt = new PointF();
            mPdfViewCtrl.convertDisplayViewPtToPageViewPt(devPt, pageViewPt, pageIndex);
            PointF pdfPointF = new PointF();
            mPdfViewCtrl.convertPageViewPtToPdfPt(pageViewPt, pdfPointF, pageIndex);
            DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
            int action = motionEvent.getActionMasked();
            switch (action) {
                case MotionEvent.ACTION_DOWN:
                    if (mDownPdfPoint == null)
                        mDownPdfPoint = new PointF();
                    mDownPdfPoint.set(pdfPointF);
                    mFormFillerActionDownTime = SystemClock.elapsedRealtime();
                    if (annot == documentManager.getCurrentAnnot()
                            && pageIndex == annot.getPage().getIndex() && isHitAnnot(annot, pdfPointF)) {
                        isDown = true;
                        mTouchBeforeAnnotCount = page.getAnnotCount();
                        isLButtonDownHandled = false;
                        mFormFiller.onLButtonDown(page, AppUtil.toFxPointF(pdfPointF), 0);
                        isLButtonDownHandled = true;
                        return true;
                    }
                    return false;
                case MotionEvent.ACTION_MOVE:
                    if (getDistanceOfPoints(pageViewPt, oldPoint) > 0 && annot == documentManager.getCurrentAnnot()
                            && pageIndex == annot.getPage().getIndex()
                            && AnnotPermissionUtil.canModifyAnnot(documentManager, annot)) {
                        oldPoint.set(pageViewPt);
                        mFormFiller.onMouseMove(page, AppUtil.toFxPointF(pdfPointF), 0);
                        return true;
                    }
                    return false;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL:
                    if (mUpPdfPoint == null)
                        mUpPdfPoint = new PointF();
                    mUpPdfPoint.set(pdfPointF);
                    if (SystemClock.elapsedRealtime() - mFormFillerActionDownTime > 150) {
                        RectF rectF = AppUtil.toRectF(annot.getRect());
                        int offset = 3;
                        if (pdfPointF.x < rectF.left + offset)
                            pdfPointF.x = rectF.left + offset;
                        if (pdfPointF.x > rectF.right - offset)
                            pdfPointF.x = rectF.right - offset;
                        if (pdfPointF.y < rectF.bottom + offset)
                            pdfPointF.y = rectF.bottom + offset;
                        if (pdfPointF.y > rectF.top - offset)
                            pdfPointF.y = rectF.top - offset;
                        onLongPressInFormFiller(page, AppUtil.toFxPointF(pdfPointF));
                        return true;
                    } else if (mAnnotationMenu.isShowing()) {
                        mAnnotationMenu.dismiss();
                        mLastPopupPdfPoint = null;
                    }
                    if (pageIndex == annot.getPage().getIndex() && (isHitAnnot(annot, pdfPointF) || isDown)) {
                        int type = FormFillerUtil.getAnnotFieldType(annot);
                        if (type == Field.e_TypeRadioButton) {
                            mTempLastCheckedIndex = FormFillerUtil.getCheckedIndex(((Widget) annot).getField());
                        }

                        isDown = false;
                        mFormFiller.onLButtonUp(page, AppUtil.toFxPointF(pdfPointF), 0);
                        mTouchAfterAnnotCount = page.getAnnotCount();
                        if (mTouchAfterAnnotCount != mTouchBeforeAnnotCount)
                            mShouldRefreshAnnotPanel = true;

                        if (type == Field.e_TypeRadioButton) {
                            if (mTempLastCheckedIndex != FormFillerUtil.getCheckedIndex(((Widget) annot).getField())) {
                                FormFillerContent formContent = new FormFillerContent(mPdfViewCtrl, (Widget) annot);
                                FormFillerModifyUndoItem undoItem = (FormFillerModifyUndoItem) createModifyUndoItem(formContent);
                                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(undoItem);
                            }
                        } else {
                            mIsModify = true;
                        }

                        if (shouldShowInputSoft(annot)) {
                            if (type == Field.e_TypeComboBox) {
                                mIsTouchComboPopup = isTouchComboPopup(pdfPointF, annot);
                                if (mIsTouchComboPopup) {
                                    return true;
                                }
                            }

                            int keybordHeight = AppKeyboardUtil.getKeyboardHeight(mParent);
                            if (keybordHeight < 0 || keybordHeight < AppDisplay.getRawScreenHeight() / 5) {
                                showSoftInput();
                            }
                        }

                        if (mIsDateField) {
                            showDatePickerDialog();
                        }
                        return true;
                    }
                    return false;
                default:
            }
        } catch (PDFException e) {
            e.printStackTrace();
            isLButtonDownHandled = true;
        }
        return false;
    }

    private com.foxit.sdk.common.fxcrt.PointF mLastLongPressFxPoint;

    private void onLongPressInFormFiller(PDFPage page, com.foxit.sdk.common.fxcrt.PointF fxPoint) throws PDFException {
        if (mLastLongPressFxPoint == null)
            mLastLongPressFxPoint = new com.foxit.sdk.common.fxcrt.PointF();
        mLastLongPressFxPoint.set(fxPoint);

        mFormFiller.onLButtonUp(page, fxPoint, 0);
        mFormFiller.onRButtonDown(page, fxPoint, 0);
        mFormFiller.onRButtonUp(page, fxPoint, 0);
    }

    private PointF mDocViewerPt = new PointF(0, 0);
    private RectF mPageDrawRect = new RectF();
    private RectF mInvalidateRect = new RectF(0, 0, 0, 0);
    private RectF mAnnotMenuRect = new RectF(0, 0, 0, 0);
    private float mThickness = 0f;


    private boolean onTouchEventByLongPress(int pageIndex, MotionEvent motionEvent, Annot annot) {
        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        if (!documentManager.canModifyForm() || !((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification() || documentManager.isSign())
            return false;
        // in pageView evX and evY
        PointF point = new PointF(motionEvent.getX(), motionEvent.getY());
        mPdfViewCtrl.convertDisplayViewPtToPageViewPt(point, point, pageIndex);
        float evX = point.x;
        float evY = point.y;

        PointF pdfPointF = new PointF();
        mPdfViewCtrl.convertPageViewPtToPdfPt(point, pdfPointF, pageIndex);

        int action = motionEvent.getAction();
        try {
            switch (action) {
                case MotionEvent.ACTION_DOWN:
                    if (annot == documentManager.getCurrentAnnot() && pageIndex == annot.getPage().getIndex()) {
                        mThickness = thicknessOnPageView(pageIndex, DEFAULT_THICKNESS);
                        RectF pageViewBBox = AppUtil.toRectF(annot.getRect());
                        mPdfViewCtrl.convertPdfRectToPageViewRect(pageViewBBox, pageViewBBox, pageIndex);
                        RectF pdfRect = AppUtil.toRectF(annot.getRect());
                        mPageViewRect.set(pdfRect.left, pdfRect.top, pdfRect.right, pdfRect.bottom);
                        mPdfViewCtrl.convertPdfRectToPageViewRect(mPageViewRect, mPageViewRect, pageIndex);
                        mPageViewRect.inset(mThickness / 2f, mThickness / 2f);

                        mCurrentCtr = isTouchControlPoint(pageViewBBox, evX, evY);

                        mDownPoint.set(evX, evY);
                        mLastPoint.set(evX, evY);
                        mDocViewerPt.set(motionEvent.getX(), motionEvent.getY());

                        if (mCurrentCtr == CTR_LT) {
                            mTouchCaptured = true;
                            mLastOper = OPER_SCALE_LT;
                            return true;
                        } else if (mCurrentCtr == CTR_T) {
                            mTouchCaptured = true;
                            mLastOper = OPER_SCALE_T;
                            return true;
                        } else if (mCurrentCtr == CTR_RT) {
                            mTouchCaptured = true;
                            mLastOper = OPER_SCALE_RT;
                            return true;
                        } else if (mCurrentCtr == CTR_R) {
                            mTouchCaptured = true;
                            mLastOper = OPER_SCALE_R;
                            return true;
                        } else if (mCurrentCtr == CTR_RB) {
                            mTouchCaptured = true;
                            mLastOper = OPER_SCALE_RB;
                            return true;
                        } else if (mCurrentCtr == CTR_B) {
                            mTouchCaptured = true;
                            mLastOper = OPER_SCALE_B;
                            return true;
                        } else if (mCurrentCtr == CTR_LB) {
                            mTouchCaptured = true;
                            mLastOper = OPER_SCALE_LB;
                            return true;
                        } else if (mCurrentCtr == CTR_L) {
                            mTouchCaptured = true;
                            mLastOper = OPER_SCALE_L;
                            return true;
                        } else if (isHitAnnot(annot, pdfPointF)) {
                            mTouchCaptured = true;
                            mLastOper = OPER_TRANSLATE;
                            return true;
                        }
                    }
                    return false;
                case MotionEvent.ACTION_MOVE:
                    if (pageIndex == annot.getPage().getIndex() && mTouchCaptured
                            && annot == documentManager.getCurrentAnnot()
                            && documentManager.canAddAnnot()
                            && ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()
                            && AnnotPermissionUtil.canModifyAnnot(documentManager, annot)) {
                        if (evX != mLastPoint.x && evY != mLastPoint.y) {
                            RectF pageViewBBox = AppUtil.toRectF(annot.getRect());
                            mPdfViewCtrl.convertPdfRectToPageViewRect(pageViewBBox, pageViewBBox, pageIndex);
                            float deltaXY = mCtlPtLineWidth + mCtlPtRadius * 2 + 2;// Judging border value
                            switch (mLastOper) {
                                case OPER_TRANSLATE: {
                                    mInvalidateRect.set(pageViewBBox);
                                    mAnnotMenuRect.set(pageViewBBox);
                                    mInvalidateRect.offset(mLastPoint.x - mDownPoint.x, mLastPoint.y - mDownPoint.y);
                                    mAnnotMenuRect.offset(evX - mDownPoint.x, evY - mDownPoint.y);
                                    PointF adjustXY = adjustScalePointF(pageIndex, mAnnotMenuRect, deltaXY);

                                    mInvalidateRect.union(mAnnotMenuRect);

                                    mInvalidateRect.inset(-deltaXY - mCtlPtDeltyXY, -deltaXY - mCtlPtDeltyXY);
                                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                                    mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mAnnotMenuRect, mAnnotMenuRect, pageIndex);
                                    if (mAnnotationMenu.isShowing()) {
                                        mAnnotationMenu.dismiss();
                                        mAnnotationMenu.update(mAnnotMenuRect);
                                    }

                                    mLastPoint.set(evX, evY);
                                    mLastPoint.offset(adjustXY.x, adjustXY.y);
                                    break;
                                }
                                case OPER_SCALE_LT: {
                                    if (evX != mLastPoint.x && evY != mLastPoint.y) {
                                        mInvalidateRect.set(mLastPoint.x, mLastPoint.y, mPageViewRect.right, mPageViewRect.bottom);
                                        mAnnotMenuRect.set(evX, evY, mPageViewRect.right, mPageViewRect.bottom);
                                        mInvalidateRect.sort();
                                        mAnnotMenuRect.sort();
                                        mInvalidateRect.union(mAnnotMenuRect);
                                        mInvalidateRect.inset(-mThickness - mCtlPtDeltyXY, -mThickness - mCtlPtDeltyXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                                        PointF adjustXY = adjustScalePointF(pageIndex, mAnnotMenuRect, deltaXY);

                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mAnnotMenuRect, mAnnotMenuRect, pageIndex);
                                        if (mAnnotationMenu.isShowing()) {
                                            mAnnotationMenu.dismiss();
                                            mAnnotationMenu.update(mAnnotMenuRect);
                                        }

                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustXY.x, adjustXY.y);
                                    }
                                    break;
                                }
                                case OPER_SCALE_T: {
                                    if (evX != mLastPoint.x && evY != mLastPoint.y) {
                                        mInvalidateRect.set(mPageViewRect.left, mLastPoint.y, mPageViewRect.right, mPageViewRect.bottom);
                                        mAnnotMenuRect.set(mPageViewRect.left, evY, mPageViewRect.right, mPageViewRect.bottom);
                                        mInvalidateRect.sort();
                                        mAnnotMenuRect.sort();
                                        mInvalidateRect.union(mAnnotMenuRect);
                                        mInvalidateRect.inset(-mThickness - mCtlPtDeltyXY, -mThickness - mCtlPtDeltyXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                                        PointF adjustXY = adjustScalePointF(pageIndex, mAnnotMenuRect, deltaXY);

                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mAnnotMenuRect, mAnnotMenuRect, pageIndex);
                                        if (mAnnotationMenu.isShowing()) {
                                            mAnnotationMenu.dismiss();
                                            mAnnotationMenu.update(mAnnotMenuRect);
                                        }

                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustXY.x, adjustXY.y);
                                    }
                                    break;
                                }
                                case OPER_SCALE_RT: {
                                    if (evX != mLastPoint.x && evY != mLastPoint.y) {

                                        mInvalidateRect.set(mPageViewRect.left, mLastPoint.y, mLastPoint.x, mPageViewRect.bottom);
                                        mAnnotMenuRect.set(mPageViewRect.left, evY, evX, mPageViewRect.bottom);
                                        mInvalidateRect.sort();
                                        mAnnotMenuRect.sort();
                                        mInvalidateRect.union(mAnnotMenuRect);
                                        mInvalidateRect.inset(-mThickness - mCtlPtDeltyXY, -mThickness - mCtlPtDeltyXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                                        PointF adjustXY = adjustScalePointF(pageIndex, mAnnotMenuRect, deltaXY);

                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mAnnotMenuRect, mAnnotMenuRect, pageIndex);
                                        if (mAnnotationMenu.isShowing()) {
                                            mAnnotationMenu.dismiss();
                                            mAnnotationMenu.update(mAnnotMenuRect);
                                        }
                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustXY.x, adjustXY.y);
                                    }
                                    break;
                                }
                                case OPER_SCALE_R: {
                                    if (evX != mLastPoint.x && evY != mLastPoint.y) {
                                        mInvalidateRect.set(mPageViewRect.left, mPageViewRect.top, mLastPoint.x, mPageViewRect.bottom);
                                        mAnnotMenuRect.set(mPageViewRect.left, mPageViewRect.top, evX, mPageViewRect.bottom);
                                        mInvalidateRect.sort();
                                        mAnnotMenuRect.sort();
                                        mInvalidateRect.union(mAnnotMenuRect);
                                        mInvalidateRect.inset(-mThickness - mCtlPtDeltyXY, -mThickness - mCtlPtDeltyXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                                        PointF adjustXY = adjustScalePointF(pageIndex, mAnnotMenuRect, deltaXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mAnnotMenuRect, mAnnotMenuRect, pageIndex);
                                        if (mAnnotationMenu.isShowing()) {
                                            mAnnotationMenu.dismiss();
                                            mAnnotationMenu.update(mAnnotMenuRect);
                                        }
                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustXY.x, adjustXY.y);
                                    }
                                    break;
                                }
                                case OPER_SCALE_RB: {
                                    if (evX != mLastPoint.x && evY != mLastPoint.y) {
                                        mInvalidateRect.set(mPageViewRect.left, mPageViewRect.top, mLastPoint.x, mLastPoint.y);
                                        mAnnotMenuRect.set(mPageViewRect.left, mPageViewRect.top, evX, evY);
                                        mInvalidateRect.sort();
                                        mAnnotMenuRect.sort();
                                        mInvalidateRect.union(mAnnotMenuRect);
                                        mInvalidateRect.inset(-mThickness - mCtlPtDeltyXY, -mThickness - mCtlPtDeltyXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                                        PointF adjustXY = adjustScalePointF(pageIndex, mAnnotMenuRect, deltaXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mAnnotMenuRect, mAnnotMenuRect, pageIndex);
                                        if (mAnnotationMenu.isShowing()) {
                                            mAnnotationMenu.dismiss();
                                            mAnnotationMenu.update(mAnnotMenuRect);
                                        }
                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustXY.x, adjustXY.y);
                                    }
                                    break;
                                }
                                case OPER_SCALE_B: {
                                    if (evX != mLastPoint.x && evY != mLastPoint.y) {
                                        mInvalidateRect.set(mPageViewRect.left, mPageViewRect.top, mPageViewRect.right, mLastPoint.y);
                                        mAnnotMenuRect.set(mPageViewRect.left, mPageViewRect.top, mPageViewRect.right, evY);
                                        mInvalidateRect.sort();
                                        mAnnotMenuRect.sort();
                                        mInvalidateRect.union(mAnnotMenuRect);
                                        mInvalidateRect.inset(-mThickness - mCtlPtDeltyXY, -mThickness - mCtlPtDeltyXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                                        PointF adjustXY = adjustScalePointF(pageIndex, mAnnotMenuRect, deltaXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mAnnotMenuRect, mAnnotMenuRect, pageIndex);
                                        if (mAnnotationMenu.isShowing()) {
                                            mAnnotationMenu.dismiss();
                                            mAnnotationMenu.update(mAnnotMenuRect);
                                        }
                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustXY.x, adjustXY.y);
                                    }
                                    break;
                                }
                                case OPER_SCALE_LB: {
                                    if (evX != mLastPoint.x && evY != mLastPoint.y) {
                                        mInvalidateRect.set(mLastPoint.x, mPageViewRect.top, mPageViewRect.right, mLastPoint.y);
                                        mAnnotMenuRect.set(evX, mPageViewRect.top, mPageViewRect.right, evY);
                                        mInvalidateRect.sort();
                                        mAnnotMenuRect.sort();
                                        mInvalidateRect.union(mAnnotMenuRect);
                                        mInvalidateRect.inset(-mThickness - mCtlPtDeltyXY, -mThickness - mCtlPtDeltyXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                                        PointF adjustXY = adjustScalePointF(pageIndex, mAnnotMenuRect, deltaXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mAnnotMenuRect, mAnnotMenuRect, pageIndex);
                                        if (mAnnotationMenu.isShowing()) {
                                            mAnnotationMenu.dismiss();
                                            mAnnotationMenu.update(mAnnotMenuRect);
                                        }
                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustXY.x, adjustXY.y);
                                    }
                                    break;
                                }
                                case OPER_SCALE_L: {
                                    if (evX != mLastPoint.x && evY != mLastPoint.y) {
                                        mInvalidateRect.set(mLastPoint.x, mPageViewRect.top, mPageViewRect.right, mPageViewRect.bottom);
                                        mAnnotMenuRect.set(evX, mPageViewRect.top, mPageViewRect.right, mPageViewRect.bottom);
                                        mInvalidateRect.sort();
                                        mAnnotMenuRect.sort();
                                        mInvalidateRect.union(mAnnotMenuRect);
                                        mInvalidateRect.inset(-mThickness - mCtlPtDeltyXY, -mThickness - mCtlPtDeltyXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mInvalidateRect, mInvalidateRect, pageIndex);
                                        mPdfViewCtrl.invalidate(AppDmUtil.rectFToRect(mInvalidateRect));

                                        PointF adjustXY = adjustScalePointF(pageIndex, mAnnotMenuRect, deltaXY);
                                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mAnnotMenuRect, mAnnotMenuRect, pageIndex);
                                        if (mAnnotationMenu.isShowing()) {
                                            mAnnotationMenu.dismiss();
                                            mAnnotationMenu.update(mAnnotMenuRect);
                                        }
                                        mLastPoint.set(evX, evY);
                                        mLastPoint.offset(adjustXY.x, adjustXY.y);
                                    }
                                    break;
                                }
                                default:
                                    break;
                            }
                        }
                        return true;
                    }
                    return false;
                case MotionEvent.ACTION_UP:
                case MotionEvent.ACTION_CANCEL:
                    boolean ret = false;
                    if (mTouchCaptured && annot == ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot() && pageIndex == annot.getPage().getIndex()) {
                        RectF pageViewRect = AppUtil.toRectF(annot.getRect());
                        mPdfViewCtrl.convertPdfRectToPageViewRect(pageViewRect, pageViewRect, pageIndex);
                        pageViewRect.inset(mThickness / 2, mThickness / 2);

                        switch (mLastOper) {
                            case OPER_TRANSLATE: {
                                mPageDrawRect.set(pageViewRect);
                                mPageDrawRect.offset(mLastPoint.x - mDownPoint.x, mLastPoint.y - mDownPoint.y);
                                break;
                            }
                            case OPER_SCALE_LT: {
                                if (!mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                                    mPageDrawRect.set(mLastPoint.x, mLastPoint.y, pageViewRect.right, pageViewRect.bottom);
                                }
                                break;
                            }
                            case OPER_SCALE_T: {
                                if (!mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                                    mPageDrawRect.set(pageViewRect.left, mLastPoint.y, pageViewRect.right, pageViewRect.bottom);
                                }
                                break;
                            }
                            case OPER_SCALE_RT: {
                                if (!mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                                    mPageDrawRect.set(pageViewRect.left, mLastPoint.y, mLastPoint.x, pageViewRect.bottom);
                                }
                                break;
                            }
                            case OPER_SCALE_R: {
                                if (!mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                                    mPageDrawRect.set(pageViewRect.left, pageViewRect.top, mLastPoint.x, pageViewRect.bottom);
                                }
                                break;
                            }
                            case OPER_SCALE_RB: {
                                if (!mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                                    mPageDrawRect.set(pageViewRect.left, pageViewRect.top, mLastPoint.x, mLastPoint.y);
                                }
                                break;
                            }
                            case OPER_SCALE_B: {
                                if (!mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                                    mPageDrawRect.set(pageViewRect.left, pageViewRect.top, pageViewRect.right, mLastPoint.y);
                                }
                                break;
                            }
                            case OPER_SCALE_LB: {
                                if (!mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                                    mPageDrawRect.set(mLastPoint.x, pageViewRect.top, pageViewRect.right, mLastPoint.y);
                                }
                                break;
                            }
                            case OPER_SCALE_L: {
                                if (!mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                                    mPageDrawRect.set(mLastPoint.x, pageViewRect.top, pageViewRect.right, pageViewRect.bottom);
                                }
                                break;
                            }
                            default:
                                break;
                        }
                        RectF viewDrawBox = new RectF(mPageDrawRect.left, mPageDrawRect.top, mPageDrawRect.right, mPageDrawRect.bottom);
                        float _lineWidth = DEFAULT_THICKNESS;
                        viewDrawBox.inset(-thicknessOnPageView(pageIndex, _lineWidth) / 2, -thicknessOnPageView(pageIndex, _lineWidth) / 2);
                        if (mLastOper != OPER_DEFAULT && !mDownPoint.equals(mLastPoint.x, mLastPoint.y)) {
                            RectF bboxRect = new RectF(viewDrawBox);
                            mPdfViewCtrl.convertPageViewRectToPdfRect(bboxRect, bboxRect, pageIndex);

                            FormFillerContent fillerContent = new FormFillerContent(mPdfViewCtrl, (Widget) annot);
                            fillerContent.mPageIndex = pageIndex;
                            fillerContent.mRectF = bboxRect;
                            modifyAnnot(fillerContent, false, false, null);
                            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(viewDrawBox, viewDrawBox, pageIndex);
                            if (mAnnotationMenu.isShowing()) {
                                mAnnotationMenu.update(viewDrawBox);
                            } else {
                                mAnnotationMenu.show(viewDrawBox);
                            }

                        } else {
                            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(viewDrawBox, viewDrawBox, pageIndex);
                            if (mAnnotationMenu.isShowing()) {
                                mAnnotationMenu.update(viewDrawBox);
                            } else {
                                mAnnotationMenu.show(viewDrawBox);
                            }
                        }

                        ret = true;
                    }

                    mTouchCaptured = false;
                    mDownPoint.set(0, 0);
                    mLastPoint.set(0, 0);
                    mLastOper = OPER_DEFAULT;
                    mCurrentCtr = CTR_NONE;
                    return ret;
                default:
            }
            return false;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private RectF mThicknessRectF = new RectF();

    private float thicknessOnPageView(int pageIndex, float thickness) {
        mThicknessRectF.set(0, 0, thickness, thickness);
        mPdfViewCtrl.convertPdfRectToPageViewRect(mThicknessRectF, mThicknessRectF, pageIndex);
        return Math.abs(mThicknessRectF.width());
    }

    private int isTouchControlPoint(RectF rect, float x, float y) {
        PointF[] ctlPts = calculateControlPoints(rect);
        RectF area = new RectF();
        int ret = -1;
        for (int i = 0; i < ctlPts.length; i++) {
            area.set(ctlPts[i].x, ctlPts[i].y, ctlPts[i].x, ctlPts[i].y);
            area.inset(-mCtlPtTouchExt, -mCtlPtTouchExt);
            if (area.contains(x, y)) {
                ret = i + 1;
            }
        }
        return ret;
    }

    /*
     *   1-----2-----3
     *   |	         |
     *   |	         |
     *   8           4
     *   |           |
     *   |           |
     *   7-----6-----5
     *   */
    RectF mMapBounds = new RectF();

    private PointF[] calculateControlPoints(RectF rect) {
        rect.sort();
        mMapBounds.set(rect);
        mMapBounds.inset(-mCtlPtRadius - mCtlPtLineWidth / 2f, -mCtlPtRadius - mCtlPtLineWidth / 2f);// control rect
        PointF p1 = new PointF(mMapBounds.left, mMapBounds.top);
        PointF p2 = new PointF((mMapBounds.right + mMapBounds.left) / 2, mMapBounds.top);
        PointF p3 = new PointF(mMapBounds.right, mMapBounds.top);
        PointF p4 = new PointF(mMapBounds.right, (mMapBounds.bottom + mMapBounds.top) / 2);
        PointF p5 = new PointF(mMapBounds.right, mMapBounds.bottom);
        PointF p6 = new PointF((mMapBounds.right + mMapBounds.left) / 2, mMapBounds.bottom);
        PointF p7 = new PointF(mMapBounds.left, mMapBounds.bottom);
        PointF p8 = new PointF(mMapBounds.left, (mMapBounds.bottom + mMapBounds.top) / 2);

        return new PointF[]{p1, p2, p3, p4, p5, p6, p7, p8};
    }

    private PointF mAdjustPointF = new PointF(0, 0);

    private PointF adjustScalePointF(int pageIndex, RectF rectF, float dxy) {
        float adjustx = 0;
        float adjusty = 0;
        if (mLastOper != OPER_TRANSLATE) {
            rectF.inset(-mThickness / 2f, -mThickness / 2f);
        }

        if ((int) rectF.left < dxy) {
            adjustx = -rectF.left + dxy;
            rectF.left = dxy;
        }
        if ((int) rectF.top < dxy) {
            adjusty = -rectF.top + dxy;
            rectF.top = dxy;
        }

        if ((int) rectF.right > mPdfViewCtrl.getPageViewWidth(pageIndex) - dxy) {
            adjustx = mPdfViewCtrl.getPageViewWidth(pageIndex) - rectF.right - dxy;
            rectF.right = mPdfViewCtrl.getPageViewWidth(pageIndex) - dxy;
        }
        if ((int) rectF.bottom > mPdfViewCtrl.getPageViewHeight(pageIndex) - dxy) {
            adjusty = mPdfViewCtrl.getPageViewHeight(pageIndex) - rectF.bottom - dxy;
            rectF.bottom = mPdfViewCtrl.getPageViewHeight(pageIndex) - dxy;
        }
        mAdjustPointF.set(adjustx, adjusty);
        return mAdjustPointF;
    }

    private double getDistanceOfPoints(PointF p1, PointF p2) {
        return Math.sqrt(Math.abs((p1.x - p2.x)
                * (p1.x - p2.x) + (p1.y - p2.y)
                * (p1.y - p2.y)));
    }

    private class Blink extends Handler implements Runnable {
        private Annot mAnnot;

        public Blink(Annot annot) {
            mAnnot = annot;
        }

        public void setAnnot(Annot annot) {
            mAnnot = annot;
        }

        @Override
        public void run() {
            if (mFNModule != null) {
                if (shouldShowInputSoft(mAnnot)) {
                    int keybordHeight = AppKeyboardUtil.getKeyboardHeight(mParent);
                    if (0 <= keybordHeight && keybordHeight < AppDisplay.getRawScreenHeight() / 5) {
                        mFNModule.showFormNavigationBar();
                    }
                } else {
                    mFNModule.showFormNavigationBar();
                }
            }

            postDelayed(Blink.this, 500);
        }
    }

    private void setEdittextLocation(Annot annot) {
        Rect rect = new Rect();
        mParent.getGlobalVisibleRect(rect);

        if (AppDisplay.isPad() && rect.top != 0 && rect.top != AppDisplay.getStatusBarHeight()) {
            if (annot != null && !annot.isEmpty()) {
                int keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mParent);
                int rawScreenHeight = AppDisplay.getRawScreenHeight();
                int navBarHeight = AppDisplay.getRealNavBarHeight();

                if (rawScreenHeight - mTempLastDisplayBBox.bottom - rect.top < (keyboardHeight + AppDisplay.dp2px(116)) + navBarHeight) {
                    mIsEdittextOffset = true;
                    mEditView.setLayoutParams(new RelativeLayout.LayoutParams(100, 100));
                    mEditView.setX(mTempLastDisplayBBox.left);
                    mEditView.setY(mTempLastDisplayBBox.bottom + 50);
                    mEditView.setSingleLine(true);
                    mEditView.setSelection(mEditView.getText().length());
                    mEditView.setBackground(null);
                    mEditView.setCursorVisible(false);
                } else {
                    mIsEdittextOffset = false;
                    mEditView.setSingleLine(false);
                    mEditView.setLayoutParams(new RelativeLayout.LayoutParams(1, 1));
                }


                if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
                    RectF rectF = new RectF(mPageViewRect);
                    rectF.inset(-100, -100);
                    mPdfViewCtrl.refresh(mLastPageIndex, AppDmUtil.rectFToRect(rectF));
                }
            }
        }
    }

    private int getFNBottomPadding() {
        Rect rect = new Rect();
        mParent.getGlobalVisibleRect(rect);
        int padding;
        int top = rect.top;
        int bottom = rect.bottom;
        int screenHeight = AppDisplay.getRawScreenHeight();

        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        int keyboardHeight;
        if (SystemUiHelper.getInstance().isFullScreenMode(uiExtensionsManager.getAttachedActivity())) {
            keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mParent, true);
        } else {
            if (SystemUiHelper.getInstance().isFullScreen())
                keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mParent, true);
            else
                keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mParent, false);
        }
        if (mIsEdittextOffset) {
            if ((screenHeight - bottom - AppDisplay.getNavBarHeight()) >= keyboardHeight) {
                padding = 0;
            } else if (screenHeight - top - (int) mTempLastDisplayBBox.bottom - 150 >= keyboardHeight && screenHeight - bottom - AppDisplay.getNavBarHeight() < keyboardHeight) {
                int[] location = new int[2];
                mParent.getLocationOnScreen(location);
                int y = location[1];

                padding = mParent.getHeight() - (screenHeight - keyboardHeight - AppDisplay.getNavBarHeight() - y);
            } else if (mParent.getHeight() - (int) mTempLastDisplayBBox.bottom > 150) {
                padding = mParent.getHeight() - (int) mTempLastDisplayBBox.bottom - 150;
            } else {
                padding = 0;
            }
        } else {
            if ((screenHeight - bottom - AppDisplay.getNavBarHeight()) >= keyboardHeight) {
                padding = 0;
            } else if (screenHeight - top > keyboardHeight && screenHeight - bottom - AppDisplay.getNavBarHeight() < keyboardHeight) {
                if (screenHeight < (bottom + AppDisplay.getNavBarHeight())) {
                    padding = keyboardHeight + AppDisplay.getNavBarHeight();
                } else {
                    padding = keyboardHeight;
                }
            } else {
                padding = 0;
            }
        }
        return padding;
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent, Annot annot) {
        if (!hasInitialized() || mFormFiller == null) return false;
        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        if (!documentManager.withAllPermission(null)) return false;
        if (documentManager.isSign() || documentManager.isXFA())
            return false;
        if (FormFillerUtil.isReadOnly(annot)) return false;

        mIsLongPressTouchEvent = true;
        try {
            final PointF pdfPointF = AppAnnotUtil.getPdfPoint(mPdfViewCtrl, pageIndex, motionEvent);

            if (annot == documentManager.getCurrentAnnot()) {
                if (pageIndex == annot.getPage().getIndex() && isHitAnnot(annot, pdfPointF)) {
                    return true;
                } else {
                    if (shouldShowInputSoft(annot)) {
                        if (mBlink != null) {
                            mBlink.removeCallbacks(mBlink);
                            mBlink = null;
                        }
                        dismissSoftInput();
                        mParent.removeView(mEditView);
                    }
                    if (shouldShowNavigation(annot)) {
                        if (mFNModule != null) {
                            mFNModule.hideFormNavigationBar();
                        }
                        resetDocViewerOffset();
                    }
                    documentManager.setCurrentAnnot(null);
                    return true;
                }
            } else {
                documentManager.setCurrentAnnot(annot);
                return true;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent, Annot annot) {
        if (!hasInitialized() || mFormFiller == null) return false;
        boolean ret = false;
        DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
        if (!documentManager.withAllPermission(null)
                || !documentManager.canFillForm()
                || !((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).isEnableModification()
                || FormFillerUtil.isReadOnly(annot)) {

            try {
                int fieldType = FormFillerUtil.getAnnotFieldType(annot);
                if (fieldType == Field.e_TypePushButton) {
                    AdditionalAction additionalAction = new AdditionalAction(annot);
                    Action action = additionalAction.getAction(AdditionalAction.e_TriggerAnnotMouseButtonPressed);
                    if (!action.isEmpty() && action.getType() == Action.e_TypeSound) {

                        PDFDictionary dict = action.getDict();
                        if (dict != null) {
                            PDFObject soundObj = dict.getElement("Sound");
                            if (soundObj != null) {
                                openSound(annot, soundObj);
                                return true;
                            }
                        }
                    }
                }
            } catch (PDFException e) {
                e.printStackTrace();
            }
            return false;
        }

        mIsLongPressTouchEvent = false;
        try {
            PointF pdfPointF = AppAnnotUtil.getPdfPoint(mPdfViewCtrl, pageIndex, motionEvent);
            PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            Annot annotTmp = AppAnnotUtil.createAnnot(page.getAnnotAtPoint(AppUtil.toFxPointF(pdfPointF), 1));
            boolean isHit = isHitAnnot(annot, pdfPointF);

            mLastTouchPoint.set(pdfPointF.x, pdfPointF.y);
            if (annot == documentManager.getCurrentAnnot()) {
                if (pageIndex == annot.getPage().getIndex() && isHit) {
                    ret = true;
                } else {
                    if (shouldShowInputSoft(annot)) {
                        if (mBlink != null) {
                            mBlink.removeCallbacks(mBlink);
                            mBlink = null;
                        }
                        dismissSoftInput();
                        mParent.removeView(mEditView);
                    }
                    if (shouldShowNavigation(annot)) {
                        if (mFNModule != null) {
                            mFNModule.hideFormNavigationBar();
                        }
                        resetDocViewerOffset();
                    }
                    documentManager.setCurrentAnnot(null);
                    ret = false;
                }
            } else {
                mIsTouchComboPopup = isTouchComboPopup(pdfPointF, annot);
                documentManager.setCurrentAnnot(annot);
                ret = true;
            }

            if (annotTmp == null || (annot == documentManager.getCurrentAnnot()
                    && pageIndex == annot.getPage().getIndex())) {
                PointF touchPointF = pdfPointF;
                if (isHit) {
                    touchPointF = adjustTouchPointF(pdfPointF, AppUtil.toRectF(annot.getRect()));
                }

                Field field = ((Widget) annot).getField();
                int type = field.getType();
                if (type == Field.e_TypeRadioButton) {
                    mTempLastCheckedIndex = FormFillerUtil.getCheckedIndex(field);
                }

                mTouchBeforeAnnotCount = page.getAnnotCount();

                PDFViewCtrl.lock();
                try {
                    isLButtonDownHandled = false;
                    mFormFiller.onLButtonDown(page, AppUtil.toFxPointF(touchPointF), 0);
                    isLButtonDownHandled = true;
                    mFormFiller.onLButtonUp(page, AppUtil.toFxPointF(touchPointF), 0);
                } finally {
                    PDFViewCtrl.unlock();
                }

                mTouchAfterAnnotCount = page.getAnnotCount();
                if (mTouchBeforeAnnotCount != mTouchAfterAnnotCount)
                    mShouldRefreshAnnotPanel = true;

                if (type == Field.e_TypeRadioButton) {
                    if (mTempLastCheckedIndex != FormFillerUtil.getCheckedIndex(field)) {
                        FormFillerContent formContent = new FormFillerContent(mPdfViewCtrl, (Widget) annot);
                        FormFillerModifyUndoItem undoItem = (FormFillerModifyUndoItem) createModifyUndoItem(formContent);
                        documentManager.addUndoItem(undoItem);
                    }
                } else {
                    mIsModify = true;
                }
            }
        } catch (PDFException e) {
            isLButtonDownHandled = true;
        }
        return ret;
    }

    /**
     * For combobox field, Under the premise that annot has been selected.
     *
     * @param point PDF point
     * @param annot current annot
     * @return true means touch popup or popup icon, false otherwise.
     */
    private boolean isTouchComboPopup(PointF point, Annot annot) {
        try {
            int fieldType = FormFillerUtil.getAnnotFieldType(annot);
            if (fieldType == Field.e_TypeComboBox) {
                RectF rectF = AppUtil.toRectF(annot.getRect());

                boolean touchAnnot = AppUtil.containsFxPoint(rectF, point.x, point.y);
                if (touchAnnot) { // touch popup icon
                    RectF popupIconRectF = new RectF(Math.max(0, rectF.right - 10), rectF.top, rectF.right, rectF.bottom);
                    return AppUtil.containsFxPoint(popupIconRectF, point.x, point.y);
                } else { // touch popup
                    return true;
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private FormFillerUndoItem createModifyUndoItem(IFormFillerContent formContent) {
        final FormFillerModifyUndoItem undoItem = new FormFillerModifyUndoItem(mPdfViewCtrl);
        undoItem.setCurrentValue(formContent.getAnnot());
        undoItem.mNeedResetChecked = false;

        undoItem.mPageIndex = formContent.getPageIndex();
        undoItem.mBBox = new RectF(formContent.getBBox());
        undoItem.mValue = formContent.getFieldValue();
        undoItem.mModifiedDate = AppDmUtil.currentDateToDocumentDate();
        undoItem.mFontId = formContent.getFontId();
        undoItem.mFontColor = formContent.getFontColor();
        undoItem.mFontSize = formContent.getFontSize();
        undoItem.mFieldName = formContent.getFieldName();
        undoItem.mFieldFlags = formContent.getFieldFlags();
        undoItem.mOptions = FormFillerUtil.cloneChoiceOptions(formContent.getOptions());
        undoItem.mCheckedIndex = formContent.getCheckedIndex();

        undoItem.mRedoBbox = new RectF(formContent.getBBox());
        undoItem.mRedoValue = formContent.getFieldValue();
        undoItem.mRedoFontId = formContent.getFontId();
        undoItem.mRedoFontColor = formContent.getFontColor();
        undoItem.mRedoFontSize = formContent.getFontSize();
        undoItem.mRedoFieldName = formContent.getFieldName();
        undoItem.mRedoFieldFlags = formContent.getFieldFlags();
        undoItem.mRedoOptions = FormFillerUtil.cloneChoiceOptions(formContent.getOptions());
        undoItem.mRedoCheckedIndex = formContent.getCheckedIndex();

        undoItem.mUndoValue = mTempValue;
        undoItem.mUndoBbox = new RectF(mTempLastBBox);
        undoItem.mUndoFontId = mTempLastFontID;
        undoItem.mUndoFontColor = mTempLastFontColor;
        undoItem.mUndoFontSize = mTempLastFontSize;
        undoItem.mUndoFieldName = mTempFieldName;
        undoItem.mUndoFieldFlags = mTempFieldFlags;
        undoItem.mUndoOptions = FormFillerUtil.cloneChoiceOptions(mTempOptions);
        undoItem.mUndoCheckedIndex = mTempLastCheckedIndex;

        return undoItem;
    }

    private PointF adjustTouchPointF(PointF pdfPointF, RectF annotRectF) {
        if (pdfPointF.x < annotRectF.left) {
            pdfPointF.x = annotRectF.left;
        }
        if (pdfPointF.x > annotRectF.right) {
            pdfPointF.x = annotRectF.right;
        }
        if (pdfPointF.y < annotRectF.bottom) {
            pdfPointF.y = annotRectF.bottom;
        }
        if (pdfPointF.y > annotRectF.top) {
            pdfPointF.y = annotRectF.top;
        }
        return pdfPointF;
    }

    @Override
    public boolean shouldViewCtrlDraw(Annot annot) {
        return true;
    }

    private PointF getPageViewOrigin(PDFViewCtrl pdfViewCtrl, int pageIndex, float x, float y) {
        PointF pagePt = new PointF(x, y);
        pdfViewCtrl.convertPageViewPtToDisplayViewPt(pagePt, pagePt, pageIndex);
        RectF rect = new RectF(0, 0, pagePt.x, pagePt.y);
        pdfViewCtrl.convertDisplayViewRectToPageViewRect(rect, rect, pageIndex);
        PointF originPt = new PointF(x - rect.width(), y - rect.height());
        return originPt;
    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        Annot annot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
        if (annot == null || annot.isEmpty() || !(annot instanceof Widget)) return;
        if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getCurrentAnnotHandler() != this)
            return;

        if (mIsLongPressTouchEvent) {
            onDrawByLongPress(annot, pageIndex, canvas);
        } else {
            onDrawBySingleTap(annot, pageIndex, canvas);
        }
    }

    private void onDrawBySingleTap(Annot annot, int pageIndex, Canvas canvas) {
        try {
            int index = annot.getPage().getIndex();
            if (index != pageIndex) return;

            if (!mIsEdittextOffset) {
                if (shouldShowInputSoft(annot)) {
                    if (mAdjustPosition) {
                        gotoPage(pageIndex, AppUtil.toRectF(annot.getRect()));
                    } else {
                        if (mOffset == 0) {
                            gotoPage(pageIndex, AppUtil.toRectF(annot.getRect()));
                        }
                    }
                }

                if (pageIndex != mPdfViewCtrl.getPageCount() - 1 &&
                        !(!mPdfViewCtrl.isContinuous() && (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_SINGLE ||
                                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING ||
                                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER))) {
                    resetDocViewerOffset();
                }

                if (AppKeyboardUtil.getKeyboardHeight(mParent) < AppDisplay.getRawScreenHeight() / 5
                        && (pageIndex == mPdfViewCtrl.getPageCount() - 1 ||
                        (!mPdfViewCtrl.isContinuous() && (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_SINGLE ||
                                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING ||
                                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER)))) {
                    resetDocViewerOffset();
                }
            }

            int fieldType = FormFillerUtil.getAnnotFieldType(annot);
            if (mFNModule != null) {
                if (fieldType != Field.e_TypePushButton) {
                    if (shouldShowInputSoft(annot)) {
                        mFNModule.showFormNavigationBarWithKeyBoard(getFNBottomPadding());
                    } else {
                        mFNModule.showFormNavigationBar();
                    }
                }
                if (((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot() == null) {
                    mFNModule.hide();
                }
            }
            canvas.save();
            canvas.setDrawFilter(new PaintFlagsDrawFilter(0, Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG));
            if (fieldType != Field.e_TypePushButton) {
                RectF bbox = AppUtil.toRectF(annot.getRect());
                mPdfViewCtrl.convertPdfRectToPageViewRect(bbox, bbox, pageIndex);
                bbox.sort();
                bbox.inset(-5, -5);

                canvas.drawLine(bbox.left, bbox.top, bbox.left, bbox.bottom, mPathPaint);
                canvas.drawLine(bbox.left, bbox.bottom, bbox.right, bbox.bottom, mPathPaint);
                canvas.drawLine(bbox.right, bbox.bottom, bbox.right, bbox.top, mPathPaint);
                canvas.drawLine(bbox.left, bbox.top, bbox.right, bbox.top, mPathPaint);

                if (mIsDateField && mDateFieldIcon != null) {
                    int bitmapWidth = mDateFieldIcon.getWidth();
                    int bitmapHeight = mDateFieldIcon.getHeight();

                    Rect srcRectF = new Rect(0, 0, bitmapWidth, bitmapHeight);
                    Rect dstRectF = new Rect((int) bbox.left, (int) bbox.top, (int) bbox.right, (int) bbox.bottom);
                    dstRectF.inset(5, 5);
                    float scale = (float) dstRectF.height() / bitmapHeight;
                    if (scale > 1) {
                        bitmapWidth *= scale;
                    }
                    dstRectF.left = dstRectF.right - bitmapWidth;
                    canvas.drawBitmap(mDateFieldIcon, srcRectF, dstRectF, mPathPaint);
                }
            }
            canvas.restore();
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void gotoPage(int pageIndex, RectF rect) {
        PointF viewpoint = new PointF(rect.left, rect.bottom);
        PointF point = new PointF(rect.left, rect.bottom);
        mPdfViewCtrl.convertPdfPtToPageViewPt(viewpoint, viewpoint, pageIndex);
        mPdfViewCtrl.convertPdfPtToPageViewPt(point, point, pageIndex);
        mPdfViewCtrl.convertPageViewPtToDisplayViewPt(viewpoint, viewpoint, pageIndex);

        int keyboardHeight = AppKeyboardUtil.getKeyboardHeight(mParent);
        if (keyboardHeight > AppDisplay.getRawScreenHeight() / 5) {
            int rawScreenHeight = AppDisplay.getRawScreenHeight();
            int navBarHeight = AppDisplay.getRealNavBarHeight();
            Rect parentRect = new Rect();
            mParent.getGlobalVisibleRect(parentRect);

            if (rawScreenHeight - viewpoint.y - parentRect.top < (keyboardHeight + AppDisplay.dp2px(116)) + navBarHeight) {
                mPageOffset = (int) (keyboardHeight - (rawScreenHeight - viewpoint.y - parentRect.top));

                if (mPageOffset != 0 && pageIndex == mPdfViewCtrl.getPageCount() - 1 ||
                        ((!mPdfViewCtrl.isContinuous() && mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_SINGLE) ||
                                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_FACING ||
                                mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_COVER)) {

                    PointF point1 = new PointF(0, mPdfViewCtrl.getPageViewHeight(pageIndex));
                    mPdfViewCtrl.convertPageViewPtToDisplayViewPt(point1, point1, pageIndex);
                    float screenHeight = AppDisplay.getActivityHeight();
                    if (point1.y <= screenHeight) {
                        int offset = mPageOffset + AppDisplay.dp2px(116) + navBarHeight;
                        setBottomOffset(offset);
                    }
                }

                PointF oriPoint = getPageViewOrigin(mPdfViewCtrl, pageIndex, point.x, point.y);
                mPdfViewCtrl.gotoPage(pageIndex,
                        oriPoint.x, oriPoint.y + mPageOffset + AppDisplay.dp2px(116) + navBarHeight);
                mAdjustPosition = false;
            } else {
                resetDocViewerOffset();
            }
        }
    }

    private RectF mBBoxInOnDraw = new RectF();
    private RectF mViewDrawRectInOnDraw = new RectF();
    private DrawFilter mDrawFilter = new PaintFlagsDrawFilter(0, Paint.ANTI_ALIAS_FLAG | Paint.FILTER_BITMAP_FLAG);

    private void onDrawByLongPress(Annot annot, int pageIndex, Canvas canvas) {
        try {
            int annotPageIndex = annot.getPage().getIndex();
            if (AppAnnotUtil.equals(mLastAnnot, annot) && annotPageIndex == pageIndex) {
                canvas.save();
                canvas.setDrawFilter(mDrawFilter);
                float thickness = thicknessOnPageView(pageIndex, DEFAULT_THICKNESS);
                mViewDrawRectInOnDraw.set(AppUtil.toRectF(annot.getRect()));
                mPdfViewCtrl.convertPdfRectToPageViewRect(mViewDrawRectInOnDraw, mViewDrawRectInOnDraw, pageIndex);
                mViewDrawRectInOnDraw.inset(thickness / 2f, thickness / 2f);
                if (mLastOper == OPER_SCALE_LT) {// SCALE
                    mBBoxInOnDraw.set(mLastPoint.x, mLastPoint.y, mViewDrawRectInOnDraw.right, mViewDrawRectInOnDraw.bottom);
                } else if (mLastOper == OPER_SCALE_T) {
                    mBBoxInOnDraw.set(mViewDrawRectInOnDraw.left, mLastPoint.y, mViewDrawRectInOnDraw.right, mViewDrawRectInOnDraw.bottom);
                } else if (mLastOper == OPER_SCALE_RT) {
                    mBBoxInOnDraw.set(mViewDrawRectInOnDraw.left, mLastPoint.y, mLastPoint.x, mViewDrawRectInOnDraw.bottom);
                } else if (mLastOper == OPER_SCALE_R) {
                    mBBoxInOnDraw.set(mViewDrawRectInOnDraw.left, mViewDrawRectInOnDraw.top, mLastPoint.x, mViewDrawRectInOnDraw.bottom);
                } else if (mLastOper == OPER_SCALE_RB) {
                    mBBoxInOnDraw.set(mViewDrawRectInOnDraw.left, mViewDrawRectInOnDraw.top, mLastPoint.x, mLastPoint.y);
                } else if (mLastOper == OPER_SCALE_B) {
                    mBBoxInOnDraw.set(mViewDrawRectInOnDraw.left, mViewDrawRectInOnDraw.top, mViewDrawRectInOnDraw.right, mLastPoint.y);
                } else if (mLastOper == OPER_SCALE_LB) {
                    mBBoxInOnDraw.set(mLastPoint.x, mViewDrawRectInOnDraw.top, mViewDrawRectInOnDraw.right, mLastPoint.y);
                } else if (mLastOper == OPER_SCALE_L) {
                    mBBoxInOnDraw.set(mLastPoint.x, mViewDrawRectInOnDraw.top, mViewDrawRectInOnDraw.right, mViewDrawRectInOnDraw.bottom);
                }
                mBBoxInOnDraw.inset(-thickness / 2f, -thickness / 2f);
                if (mLastOper == OPER_TRANSLATE || mLastOper == OPER_DEFAULT) {// TRANSLATE or DEFAULT
                    mBBoxInOnDraw = AppUtil.toRectF(annot.getRect());
                    mPdfViewCtrl.convertPdfRectToPageViewRect(mBBoxInOnDraw, mBBoxInOnDraw, pageIndex);
                    float dx = mLastPoint.x - mDownPoint.x;
                    float dy = mLastPoint.y - mDownPoint.y;

                    mBBoxInOnDraw.offset(dx, dy);
                }
                DocumentManager documentManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager();
                if (annot == documentManager.getCurrentAnnot()) {
                    float radius = mCtlPtRadius;
                    if (AnnotPermissionUtil.canModifyAnnot(documentManager, annot)) {
                        drawControlPoints(canvas, mBBoxInOnDraw);
                    } else {
                        radius = 0;
                    }
                    // add Control Imaginary
                    drawControlImaginary(canvas, mBBoxInOnDraw, radius);
                }
                canvas.restore();
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private void drawControlPoints(Canvas canvas, RectF rectBBox) {
        PointF[] ctlPts = calculateControlPoints(rectBBox);
        mCtlPtPaint.setStrokeWidth(mCtlPtLineWidth);
        for (PointF ctlPt : ctlPts) {
            mCtlPtPaint.setColor(Color.WHITE);
            mCtlPtPaint.setStyle(Paint.Style.FILL);
            canvas.drawCircle(ctlPt.x, ctlPt.y, mCtlPtRadius, mCtlPtPaint);
            mCtlPtPaint.setColor(DEFAULT_COLOR);
            mCtlPtPaint.setStyle(Paint.Style.STROKE);
            canvas.drawCircle(ctlPt.x, ctlPt.y, mCtlPtRadius, mCtlPtPaint);
        }
    }

    private Path mImaginaryPath = new Path();

    private void drawControlImaginary(Canvas canvas, RectF rectBBox, float radius) {
        PointF[] ctlPts = calculateControlPoints(rectBBox);
        mFrmPaint.setStrokeWidth(mCtlPtLineWidth);
        mImaginaryPath.reset();
        // set path
        pathAddLine(mImaginaryPath, ctlPts[0].x + radius, ctlPts[0].y, ctlPts[1].x - radius, ctlPts[1].y);
        pathAddLine(mImaginaryPath, ctlPts[1].x + radius, ctlPts[1].y, ctlPts[2].x - radius, ctlPts[2].y);
        pathAddLine(mImaginaryPath, ctlPts[2].x, ctlPts[2].y + radius, ctlPts[3].x, ctlPts[3].y - radius);
        pathAddLine(mImaginaryPath, ctlPts[3].x, ctlPts[3].y + radius, ctlPts[4].x, ctlPts[4].y - radius);
        pathAddLine(mImaginaryPath, ctlPts[4].x - radius, ctlPts[4].y, ctlPts[5].x + radius, ctlPts[5].y);
        pathAddLine(mImaginaryPath, ctlPts[5].x - radius, ctlPts[5].y, ctlPts[6].x + radius, ctlPts[6].y);
        pathAddLine(mImaginaryPath, ctlPts[6].x, ctlPts[6].y - radius, ctlPts[7].x, ctlPts[7].y + radius);
        pathAddLine(mImaginaryPath, ctlPts[7].x, ctlPts[7].y - radius, ctlPts[0].x, ctlPts[0].y + radius);

        canvas.drawPath(mImaginaryPath, mFrmPaint);
    }

    private void pathAddLine(Path path, float start_x, float start_y, float end_x, float end_y) {
        path.moveTo(start_x, start_y);
        path.lineTo(end_x, end_y);
    }

    private RectF mViewDrawRect = new RectF(0, 0, 0, 0);
    private RectF mDocViewerBBox = new RectF(0, 0, 0, 0);

    protected void onDrawForControls(Canvas canvas) {
        Annot curAnnot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
        if (curAnnot instanceof Widget
                && ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getCurrentAnnotHandler() == this) {
            if (mIsLongPressTouchEvent) {
                try {
                    int annotPageIndex = curAnnot.getPage().getIndex();
                    if (mPdfViewCtrl.isPageVisible(annotPageIndex)) {
                        float thickness = thicknessOnPageView(annotPageIndex, DEFAULT_THICKNESS);
                        mViewDrawRect.set(AppUtil.toRectF(curAnnot.getRect()));
                        mPdfViewCtrl.convertPdfRectToPageViewRect(mViewDrawRect, mViewDrawRect, annotPageIndex);
                        mViewDrawRect.inset(thickness / 2f, thickness / 2f);
                        if (mLastOper == OPER_SCALE_LT) {
                            mDocViewerBBox.left = mLastPoint.x;
                            mDocViewerBBox.top = mLastPoint.y;
                            mDocViewerBBox.right = mViewDrawRect.right;
                            mDocViewerBBox.bottom = mViewDrawRect.bottom;
                        } else if (mLastOper == OPER_SCALE_T) {
                            mDocViewerBBox.left = mViewDrawRect.left;
                            mDocViewerBBox.top = mLastPoint.y;
                            mDocViewerBBox.right = mViewDrawRect.right;
                            mDocViewerBBox.bottom = mViewDrawRect.bottom;
                        } else if (mLastOper == OPER_SCALE_RT) {
                            mDocViewerBBox.left = mViewDrawRect.left;
                            mDocViewerBBox.top = mLastPoint.y;
                            mDocViewerBBox.right = mLastPoint.x;
                            mDocViewerBBox.bottom = mViewDrawRect.bottom;
                        } else if (mLastOper == OPER_SCALE_R) {
                            mDocViewerBBox.left = mViewDrawRect.left;
                            mDocViewerBBox.top = mViewDrawRect.top;
                            mDocViewerBBox.right = mLastPoint.x;
                            mDocViewerBBox.bottom = mViewDrawRect.bottom;
                        } else if (mLastOper == OPER_SCALE_RB) {
                            mDocViewerBBox.left = mViewDrawRect.left;
                            mDocViewerBBox.top = mViewDrawRect.top;
                            mDocViewerBBox.right = mLastPoint.x;
                            mDocViewerBBox.bottom = mLastPoint.y;
                        } else if (mLastOper == OPER_SCALE_B) {
                            mDocViewerBBox.left = mViewDrawRect.left;
                            mDocViewerBBox.top = mViewDrawRect.top;
                            mDocViewerBBox.right = mViewDrawRect.right;
                            mDocViewerBBox.bottom = mLastPoint.y;
                        } else if (mLastOper == OPER_SCALE_LB) {
                            mDocViewerBBox.left = mLastPoint.x;
                            mDocViewerBBox.top = mViewDrawRect.top;
                            mDocViewerBBox.right = mViewDrawRect.right;
                            mDocViewerBBox.bottom = mLastPoint.y;
                        } else if (mLastOper == OPER_SCALE_L) {
                            mDocViewerBBox.left = mLastPoint.x;
                            mDocViewerBBox.top = mViewDrawRect.top;
                            mDocViewerBBox.right = mViewDrawRect.right;
                            mDocViewerBBox.bottom = mViewDrawRect.bottom;
                        }
                        mDocViewerBBox.inset(-thickness / 2f, -thickness / 2f);
                        if (mLastOper == OPER_TRANSLATE || mLastOper == OPER_DEFAULT) {
                            mDocViewerBBox = AppUtil.toRectF(curAnnot.getRect());
                            mPdfViewCtrl.convertPdfRectToPageViewRect(mDocViewerBBox, mDocViewerBBox, annotPageIndex);

                            float dx = mLastPoint.x - mDownPoint.x;
                            float dy = mLastPoint.y - mDownPoint.y;

                            mDocViewerBBox.offset(dx, dy);
                        }

                        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mDocViewerBBox, mDocViewerBBox, annotPageIndex);
                        mAnnotationMenu.update(mDocViewerBBox);
                        if (mPropertyBar.isShowing()) {
                            RectF rectF = new RectF(mDocViewerBBox);
                            mPropertyBar.update(rectF);
                        }
                    }
                } catch (PDFException e) {
                    e.printStackTrace();
                }
            } else if (mLastPopupPdfPoint != null) {
                RectF rectF = getPopupMenuRect(mLastPopupPdfPoint);
                mAnnotationMenu.update(rectF);
            } else {
                if (mJSActionCallback != null) {
                    if (mPdfViewCtrl.isPageVisible(mLastPageIndex)) {
                        try {
                            RectF annotRect = AppUtil.toRectF(curAnnot.getRect());
                            RectF displayRect = new RectF();
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRect, displayRect, mLastPageIndex);
                            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(displayRect, displayRect, mLastPageIndex);
                            mAnnotationMenu.update(displayRect);
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    }
                }
            }
        }
    }

    @Override
    public void addAnnot(int pageIndex, AnnotContent contentSupplier, boolean addUndo,
                         Event.Callback result) {
    }

    @Override
    public void modifyAnnot(Annot annot, AnnotContent content, boolean addUndo, Event.Callback result) {
    }

    @Override
    public void removeAnnot(Annot annot, boolean addUndo, Event.Callback result) {
    }

    protected boolean shouldShowInputSoft(Annot annot) {
        if (annot == null) return false;
        if (!(annot instanceof Widget)) return false;
        int type = FormFillerUtil.getAnnotFieldType(annot);
        try {
            if ((type == Field.e_TypeTextField) ||
                    (type == Field.e_TypeComboBox && (((Widget) annot).getField().getFlags() & Field.e_FlagComboEdit) != 0))
                return true;

        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private void resetDocViewerOffset() {
        if (mPageOffset != 0) {
            mPageOffset = 0;
            setBottomOffset(0);
        }
    }

    private void setBottomOffset(int offset) {
        if (mOffset == -offset)
            return;
        mOffset = -offset;
        mPdfViewCtrl.layout(0, mOffset, mPdfViewCtrl.getWidth(), mPdfViewCtrl.getHeight() + mOffset);
    }

    protected boolean onKeyBack() {
        Annot curAnnot = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot();
        try {
            if (curAnnot == null) return false;
            if (curAnnot.getType() != Annot.e_Widget) return false;
            Field field = ((Widget) curAnnot).getField();
            if (field == null || field.isEmpty()) return false;
            int type = field.getType();
            if (type != Field.e_TypeSignature && type != Field.e_TypeUnknown) {
                ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().setCurrentAnnot(null);
                navigationDismiss();
                return true;
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }

        return false;
    }

    protected void onLayoutChange(View v, int newWidth, int newHeight, int oldWidth, int oldHeight) {
        if (newWidth != oldWidth || newHeight != oldHeight) {
            mAdjustPosition = true;

            if (mLastAnnot != null && !mLastAnnot.isEmpty()) {
                if (mIsEdittextOffset) {
                    RectF pageViewRect = new RectF(mTempLastBBox);
                    mPdfViewCtrl.convertPdfRectToPageViewRect(pageViewRect, pageViewRect, mLastPageIndex);
                    mTempLastDisplayBBox = new RectF(pageViewRect);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(mTempLastDisplayBBox, mTempLastDisplayBBox, mLastPageIndex);
                }
            }
        }
    }

    private ChoiceOptionsDialog mChoiceOptionsDialog;

    private void showOptionsDialog(final Annot annot,
                                   ChoiceOptionsAdapter.SelectMode selectMode,
                                   ArrayList<ChoiceItemInfo> itemInfos,
                                   IResult<ArrayList<ChoiceItemInfo>, Object, Object> pickOptionsCallback) {
        if (AppDisplay.isPad()) {
            mPropertyBar.setDismissWithResetProperties(false);
            mPropertyBar.dismiss();
        }

        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        Activity context = uiExtensionsManager.getAttachedActivity();
        if (context == null) return;

        uiExtensionsManager.stopHideToolbarsTimer();
        mChoiceOptionsDialog = new ChoiceOptionsDialog(context, mPdfViewCtrl);
        mChoiceOptionsDialog.init(selectMode, itemInfos, pickOptionsCallback);
        mChoiceOptionsDialog.showDialog();
    }

    void onFontColorChanged(int color) {
        try {
            if (mLastAnnot != null && !mLastAnnot.isEmpty()) {
                Field field = ((Widget) mLastAnnot).getField();
                DefaultAppearance da;
                if (field.getType() == Field.e_TypeRadioButton)
                    da = ((Widget) mLastAnnot).getControl().getDefaultAppearance();
                else
                    da = field.getDefaultAppearance();

                if (color != da.getText_color()) {
                    FormFillerContent fillerContent = new FormFillerContent(mPdfViewCtrl, (Widget) mLastAnnot);
                    fillerContent.mFontColor = color;
                    modifyAnnot(fillerContent, false, false, null);
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void onFontSizeChanged(float size) {
        try {
            if (mLastAnnot != null && !mLastAnnot.isEmpty()) {
                Field field = ((Widget) mLastAnnot).getField();
                DefaultAppearance da;
                if (field.getType() == Field.e_TypeRadioButton)
                    da = ((Widget) mLastAnnot).getControl().getDefaultAppearance();
                else
                    da = field.getDefaultAppearance();
                if (size != da.getText_size()) {
                    FormFillerContent fillerContent = new FormFillerContent(mPdfViewCtrl, (Widget) mLastAnnot);
                    fillerContent.mFontSize = size;
                    modifyAnnot(fillerContent, false, false, null);
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void onFontNameChanged(String name) {
        try {
            if (mLastAnnot != null && !mLastAnnot.isEmpty()) {
                Field field = ((Widget) mLastAnnot).getField();
                DefaultAppearance da;
                if (field.getType() == Field.e_TypeRadioButton)
                    da = ((Widget) mLastAnnot).getControl().getDefaultAppearance();
                else
                    da = field.getDefaultAppearance();

                int fontId = mTextUtil.getSupportFontID(name);
                if (fontId != mTextUtil.getSupportFontID(da, mPdfViewCtrl.getDoc())) {
                    FormFillerContent fillerContent = new FormFillerContent(mPdfViewCtrl, (Widget) mLastAnnot);
                    fillerContent.mFontId = fontId;
                    modifyAnnot(fillerContent, false, false, null);
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void onFieldNameChanged(String name) {
        try {
            if (mLastAnnot != null && !mLastAnnot.isEmpty()) {
                String fieldName = ((Widget) mLastAnnot).getField().getName();
                if (!name.equals(fieldName)) {
                    mIsModify = true;
                    mLastFieldName = name;
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    public void setPropertyBar(PropertyBar propertyBar) {
        mPropertyBar = propertyBar;
    }

    protected interface FillerFocusEventListener {
        void focusGotOnControl(Control control, String filedValue);

        void focusLostFromControl(Control control, String filedValue);
    }

    protected abstract static class FillerPopupMenuListener {
        void appendPopupMenuItem(int itemId, String defaultString) {
        }

        void enablePopupMenuItem(int itemId, boolean isEnabled) {
        }

        void showPopupMenu(com.foxit.sdk.common.fxcrt.PointF point) {
        }

        void setClipboardText(String text) {
        }

        String getClipboardText() {
            return null;
        }
    }

    private void setPropChekced(boolean isSelected) {
        mPropCheckTrack.setSelected(isSelected);
        RelativeLayout.LayoutParams params = (RelativeLayout.LayoutParams) mPropCheckThumb.getLayoutParams();
        if (params == null) return;
        params.removeRule(isSelected ? RelativeLayout.ALIGN_PARENT_LEFT : RelativeLayout.ALIGN_PARENT_RIGHT);
        params.addRule(isSelected ? RelativeLayout.ALIGN_PARENT_RIGHT : RelativeLayout.ALIGN_PARENT_LEFT, RelativeLayout.TRUE);
        mPropCheckThumb.setLayoutParams(params);
    }

    private ColorStateList getPropTrackColorStateList() {
        int disabled = AppResource.getColor(mContext, R.color.p1);
        int selected = ThemeConfig.getInstance(mContext).getPrimaryColor();
        int normal = AppResource.getColor(mContext, R.color.p1);
        return AppResource.createColorStateList(selected, disabled, normal);
    }

    void onThemeColorChanged() {
        if (mPropertyBar != null)
            mPropertyBar.updateTheme();
        if (mChoiceOptionsDialog != null)
            mChoiceOptionsDialog.themeColorChanged();
        if (mFileSelectDialog != null)
            mFileSelectDialog.dismiss();
    }

    void prepareFillerPopupMenu() {
        mMenuText.clear();
        int size = mMenuItemArray.size();
        for (int i = 0; i < size; i++) {
            mMenuText.add(mMenuItemArray.keyAt(i));
        }
        mAnnotationMenu.setMenuItems(mMenuText);
        initAnnotationMenuListener();
    }

    private void initAnnotationMenuListener() {
        if (mMenuClickListener == null)
            mMenuClickListener = new AnnotMenu.ClickListener() {
                @Override
                public void onAMClick(int btType) {
                    try {
                        Annot annot = mLastAnnot;
                        if (btType == AnnotMenu.AM_BT_DELETE) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_Annot_PopMenu_Delete);
                            if (annot == ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().getCurrentAnnot()) {
                                deleteAnnot(annot, true, null);
                            }
                        } else if (btType == AnnotMenu.AM_BT_STYLE) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).onUIInteractElementClicked(IUIInteractionEventListener.Reading_Annot_PopMenu_Appearance);
                            int pageIndex = annot.getPage().getIndex();
                            RectF annotRectF = AppUtil.toRectF(annot.getRect());
                            mPdfViewCtrl.convertPdfRectToPageViewRect(annotRectF, annotRectF, pageIndex);
                            mPdfViewCtrl.convertPageViewRectToDisplayViewRect(annotRectF, annotRectF, pageIndex);

                            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                            RectF rectF = new RectF();
                            if (SystemUiHelper.getInstance().isStatusBarShown(uiExtensionsManager.getAttachedActivity())) {
                                rectF.set(AppUtil.toGlobalVisibleRectF(uiExtensionsManager.getRootView(), annotRectF));
                            } else {
                                rectF.set(annotRectF);
                            }
                            if (AppDisplay.isPad()) {
                                int rootWidth = uiExtensionsManager.getRootView().getWidth();
                                int rootHeight = uiExtensionsManager.getRootView().getHeight();
                                int maxHeight = Math.min(rootWidth, rootHeight) / 2;
                                mPropertyBar.setMaxHeight(maxHeight);
                            }
                            mPropertyBar.show(rectF, false);
                        } else if (btType == AnnotMenu.PM_BT_CUT) {
                            mFormFiller.onCut();
                        } else if (btType == AnnotMenu.PM_BT_COPY) {
                            mFormFiller.onCopy();
                        } else if (btType == AnnotMenu.PM_BT_PASTE) {
                            mFormFiller.onPaste();
                        } else if (btType == AnnotMenu.PM_BT_DELETE) {
                            mFormFiller.onDelete();
                        } else if (btType == AnnotMenu.PM_BT_SELECT_ALL) {
                            mFormFiller.onSelectAll();
                            mPdfViewCtrl.postDelayed(new Runnable() {
                                @Override
                                public void run() {
                                    try {
                                        PDFPage page = mLastAnnot.getPage();
                                        mFormFiller.onRButtonDown(page, mLastLongPressFxPoint, 0);
                                        mFormFiller.onRButtonUp(page, mLastLongPressFxPoint, 0);
                                    } catch (PDFException e) {
                                        e.printStackTrace();
                                    }
                                }
                            }, 50);
                        } else if (btType == AnnotMenu.AM_BT_GALERY) {
                            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                            AppIntentUtil.selectImageFromGallery(uiExtensionsManager.getAttachedActivity(), ActRequestCode.REQ_FILE_FROM_GALLERY);
                        } else if (btType == AnnotMenu.AM_BT_CAMERA) {
                            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                            AppIntentUtil.selectImageFromCamera(uiExtensionsManager,
                                    mCaptureImgPath, ActRequestCode.REQ_CAMERA_PERMISSION, ActRequestCode.REQ_FILE_FROM_CAMERA);
                        } else if (btType == AnnotMenu.AM_BT_FILE) {
                            showSelectFileDialog();
                        } else if (btType == AnnotMenu.AM_BT_CLEAR) {
                            modifyImageField(true, "", mLastAnnot);
                        }
                        mAnnotationMenu.dismiss();
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }
            };
        mAnnotationMenu.setListener(mMenuClickListener);
    }

    private void showSelectFileDialog() {
        mFileSelectDialog = getFileSelectDialog();
        mFileSelectDialog.init(new FileFilter() {
            @Override
            public boolean accept(File pathname) {
                return !(pathname.isHidden() || !AppFileUtil.canRead(pathname))
                        && !(pathname.isFile() && !isPictures(pathname));
            }
        }, true);

        mFileSelectDialog.setListener(new MatchDialog.DialogListener() {
            @Override
            public void onResult(long btType) {
            }

            @Override
            public void onBackClick() {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
                mFileSelectDialog.dismiss();
            }

            @Override
            public void onTitleRightButtonClick() {
                if (mFileSelectDialog.getSelectedFiles().size() > 0) {
                    String filepath = mFileSelectDialog.getSelectedFiles().get(0).path;
                    modifyImageField(false, filepath, mLastAnnot);
                }
                mFileSelectDialog.dismiss();
            }
        });
        mFileSelectDialog.setOnDLCancelListener(new DialogInterface.OnCancelListener() {
            @Override
            public void onCancel(DialogInterface dialog) {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
            }
        });

        mFileSelectDialog.resetWH();
        mFileSelectDialog.showDialog();
    }

    private UIFileSelectDialog getFileSelectDialog() {
        if (mFileSelectDialog == null) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            mFileSelectDialog = new UIFileSelectDialog(uiExtensionsManager.getAttachedActivity());
            mFileSelectDialog.init(new FileFilter() {
                @Override
                public boolean accept(File pathname) {
                    return !(pathname.isHidden() || !AppFileUtil.canRead(pathname))
                            && !(pathname.isFile() && !isPictures(pathname));
                }
            }, true);
            mFileSelectDialog.setCanceledOnTouchOutside(true);
        } else {
            AppThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                @Override
                public void run() {
                    mFileSelectDialog.notifyDataSetChanged();
                }
            });
        }
        return mFileSelectDialog;
    }

    private boolean isPictures(File file) {
        String pathName = file.getName().toLowerCase();
        return pathName.endsWith(".bmp")
                || pathName.endsWith(".jpg")
                || pathName.endsWith(".png")
                || pathName.endsWith(".gif")
                || pathName.endsWith(".tif")
                || pathName.endsWith(".jpeg");
    }


    private boolean isEditingText() {
        return mEditView != null && mEditView.getParent() != null;
    }

    void onConfigurationChanged(Configuration newConfig) {
        int orientation = newConfig.orientation;
        if (!AppDisplay.isPad()) {
            if (isEditingText() && mLastOrientation != orientation) {
                if (orientation == Configuration.ORIENTATION_LANDSCAPE) {
                    UIExtensionsManager extensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                    int keyboardHeight = AppKeyboardUtil.getKeyboardHeight(extensionsManager.getRootView());
                    if (keyboardHeight > AppDisplay.getRawScreenHeight() / 4 && mAnnotationMenu.isShowing())
                        mAnnotationMenu.dismiss();
                } else {
                    if (!mAnnotationMenu.isShowing() && mLastPopupPdfPoint != null) {
                        RectF rectF = getPopupMenuRect(mLastPopupPdfPoint);
                        mAnnotationMenu.show(rectF);
                    }
                }
            }

            mLastOrientation = orientation;
        } else {
            if (mFNModule != null) {
                mFNModule.onConfigurationChanged();
            }
        }

        if (mFileSelectDialog != null && mFileSelectDialog.isShowing()) {
            mFileSelectDialog.resetWH();
            mFileSelectDialog.showDialog();
        }
    }

    private class JSActionCallback implements AnnotActionHandler.IJSActionCallback {

        private final int mPageIndex;
        private final Widget mWidget;

        public JSActionCallback(int pageIndex, Widget widget) {
            mPageIndex = pageIndex;
            mWidget = widget;
            prepareImageMenu(mWidget);
        }

        @Override
        public void browseFile() {
            showImageMenu();
        }

        @Override
        public void browseFile(boolean is_open_dialog, String file_format, String file_filter) {
            showImageMenu();
        }

        private void showImageMenu() {
            if (mWidget != null && !mWidget.isEmpty()) {
                try {
                    RectF annotRect = AppUtil.toRectF(mWidget.getRect());
                    RectF displayRect = new RectF();
                    mPdfViewCtrl.convertPdfRectToPageViewRect(annotRect, displayRect, mPageIndex);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(displayRect, displayRect, mPageIndex);
                    mAnnotationMenu.show(displayRect);
                } catch (PDFException e) {
                    e.printStackTrace();
                }
            }
        }
    }

    void onActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
        if (resultCode == Activity.RESULT_OK) {
            String path = null;
            if (requestCode == ActRequestCode.REQ_FILE_FROM_GALLERY && data != null) {
                path = AppFileUtil.getFilePathFromUri(mContext, data.getData());
            } else if (requestCode == ActRequestCode.REQ_FILE_FROM_CAMERA) {
                path = mCaptureImgPath;
            }

            if (!AppUtil.isEmpty(path)) {
                if (mLastAnnot == null || mLastAnnot.isEmpty()) return;
                modifyImageField(false, path, mLastAnnot);
            }
        } else {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
        }
    }

    void onRequestPermissionsResult(int requestCode, String[] permissions, int[] grantResults) {
        if (verifyPermissions(grantResults)) {
            if (requestCode == ActRequestCode.REQ_CAMERA_PERMISSION) {
                UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
                AppIntentUtil.selectImageFromCamera(uiExtensionsManager,
                        mCaptureImgPath, ActRequestCode.REQ_CAMERA_PERMISSION, ActRequestCode.REQ_FILE_FROM_CAMERA);
            }
        } else {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
            UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_permission_denied));
        }
    }

    private boolean verifyPermissions(int[] grantResults) {
        for (int grantResult : grantResults) {
            if (grantResult != PackageManager.PERMISSION_GRANTED) {
                return false;
            }
        }
        return true;
    }

    private void showDatePickerDialog() {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();

        final Calendar calendar = Calendar.getInstance();
        Date date = mDateFieldValue == null ? new Date() : mDateFieldValue;
        calendar.setTime(date);
        int year = calendar.get(Calendar.YEAR);
        if (year == 1970) {
            year = Calendar.getInstance().get(Calendar.YEAR);
        }
        DatePickerDialog datePickerDialog = new DatePickerDialog(uiExtensionsManager.getAttachedActivity(),
                new DatePickerDialog.OnDateSetListener() {
                    @Override
                    public void onDateSet(DatePicker view, int year, int month, int dayOfMonth) {
                        calendar.set(year, month, dayOfMonth);
                        Date pickDate = calendar.getTime();
                        try {
                            resetField((Widget) mLastAnnot, true, false);
                            DateFormat dateFormat = new SimpleDateFormat(mDatePattern, Locale.ENGLISH);
                            char[] date = dateFormat.format(pickDate).toCharArray();
                            for (char c : date) {
                                mFormFiller.onChar(c, 0);
                            }
                            mDateFieldValue = pickDate;
                        } catch (Exception ignored) {
                        }
                    }
                }, year,
                calendar.get(Calendar.MONTH),
                calendar.get(Calendar.DAY_OF_MONTH));
        datePickerDialog.show();
    }

    private IAudioPlayView mAudioPlayView;
    private FxProgressDialog mProgressDlg;

    private void openSound(final Annot annot, PDFObject soundObj) {
        showProgressDlg();

        final String cachePath = getCachePath(annot);
        if (TextUtils.isEmpty(cachePath) || !new File(cachePath).exists()) {
            OpenSoundTask openSoundTask = new OpenSoundTask(soundObj, cachePath, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    dismissProgressDlg();
                    if (success) {
                        openFile(cachePath);
                    } else {
                        UIToast.getInstance(mContext).show(R.string.rv_document_open_failed);
                    }
                }
            });
            mPdfViewCtrl.addTask(openSoundTask);
        } else {
            dismissProgressDlg();
            openFile(cachePath);
        }
    }

    private void openFile(String path) {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);

        IAudioPlayView playView = createAudioPlayView();
        if (playView != null)
            playView.startPlay(path);
    }

    IAudioPlayView createAudioPlayView() {
        if (mAudioPlayView == null) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            SoundModule soundModule = (SoundModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_SOUND);
            if (soundModule != null) {
                mAudioPlayView = soundModule.createPlayView();
            }
        }
        return mAudioPlayView;
    }

    private String getCachePath(Annot annot) {
        String parent = AppFileUtil.getDiskCachePath(mContext)
                + File.separatorChar
                + "form_sound"
                + File.separator;
        File file = new File(parent);
        if (!file.exists()) {
            file.mkdirs();
        }
        String child = AppAnnotUtil.getAnnotUniqueID(annot) + ".wav";
        return parent + child;
    }

    private void showProgressDlg() {
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        if (mProgressDlg == null && uiExtensionsManager.getAttachedActivity() != null) {
            mProgressDlg = new FxProgressDialog(uiExtensionsManager.getAttachedActivity(), AppResource.getString(mContext, R.string.fx_string_opening));
        }
        if (mProgressDlg != null && !mProgressDlg.isShowing()) {
            mProgressDlg.show();
        }
    }

    private void dismissProgressDlg() {
        if (mProgressDlg != null && mProgressDlg.isShowing()) {
            mProgressDlg.dismiss();
            mProgressDlg = null;
        }
    }

}
