/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.common;

import android.annotation.SuppressLint;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapShader;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Rect;
import android.graphics.RectF;
import android.graphics.Shader;
import android.graphics.drawable.ShapeDrawable;
import android.graphics.drawable.shapes.RectShape;
import android.graphics.drawable.shapes.RoundRectShape;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;

import com.foxit.uiextensions.R;
import com.foxit.uiextensions.utils.AppResource;

public class UIMagnifierView extends View {
    //default scale factor
    private static final int FACTOR = 2;
    private static final int BOUND_W = 4;
    //default radius
    private static int mDefaultRadius = 300;
    private final Matrix mMatrix = new Matrix();
    private  ShapeDrawable mDrawable;
    private  ShapeDrawable mBoundDrawable;
    private int mTopBarHeight;
    private int mTranslateY = -mDefaultRadius / 3;
    private int mBoundPainterColorId;
    private View mTargetView;
    private RectF mBoundDrawableInset;

    public UIMagnifierView(Context context) {
        super(context);
        initView();
    }

    private void initView() {
        mDrawable = new ShapeDrawable(new RectShape());
        float[] outerR = new float[] { 8, 8, 8, 8, 8, 8, 8, 8 };
        mDefaultRadius = (int) AppResource.getDimension(getContext(),R.dimen.ux_margin_184dp);
        mTranslateY = -mDefaultRadius / 3;
        mBoundDrawableInset = new RectF(BOUND_W, BOUND_W, mDefaultRadius + BOUND_W, mDefaultRadius / 2f + BOUND_W);
        float[] innerR = new float[] { 8, 8, 8, 8, 8, 8, 8, 8 };
        mBoundDrawable = new ShapeDrawable(new RoundRectShape(outerR, mBoundDrawableInset, innerR));
        setBoundPainterColor(R.color.ux_color_text_disabled);
        mTopBarHeight = (int) AppResource.getDimension(getContext(),R.dimen.ux_margin_50dp);
        mOffsetY = (int) AppResource.getDimension(getContext(),R.dimen.ux_margin_20dp);
    }

    private void  createShader(int x, int y) {
        if (mTargetView == null)return;
        //get Page bitmap
        ViewGroup docView = (ViewGroup) mTargetView;

        docView.setDrawingCacheEnabled(true);
        docView.buildDrawingCache();
        Bitmap bmp = docView.getDrawingCache();
        Bitmap bitmap = Bitmap.createBitmap(bmp, x - mDefaultRadius / 2, y - mDefaultRadius / 4, mDefaultRadius, mDefaultRadius / 2);
        docView.destroyDrawingCache();

        //bitmapShader
        BitmapShader shader = new BitmapShader(
                Bitmap.createScaledBitmap(bitmap, bitmap.getWidth()*FACTOR,
                        bitmap.getHeight()*FACTOR, true), Shader.TileMode.CLAMP, Shader.TileMode.CLAMP);

        mDrawable.getPaint().setShader(shader);
    }

    @SuppressLint("ClickableViewAccessibility")
    @Override
    public boolean onTouchEvent(MotionEvent event) {
        return onTouchEvent((int)event.getX(), (int)event.getY());
    }

    private Rect mDrawableRect;
    private int mOffsetY;

    public boolean onTouchEvent(int x, int y) {
        if (mTargetView != null){
            updateViewData();
            if (x < mDefaultRadius / 2 + BOUND_W) {
                x = mDefaultRadius / 2 + BOUND_W;
            }
            if (x > mTargetView.getWidth() - (mDefaultRadius / 2 + BOUND_W)) {
                x = mTargetView.getWidth() - (mDefaultRadius / 2 + BOUND_W);
            }
            if (y < mDefaultRadius / 2 + BOUND_W) {
                y = mDefaultRadius / 2 + BOUND_W;
            }
            if (y > mTargetView.getHeight() - (mDefaultRadius / 4 + BOUND_W)) {
                y = mTargetView.getHeight() - (mDefaultRadius / 4 + BOUND_W) ;
            }

            createShader(x, y);

            //start position to draw shader
            mMatrix.setTranslate(-mDefaultRadius / 2f, mTranslateY);
            mDrawable.getPaint().getShader().setLocalMatrix(mMatrix);

            //out bound rectangle
            mDrawable.setBounds(x - mDefaultRadius / 2, y - mOffsetY - mDefaultRadius / 2, x + mDefaultRadius / 2, y - mOffsetY);
            mBoundDrawable.getPaint().setStyle(Paint.Style.STROKE);
            mBoundDrawable.getPaint().setStrokeWidth(BOUND_W);
            mBoundDrawable.getPaint().setColor(AppResource.getColor(getContext(),mBoundPainterColorId));
            mBoundDrawable.getPaint().setStrokeJoin(Paint.Join.ROUND);

            if (mDrawableRect == null){
                mDrawableRect = new Rect();
            }
            mDrawable.copyBounds(mDrawableRect);
            mDrawableRect.inset(-BOUND_W,-BOUND_W);
            mBoundDrawableInset.set(mBoundDrawableInset.left, mBoundDrawableInset.top,mDefaultRadius + BOUND_W,  mDefaultRadius / 2f + BOUND_W);
            mBoundDrawable.setBounds(mDrawableRect);

            invalidate();
        }

        return false;
    }

    @Override
    public void onDraw(Canvas canvas) {
        super.onDraw(canvas);
        mDrawable.draw(canvas);
        mBoundDrawable.draw(canvas);
    }

    public void setBoundPainterColor(int id){
        mBoundPainterColorId = id;
    }

    public void setTargetView(View view){
        mTargetView = view;
    }

    private void updateViewData() {
        if (mTargetView != null){
            if (300 > mTargetView.getWidth() * (1/3f)){
                mDefaultRadius = Math.min(mDefaultRadius,300);
            }else {
                mDefaultRadius = Math.max(mDefaultRadius,300);
            }
            mTranslateY = -mDefaultRadius / 3;
        }
    }
}
