/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.caret;

import android.app.Activity;
import android.app.Dialog;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.PointF;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffXfermode;
import android.graphics.Rect;
import android.graphics.RectF;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.view.inputmethod.EditorInfo;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.DateTime;
import com.foxit.sdk.common.Progressive;
import com.foxit.sdk.common.Range;
import com.foxit.sdk.pdf.PDFPage;
import com.foxit.sdk.pdf.TextPage;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.annots.Caret;
import com.foxit.sdk.pdf.annots.StrikeOut;
import com.foxit.sdk.pdf.graphics.GraphicsObject;
import com.foxit.sdk.pdf.graphics.TextObject;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.uiextensions.IUIInteractionEventListener;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.common.EditAnnotEvent;
import com.foxit.uiextensions.annots.common.EditAnnotTask;
import com.foxit.uiextensions.annots.textmarkup.TextMarkupContentAbs;
import com.foxit.uiextensions.annots.textmarkup.TextSelector;
import com.foxit.uiextensions.annots.textmarkup.strikeout.StrikeoutEvent;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IToolSupply;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolItemBean;
import com.foxit.uiextensions.controls.toolbar.ToolProperty;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.impl.ToolSupplyImpl;
import com.foxit.uiextensions.controls.toolbar.impl.UIColorItem;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.SystemUiHelper;

public class CaretToolHandler implements ToolHandler {
    private final Context mContext;
    private final PDFViewCtrl mPdfViewCtrl;
    private final UIExtensionsManager mUiextensionsManager;

    private PropertyBar.PropertyChangeListener mPropertyChangeListener;
    private final PropertyBar mPropertyBar;

    private boolean mIsInsertTextModule;
    private int[] mColors;
    int mColor;
    int mOpacity;

    private Dialog mDialog;
    private EditText mDlgContent;

    private final RectF mCharSelectedRectF;
    private final TextSelector mTextSelector;
    private int mSelectedPageIndex;
    private boolean mIsContinuousCreate = true;
    private final Paint mPaint;
    private boolean mRPLCreating = false;
    private boolean mSelecting = false;
    private int mCaretRotate = 0;

    public CaretToolHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiextensionsManager = (UIExtensionsManager) pdfViewCtrl.getUIExtensionsManager();
        mPropertyBar = mUiextensionsManager.getMainFrame().getPropertyBar();

        mPaint = new Paint();
        mPaint.setAntiAlias(true);
        mPaint.setXfermode(new PorterDuffXfermode(PorterDuff.Mode.MULTIPLY));
        mTextSelector = new TextSelector(mPdfViewCtrl);
        mCharSelectedRectF = new RectF();

        mRPLCreating = false;
    }

    protected void setPropertyChangeListener(PropertyBar.PropertyChangeListener propertyChangeListener) {
        mPropertyChangeListener = propertyChangeListener;
    }

    protected void removePropertyBarListener() {
        mPropertyChangeListener = null;
    }

    protected void init(boolean isInsertTextModule) {
        mIsInsertTextModule = isInsertTextModule;
        mColors = PropertyBar.PB_COLORS_TOOL_DEFAULT;
        mOpacity = 100;

        if (mIsInsertTextModule) {
            mColor = mUiextensionsManager.getConfig().uiSettings.annotations.insert.color;
            mOpacity = (int) (mUiextensionsManager.getConfig().uiSettings.annotations.insert.opacity * 100);
        } else {
            mColor = mUiextensionsManager.getConfig().uiSettings.annotations.replace.color;
            mOpacity = (int) (mUiextensionsManager.getConfig().uiSettings.annotations.replace.opacity * 100);
        }
    }

    protected void changeCurrentColor(int currentColor) {
        mColor = currentColor;
        setProItemColor(currentColor);
    }

    protected void changeCurrentOpacity(int opacity) {
        mOpacity = opacity;
        if (mCurToolItem == null) return;
        mCurToolItem.property.opacity = opacity;
    }

    private void setProItemColor(int color) {
        if (mCurToolItem == null) return;
        mCurToolItem.property.color = color;
        ((UIColorItem) mCurToolItem.toolItem).setAlphaColorBg(color);
    }

    @Override
    public String getType() {
        if (mIsInsertTextModule)
            return ToolHandler.TH_TYPR_INSERTTEXT;
        return ToolHandler.TH_TYPE_REPLACE;
    }

    @Override
    public void onActivate() {
        mTextSelector.clear();
        mCharSelectedRectF.setEmpty();
        resetPropertyBar();
    }

    @Override
    public void onDeactivate() {
        mCharSelectedRectF.setEmpty();
    }

    @Override
    public boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiextensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        if (!mIsInsertTextModule){
            if (!handled){
                handled = handleTextReplaceTapConfirmed(pageIndex, motionEvent);
            }
        }else {
            if (!handled){
                motionEvent.setAction(MotionEvent.ACTION_UP);
                handled = onCaretToolTouch(pageIndex, motionEvent);
            }
        }
        return handled;
    }

    private boolean handleTextReplaceTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        PDFPage page = mUiextensionsManager.getDocumentManager().getPage(pageIndex, false);
        if (page == null) {
            return true;
        }
        try {
            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);
            PointF pagePt = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
            PointF pdfPt = new PointF();
            mPdfViewCtrl.convertPageViewPtToPdfPt(pagePt, pdfPt, pageIndex);
            Range range = textPage.getWordAtPos(pdfPt.x,pdfPt.y,10);
            if (range.getSegmentCount() == 1){
                int start = range.getSegmentStart(0);
                int end = range.getSegmentEnd(0);
                mSelecting = true;
                mRPLCreating = true;
                mSelectedPageIndex = pageIndex;
                mTextSelector.start(page, start);
                mTextSelector.update(page, end);
                invalidateTouch(pageIndex, mTextSelector);
                onSelectRelease(pageIndex, mTextSelector);
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return true;
    }

    @Override
    public boolean isContinueAddAnnot() {
        return mIsContinuousCreate;
    }

    @Override
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        mIsContinuousCreate = continueAddAnnot;
    }

    @Override
    public boolean onLongPress(int pageIndex, MotionEvent motionEvent) {
        if (mUiextensionsManager.getDocumentManager().getCurrentAnnot() != null){
            return mUiextensionsManager.defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
        onCaretToolTouch(pageIndex, motionEvent);
        mPdfViewCtrl.capturePageViewOnTouch(motionEvent);
        return true;
    }

    private int getCharIndexAtPoint(int pageIndex, PointF point) {
        int index = 0;
        try {
            PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            index = getCharIndexAtPoint(page, point);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return index;
    }

     void resetPropertyBar() {
        mPropertyBar.clearPropertyTitle();
        long supportProperty = PropertyBar.PROPERTY_COLOR | PropertyBar.PROPERTY_OPACITY;
        System.arraycopy(PropertyBar.PB_COLORS_TOOL_DEFAULT, 0, mColors, 0, mColors.length);
        mPropertyBar.setColors(mColors);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_COLOR, mColor);
        mPropertyBar.setProperty(PropertyBar.PROPERTY_OPACITY, mOpacity);
        mPropertyBar.reset(supportProperty);
        mPropertyBar.setPropertyChangeListener(mPropertyChangeListener);
    }

    private boolean onSelectDown(int pageIndex, PointF point, TextSelector selectInfo) {
        if (selectInfo == null) return false;
        int index = getCharIndexAtPoint(pageIndex, point);
        if (index >= 0) {
            selectInfo.setStart(index);
            selectInfo.setEnd(index);

            return true;
        }
        return false;
    }

    private boolean onSelectMove(int pageIndex, PointF point, TextSelector selectInfo) {
        if (selectInfo == null || selectInfo.getStart() < 0)
            return false;
        if (mSelectedPageIndex != pageIndex)
            return false;

        int index = getCharIndexAtPoint(pageIndex, point);
        if (index >= 0) {
            selectInfo.setEnd(index);
            return true;
        }
        return false;
    }

    private boolean onSelectRelease(final int pageIndex, final TextSelector selectorInfo) {

        if (!mIsInsertTextModule && mRPLCreating) {
            if (selectorInfo.getStart() >= 0 && selectorInfo.getEnd() >= 0) {
                selectorInfo.computeSelected(mUiextensionsManager.getDocumentManager().getPage(pageIndex, false), selectorInfo.getStart(), selectorInfo.getEnd());
                if (selectorInfo.getRectFList().size() > 0) {
                    mCharSelectedRectF.set(selectorInfo.getRectFList().get(selectorInfo.getRectFList().size() - 1));
                }
            }
            View.OnClickListener cancelClickListener = new View.OnClickListener() {

                @Override
                public void onClick(View v) {
                    mDialog.dismiss();
                    AppUtil.dismissInputSoft(mDlgContent);
                    selectorInfo.clear();
                }
            };
            View.OnClickListener okClickListener = new View.OnClickListener() {

                @Override
                public void onClick(View v) {
                    //add Caret Annotation
                    addCaretAnnot(pageIndex, getCaretRectFromSelectRect(selectorInfo, pageIndex, null), mCaretRotate, mTextSelector, null);
                    mDialog.dismiss();
                    AppUtil.dismissInputSoft(mDlgContent);

                    if (!mIsContinuousCreate) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
                    }
                }
            };
            initDialog(cancelClickListener, okClickListener);
            mDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
                @Override
                public void onDismiss(DialogInterface dialog) {
                    mSelecting = false;
                    mRPLCreating = false;
                    RectF selectedRectF = new RectF(mCharSelectedRectF);
                    mPdfViewCtrl.convertPdfRectToPageViewRect(selectedRectF, selectedRectF, pageIndex);
                    Rect selectedRect = new Rect();
                    selectedRectF.roundOut(selectedRect);
                    getInvalidateRect(selectedRect);
                    mPdfViewCtrl.invalidate(selectedRect);
                    mCharSelectedRectF.setEmpty();
                }
            });

            return true;
        }

        return false;
    }

    private int getCharIndexAtPoint(PDFPage page, PointF pdfPt) {
        int index = 0;
        try {
            TextPage textSelect = new TextPage(page, TextPage.e_ParseTextNormal);
            index = textSelect.getIndexAtPos(pdfPt.x, pdfPt.y, 10);
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return index;
    }

    //Non standard dictionary entry, Adobe Acrobat and Foxit RDK support, but not supported by the Foxit Phantom
    private void setCaretRotate(Annot caret, int rotate) {
        if (!(caret instanceof Caret))
            return;
        if (rotate < Constants.e_Rotation0 || rotate > Constants.e_RotationUnknown)
            rotate = 0;
        try {
            caret.getDict().setAt("Rotate", PDFObject.createFromInteger(360 - rotate * 90));
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    private RectF getCaretRectFromSelectRect(TextSelector selector, int pageIndex, PointF point) {
        try {
            PDFPage page = mPdfViewCtrl.getDoc().getPage(pageIndex);
            if (!page.isParsed()) {
                Progressive progressive = page.startParse(PDFPage.e_ParsePageNormal, null, false);
                int state = Progressive.e_ToBeContinued;
                while (state == Progressive.e_ToBeContinued) {
                    state = progressive.resume();
                }
            }
            TextPage textSelect = new TextPage(page, TextPage.e_ParseTextNormal);
            int start = Math.min(selector.getStart(), selector.getEnd());
            int end = Math.max(selector.getStart(), selector.getEnd());
            int nCount = textSelect.getTextRectCount(start, end - start + 1);
            RectF docSelectedRectF = AppUtil.toRectF(textSelect.getTextRect(nCount - 1));

            PointF pointF = new PointF((docSelectedRectF.left + docSelectedRectF.right) / 2, (docSelectedRectF.top + docSelectedRectF.bottom) / 2);
            GraphicsObject graphicsObject = page.getGraphicsObjectAtPoint(AppUtil.toFxPointF(pointF), 0.5f, GraphicsObject.e_TypeText);
            RectF textRect = null;
            if (graphicsObject != null) {
                TextObject textObject = graphicsObject.getTextObject();
                if (textObject != null) {
                    textRect = AppUtil.toRectF(textObject.getRect());

                    graphicsObject.delete();
                    textObject.delete();
                }
            }

            mCaretRotate = textSelect.getBaselineRotation(nCount - 1) % Constants.e_RotationUnknown;
            RectF caretRect = new RectF();
            float w, h;
            if (mCaretRotate % 2 != 0) {
                w = textRect != null ? Math.abs(textRect.width()) * 2 / 3 : (docSelectedRectF.right - docSelectedRectF.left);
                h = w * 2 / 3;
            } else {
                h = textRect != null ? Math.abs(textRect.height()) * 2 / 3 : (docSelectedRectF.top - docSelectedRectF.bottom);
                w = h * 2 / 3;
            }
            float offsetY = h * 0.9f;
            float offsetX = w * 0.9f;
            switch (mCaretRotate) {
                case Constants.e_Rotation0: {
                    if ((point != null && point.x - docSelectedRectF.left <= (docSelectedRectF.right - docSelectedRectF.left) / 2)) {
                        caretRect.set(docSelectedRectF.left - w / 2, docSelectedRectF.bottom + h, docSelectedRectF.left + w / 2, docSelectedRectF.bottom);
                    } else {
                        caretRect.set(docSelectedRectF.right - w / 2, docSelectedRectF.bottom + h, docSelectedRectF.right + w / 2, docSelectedRectF.bottom);
                    }

                    caretRect.offset(0, 0 - offsetY);
                }
                break;
                case Constants.e_Rotation90: {
                    if ((point != null && point.y - docSelectedRectF.bottom >= (docSelectedRectF.top - docSelectedRectF.bottom) / 2)) {
                        caretRect.set(docSelectedRectF.left, docSelectedRectF.top + h / 2, docSelectedRectF.left + w, docSelectedRectF.top - h / 2);
                    } else {
                        caretRect.set(docSelectedRectF.left, docSelectedRectF.bottom + h / 2, docSelectedRectF.left + w, docSelectedRectF.bottom - h / 2);
                    }
                    caretRect.offset(0 - offsetX, 0);
                }
                break;
                case Constants.e_Rotation180: {
                    if ((point != null && point.x - docSelectedRectF.left >= (docSelectedRectF.right - docSelectedRectF.left) / 2)) {
                        caretRect.set(docSelectedRectF.right - w / 2, docSelectedRectF.top, docSelectedRectF.right + w / 2, docSelectedRectF.top - h);
                    } else {
                        caretRect.set(docSelectedRectF.left - w / 2, docSelectedRectF.top, docSelectedRectF.left + w / 2, docSelectedRectF.top - h);
                    }
                    caretRect.offset(0, offsetY);

                }
                break;
                case Constants.e_Rotation270: {
                    if ((point != null && point.y - docSelectedRectF.bottom <= (docSelectedRectF.top - docSelectedRectF.bottom) / 2)) {
                        caretRect.set(docSelectedRectF.right - w, docSelectedRectF.bottom + h / 2, docSelectedRectF.right, docSelectedRectF.bottom - h / 2);
                    } else {
                        caretRect.set(docSelectedRectF.right - w, docSelectedRectF.top + h / 2, docSelectedRectF.right, docSelectedRectF.top - h / 2);
                    }
                    caretRect.offset(offsetX, 0);
                }
                break;
                default:
                    break;
            }

            return caretRect;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;

    }

    @Override
    public boolean onTouchEvent(final int pageIndex, MotionEvent motionEvent) {
        boolean handled = mUiextensionsManager.defaultTouchEvent(pageIndex, motionEvent);
        if (!handled && motionEvent.getActionMasked() != MotionEvent.ACTION_DOWN){
            handled = onCaretToolTouch(pageIndex, motionEvent);
        }
        return handled;
    }

    private boolean onCaretToolTouch(int pageIndex, MotionEvent motionEvent) {
        PointF point = AppAnnotUtil.getPageViewPoint(mPdfViewCtrl, pageIndex, motionEvent);
        try {
            switch (motionEvent.getAction()) {
                case MotionEvent.ACTION_DOWN: {
                    mTextSelector.clear();
                    mSelectedPageIndex = pageIndex;
                    PointF docPoint = new PointF(point.x, point.y);
                    mPdfViewCtrl.convertPageViewPtToPdfPt(docPoint, docPoint, pageIndex);
                    mRPLCreating = onSelectDown(pageIndex, docPoint, mTextSelector);
                    mSelecting = mRPLCreating;
                    return mRPLCreating;
                }
                case MotionEvent.ACTION_MOVE: {
                    if (mRPLCreating) {
                        PointF docPoint = new PointF(point.x, point.y);
                        mPdfViewCtrl.convertPageViewPtToPdfPt(docPoint, docPoint, pageIndex);
                        if (onSelectMove(pageIndex, docPoint, mTextSelector)) {
                            mTextSelector.computeSelected(mPdfViewCtrl.getDoc().getPage(pageIndex), mTextSelector.getStart(), mTextSelector.getEnd());
                            invalidateTouch(pageIndex, mTextSelector);
                            return true;
                        }
                    }
                }
                break;
                case MotionEvent.ACTION_CANCEL:
                case MotionEvent.ACTION_UP:
                    if (!mIsInsertTextModule && mRPLCreating){
                        try{
                            PDFPage page = mUiextensionsManager.getDocumentManager().getPage(pageIndex, false);
                            TextPage textPage = new TextPage(page, TextPage.e_ParseTextNormal);
                            if (mTextSelector.getStart() == mTextSelector.getEnd()){
                                PointF pdfPt = new PointF(point.x, point.y);
                                mPdfViewCtrl.convertPageViewPtToPdfPt(pdfPt, pdfPt, pageIndex);
                                Range range = textPage.getWordAtPos(pdfPt.x,pdfPt.y,10);
                                if (range.getSegmentCount() == 1){
                                    int start = range.getSegmentStart(0);
                                    int end = range.getSegmentEnd(0);
                                    mSelecting = true;
                                    mRPLCreating = true;
                                    mSelectedPageIndex = pageIndex;
                                    mTextSelector.start(page, start);
                                    mTextSelector.update(page, end);
                                    invalidateTouch(pageIndex, mTextSelector);
                                }
                            }
                        } catch (Exception ignored) {
                        }
                        return onSelectRelease(pageIndex, mTextSelector);
                    }else if (mIsInsertTextModule){
                        mRPLCreating = false;
                        mSelecting = false;
                        return addInsertText(pageIndex, point);
                    }
                default:
                    break;
            }

        } catch (PDFException e) {
            e.printStackTrace();
        }
        return false;
    }

    private boolean addInsertText(final int pageIndex, PointF point) {
        mSelectedPageIndex = pageIndex;
        PointF docPoint = new PointF(point.x, point.y);
        mPdfViewCtrl.convertPageViewPtToPdfPt(docPoint, docPoint, pageIndex);

        final PDFPage page = mUiextensionsManager.getDocumentManager().getPage(pageIndex, false);
        int index = getCharIndexAtPoint(page, docPoint);
        if (index == -1 ) {
            if (mTextSelector.getStart() == -1){
                return true;
            }else {
                index = mTextSelector.getEnd();
            }
        }
        mTextSelector.clear();
        if (index >= 0) {
            mSelecting = true;
            mTextSelector.setStart(index);
            mTextSelector.setEnd(index);

            mTextSelector.computeSelected(page, mTextSelector.getStart(), mTextSelector.getEnd());
            mCharSelectedRectF.set(mTextSelector.getBbox());
            invalidateTouch(pageIndex, mTextSelector);
            final PointF pointTemp = new PointF(point.x, point.y);
            View.OnClickListener cancelClickListener = new View.OnClickListener() {

                @Override
                public void onClick(View v) {
                    mDialog.dismiss();
                    AppUtil.dismissInputSoft(mDlgContent);

                }
            };
            View.OnClickListener okClickListener = new View.OnClickListener() {

                @Override
                public void onClick(View v) {

                    PointF pdfPoint = new PointF(pointTemp.x, pointTemp.y);
                    mPdfViewCtrl.convertPageViewPtToPdfPt(pdfPoint, pdfPoint, pageIndex);
                    addCaretAnnot(pageIndex, getCaretRectFromSelectRect(mTextSelector, pageIndex, pdfPoint), mCaretRotate, mTextSelector, null);
                    mDialog.dismiss();
                    AppUtil.dismissInputSoft(mDlgContent);
                    if (!mIsContinuousCreate) {
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).setCurrentToolHandler(null);
                    }
                }
            };
            initDialog(cancelClickListener, okClickListener);
            mDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
                @Override
                public void onDismiss(DialogInterface dialog) {
                    mSelecting = false;
                    RectF selectedRectF = new RectF(mCharSelectedRectF);
                    clearSelectedRectF();
                    mPdfViewCtrl.convertPdfRectToPageViewRect(selectedRectF, selectedRectF, pageIndex);
                    mPdfViewCtrl.convertPageViewRectToDisplayViewRect(selectedRectF, selectedRectF, pageIndex);
                    Rect selectedRect = new Rect();
                    selectedRectF.roundOut(selectedRect);
                    getInvalidateRect(selectedRect);
                    mPdfViewCtrl.invalidate(selectedRect);
                }
            });

            return true;
        }
        return false;
    }

    private void initDialog(View.OnClickListener cancelClickListener, View.OnClickListener okClickListener) {
        if (mPdfViewCtrl.getUIExtensionsManager() == null) return;
        final Activity context = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAttachedActivity();
        if (context == null) return;
        View mView = View.inflate(mContext, R.layout.rd_note_dialog_edit, null);
        final TextView dialogTitle = (TextView) mView.findViewById(R.id.rd_note_dialog_edit_title);
        mDlgContent = (EditText) mView.findViewById(R.id.rd_note_dialog_edit);
        if (AppDisplay.isPad()) { // SDKRD-9313
            mDlgContent.setImeOptions(EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }
        final Button cancelButton = (Button) mView.findViewById(R.id.rd_note_dialog_edit_cancel);
        final Button applayButton = (Button) mView.findViewById(R.id.rd_note_dialog_edit_ok);

        mView.setLayoutParams(new ViewGroup.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.WRAP_CONTENT));
        mDialog = new Dialog(context, R.style.rv_dialog_style);
        mDialog.setContentView(mView, new ViewGroup.LayoutParams(AppDisplay.getUITextEditDialogWidth(), ViewGroup.LayoutParams.WRAP_CONTENT));
        mDlgContent.setMaxLines(10);

        mDialog.getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN);
        mDialog.getWindow().setBackgroundDrawableResource(R.drawable.dlg_title_bg_4circle_corner_white);
        mDialog.setOnDismissListener(new DialogInterface.OnDismissListener() {
            @Override
            public void onDismiss(DialogInterface dialog) {
                if (SystemUiHelper.getInstance().isFullScreen()){
                    if (AppDisplay.isPad())
                        SystemUiHelper.getInstance().hideSystemUI(context);
                    else
                        SystemUiHelper.getInstance().hideStatusBar(context);
                }
            }
        });

        if (mIsInsertTextModule) {
            dialogTitle.setText(mContext.getApplicationContext().getString(R.string.fx_string_inserttext));
        } else {
            dialogTitle.setText(mContext.getApplicationContext().getString(R.string.fx_string_replacetext));
        }
        cancelButton.setOnClickListener(cancelClickListener);
        applayButton.setOnClickListener(okClickListener);
        mDialog.show();
        AppUtil.showSoftInput(mDlgContent);
    }


    protected void addAnnot(final int pageIndex, final CaretAnnotContent content, boolean addUndo, final Event.Callback result) {
        CaretAddUndoItem undoItem = new CaretAddUndoItem(mPdfViewCtrl);
        undoItem.setCurrentValue(content);
        try {
            //step 1 add annot to pdf
            final PDFPage page = mUiextensionsManager.getDocumentManager().getPage(pageIndex, false);
            if (page == null) return;
            RectF docRect = content.getBBox();
            if (docRect == null) return;

            final Caret caret = (Caret) AppAnnotUtil.createAnnot(page.addAnnot(Annot.e_Caret, AppUtil.toFxRectF(docRect)), Annot.e_Caret);
            if (caret == null) return;

            undoItem.setCurrentValue(content);
            undoItem.mRotate = content.getRotate();
            undoItem.mTextSelector = new TextSelector(mPdfViewCtrl);
            undoItem.mTextSelector.setStart(content.getTextSelector().getStart());
            undoItem.mTextSelector.setEnd(content.getTextSelector().getEnd());
            undoItem.mTextSelector.computeSelected(page, undoItem.mTextSelector.getStart(), undoItem.mTextSelector.getEnd());
            undoItem.mIsReplace = !mIsInsertTextModule;
            addAnnot(caret, undoItem, addUndo, result);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    protected void addAnnot(final Annot annot, final CaretAddUndoItem undoItem, final boolean addUndo, final Event.Callback result) {
        //add replace strikeout
        if (!mIsInsertTextModule) {
            TextMarkupContentAbs strikeoutAbs = new TextMarkupContentAbs() {
                @Override
                public TextSelector getTextSelector() {
                    return undoItem.mTextSelector;
                }

                @Override
                public int getPageIndex() {
                    return undoItem.mPageIndex;
                }

                @Override
                public int getType() {
                    return Annot.e_StrikeOut;
                }

                @Override
                public String getIntent() {
                    return "StrikeOutTextEdit";
                }

                @Override
                public String getAuthor() {
                    return null;
                }

                @Override
                public int getColor() {
                    return (int) undoItem.mColor;
                }

                @Override
                public int getFillColor() {
                    return 0;
                }

                @Override
                public int getOpacity() {
                    return (int) (undoItem.mOpacity * 255f + 0.5f);
                }

                @Override
                public String getSubject() {
                    return "Replace";
                }
            };

            AnnotHandler annotHandler = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getAnnotHandlerByType(Annot.e_StrikeOut);
            annotHandler.addAnnot(undoItem.mPageIndex, strikeoutAbs, false, new Event.Callback() {
                @Override
                public void result(Event event, boolean success) {
                    if (!(event instanceof StrikeoutEvent)) return;
                    undoItem.strikeOutEvent = (StrikeoutEvent) event;
                }
            });
        }

        CaretEvent event = new CaretEvent(EditAnnotEvent.EVENTTYPE_ADD, undoItem, (Caret) annot, mPdfViewCtrl);
        EditAnnotTask task = new EditAnnotTask(event, new Event.Callback() {
            @Override
            public void result(Event event, boolean success) {
                if (success) {
                    try {
                        final PDFPage page = annot.getPage();
                        final int pageIndex = page.getIndex();
                        ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, annot);
                        if (undoItem != null && undoItem.strikeOutEvent != null && undoItem.strikeOutEvent.mAnnot != null)
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().onAnnotAdded(page, undoItem.strikeOutEvent.mAnnot);

                        if (addUndo) {
                            ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager()).getDocumentManager().addUndoItem(undoItem);
                        }

                        //step 2 invalidate page view
                        if (mPdfViewCtrl.isPageVisible(pageIndex)) {
                            RectF viewRect = AppUtil.toRectF(annot.getRect());
                            if (!mIsInsertTextModule) {
                                StrikeOut strikeOut = AppAnnotUtil.getStrikeOutFromCaret((Caret) annot);
                                if (strikeOut != null && !strikeOut.isEmpty()) {
                                    RectF sto_Rect = AppUtil.toRectF(strikeOut.getRect());
                                    sto_Rect.union(viewRect);

                                    viewRect.set(sto_Rect);
                                }
                            }
                            mPdfViewCtrl.convertPdfRectToPageViewRect(viewRect, viewRect, pageIndex);
                            Rect rect = new Rect();
                            viewRect.roundOut(rect);
                            rect.inset(-10, -10);
                            mPdfViewCtrl.refresh(pageIndex, rect);
                        }

                        if (result != null) {
                            result.result(null, success);
                        }
                    } catch (PDFException e) {
                        e.printStackTrace();
                    }
                }
            }
        });
        mPdfViewCtrl.addTask(task);
    }

    private void addCaretAnnot(final int pageIndex, final RectF annotRect, final int rotate, final TextSelector textSelector, final Event.Callback result) {
        final DateTime dateTime = AppDmUtil.currentDateToDocumentDate();
        CaretAnnotContent caretAnnotContent = new CaretAnnotContent(mPdfViewCtrl, mIsInsertTextModule) {
            @Override
            public int getPageIndex() {
                return pageIndex;
            }

            @Override
            public int getType() {
                return Annot.e_Caret;
            }

            @Override
            public float getLineWidth() {
                return 0;
            }

            @Override
            public int getRotate() {
                return rotate;
            }

            @Override
            public TextSelector getTextSelector() {
                return textSelector;
            }

            @Override
            public DateTime getCreatedDate() {
                return dateTime;
            }

            @Override
            public RectF getBBox() {
                return annotRect;
            }

            @Override
            public int getColor() {
                return mColor;
            }

            @Override
            public int getFillColor() {
                return 0;
            }

            @Override
            public int getOpacity() {
                return AppDmUtil.opacity100To255(mOpacity);
            }


            @Override
            public DateTime getModifiedDate() {
                return dateTime;
            }

            @Override
            public String getContents() {
                return mDlgContent.getText().toString();
            }

        };

        addAnnot(pageIndex, caretAnnotContent, true, result);

    }

    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mSelectedPageIndex != pageIndex) return;
        if (mIsInsertTextModule && !mRPLCreating) {
            if (mSelecting && mTextSelector != null && !(mCharSelectedRectF.left >= mCharSelectedRectF.right || mCharSelectedRectF.top <= mCharSelectedRectF.bottom)) {
                mPaint.setColor(calColorByMultiply(0x73C1E1, 150));
                Rect clipRect = canvas.getClipBounds();
                RectF tmp = new RectF(mTextSelector.getBbox());
                mPdfViewCtrl.convertPdfRectToPageViewRect(tmp, tmp, pageIndex);
                Rect r = new Rect();
                tmp.round(r);
                if (r.intersect(clipRect)) {
                    canvas.save();
                    canvas.drawRect(r, mPaint);

                    if (mTextSelector.getRectFList().size() > 0) {
                        RectF start = new RectF(mTextSelector.getRectFList().get(0));
                        RectF end = new RectF(mTextSelector.getRectFList().get(mTextSelector.getRectFList().size() - 1));
                        mPdfViewCtrl.convertPdfRectToPageViewRect(start, start, pageIndex);
                        mPdfViewCtrl.convertPdfRectToPageViewRect(end, end, pageIndex);

                        mPaint.setARGB(255, 76, 121, 164);
                        canvas.drawLine(start.left, start.top, start.left, start.bottom, mPaint);
                        canvas.drawLine(end.right, end.top, end.right, end.bottom, mPaint);
                    }

                    canvas.restore();
                }
            }
        } else {
            if (mSelecting && mTextSelector != null && mTextSelector.getStart() >= 0) {
                mPaint.setColor(calColorByMultiply(0x73C1E1, 150));
                Rect clipRect = canvas.getClipBounds();
                for (RectF rect : mTextSelector.getRectFList()) {
                    RectF tmp = new RectF(rect);
                    mPdfViewCtrl.convertPdfRectToPageViewRect(tmp, tmp, pageIndex);
                    Rect r = new Rect();
                    tmp.round(r);
                    if (r.intersect(clipRect)) {
                        canvas.save();
                        canvas.drawRect(r, mPaint);
                        canvas.restore();
                    }
                }
                if (mTextSelector.getRectFList().size() > 0) {
                    RectF start = new RectF(mTextSelector.getRectFList().get(0));
                    RectF end = new RectF(mTextSelector.getRectFList().get(mTextSelector.getRectFList().size() - 1));
                    mPdfViewCtrl.convertPdfRectToPageViewRect(start, start, pageIndex);
                    mPdfViewCtrl.convertPdfRectToPageViewRect(end, end, pageIndex);

                    mPaint.setARGB(255, 76, 121, 164);
                    canvas.drawLine(start.left, start.top, start.left, start.bottom, mPaint);
                    canvas.drawLine(end.right, end.top, end.right, end.bottom, mPaint);
                }
            }
        }

    }

    private final RectF mTmpRect = new RectF();

    private void invalidateTouch(int pageIndex, TextSelector textSelector) {

        if (textSelector == null) return;
        RectF rectF = new RectF();
        rectF.set(textSelector.getBbox());
        mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, rectF, pageIndex);
        mPdfViewCtrl.convertPageViewRectToDisplayViewRect(rectF, rectF, pageIndex);
        RectF rF = calculate(rectF, mTmpRect);
        Rect rect = new Rect();
        rF.roundOut(rect);
        getInvalidateRect(rect);
        mPdfViewCtrl.invalidate(rect);
        mTmpRect.set(rectF);
    }

    private RectF calculate(RectF desRectF, RectF srcRectF) {
        RectF mTmpDesRect = new RectF();
        if (srcRectF.isEmpty()) return desRectF;
        int count = 0;
        if (desRectF.left == srcRectF.left && desRectF.top == srcRectF.top) count++;
        if (desRectF.right == srcRectF.right && desRectF.top == srcRectF.top) count++;
        if (desRectF.left == srcRectF.left && desRectF.bottom == srcRectF.bottom) count++;
        if (desRectF.right == srcRectF.right && desRectF.bottom == srcRectF.bottom) count++;
        mTmpDesRect.set(desRectF);
        if (count == 2) {
            mTmpDesRect.union(srcRectF);
            RectF rectF = new RectF();
            rectF.set(mTmpDesRect);
            mTmpDesRect.intersect(srcRectF);
            rectF.intersect(mTmpDesRect);
            return rectF;
        } else if (count == 3 || count == 4) {
            return mTmpDesRect;
        } else {
            mTmpDesRect.union(srcRectF);
            return mTmpDesRect;
        }
    }

    private void getInvalidateRect(Rect rect) {
        rect.top -= 20;
        rect.bottom += 20;
        rect.left -= 20 / 2;
        rect.right += 20 / 2;
        rect.inset(-20, -20);
    }

    private void clearSelectedRectF() {
        mTextSelector.clear();
        mCharSelectedRectF.setEmpty();
    }

    private int calColorByMultiply(int color, int opacity) {
        int rColor = color | 0xFF000000;
        int r = (rColor & 0xFF0000) >> 16;
        int g = (rColor & 0xFF00) >> 8;
        int b = (rColor & 0xFF);
        float rOpacity = opacity / 255.0f;
        r = (int) (r * rOpacity + 255 * (1 - rOpacity));
        g = (int) (g * rOpacity + 255 * (1 - rOpacity));
        b = (int) (b * rOpacity + 255 * (1 - rOpacity));
        rColor = (rColor & 0xFF000000) | (r << 16) | (g << 8) | (b);
        return rColor;
    }

    private int mLastColor;
    private int mLastOpacity;
    private IToolSupply mToolSupply;
    private ToolItemBean mCurToolItem;
    private PropertyBar.PropertyChangeListener mCustomPropertyListener;

    PropertyBar.PropertyChangeListener getCustomPropertyListener(){
        return mCustomPropertyListener;
    }

    IToolSupply getToolSupply() {
        if (mToolSupply == null)
            mToolSupply = new CaretToolSupply(mContext);
        return mToolSupply;
    }

    private class CaretToolSupply extends ToolSupplyImpl {

        public CaretToolSupply(Context context) {
            super(context);
        }

        @Override
        public int getToolBackgroundResource(int toolType) {
            return mIsInsertTextModule ? R.drawable.comment_tool_insert_text_bg
                    : R.drawable.comment_tool_replace_text_bg;
        }

        @Override
        public int getToolForegroundResource(int toolType) {
            return mIsInsertTextModule ? R.drawable.comment_tool_insert_text_src
                    : R.drawable.comment_tool_replace_text_src;
        }

        @Override
        public ToolProperty createToolProperty(int toolType) {
            ToolProperty property = new ToolProperty();
            property.type = mIsInsertTextModule ? ToolConstants.Insert_Text : ToolConstants.Replace_Text;
            property.color = mColor;
            property.opacity = mOpacity;
            return property;
        }

        @Override
        public String getToolName(int toolType) {
            return mIsInsertTextModule ? JsonConstants.TYPE_CARET : JsonConstants.TYPE_REPLACE;
        }

        @Override
        public void onClick(ToolItemBean itemBean) {
            mCurToolItem = itemBean;
            if (itemBean.toolItem.isSelected()) {
                if (mUiextensionsManager.getMainFrame().getCurrentTab() == ToolbarItemConfig.ITEM_COMMENT_TAB) {
                    if(itemBean.type == ToolConstants.Insert_Text){
                        mUiextensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_InsertText);
                    }else{
                        mUiextensionsManager.onUIInteractElementClicked(IUIInteractionEventListener.Reading_CommentBar_ReplaceText);
                    }

                }
                mLastColor = mColor;
                mLastOpacity = mOpacity;

                ToolProperty property = itemBean.property;
                if (property == null) {
                    property = createToolProperty(itemBean.type);
                    itemBean.property = property;
                }
                mColor = property.color;
                mOpacity = property.opacity;
                mUiextensionsManager.setCurrentToolHandler(CaretToolHandler.this);
            } else {
                if (mUiextensionsManager.getCurrentToolHandler() == CaretToolHandler.this){
                    mColor = mLastColor;
                    mOpacity = mLastOpacity;
                    mCurToolItem = null;
                    mUiextensionsManager.setCurrentToolHandler(null);
                }
            }
        }

        @Override
        public void resetPropertyBar(ToolItemBean itemBean, final PropertyBar.PropertyChangeListener propertyChangeListener) {
            mCustomPropertyListener = propertyChangeListener;
            mCurToolItem = itemBean;
            mLastColor = mColor;
            mLastOpacity = mOpacity;

            ToolProperty property = itemBean.property;
            if (property == null) {
                property = createToolProperty(itemBean.type);
                itemBean.property = property;
            }
            mColor = property.color;
            mOpacity = property.opacity;

            CaretToolHandler.this.resetPropertyBar();
            mPropertyBar.setDismissListener(new PropertyBar.DismissListener() {
                @Override
                public void onDismiss() {
                    mPropertyBar.setDismissListener(null);
                    mColor = mLastColor;
                    mOpacity = mLastOpacity;
                    mCurToolItem = null;
                    mCustomPropertyListener = null;
                }
            });
        }

        @Override
        public PropertyBar getPropertyBar() {
            return mPropertyBar;
        }
    }

}
