/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots.caret;

import android.content.Context;
import android.graphics.Canvas;

import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.uiextensions.IThemeEventListener;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.ToolHandler;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.config.modules.ModulesConfig;
import com.foxit.uiextensions.config.modules.annotations.AnnotationsConfig;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.propertybar.imp.AnnotMenuImpl;
import com.foxit.uiextensions.controls.propertybar.imp.PropertyBarImpl;
import com.foxit.uiextensions.controls.toolbar.ToolConstants;
import com.foxit.uiextensions.controls.toolbar.ToolbarItemConfig;
import com.foxit.uiextensions.controls.toolbar.drag.UIToolBaseBar;
import com.foxit.uiextensions.utils.AppAnnotUtil;

public class CaretModule implements Module, PropertyBar.PropertyChangeListener {

    private final Context mContext;
    private PDFViewCtrl mPdfViewCtrl;

    private CaretToolHandler mIS_ToolHandler;
    private CaretToolHandler mRP_ToolHandler;
    private CaretAnnotHandler mAnnotHandler;
    private PDFViewCtrl.UIExtensionsManager mUiExtensionsManager;

    public CaretModule(Context context, PDFViewCtrl pdfViewCtrl, PDFViewCtrl.UIExtensionsManager uiExtensionsManager) {
        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mUiExtensionsManager = uiExtensionsManager;
    }

    @Override
    public boolean loadModule() {
        mAnnotHandler = new CaretAnnotHandler(mContext, mPdfViewCtrl);
        mAnnotHandler.setPropertyChangeListener(this);
        mAnnotHandler.setAnnotMenu(new AnnotMenuImpl(mContext, mPdfViewCtrl));
        mAnnotHandler.setPropertyBar(new PropertyBarImpl(mContext, mPdfViewCtrl));

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            ModulesConfig modulesConfig = uiExtensionsManager.getConfig().modules;
            AnnotationsConfig annotConfig = modulesConfig.annotations;

            if (annotConfig.isLoadInsertText) {
                mIS_ToolHandler = new CaretToolHandler(mContext, mPdfViewCtrl);
                mIS_ToolHandler.setPropertyChangeListener(this);
                mIS_ToolHandler.init(true);
                mAnnotHandler.setToolHandler("Insert Text", mIS_ToolHandler);
                uiExtensionsManager.registerToolHandler(mIS_ToolHandler);
                uiExtensionsManager.getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_COMMENT, ToolConstants.Insert_Text, mIS_ToolHandler.getToolSupply());

                uiExtensionsManager.addCreatePropertyChangedListener(mAnnotHandler.getType(), mCreatePropertyChangedListener);
            }

            if (annotConfig.isLoadReplaceText) {
                mRP_ToolHandler = new CaretToolHandler(mContext, mPdfViewCtrl);
                mRP_ToolHandler.setPropertyChangeListener(this);
                mRP_ToolHandler.init(false);
                mAnnotHandler.setToolHandler("Replace", mRP_ToolHandler);
                uiExtensionsManager.registerToolHandler(mRP_ToolHandler);
                uiExtensionsManager.getToolsManager()
                        .addToolItem(UIToolBaseBar.USAGE_TYPE_COMMENT, ToolConstants.Replace_Text, mRP_ToolHandler.getToolSupply());

                uiExtensionsManager.addCreatePropertyChangedListener(mAnnotHandler.getType(), mCreatePropertyChangedListener);
            }

            uiExtensionsManager.registerAnnotHandler(mAnnotHandler);
            uiExtensionsManager.registerModule(this);
            uiExtensionsManager.registerThemeEventListener(mThemeEventListener);
        }
        mPdfViewCtrl.registerRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.registerDrawEventListener(mDrawEventListener);
        return true;
    }

    @Override
    public boolean unloadModule() {
        mAnnotHandler.removePropertyBarListener();

        if (mUiExtensionsManager != null && mUiExtensionsManager instanceof UIExtensionsManager) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            if (mRP_ToolHandler != null) {
                mRP_ToolHandler.removePropertyBarListener();
                uiExtensionsManager.unregisterToolHandler(mRP_ToolHandler);
            }
            if (mIS_ToolHandler != null) {
                mIS_ToolHandler.removePropertyBarListener();
                uiExtensionsManager.unregisterToolHandler(mIS_ToolHandler);
            }
            uiExtensionsManager.unregisterAnnotHandler(mAnnotHandler);
            uiExtensionsManager.unregisterThemeEventListener(mThemeEventListener);
            uiExtensionsManager.removeCreatePropertyChangedListener(mAnnotHandler.getType());
        }

        mPdfViewCtrl.unregisterRecoveryEventListener(memoryEventListener);
        mPdfViewCtrl.unregisterDrawEventListener(mDrawEventListener);
        return true;
    }

    @Override
    public String getName() {
        return Module.MODULE_NAME_CARET;
    }

    private boolean needHandleByTool(UIExtensionsManager uiExtensionsManager,ToolHandler toolHandler , AnnotHandler annotHandler) {
        return uiExtensionsManager.getCurrentToolHandler() != null && uiExtensionsManager.getCurrentToolHandler() == toolHandler
                && (mAnnotHandler != annotHandler || mAnnotHandler.getAnnotMenu().isShowing());
    }

    @Override
    public void onValueChanged(long property, int value) {
        UIExtensionsManager uiExtensionsManager = ((UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager());
        AnnotHandler currentAnnotHandler = uiExtensionsManager.getCurrentAnnotHandler();
        if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {

            if (needHandleByTool(uiExtensionsManager, mIS_ToolHandler, currentAnnotHandler)) {
                mIS_ToolHandler.changeCurrentColor(value);
            } else if (needHandleByTool(uiExtensionsManager, mRP_ToolHandler, currentAnnotHandler)) {
                mRP_ToolHandler.changeCurrentColor(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onColorValueChanged(value);
            } else if (mIS_ToolHandler.getCustomPropertyListener() != null) {
                mIS_ToolHandler.changeCurrentColor(value);
                mIS_ToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            } else if (mRP_ToolHandler.getCustomPropertyListener() != null) {
                mRP_ToolHandler.changeCurrentColor(value);
                mRP_ToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        } else if (property == PropertyBar.PROPERTY_OPACITY) {
            if (needHandleByTool(uiExtensionsManager, mIS_ToolHandler, currentAnnotHandler)) {
                mIS_ToolHandler.changeCurrentOpacity(value);
            } else if (needHandleByTool(uiExtensionsManager, mRP_ToolHandler, currentAnnotHandler)) {
                mRP_ToolHandler.changeCurrentOpacity(value);
            } else if (currentAnnotHandler == mAnnotHandler) {
                mAnnotHandler.onOpacityValueChanged(value);
            } else if (mIS_ToolHandler.getCustomPropertyListener() != null) {
                mIS_ToolHandler.changeCurrentOpacity(value);
                mIS_ToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            } else if (mRP_ToolHandler.getCustomPropertyListener() != null) {
                mRP_ToolHandler.changeCurrentOpacity(value);
                mRP_ToolHandler.getCustomPropertyListener().onValueChanged(property, value);
            }
        }
    }

    public ToolHandler getISToolHandler() {
        return mIS_ToolHandler;
    }

    public ToolHandler getRPToolHandler() {
        return mRP_ToolHandler;
    }

    public AnnotHandler getAnnotHandler() {
        return mAnnotHandler;
    }

    @Override
    public void onValueChanged(long property, float value) {
    }

    @Override
    public void onValueChanged(long property, String value) {
    }

    private PDFViewCtrl.IDrawEventListener mDrawEventListener = new PDFViewCtrl.IDrawEventListener() {

        @Override
        public void onDraw(int pageIndex, Canvas canvas) {
            mAnnotHandler.onDrawForControls(canvas);
        }
    };

    private PDFViewCtrl.IRecoveryEventListener memoryEventListener = new PDFViewCtrl.IRecoveryEventListener() {
        @Override
        public void onWillRecover() {
            if (mAnnotHandler.getAnnotMenu() != null && mAnnotHandler.getAnnotMenu().isShowing()) {
                mAnnotHandler.getAnnotMenu().dismiss();
            }

            if (mAnnotHandler.getPropertyBar() != null && mAnnotHandler.getPropertyBar().isShowing()) {
                mAnnotHandler.getPropertyBar().dismiss();
            }
        }

        @Override
        public void onRecovered() {
        }
    };

    private IThemeEventListener mThemeEventListener = new IThemeEventListener() {
        @Override
        public void onThemeColorChanged(String type, int color) {
            mAnnotHandler.updateTheme();
        }
    };

    private final PropertyBar.CreatePropertyChangedListener mCreatePropertyChangedListener = new PropertyBar.CreatePropertyChangedListener() {
        @Override
        public void onCreateValueChanged(long property, Object value) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if (property == PropertyBar.PROPERTY_COLOR || property == PropertyBar.PROPERTY_SELF_COLOR) {
                if (AppAnnotUtil.isReplaceCaret(curAnnot))
                    mRP_ToolHandler.mColor = (int) value;
                else
                    mIS_ToolHandler.mColor = (int) value;
            } else if (property == PropertyBar.PROPERTY_OPACITY) {
                if (AppAnnotUtil.isReplaceCaret(curAnnot))
                    mRP_ToolHandler.mOpacity = (int) value;
                else
                    mIS_ToolHandler.mOpacity = (int) value;
            }
        }

        @Override
        public int getToolType() {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if (AppAnnotUtil.isReplaceCaret(curAnnot)) {
                return ToolConstants.Replace_Text;
            }
            return ToolConstants.Insert_Text;
        }

        @Override
        public int getTabPosition() {
            return ToolbarItemConfig.ITEM_COMMENT_TAB;
        }

        @Override
        public void resetToolPropertyBar() {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mUiExtensionsManager;
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if (AppAnnotUtil.isReplaceCaret(curAnnot))
                mRP_ToolHandler.resetPropertyBar();
            else
                mIS_ToolHandler.resetPropertyBar();
        }
    };

}
