/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions.annots;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;

import com.foxit.sdk.ActionCallback;
import com.foxit.sdk.ButtonItem;
import com.foxit.sdk.DialogDescriptionConfig;
import com.foxit.sdk.IdentityProperties;
import com.foxit.sdk.MediaPlayerCallback;
import com.foxit.sdk.MenuItemConfig;
import com.foxit.sdk.MenuItemEx;
import com.foxit.sdk.MenuItemExArray;
import com.foxit.sdk.MenuListArray;
import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.PlayerArgs;
import com.foxit.sdk.PrintParams;
import com.foxit.sdk.SOAPRequestProperties;
import com.foxit.sdk.SOAPResponseInfo;
import com.foxit.sdk.SearchIndexConfig;
import com.foxit.sdk.SearchOption;
import com.foxit.sdk.common.Constants;
import com.foxit.sdk.common.Range;
import com.foxit.sdk.common.WStringArray;
import com.foxit.sdk.common.fxcrt.PointF;
import com.foxit.sdk.common.fxcrt.RectF;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.Signature;
import com.foxit.sdk.pdf.actions.Destination;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.uiextensions.Module;
import com.foxit.uiextensions.R;
import com.foxit.uiextensions.UIExtensionsManager;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.modules.pagenavigation.PageNavigationModule;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppLatch;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

/**
 * Class that define the action handler associated with an annotation. Annotation such as Widget
 * annotation, may contains its own action, and this need to be handled when the use tap on it or input text on it.
 */
public class AnnotActionHandler extends ActionCallback {
    private final Context mContext;
    private PDFViewCtrl mPdfViewCtrl;
    private String version;//"Version 8.1";
    private IJSActionCallback mJSActionCallback;

    public AnnotActionHandler(Context context, PDFViewCtrl pdfViewCtrl) {
        this.mContext = context;
        this.mPdfViewCtrl = pdfViewCtrl;
        try {
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
            version = pInfo.versionName;
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
        }
    }

    @Override
    public void release() {
        mPdfViewCtrl = null;
    }

    public void reset() {
        //
    }

    @Override
    public boolean invalidateRect(PDFDoc document, int page_index, RectF pdf_rect) {
        if (mPdfViewCtrl == null || mPdfViewCtrl.getDoc() == null) return false;

        try {
            if (mPdfViewCtrl.isPageVisible(page_index)) {

                int globalWidth = mPdfViewCtrl.getDisplayViewWidth();
                int globalHeight = mPdfViewCtrl.getDisplayViewHeight();
                android.graphics.RectF globalRect = new android.graphics.RectF(0, 0, globalWidth, globalHeight);

                android.graphics.RectF rectF = AppUtil.toRectF(pdf_rect);
                android.graphics.RectF viewRect = new android.graphics.RectF();
                mPdfViewCtrl.convertPdfRectToPageViewRect(rectF, viewRect, page_index);

                android.graphics.RectF displayRect = new android.graphics.RectF();
                mPdfViewCtrl.convertPageViewRectToDisplayViewRect(viewRect, displayRect, page_index);
                if (!android.graphics.RectF.intersects(globalRect, displayRect)) return false;

                viewRect.inset(-5, -5);
                mPdfViewCtrl.refresh(page_index, AppDmUtil.rectFToRect(viewRect));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return false;
    }

    @Override
    public int getCurrentPage(PDFDoc document) {
        return 0;
    }

    @Override
    public void setCurrentPage(PDFDoc document, int page_index) {
        if (mPdfViewCtrl == null || mPdfViewCtrl.getDoc() == null || document == null || page_index == -1) return;
        mPdfViewCtrl.gotoPage(page_index, 0, 0);
    }

    @Override
    public void setCurrentPage(PDFDoc document, Destination destination) {
        if (mPdfViewCtrl == null || mPdfViewCtrl.getDoc() == null || document == null || destination == null) return;
        try {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            MainFrame mainFrame = (MainFrame) uiExtensionsManager.getMainFrame();
            boolean isToolbarsVisible = mainFrame.isToolbarsVisible();
            int offset;
            if (isToolbarsVisible) {
                offset = mainFrame.getTopBarHeight();
            } else {
                offset = AppResource.getDimensionPixelSize(mContext, R.dimen.ux_margin_20dp);
            }
            int pageIndex = destination.getPageIndex(document);
            Annot curAnnot = uiExtensionsManager.getDocumentManager().getCurrentAnnot();
            if(curAnnot != null && curAnnot.getPage().getIndex() != pageIndex){
                uiExtensionsManager.getDocumentManager().setCurrentAnnot(null);
            }
            android.graphics.PointF pdfPoint = new android.graphics.PointF(destination.getLeft(), destination.getTop());
            android.graphics.PointF viewPoint = new android.graphics.PointF();
            mPdfViewCtrl.convertPdfPtToPageViewPt(pdfPoint, viewPoint, pageIndex);
            mPdfViewCtrl.gotoPage(pageIndex, viewPoint.x, viewPoint.y - offset);
        } catch (PDFException e) {
            e.printStackTrace();
        }
    }

    @Override
    public int getPageRotation(PDFDoc document, int page_index) {
        return Constants.e_Rotation0;
    }

    @Override
    public boolean setPageRotation(PDFDoc document, int page_index, int rotation) {
        return false;
    }

    @Override
    public boolean executeNamedAction(PDFDoc document, String named_action) {
        if (AppUtil.isEmpty(named_action))
            return false;

        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        IViewSettingsWindow settingsWindow = uiExtensionsManager.getMainFrame().getSettingWindow();

        // MOBRD-7847
        if (named_action.equalsIgnoreCase("FitPage")) {
            mPdfViewCtrl.setZoomMode(PDFViewCtrl.ZOOMMODE_FITPAGE);
            settingsWindow.setProperty(IViewSettingsWindow.TYPE_FIT_PAGE, PDFViewCtrl.ZOOMMODE_FITPAGE);
        } else if (named_action.equalsIgnoreCase("FitWidth")) {
            mPdfViewCtrl.setZoomMode(PDFViewCtrl.ZOOMMODE_FITWIDTH);
            settingsWindow.setProperty(IViewSettingsWindow.TYPE_FIT_WIDTH, PDFViewCtrl.ZOOMMODE_FITWIDTH);
        }
        return true;
    }

    @Override
    public boolean setDocChangeMark(PDFDoc document, boolean change_mark) {
        if (mPdfViewCtrl == null) return false;
        UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
        uiExtensionsManager.getDocumentManager().setDocModified(true);

        if (mPdfViewCtrl.getDoc() == null) return true;
        try {
            if (document != null && !document.isEmpty()) {
                if (document.getPageCount() != mPdfViewCtrl.getPageCount()) {
                    mPdfViewCtrl.updatePagesLayout();
                    PageNavigationModule module = (PageNavigationModule) uiExtensionsManager.getModuleByName(Module.MODULE_NAME_PAGENAV);
                    if (module != null) {
                        module.resetJumpView();
                    }
                }
            }
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return true;
    }

    @Override
    public boolean getDocChangeMark(PDFDoc document) {
        return true;
    }

    @Override
    public int getOpenedDocCount() {
        return 1;
    }

    @Override
    public PDFDoc getOpenedDoc(int index) {
        return null;
    }

    @Override
    public PDFDoc getCurrentDoc() {
//        if (mPdfViewCtrl != null) { //todo SDKRD-8708
//            return mPdfViewCtrl.getDoc();
//        }
        return null;
    }

    @Override
    public PDFDoc createBlankDoc() {
        return null;
    }

    @Override
    public void closeDoc(PDFDoc document, boolean is_prompt_to_save) {
    }

    @Override
    public PDFDoc openDoc(String file_path, String password) {
        try {
            PDFDoc doc = new PDFDoc(file_path);
            doc.load(password.getBytes() == null ? null : password.getBytes());
            return doc;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public boolean beep(int type) {
//        String strType;
//        switch (type)
//        {
//            case 0:
//                strType = "Error";
//                break;
//            case 1:
//                strType = "Warning";
//                break;
//            case 2:
//                strType = "Question";
//                break;
//            case 3:
//                strType = "Status";
//                break;
//            case 4:
//                strType = "Default";
//                break;
//            default:
//                strType = "not defined";
//        }
        return false;
    }

    @Override
    public String response(String question, String title, String default_value, String label, boolean is_password) {
        return "";
    }

    @Override
    public String getFilePath(PDFDoc document) {
        return mPdfViewCtrl != null ? mPdfViewCtrl.getFilePath() : "";
    }

    @Override
    public boolean isLocalFile(PDFDoc document) {
        return false;
    }

    @Override
    public String getAttachmentsFilePath(PDFDoc pdf_doc, String name) {
        return "";
    }

    @Override
    public String getExtractedEmbeddedFilePath(PDFDoc pdf_doc, String name) {
        return "";
    }

    @Override
    public boolean print(PDFDoc document, boolean is_ui, Range page_range, boolean is_silent, boolean is_shrunk_to_fit, boolean is_printed_as_image, boolean is_reversed, boolean is_to_print_annots) {
        return true;
    }

    @Override
    public boolean print(PDFDoc document, PrintParams print_params) {
        return false;
    }

    @Override
    public boolean submitForm(PDFDoc document, byte[] form_data, String url, int file_format_type) {
        showPromptDialog(AppResource.getString(mContext, R.string.unsupported_to_submit_form_tip));
        return false;
    }

    @Override
    public boolean launchURL(String url) {
        showPromptDialog(AppResource.getString(mContext, R.string.unsupported_to_launch_url_tip));
        return false;
    }

    @Override
    public String browseFile() {
        AppThreadManager.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (mJSActionCallback != null) {
                    mJSActionCallback.browseFile();
                    return;
                }
                showPromptDialog(AppResource.getString(mContext, R.string.unsupported_to_browse_file_tip));
            }
        });
        return "";
    }

    @Override
    public String browseFile(final boolean is_open_dialog, final String file_format, final String file_filter) {
        if (is_open_dialog) {
            AppThreadManager.getInstance().runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    if (mJSActionCallback != null) {
                        mJSActionCallback.browseFile(is_open_dialog, file_format, file_filter);
                        return;
                    }
                    showPromptDialog(AppResource.getString(mContext, R.string.unsupported_to_browse_file_tip));
                }
            });
            return "";
        } else {
            if (AppUtil.isEmpty(file_format)) {
                showPromptDialog(AppResource.getString(mContext, R.string.unsupported_to_browse_file_tip));
                return "";
            }

            String filePath = mPdfViewCtrl.getFilePath();
            String fileFolder = AppFileUtil.getFileFolder(filePath);
            String fileName = AppFileUtil.getFileNameWithoutExt(filePath);
            final String newPath = AppFileUtil.getFileDuplicateName(fileFolder + "/" + fileName + "." + file_format);

            AppThreadManager.getInstance().runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    UIToast.getInstance(mContext).show(AppResource.getString(mContext, R.string.fx_string_saving) + newPath);
                }
            });
            return newPath;
        }
    }

    @Override
    public int getLanguage() {
        return e_LanguageENU;
    }

    @Override
    public int alert(final String msg, String title, int type, int icon) {
        String toastMsg = AppResource.getString(mContext, R.string.action_alert_msg, msg);
        showPromptDialog(toastMsg);
        return 1;
    }

    @Override
    public IdentityProperties getIdentityProperties() {
        String author = "Foxit";
        if (mPdfViewCtrl != null) {
            UIExtensionsManager uiExtensionsManager = (UIExtensionsManager) mPdfViewCtrl.getUIExtensionsManager();
            author = uiExtensionsManager.getAnnotAuthor();
        }
        IdentityProperties identityProperties = new IdentityProperties();
        identityProperties.setName(author);
        return identityProperties;
    }

    @Override
    public String popupMenu(MenuListArray menus, boolean is_selected_item) {
        if (menus == null || menus.getSize() == 0) {
            return "";
        }
        return menus.getAt(0).getName();
    }

    @Override
    public boolean setIdentityProperties(IdentityProperties identity_properties) {
        return true;
    }

    @Override
    public MenuItemEx popupMenuEx(MenuItemExArray menus, boolean is_selected_item) {
        if (menus == null || menus.getSize() == 0) {
            return null;
        }
        return menus.getAt(0);
    }

    @Override
    public String getAppInfo(int type) {
        String info = "";
        switch (type) {
            case e_AppInfoTypeFormsVersion:
                info = "7.3";
                break;
            case e_AppInfoTypeViewerType:
                info = "Exchange-Pro";
                break;
            case e_AppInfoTypeViewerVariation:
                info = "Full";
                break;
            case e_AppInfoTypeViewerVersion:
                info = "11.007";
                break;
            case e_AppInfoTypeAppVersion:
                info = version;
                break;
            default:
        }
        return info;
    }

    @Override
    public boolean mailData(java.lang.Object data, boolean is_ui, String to, String subject, String cc, String bcc, String message) {
        return false;
    }

    @Override
    public int verifySignature(PDFDoc document, Signature pdf_signature) {
        return -1;
    }

    @Override
    public String getUntitledBookmarkName() {
        return "Untitled";
    }

    @Override
    public WStringArray getPrinterNameList() {
        return new WStringArray();
    }

    @Override
    public boolean addToolButton(ButtonItem button_item) {
        return false;
    }

    @Override
    public boolean removeToolButtom(String button_name) {
        return false;
    }

    @Override
    public MenuListArray getMenuItemNameList() {
        return new MenuListArray();
    }

    @Override
    public boolean addMenuItem(MenuItemConfig menu_item_config, boolean is_prepend) {
        return false;
    }

    @Override
    public boolean addSubMenu(MenuItemConfig menu_item_config) {
        return false;
    }

    @Override
    public boolean showDialog(DialogDescriptionConfig dlg_config) {
        return false;
    }

    @Override
    public boolean getFullScreen() {
        return false;
    }

    @Override
    public void setFullScreen(boolean is_full_screen) {
    }

    @Override
    public void onFieldValueChanged(String field_name, int type, WStringArray value_before_changed, WStringArray value_after_changed) {
    }

    @Override
    public void updateLogicalLabel() {
    }

    @Override
    public int mailDoc(PDFDoc document, String to_address, String cc_address, String bcc_address, String subject, String message, boolean is_ui) {
        return -1;
    }

    @Override
    public String getTemporaryFileName(PDFDoc document, String file_suffix_name) {
        return "";
    }

    @Override
    public MediaPlayerCallback openMediaPlayer(PlayerArgs player_args) {
        return null;
    }

    @Override
    public String getTemporaryDirectory() {
        return "";
    }

    @Override
    public void scroll(PointF point) {
    }

    @Override
    public void selectPageNthWord(int page_index, int start_offset, int end_offset, boolean is_show_selection) {
    }

    @Override
    public PointF getMousePosition() {
        return new PointF();
    }

    @Override
    public RectF getPageWindowRect() {
        return new RectF();
    }

    @Override
    public int getLayoutMode() {
        return ActionCallback.e_LayoutModeContinuous;
    }

    @Override
    public void setLayoutMode(int layout_mode, boolean is_cover_mode) {
    }

    @Override
    public float getPageScale() {
        return 1.0f;
    }

    @Override
    public void setPageScale(int zoom_mode, Destination dest) {
    }

    @Override
    public int getPageZoomMode() {
        return Destination.e_ZoomFitBHorz;
    }

    @Override
    public void query(String keywords, int search_scope, SearchOption search_option, String di_path) {
    }

    @Override
    public SearchIndexConfig addSearchIndex(String di_path, boolean is_selected) {
        return new SearchIndexConfig();
    }

    @Override
    public boolean removeSearchIndex(SearchIndexConfig search_index_config) {
        return true;
    }

    @Override
    public WStringArray getSignatureAPStyleNameList() {
        return new WStringArray();
    }

    @Override
    public SOAPResponseInfo soapRequest(SOAPRequestProperties request_params) {
        return null;
    }

    @Override
    public void enablePageLoop(boolean is_loop) {
    }

    @Override
    public boolean isPageLoop() {
        return false;
    }

    @Override
    public void setDefaultPageTransitionMode(String trans_type, String trans_di) {
    }

    @Override
    public boolean isCurrentDocOpenedInBrowser() {
        return false;
    }

    @Override
    public void postMessageToHtml(WStringArray message) {
    }

    public void setJSActionCallback(IJSActionCallback jsActionCallback) {
        mJSActionCallback = jsActionCallback;
    }

    public interface IJSActionCallback {
        void browseFile();

        void browseFile(boolean is_open_dialog, String file_format, String file_filter);
    }

    private void showPromptDialog(final String tips) {
        AppThreadManager.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                UIToast.getInstance(mContext).showCustomToast(tips);
//                if (mPdfViewCtrl.getAttachedActivity() != null) {
//                    final UITextEditDialog dialog = new UITextEditDialog(mPdfViewCtrl.getAttachedActivity(), UIDialog.NO_INPUT);
//                    dialog.setTitle(AppResource.getString(mContext,R.string.fx_string_warning));
//                    dialog.getPromptTextView().setText(tips);
//                    dialog.getCancelButton().setVisibility(View.GONE);
//                    dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
//                        @Override
//                        public void onClick(View v) {
//                            dialog.dismiss();
//                        }
//                    });
//                    dialog.show();
//                } else {
//                    UIToast.getInstance(mContext).show(tips);
//                }
            }
        });
    }

}
