/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.uiextensions;

import android.Manifest;
import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Fragment;
import android.app.ProgressDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.content.res.Configuration;
import android.graphics.Canvas;
import android.graphics.Point;
import android.graphics.PointF;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.Message;
import android.text.InputType;
import android.text.TextUtils;
import android.util.SparseArray;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.ScaleGestureDetector;
import android.view.View;
import android.view.ViewGroup;
import android.view.Window;
import android.view.WindowManager;
import android.widget.RelativeLayout;

import com.foxit.sdk.PDFException;
import com.foxit.sdk.PDFViewCtrl;
import com.foxit.sdk.Task;
import com.foxit.sdk.addon.xfa.XFADoc;
import com.foxit.sdk.addon.xfa.XFAWidget;
import com.foxit.sdk.common.Range;
import com.foxit.sdk.common.fxcrt.FileReaderCallback;
import com.foxit.sdk.common.fxcrt.FileWriterCallback;
import com.foxit.sdk.pdf.PDFDoc;
import com.foxit.sdk.pdf.annots.Annot;
import com.foxit.sdk.pdf.objects.PDFDictionary;
import com.foxit.sdk.pdf.objects.PDFObject;
import com.foxit.uiextensions.annots.AnnotActionHandler;
import com.foxit.uiextensions.annots.AnnotHandler;
import com.foxit.uiextensions.annots.IAnnotationsPermission;
import com.foxit.uiextensions.annots.caret.CaretModule;
import com.foxit.uiextensions.annots.circle.CircleModule;
import com.foxit.uiextensions.annots.fileattachment.FileAttachmentModule;
import com.foxit.uiextensions.annots.fillsign.FillSignModule;
import com.foxit.uiextensions.annots.fillsign.FillSignToolHandler;
import com.foxit.uiextensions.annots.form.FormFillerModule;
import com.foxit.uiextensions.annots.form.FormFillerToolHandler;
import com.foxit.uiextensions.annots.form.FormNavigationModule;
import com.foxit.uiextensions.annots.freetext.callout.CalloutModule;
import com.foxit.uiextensions.annots.freetext.textbox.TextBoxModule;
import com.foxit.uiextensions.annots.freetext.typewriter.TypewriterModule;
import com.foxit.uiextensions.annots.ink.EraserModule;
import com.foxit.uiextensions.annots.ink.InkDrawToolType;
import com.foxit.uiextensions.annots.ink.InkModule;
import com.foxit.uiextensions.annots.ink.InkToolHandler;
import com.foxit.uiextensions.annots.line.LineModule;
import com.foxit.uiextensions.annots.link.LinkModule;
import com.foxit.uiextensions.annots.multimedia.PhoneStateBroadCastReceiver;
import com.foxit.uiextensions.annots.multimedia.screen.image.PDFImageModule;
import com.foxit.uiextensions.annots.multimedia.screen.multimedia.MultimediaModule;
import com.foxit.uiextensions.annots.multimedia.sound.SoundModule;
import com.foxit.uiextensions.annots.multiselect.MultiSelectModule;
import com.foxit.uiextensions.annots.note.NoteModule;
import com.foxit.uiextensions.annots.polygon.PolygonModule;
import com.foxit.uiextensions.annots.polygon.PolygonToolHandler;
import com.foxit.uiextensions.annots.polyline.PolyLineModule;
import com.foxit.uiextensions.annots.polyline.PolyLineToolHandler;
import com.foxit.uiextensions.annots.popup.PopupModule;
import com.foxit.uiextensions.annots.redaction.RedactModule;
import com.foxit.uiextensions.annots.redaction.RedactToolHandler;
import com.foxit.uiextensions.annots.square.SquareModule;
import com.foxit.uiextensions.annots.stamp.StampModule;
import com.foxit.uiextensions.annots.textmarkup.highlight.HighlightModule;
import com.foxit.uiextensions.annots.textmarkup.squiggly.SquigglyModule;
import com.foxit.uiextensions.annots.textmarkup.strikeout.StrikeoutModule;
import com.foxit.uiextensions.annots.textmarkup.underline.UnderlineModule;
import com.foxit.uiextensions.config.Config;
import com.foxit.uiextensions.config.JsonConstants;
import com.foxit.uiextensions.config.modules.annotations.AnnotationsConfig;
import com.foxit.uiextensions.config.permissions.PermissionsManager;
import com.foxit.uiextensions.config.uisettings.UISettingsManager;
import com.foxit.uiextensions.controls.dialog.AppDialogManager;
import com.foxit.uiextensions.controls.dialog.MatchDialog;
import com.foxit.uiextensions.controls.dialog.UIDialog;
import com.foxit.uiextensions.controls.dialog.UIPopoverWin;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.controls.dialog.fileselect.UIFolderSelectDialog;
import com.foxit.uiextensions.controls.menu.IMenuView;
import com.foxit.uiextensions.controls.menu.MenuViewManager;
import com.foxit.uiextensions.controls.propertybar.IViewSettingsWindow;
import com.foxit.uiextensions.controls.propertybar.PropertyBar;
import com.foxit.uiextensions.controls.toolbar.IBarsHandler;
import com.foxit.uiextensions.controls.toolbar.ToolItemsManager;
import com.foxit.uiextensions.controls.toolbar.impl.BarsHandlerImpl;
import com.foxit.uiextensions.event.DocEventListener;
import com.foxit.uiextensions.event.IUISaveasEventListener;
import com.foxit.uiextensions.event.PageEventListener;
import com.foxit.uiextensions.home.local.LocalModule;
import com.foxit.uiextensions.modules.AutoFlipModule;
import com.foxit.uiextensions.modules.ReflowModule;
import com.foxit.uiextensions.modules.SearchModule;
import com.foxit.uiextensions.modules.UndoModule;
import com.foxit.uiextensions.modules.compare.CompareHandler;
import com.foxit.uiextensions.modules.compare.ComparisonModule;
import com.foxit.uiextensions.modules.crop.CropModule;
import com.foxit.uiextensions.modules.doc.docinfo.DocInfoModule;
import com.foxit.uiextensions.modules.doc.saveas.DocSaveAsModule;
import com.foxit.uiextensions.modules.dynamicxfa.DynamicXFAModule;
import com.foxit.uiextensions.modules.dynamicxfa.DynamicXFAWidgetHandler;
import com.foxit.uiextensions.modules.dynamicxfa.IXFAPageEventListener;
import com.foxit.uiextensions.modules.dynamicxfa.IXFAWidgetEventListener;
import com.foxit.uiextensions.modules.more.MoreMenuModule;
import com.foxit.uiextensions.modules.pagenavigation.PageNavigationModule;
import com.foxit.uiextensions.modules.panel.IPanelManager;
import com.foxit.uiextensions.modules.panel.annot.AnnotPanelModule;
import com.foxit.uiextensions.modules.panel.bookmark.ReadingBookmarkModule;
import com.foxit.uiextensions.modules.panel.filespec.FileSpecPanelModule;
import com.foxit.uiextensions.modules.panel.outline.OutlineModule;
import com.foxit.uiextensions.modules.panel.signature.SignaturePanelModule;
import com.foxit.uiextensions.modules.panzoom.PanZoomModule;
import com.foxit.uiextensions.modules.print.IPrintResultCallback;
import com.foxit.uiextensions.modules.print.PDFPrint;
import com.foxit.uiextensions.modules.print.PDFPrintAdapter;
import com.foxit.uiextensions.modules.print.PrintModule;
import com.foxit.uiextensions.modules.print.XFAPrintAdapter;
import com.foxit.uiextensions.modules.signature.SignatureModule;
import com.foxit.uiextensions.modules.signature.SignatureToolHandler;
import com.foxit.uiextensions.modules.textselect.BlankSelectToolHandler;
import com.foxit.uiextensions.modules.textselect.TextSelectModule;
import com.foxit.uiextensions.modules.textselect.TextSelectToolHandler;
import com.foxit.uiextensions.modules.thumbnail.ThumbnailModule;
import com.foxit.uiextensions.modules.tts.TTSModule;
import com.foxit.uiextensions.pdfreader.ILayoutChangeListener;
import com.foxit.uiextensions.pdfreader.ILifecycleEventListener;
import com.foxit.uiextensions.pdfreader.IMainFrame;
import com.foxit.uiextensions.pdfreader.IStateChangeListener;
import com.foxit.uiextensions.pdfreader.config.ActRequestCode;
import com.foxit.uiextensions.pdfreader.config.ReadStateConfig;
import com.foxit.uiextensions.pdfreader.impl.MainFrame;
import com.foxit.uiextensions.security.digitalsignature.DigitalSignatureModule;
import com.foxit.uiextensions.security.standard.PasswordModule;
import com.foxit.uiextensions.security.trustcertificate.TrustCertificateModule;
import com.foxit.uiextensions.theme.BaseThemeAdapter;
import com.foxit.uiextensions.theme.IThemeChangeObserver;
import com.foxit.uiextensions.theme.ThemeConfig;
import com.foxit.uiextensions.utils.AppAnnotUtil;
import com.foxit.uiextensions.utils.AppDarkUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppDmUtil;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppSharedPreferences;
import com.foxit.uiextensions.utils.AppStorageManager;
import com.foxit.uiextensions.utils.AppUtil;
import com.foxit.uiextensions.utils.Event;
import com.foxit.uiextensions.utils.UIToast;
import com.foxit.uiextensions.utils.thread.AppThreadManager;

import java.io.File;
import java.io.FileFilter;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

import androidx.annotation.IntDef;
import androidx.annotation.NonNull;
import androidx.core.content.ContextCompat;

import static com.foxit.sdk.common.Constants.e_ErrPassword;
import static com.foxit.sdk.common.Constants.e_ErrSuccess;

/**
 * Class <CODE>UIExtensionsManager</CODE> represents a UI extensions manager.
 * <p>
 * The <CODE>UIExtensionsManager</CODE> class is mainly used for managing the UI extensions which implement {@link IPDFReader} interface, it implements the {@link PDFViewCtrl.UIExtensionsManager}
 * interface that is a listener to listen common interaction events and view event, and will dispatch some events to UI extensions, it also defines functions to manage the UI extensions.
 */
public class UIExtensionsManager implements PDFViewCtrl.UIExtensionsManager, IPDFReader, IThemeChangeObserver {

    /**
     * Interface used to allow the user to run some code
     * when the current {@link ToolHandler} has changed.
     */
    public interface ToolHandlerChangedListener {
        /**
         * Called when current {@link ToolHandler} has changed.
         *
         * @param oldToolHandler The old tool handler.
         * @param newToolHandler The new tool handler.
         */
        void onToolHandlerChanged(ToolHandler oldToolHandler, ToolHandler newToolHandler);
    }

    /**
     * Interface used to allow the user to run some code
     * when the device configuration changes
     */
    public interface ConfigurationChangedListener {
        /**
         * Called when {@link UIExtensionsManager#onConfigurationChanged(Activity, Configuration)} is called.
         *
         * @param newConfig The new device configuration.
         */
        void onConfigurationChanged(Configuration newConfig);
    }

    /**
     * Interface used to allow the user to run some code
     * when do some operations on menu
     */
    public interface MenuEventListener {
        /**
         * Called when {@link #triggerDismissMenuEvent()} is called.
         */
        void onTriggerDismissMenu();
    }

    public interface ITouchEventListener {
        boolean onTouchEvent(int pageIndex, MotionEvent motionEvent);

        boolean onSingleTapConfirmed(int pageIndex, MotionEvent motionEvent);

        boolean onLongPress(int pageIndex, MotionEvent motionEvent);
    }

    private ILinkEventListener mLinkEventListener = null;
    /**
     * link type: annotation
     */
    public static final int LINKTYPE_ANNOT = 0;
    /**
     * link type: text
     */
    public static final int LINKTYPE_TEXT = 1;

    /**
     * night color mode: default
     */
    public static final int NIGHTCOLORMODE_DEFAULT = 0;

    /**
     * night color mode: mapping gray
     */
    public static final int NIGHTCOLORMODE_MAPPINGGRAY = 1;

    /**
     * Class definition for link information
     */
    public static class LinkInfo {

        /**
         * should be one of {@link #LINKTYPE_ANNOT}, {@link #LINKTYPE_TEXT}
         */
        public int linkType;
        /**
         * Should be link annotation or text link.
         */
        public Object link;
    }

    /**
     * Interface used to allow the user to run some code when do some operations on link
     */
    public interface ILinkEventListener {
        /**
         * Called when tap a link.
         *
         * @param linkInfo The link information of the tapped link.
         * @return Return <code>true</code> to prevent this event from being propagated
         * further, or <code>false</code> to indicate that you have not handled
         * this event and it should continue to be propagated by Foxit.
         */
        boolean onLinkTapped(LinkInfo linkInfo);
    }

    /**
     * Set link event listener.
     *
     * @param listener The specified link event listener.
     */
    public void setLinkEventListener(ILinkEventListener listener) {
        mLinkEventListener = listener;
    }

    /**
     * Interface used to allow the user to run some code
     * when close document and exit the current activity
     */
    public interface OnFinishListener {
        /**
         * Usually called when close document and exit the current activity.
         */
        void onFinish();
    }

    /**
     * Set the {@link OnFinishListener} to be invoked when the document closed and current activity has exited.
     *
     * @param listener the {@link OnFinishListener} to use.
     */
    public void setOnFinishListener(OnFinishListener listener) {
        onFinishListener = listener;
    }

    /**
     * Get link event listener object.
     *
     * @return The link event listener object.
     */
    public ILinkEventListener getLinkEventListener() {
        return mLinkEventListener;
    }

    private ToolHandler mCurToolHandler = null;
    private PDFViewCtrl mPdfViewCtrl = null;
    private List<Module> mModules = new CopyOnWriteArrayList<>();
    private HashMap<String, ToolHandler> mToolHandlerList;
    private SparseArray<AnnotHandler> mAnnotHandlerList;
    private ArrayList<ToolHandlerChangedListener> mHandlerChangedListeners;
    private ArrayList<ILayoutChangeListener> mLayoutChangedListeners;
    // IXFAPageEventListener
    private ArrayList<IXFAPageEventListener> mXFAPageEventListeners;
    private ArrayList<IXFAWidgetEventListener> mXFAWidgetEventListener;
    private ArrayList<ConfigurationChangedListener> mConfigurationChangedListeners;
    private ArrayList<IInteractionEventListener> mInteractionListeners;
    private ArrayList<IThemeEventListener> mThemeEventListeners;
    private ArrayList<MenuEventListener> mMenuEventListeners;
    private SparseArray<IUISaveasEventListener> mSaveEventListeners;
    private SparseArray<PropertyBar.CreatePropertyChangedListener> mCreatePropertyChangedListener;

    private Activity mAttachActivity = null;
    private Context mContext;
    private ViewGroup mParent;
    private Config mConfig;
    private boolean continueAddAnnot = false;
    private boolean mEnableLinkAnnot = true;
    private boolean mEnableLinkHighlight = true;
    private boolean mEnableFormHighlight = true;
    private boolean mIsUseLogicalPage = false;
    private long mFormHighlightColor = 0x200066cc;
    private long mLinkHighlightColor = 0x16007FFF;
    private int mSelectHighlightColor = 0xFFACDAED;
    private IPanelManager mPanelManager = null;
    private ToolItemsManager mBarToolsManager;

    private DocumentManager documentManager;

    private String mAnnotAuthor;
    private String currentFileCachePath = null;
    private String mSavePath = null; // default save path
    private String mDocPath = "";
    private String mRealDocPath = null;
    private String mProgressMsg = null;
    private int mState = ReadStateConfig.STATE_NORMAL;
    private int mSaveFlag = PDFDoc.e_SaveFlagRemoveRedundantObjects;//todo SDKRD-15323
    private boolean bDocClosed = false;
    private boolean mPasswordError = false;
    private boolean isSaveDocInCurPath = false;
    protected boolean isCloseDocAfterSaving = false;

    private List<ILifecycleEventListener> mLifecycleEventList;
    private List<IPolicyEventListener> mPolicyEventListeners;
    private List<IStateChangeListener> mStateChangeEventList;

    private MainFrame mMainFrame;
    private IBarsHandler mBaseBarMgr;
    private MenuViewManager mCustomViewMgr;
    private ProgressDialog mProgressDlg;
    private OnFinishListener onFinishListener;
    private BackEventListener mBackEventListener = null;
    private AlertDialog mSaveAlertDlg;
    private UIFolderSelectDialog mFolderSelectDialog;

    private RelativeLayout mActivityLayout;
    private String mUserSavePath = null;
    private boolean mIsDocOpened = false;
    private static final int TOOL_BAR_SHOW_TIME = 8000;

    // for compare docuemnt
    private boolean mOldFileIsTwoColumnLeft = false;
    private int mOldPageLayout;
    private boolean mIsCompareDoc = false;
    private boolean mIsAutoSaveDoc = false;
    private boolean mCanUpdateAnnotDefaultProperties = false;
    private int mCurNightMode;
    private int mNightColorMode = NIGHTCOLORMODE_DEFAULT;
    private int mPageColorMode = NIGHTCOLORMODE_MAPPINGGRAY;
    private @OpenFileType
    int mOpenFileType = OPEN_FILE_TYPE_PATH;
    private static final int OPEN_FILE_TYPE_PATH = 0;
    private static final int OPEN_FILE_TYPE_STREAM = 1;
    private long mInkTimeOut = -1;

    @IntDef({OPEN_FILE_TYPE_PATH, OPEN_FILE_TYPE_STREAM})
    @Retention(RetentionPolicy.SOURCE)
    @interface OpenFileType {
    }

    private FileReaderCallback mFileReader = null;
    private FileWriterCallback mFileWriter = null;

    private boolean mAutoFullScreen = true;

    /**
     * Instantiates a new UI extensions manager.
     *
     * @param context     A <CODE>Context</CODE> object which species the context.
     * @param pdfViewCtrl A <CODE>PDFViewCtrl</CODE> object which species the PDF view control.
     */
    public UIExtensionsManager(Context context, PDFViewCtrl pdfViewCtrl) {
        init(context, pdfViewCtrl, null);
    }

    /**
     * Instantiates a new UI extensions manager with modules config.
     *
     * @param context     A <CODE>Context</CODE> object which species the context.
     * @param pdfViewCtrl A <CODE>PDFViewCtrl</CODE> object which species the PDF view control.
     * @param config      A <CODE>Config</CODE> object which species a modules loading config,
     *                    if null, UIExtension manager will load all modules by default, and equal to {@link #UIExtensionsManager(Context, PDFViewCtrl)}.
     */
    public UIExtensionsManager(Context context, PDFViewCtrl pdfViewCtrl, Config config) {
        init(context, pdfViewCtrl, config);
    }

    private void init(Context context, PDFViewCtrl pdfViewCtrl, Config config) {
        AppUtil.initContext(context);
        AppUtil.requireNonNull(pdfViewCtrl, "PDF view control can't be null");
        AppDisplay.Instance(context);

        mContext = context;
        mPdfViewCtrl = pdfViewCtrl;
        mPdfViewCtrl.setBackgroundColor(AppResource.getColor(mContext, R.color.ux_bg_color_docviewer));
        documentManager = new DocumentManager(pdfViewCtrl);
        int uiMode = context.getResources().getConfiguration().uiMode;
        mCurNightMode = uiMode & Configuration.UI_MODE_NIGHT_MASK;

        mAnnotAuthor = AppDmUtil.getAnnotAuthor();
        mToolHandlerList = new HashMap<>(8);
        mAnnotHandlerList = new SparseArray<>(8);
        mHandlerChangedListeners = new ArrayList<>();
        mLayoutChangedListeners = new ArrayList<>();
        mXFAPageEventListeners = new ArrayList<>();
        mXFAWidgetEventListener = new ArrayList<>();
        mConfigurationChangedListeners = new ArrayList<>();
        mInteractionListeners = new ArrayList<>();
        mThemeEventListeners = new ArrayList<>();
        mLifecycleEventList = new CopyOnWriteArrayList<>();
        mStateChangeEventList = new ArrayList<>();
        mPolicyEventListeners = new ArrayList<>();
        mMenuEventListeners = new ArrayList<>();
        mSaveEventListeners = new SparseArray<>();
        mCreatePropertyChangedListener = new SparseArray<>();

        pdfViewCtrl.registerDocEventListener(mDocEventListener);
        pdfViewCtrl.registerRecoveryEventListener(mRecoveryEventListener);
        pdfViewCtrl.registerDoubleTapEventListener(mDoubleTapEventListener);
        pdfViewCtrl.registerTouchEventListener(mTouchEventListener);
        pdfViewCtrl.registerPageEventListener(mPageEventListener);
        if (ThemeConfig.getInstance(context).getAdapter() == null) {
            ThemeConfig.getInstance(context).setAdapter(new BaseThemeAdapter());
        }
        ThemeConfig.getInstance(context).getAdapter().registerThemeChangeObserver(this);

        pdfViewCtrl.setUIExtensionsManager(this);
        registerMenuEventListener(mMenuEventListener);

        if (config == null) {
            mConfig = new Config();
        } else {
            mConfig = config;
        }

        mMainFrame = new MainFrame(mContext, mConfig);
        mBaseBarMgr = new BarsHandlerImpl(mContext, mMainFrame);

        if (mActivityLayout == null) {
            mActivityLayout = new RelativeLayout(mContext);
        } else {
            mActivityLayout.removeAllViews();
            mActivityLayout = new RelativeLayout(mContext);
        }
        mActivityLayout.setId(R.id.rd_main_id);
        RelativeLayout.LayoutParams params = new RelativeLayout.LayoutParams(
                RelativeLayout.LayoutParams.MATCH_PARENT, RelativeLayout.LayoutParams.MATCH_PARENT);
        mBarToolsManager = new ToolItemsManager();
        mMainFrame.init(this);
        mPanelManager = mMainFrame.getPanelManager();
        mMainFrame.addDocView(mPdfViewCtrl);
        mActivityLayout.addView(mMainFrame.getContentView(), params);
        mParent = mMainFrame.getContentView();
        mParent.addOnLayoutChangeListener(new View.OnLayoutChangeListener() {
            @Override
            public void onLayoutChange(View v, int left, int top, int right, int bottom, int oldLeft, int oldTop, int oldRight, int oldBottom) {
                int newWidth = right - left;
                int newHeight = bottom - top;
                int oldWidth = oldRight - oldLeft;
                int oldHeight = oldBottom - oldTop;
                if (oldWidth != newWidth || oldHeight != newHeight){
                    mPdfViewCtrl.postPageContainer();
                }
                UIExtensionsManager.this.onLayoutChange(v, newWidth, newHeight, oldWidth, oldHeight);
            }
        });
        if (mMainFrame.getAttachedActivity() != null) {
            setAttachedActivity(mMainFrame.getAttachedActivity());
        }
        loadAllModules();
        initActionHandler();
    }

    private void initActionHandler() {
        AnnotActionHandler annotActionHandler = (AnnotActionHandler) getDocumentManager().getActionCallback();
        if (annotActionHandler == null) {
            annotActionHandler = new AnnotActionHandler(mContext, mPdfViewCtrl);
        }
        getDocumentManager().setActionCallback(annotActionHandler);
    }

    protected boolean loadModule(String name) {
        return false;
    }

    protected boolean unloadModule(String name) {
        for (int i = 0; i < mModules.size(); i ++) {
            Module module = mModules.get(i);
            if (AppUtil.isEqual(module.getName(), name)) {
                module.unloadModule();
                mModules.remove(module);

                changeState(getState());
                return true;
            }
        }
        return false;
    }

    /**
     * By default, all modules will be loaded..
     */
    protected void loadAllModules() {
        if (mConfig.modules.isLoadTextSelection) {
            //text select module
            TextSelectModule tsModule = new TextSelectModule(mContext, mPdfViewCtrl, this);
            tsModule.loadModule();
        }

        if (mConfig.modules.isLoadAnnotations || mConfig.modules.isLoadSignature) {
            registerToolHandler(new BlankSelectToolHandler(mContext, mPdfViewCtrl, this));
        }

        //undo&redo module
        UndoModule undoModule = new UndoModule(mContext, mPdfViewCtrl, this);
        undoModule.loadModule();

        //link module
        LinkModule linkModule = new LinkModule(mContext, mPdfViewCtrl, this);
        linkModule.loadModule();

        DocInfoModule docInfoModule = new DocInfoModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
        docInfoModule.loadModule();
        DocSaveAsModule saveAsModule = new DocSaveAsModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
        saveAsModule.loadModule();
        PrintModule printModule = new PrintModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
        printModule.loadModule();
        // above three modules are required to be loaded before more menu module
        MoreMenuModule moreMenuModule = new MoreMenuModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
        moreMenuModule.loadModule();

        if (mConfig.modules.isLoadAnnotations) {
            final AnnotationsConfig annotConfig = mConfig.modules.annotations;
            mPdfViewCtrl.addTask(new Task(mLoadModuleCallback) {
                @Override
                protected void execute() {
                    if (annotConfig.isLoadSquiggly) {
                        //squiggly annotation module
                        SquigglyModule sqgModule = new SquigglyModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        sqgModule.loadModule();
                    }

                    if (annotConfig.isLoadStrikeout || annotConfig.isLoadReplaceText) {
                        //strikeout annotation module
                        StrikeoutModule stoModule = new StrikeoutModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        stoModule.loadModule();
                    }

                    if (annotConfig.isLoadUnderline) {
                        //underline annotation module
                        UnderlineModule unlModule = new UnderlineModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        unlModule.loadModule();
                    }

                    if (annotConfig.isLoadHighlight) {
                        //highlight annotation module
                        HighlightModule hltModule = new HighlightModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        hltModule.loadModule();
                    }

//            if (annotConfig.isLoadNote()) {
                    //note annotation module
                    NoteModule noteModule = new NoteModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                    noteModule.loadModule();
//            }

                    if (annotConfig.isLoadCircle) {
                        //circle module
                        CircleModule circleModule = new CircleModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        circleModule.loadModule();
                    }

                    if (annotConfig.isLoadSquare) {
                        //square module
                        SquareModule squareModule = new SquareModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        squareModule.loadModule();
                    }

                    if (annotConfig.isLoadTypewriter) {
                        //freetext: typewriter
                        TypewriterModule typewriterModule = new TypewriterModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        typewriterModule.loadModule();
                    }

                    if (annotConfig.isLoadCallout) {
                        // freetext: callout
                        CalloutModule calloutModule = new CalloutModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        calloutModule.loadModule();
                    }

                    if (annotConfig.isLoadStamp) {
                        //stamp module
                        StampModule stampModule = new StampModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                        stampModule.loadModule();
                    }

                    if (annotConfig.isLoadInsertText || annotConfig.isLoadReplaceText) {
                        //Caret module
                        CaretModule caretModule = new CaretModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        caretModule.loadModule();
                    }

                    if (annotConfig.isLoadPencil || annotConfig.isLoadEraser) {
                        //ink(pencil) module
                        InkModule inkModule = new InkModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        inkModule.loadModule();
                    }

                    if (annotConfig.isLoadEraser) {
                        //eraser module
                        EraserModule eraserModule = new EraserModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        eraserModule.loadModule();
                    }

                    if (annotConfig.isLoadLine || annotConfig.isLoadArrow || annotConfig.isLoadMeasure) {
                        //Line module
                        LineModule lineModule = new LineModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        lineModule.loadModule();
                    }

                    if (annotConfig.isLoadFileattachment) {
                        //FileAttachment module
                        FileAttachmentModule fileAttachmentModule = new FileAttachmentModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        fileAttachmentModule.loadModule();
                    }

                    if (annotConfig.isLoadPolygon || annotConfig.isLoadCloud) {
                        //Polygon module
                        PolygonModule polygonModule = new PolygonModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        polygonModule.loadModule();
                    }

                    if (annotConfig.isLoadPolyLine) {
                        //PolyLine module
                        PolyLineModule polyLineModule = new PolyLineModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        polyLineModule.loadModule();
                    }

                    if (annotConfig.isLoadTextbox) {
                        //Textbox module
                        TextBoxModule textBoxModule = new TextBoxModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        textBoxModule.loadModule();
                    }

                    if (annotConfig.isLoadImage) {
                        // Image module
                        PDFImageModule imageModule = new PDFImageModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        imageModule.loadModule();
                    }

                    if (annotConfig.isLoadVideo || annotConfig.isLoadAudio) {
                        // multimedia module
                        MultimediaModule imageModule = new MultimediaModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        imageModule.loadModule();
                    }

                    if (annotConfig.isLoadAudio) {
                        SoundModule soundModule = new SoundModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                        soundModule.loadModule();
                    }

                    //popup annot module
                    PopupModule popupModule = new PopupModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                    popupModule.loadModule();
                }
            });
            if (annotConfig.isLoadRedaction) {
                //Redact module
                RedactModule redactModule = new RedactModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                redactModule.loadModule();
            }
        }

        if (mConfig.modules.isLoadPageNavigation) {
            //page navigation module
            PageNavigationModule pageNavigationModule = new PageNavigationModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
            pageNavigationModule.loadModule();
        }

        if (mConfig.modules.isLoadSearch) {
            SearchModule searchModule = new SearchModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
            searchModule.loadModule();
        }

        if (mConfig.modules.isLoadReadingBookmark) {
            ReadingBookmarkModule readingBookmarkModule = new ReadingBookmarkModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
            readingBookmarkModule.loadModule();
        }

        if (mConfig.modules.isLoadThumbnail) {
            ThumbnailModule thumbnailModule = new ThumbnailModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
            thumbnailModule.loadModule();
        }

        if (mConfig.modules.isLoadMultiSelect && mConfig.modules.isLoadAnnotations) {
            MultiSelectModule multiSelectModule = new MultiSelectModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
            multiSelectModule.loadModule();
        }

        TTSModule ttsModule = new TTSModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
        ttsModule.loadModule();

        ReflowModule reflowModule = new ReflowModule(mContext, mParent, mPdfViewCtrl, com.foxit.uiextensions.UIExtensionsManager.this);
        reflowModule.loadModule();

        mPdfViewCtrl.addTask(new Task(mLoadModuleCallback) {
            @Override
            protected void execute() {
                if (mConfig.modules.isLoadForm) {
                    //form annotation module
                    FormFillerModule formFillerModule = new FormFillerModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                    formFillerModule.loadModule();
                }

                if (mConfig.modules.isLoadSignature) {
                    //signature module
                    SignatureModule signatureModule = new SignatureModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                    signatureModule.loadModule();

                    DigitalSignatureModule dsgModule = new DigitalSignatureModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                    dsgModule.loadModule();

                    TrustCertificateModule trustCertModule = new TrustCertificateModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                    trustCertModule.loadModule();
                }

                if (mConfig.modules.isLoadFillSign || mConfig.modules.isLoadSignature) {
                    //fillsign module
                    FillSignModule fillSignModule = new FillSignModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                    fillSignModule.loadModule();
                }

                if (mConfig.modules.isLoadOutline) {
                    OutlineModule outlineModule = new OutlineModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                    outlineModule.loadModule();
                }

                if (mConfig.modules.isLoadAnnotations) {
                    //annot panel
                    AnnotPanelModule annotPanelModule = new AnnotPanelModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                    annotPanelModule.loadModule();
                }

                if (mConfig.modules.isLoadAttachment) {
                    FileSpecPanelModule fileSpecPanelModule = new FileSpecPanelModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                    fileSpecPanelModule.loadModule();
                }

                if (mConfig.modules.isLoadFileEncryption) {
                    //password module
                    PasswordModule passwordModule = new PasswordModule(mContext, mPdfViewCtrl, UIExtensionsManager.this);
                    passwordModule.loadModule();
                }

                if (!mConfig.uiSettings.disableFormNavigationBar) {
                    //form navigation module
                    FormNavigationModule formNavigationModule = new FormNavigationModule(mContext, mParent, UIExtensionsManager.this);
                    formNavigationModule.loadModule();
                }

                if (mConfig.modules.isLoadSignature) {
                    // signature panel
                    SignaturePanelModule signaturePanelModule = new SignaturePanelModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                    signaturePanelModule.loadModule();
                }

                CropModule cropModule = new CropModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                cropModule.loadModule();

                PanZoomModule panZoomModule = new PanZoomModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                panZoomModule.loadModule();

                DynamicXFAModule dynamicXFAModule = new DynamicXFAModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                dynamicXFAModule.loadModule();

                ComparisonModule comparisonModule = new ComparisonModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                comparisonModule.loadModule();

                AutoFlipModule autoFlipModule = new AutoFlipModule(mContext, mParent, mPdfViewCtrl, UIExtensionsManager.this);
                autoFlipModule.loadModule();
            }
        });

        //json config
        PermissionsManager permissionsManager = new PermissionsManager(mContext, mPdfViewCtrl);
        permissionsManager.setPermissions();

        UISettingsManager uiSettingsManager = new UISettingsManager(mContext, mPdfViewCtrl);
        uiSettingsManager.setUISettings();
    }

    private final Task.CallBack mLoadModuleCallback = new Task.CallBack() {
        int doneCount = 0;
        @Override
        public void result(Task task) {
            if (!mConfig.modules.isLoadAnnotations || ++doneCount == 2) {
                setToolHandlerContinueAddAnnot();
                mMainFrame.asyncInitTaskFinished();
            }
        }
    };

    /**
     * @return the root view
     */
    public ViewGroup getRootView() {
        return mParent;
    }

//    @Override
//    protected void finalize() throws Throwable {
//        super.finalize();
//    }

    /**
     * Register a callback to be invoked when the tool handler changed.
     * <p>
     * Note: This method is only used within RDK
     */
    public void registerToolHandlerChangedListener(ToolHandlerChangedListener listener) {
        mHandlerChangedListeners.add(listener);
    }

    /**
     * Unregister the {@link ToolHandler} changed listener.
     * <p>
     * Note: This method is only used within RDK
     *
     * @param listener a {@link ToolHandlerChangedListener} to use.
     */
    public void unregisterToolHandlerChangedListener(ToolHandlerChangedListener listener) {
        mHandlerChangedListeners.remove(listener);
    }

    private void onToolHandlerChanged(ToolHandler lastTool, ToolHandler currentTool) {
        for (ToolHandlerChangedListener listener : mHandlerChangedListeners) {
            listener.onToolHandlerChanged(lastTool, currentTool);
        }
    }

    private void onkeyDown(ToolHandler lastTool, ToolHandler currentTool) {
        for (ToolHandlerChangedListener listener : mHandlerChangedListeners) {
            listener.onToolHandlerChanged(lastTool, currentTool);
        }
    }

    /**
     * Register a callback to be invoked when the interaction event happened.
     *
     * @param listener the interaction event {@link IInteractionEventListener}
     */
    public void registerInteractionListener(IInteractionEventListener listener) {
        mInteractionListeners.add(listener);
    }

    /**
     * Unregister the interaction event listener.
     *
     * @param listener the interaction event {@link IInteractionEventListener}
     */
    public void unregisterInteractionListener(IInteractionEventListener listener) {
        mInteractionListeners.remove(listener);
    }

    /**
     * Register a callback to be invoked when the theme changed.
     *
     * @param listener the theme event {@link IThemeEventListener}
     */
    public void registerThemeEventListener(IThemeEventListener listener) {
        mThemeEventListeners.add(listener);
    }

    /**
     * Unregister the theme event listener.
     * @param listener the theme event {@link IThemeEventListener}
     */
    public void unregisterThemeEventListener(IThemeEventListener listener) {
        mThemeEventListeners.remove(listener);
    }

    /**
     * Register a callback to be invoked when the configuration changed.
     *
     * @param listener the configuration changed {@link ConfigurationChangedListener}
     */
    public void registerConfigurationChangedListener(ConfigurationChangedListener listener) {
        mConfigurationChangedListeners.add(listener);
    }

    /**
     * unregister the specified {@link ConfigurationChangedListener}.
     *
     * @param listener the specified {@link ConfigurationChangedListener}
     */
    public void unregisterConfigurationChangedListener(ConfigurationChangedListener listener) {
        mConfigurationChangedListeners.remove(listener);
    }

    /**
     * Register a xfa page event listener.
     *
     * @param listener An <CODE>IPageEventListener</CODE> object to be registered.
     */
    public void registerXFAPageEventListener(IXFAPageEventListener listener) {
        mXFAPageEventListeners.add(listener);
    }

    /**
     * Unregister a xfa page event listener.
     *
     * @param listener An <CODE>IPageEventListener</CODE> object to be unregistered.
     */
    public void unregisterXFAPageEventListener(IXFAPageEventListener listener) {
        mXFAPageEventListeners.remove(listener);
    }

    /**
     * Called when the specified xfa page has removed.
     */
    public void onXFAPageRemoved(boolean isSuccess, int pageIndex) {
        for (IXFAPageEventListener listener : mXFAPageEventListeners) {
            listener.onPagesRemoved(isSuccess, pageIndex);
        }
    }

    /**
     * Called when a xfa page has added in the specified position.
     */
    public void onXFAPagesInserted(boolean isSuccess, int pageIndex) {
        for (IXFAPageEventListener listener : mXFAPageEventListeners) {
            listener.onPagesInserted(isSuccess, pageIndex);
        }
    }


    /**
     * Register a xfa widget event listener.
     *
     * @param listener An <CODE>IXFAWidgetEventListener</CODE> object to be registered.
     */
    public void registerXFAWidgetEventListener(IXFAWidgetEventListener listener) {
        mXFAWidgetEventListener.add(listener);
    }

    /**
     * Unregister a xfa widget event listener.
     *
     * @param listener An <CODE>IXFAWidgetEventListener</CODE> object to be unregistered.
     */
    public void unregisterXFAWidgetEventListener(IXFAWidgetEventListener listener) {
        mXFAWidgetEventListener.remove(listener);
    }

    /**
     * Called when a {@link XFAWidget} has added.
     */
    public void onXFAWidgetAdded(XFAWidget xfaWidget) {
        for (IXFAWidgetEventListener listener : mXFAWidgetEventListener) {
            listener.onXFAWidgetAdded(xfaWidget);
        }
    }

    /**
     * Called when a {@link XFAWidget} will be removed.
     */
    public void onXFAWidgetWillRemove(XFAWidget xfaWidget) {
        for (IXFAWidgetEventListener listener : mXFAWidgetEventListener) {
            listener.onXFAWidgetWillRemove(xfaWidget);
        }
    }

    /// @cond DEV
    public void registerLayoutChangeListener(ILayoutChangeListener listener) {
        mLayoutChangedListeners.add(listener);
    }

    public void unregisterLayoutChangeListener(ILayoutChangeListener listener) {
        mLayoutChangedListeners.remove(listener);
    }

    private void onLayoutChange(View view, int newWidth, int newHeight, int oldWidth, int oldHeight) {

        //for (ILayoutChangeListener listener : mLayoutChangedListeners) { // MOBRD-2722: java.util.ConcurrentModificationException:
        for (int i = 0; i < mLayoutChangedListeners.size(); i++) {
            ILayoutChangeListener listener = mLayoutChangedListeners.get(i);
            listener.onLayoutChange(view, newWidth, newHeight, oldWidth, oldHeight);
        }
    }
    /// @endcond

    /**
     * Set the current tool handler.
     *
     * @param toolHandler A <CODE>ToolHandler</CODE> object which specifies the current tool handler.
     */
    public void setCurrentToolHandler(ToolHandler toolHandler) {
        if (toolHandler == null && mCurToolHandler == null) {
            return;
        }

        boolean canAdd = true;
        if (toolHandler != null && !toolHandler.getType().equals(ToolHandler.TH_TYPE_SELECT_ANNOTATIONS)) {
            if (toolHandler.getType().equals(ToolHandler.TH_TYPE_SIGNATURE)
                    || toolHandler.getType().equals(ToolHandler.TH_TYPE_FILLSIGN)) {
                canAdd = getDocumentManager().canAddSignature()&& ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).isEnableModification();
            } else if(toolHandler.getType().equals(ToolHandler.TH_TYPE_SCREEN_AUDIO)
                    || toolHandler.getType().equals(ToolHandler.TH_TYPE_SCREEN_VIDEO)
                    || toolHandler.getType().equals(ToolHandler.TH_TYPE_LINK)){
                canAdd = getDocumentManager().canEdit()&& ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).isEnableModification();
            }else {
                // Now, others are all annotation tool handler
                canAdd = getDocumentManager().canAddAnnot()&& ((UIExtensionsManager)mPdfViewCtrl.getUIExtensionsManager()).isEnableModification();
            }
        }

        if (!canAdd || (toolHandler != null && mCurToolHandler != null && mCurToolHandler.getType().equals(toolHandler.getType()))) {
            return;
        }
        ToolHandler lastToolHandler = mCurToolHandler;
        if (lastToolHandler != null) {
            lastToolHandler.onDeactivate();
        }

        if (toolHandler != null) {
            if (getDocumentManager().getCurrentAnnot() != null) {
                getDocumentManager().setCurrentAnnot(null);
            }
        }

        mCurToolHandler = toolHandler;
        if (mCurToolHandler != null) {
            mCurToolHandler.onActivate();
        }

        changeToolBarState(lastToolHandler, mCurToolHandler);
        onToolHandlerChanged(lastToolHandler, mCurToolHandler);
    }

    /**
     * Get the current tool handler.
     *
     * @return A <CODE>ToolHandler</CODE> object which specifies the current tool handler.
     */
    public ToolHandler getCurrentToolHandler() {
        return mCurToolHandler;
    }

    /**
     * Register the specified {@link ToolHandler} to current UI extensions manager.
     *
     * @param handler A <CODE>ToolHandler</CODE> object to be registered.
     */
    public void registerToolHandler(ToolHandler handler) {
        mToolHandlerList.put(handler.getType(), handler);
    }

    /**
     * Unregister the specified {@link ToolHandler} from current UI extensions manager.
     * <p>
     * Note: This method is only used within RDK
     *
     * @param handler A <CODE>ToolHandler</CODE> object to be unregistered.
     */
    public void unregisterToolHandler(ToolHandler handler) {
        mToolHandlerList.remove(handler.getType());
    }

    /**
     * get the specified {@link ToolHandler} from current UI extensions manager.
     *
     * @param type The tool handler type, refer to function {@link ToolHandler#getType()}.
     * @return A <CODE>ToolHandler</CODE> object with specified type.
     */
    public ToolHandler getToolHandlerByType(String type) {
        return mToolHandlerList.get(type);
    }

    /**
     * Register the specified {@link AnnotHandler} to current UI extensions manager.
     *
     * @param handler A {@link AnnotHandler} to use.
     */
    public void registerAnnotHandler(AnnotHandler handler) {
        mAnnotHandlerList.put(handler.getType(), handler);
    }

    /**
     * Unregister the specified {@link AnnotHandler} from current UI extensions manager.
     *
     * @param handler A {@link AnnotHandler} to use.
     */
    public void unregisterAnnotHandler(AnnotHandler handler) {
        mAnnotHandlerList.remove(handler.getType());
    }

    /**
     * @return Get the current {@link AnnotHandler}
     */
    public AnnotHandler getCurrentAnnotHandler() {
        Annot curAnnot = getDocumentManager().getCurrentAnnot();
        if (curAnnot == null) {
            return null;
        }

        return getAnnotHandlerByType(getDocumentManager().getAnnotHandlerType(curAnnot));
    }

    /**
     * Get the specified {@link AnnotHandler} from current UI extensions manager.
     *
     * @param type The type of {@link AnnotHandler}, refer to {@link AnnotHandler#getType()};
     * @return A {@link AnnotHandler} object with specified type.
     */
    public AnnotHandler getAnnotHandlerByType(int type) {
        return mAnnotHandlerList.get(type);
    }

    /**
     * Register the specified module to current UI extensions manager.
     * <p>
     * Note: This method is only used within RDK
     *
     * @param module A <CODE>Module</CODE> object to be registered.
     */
    public void registerModule(Module module) {
        mModules.add(module);
    }

    /**
     * Unregister the specified module from current UI extensions manager.
     * Note: This method is only used within RDK
     *
     * @param module A <CODE>Module</CODE> object to be unregistered.
     */
    public void unregisterModule(Module module) {
        mModules.remove(module);
    }


    /**
     * Get the specified module from current UI extensions manager.
     *
     * @param name The specified module name, refer to {@link Module#getName()}.
     * @return A <CODE>Module</CODE> object with specified module name.
     */
    public Module getModuleByName(String name) {
        if (mModules == null)
            return null;
        for (Module module : mModules) {
            String moduleName = module.getName();
            if (moduleName != null && moduleName.compareTo(name) == 0)
                return module;
        }
        return null;
    }

    /**
     * Whether or not the annotation can be created continuously.
     *
     * @return <CODE>True</CODE> the annotation can be created continuously or otherwise.
     */
    public boolean isContinueAddAnnot() {
        return continueAddAnnot;
    }

    /**
     * Set whether the annotation can be created continuously. The default is false.
     *
     * @param continueAddAnnot whether the annot can be created continuously.
     */
    public void setContinueAddAnnot(boolean continueAddAnnot) {
        this.continueAddAnnot = continueAddAnnot;
    }

    private void setToolHandlerContinueAddAnnot() {
        try {
            //for (Map.Entry<String, ToolHandler> entry : mToolHandlerList.entrySet()) {
            ArrayList<ToolHandler> toolHandlers = new ArrayList<>(mToolHandlerList.values());
            for (int i = 0; i < toolHandlers.size(); i++) {
                ToolHandler toolHandler = toolHandlers.get(i);
                toolHandler.setContinueAddAnnot(continueAddAnnot);
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Enable link annotation action event.
     *
     * @param enable True means link annotation action event can be triggered, false for else.
     */
    public void enableLinks(boolean enable) {
        mEnableLinkAnnot = enable;
    }

    /**
     * Check whether link annotation action event can be triggered.
     *
     * @return True means link annotation action event can be triggered, false for else.
     */
    public boolean isLinksEnabled() {
        return mEnableLinkAnnot;
    }

    /**
     * Check whether link highlight can be display.
     *
     * @return True means link highlight can be displayed, false for else.
     */
    public boolean isLinkHighlightEnabled() {
        return mEnableLinkHighlight;
    }

    /**
     * Enable highlight color of link annotation
     *
     * @param enable True means highlight color of link annotation can be displayed, false for else.
     */
    public void enableLinkHighlight(boolean enable) {
        this.mEnableLinkHighlight = enable;
    }


    /**
     * get the highlight color of link annotation.
     *
     * @return the highlight color
     */
    public long getLinkHighlightColor() {
        return mLinkHighlightColor;
    }

    /**
     * Set the highlight color of link annotation.
     *
     * @param color the highlight color to be set
     */
    public void setLinkHighlightColor(long color) {
        this.mLinkHighlightColor = color;
    }

    /**
     * Get the highlight color of form field.
     *
     * @return the highlight color of form field.
     */
    public long getFormHighlightColor() {
        return mFormHighlightColor;
    }

    /**
     * Set form highlight color.
     * If the document is opened, please call function {@link PDFViewCtrl#updatePagesLayout()} after setting the new value.
     *
     * @param color the form highlight color to be set
     */
    public void setFormHighlightColor(long color) {
        this.mFormHighlightColor = color;
        FormFillerModule formModule = (FormFillerModule) getModuleByName(Module.MODULE_NAME_FORMFILLER);
        if (formModule != null)
            formModule.setFormHighlightColor(color);
    }

    /**
     * Check whether form highlight can be displayed.
     *
     * @return True means form highlight can be displayed, false for else.
     */
    public boolean isFormHighlightEnable() {
        return mEnableFormHighlight;
    }

    /**
     * Enable the highlight color of form field.
     * If the document is opened, please call function {@link PDFViewCtrl#updatePagesLayout()} after setting the new value.
     *
     * @param enable True means highlight color of form field. can be displayed, false for else.
     */
    public void enableFormHighlight(boolean enable) {
        this.mEnableFormHighlight = enable;
        FormFillerModule formModule = (FormFillerModule) getModuleByName(Module.MODULE_NAME_FORMFILLER);
        if (formModule != null)
            formModule.enableFormHighlight(enable);
    }

    /**
     * Set highlight color (including alpha) when select text.
     *
     * @param color The highlight color to be set.
     */
    public void setSelectionHighlightColor(int color) {
        mSelectHighlightColor = color;
    }

    /**
     * Get highlight color (including alpha) when text has selected.
     *
     * @return The highlight color.
     */
    public int getSelectionHighlightColor() {
        return mSelectHighlightColor;
    }

    /**
     * Whether to use logical pages, default is false.
     *
     * @param useLogicalPageNumbers True means use logical page numbers, false for else.
     */
    public void setUseLogicalPageNumbers(boolean useLogicalPageNumbers) {
        mIsUseLogicalPage = useLogicalPageNumbers;
    }

    /**
     * Whether to use logical pages.
     * @return True means use logical page numbers, false for else.
     */
    public boolean isUseLogicalPageNumbers() {
        return mIsUseLogicalPage;
    }

    /**
     * Get current selected text content from text select tool handler.
     *
     * @return The current selected text content.
     */
    public String getCurrentSelectedText() {
        ToolHandler selectionTool = getToolHandlerByType(ToolHandler.TH_TYPE_TEXTSELECT);
        if (selectionTool != null) {
            return ((TextSelectToolHandler) selectionTool).getCurrentSelectedText();
        }

        return null;
    }

    /**
     * Register a callback to be invoked when the menu event has triggered.
     */
    public void registerMenuEventListener(MenuEventListener listener) {
        mMenuEventListeners.add(listener);
    }

    /**
     * Unregister the specified {@link MenuEventListener}
     */
    public void unregisterMenuEventListener(MenuEventListener listener) {
        mMenuEventListeners.remove(listener);
    }

    /**
     * Called when menu has dismissed.
     */
    public void triggerDismissMenuEvent() {
        for (MenuEventListener listener : mMenuEventListeners) {
            listener.onTriggerDismissMenu();
        }
    }

    private static final int MSG_COUNT_DOWN = 121;
    private boolean isCountDown = false;

    private final Handler mHandler = new Handler(Looper.getMainLooper()) {
        @Override
        public void handleMessage(Message msg) {
            super.handleMessage(msg);
            if (msg.what == MSG_COUNT_DOWN) {
                if (mMainFrame != null && mMainFrame.canFullScreen()) {
                    mMainFrame.hideToolbars();
                }
            }
        }
    };

    /// @cond DEV
    public void startHideToolbarsTimer() {
        if (!mAutoFullScreen) return;
        if (mConfig.uiSettings.fullscreen && mMainFrame != null) {
            if ((mMainFrame.canFullScreen() && getCurrentToolHandler() == null)
                    || (mState == ReadStateConfig.STATE_REDACT && !mMainFrame.isShowFullScreenUI())) {
                stopHideToolbarsTimer();
                isCountDown = mHandler.sendEmptyMessageDelayed(MSG_COUNT_DOWN, TOOL_BAR_SHOW_TIME);
            }
        }
    }

    public void stopHideToolbarsTimer() {
        if (!mAutoFullScreen) return;
        if (isCountDown) {
            mHandler.removeMessages(MSG_COUNT_DOWN);
            isCountDown = false;
        }
    }

    public void resetHideToolbarsTimer() {
        if (!mAutoFullScreen) return;
        if (mConfig.uiSettings.fullscreen
                && mMainFrame != null
                && mMainFrame.isToolbarsVisible()
                && getCurrentToolHandler() == null) {
            startHideToolbarsTimer();
        }
    }
    /// @endcond

    /**
     * Set whether to automatically full screen, the default is true.
     *
     * @param enable If you set true, it will automatically full screen after a few seconds,
     *               hiding topbar and bottommbar,false otherwise.
     */
    public void enableAutoFullScreen(boolean enable) {
        mAutoFullScreen = enable;
    }

    @Override
    public boolean onTouchEvent(int pageIndex, MotionEvent motionEvent) {
        if (AppDisplay.isPad() && getPanelManager().isShowingPanel()) {
            getPanelManager().hidePanel();
            return true;
        }
        InkToolHandler inkHandler = (InkToolHandler) getToolHandlerByType(ToolHandler.TH_TYPE_INK);
        if (inkHandler != null && inkHandler.getCurToolItem() != null && inkHandler.getCurToolItem().toolItem.isSelected()) {
            ToolHandler currentToolHandler = getCurrentToolHandler();
            if (motionEvent.getToolType(0) == MotionEvent.TOOL_TYPE_ERASER) {
                if (currentToolHandler != null && !currentToolHandler.getType().equals(ToolHandler.TH_TYPE_ERASER)) {
                    ToolHandler handler = getToolHandlerByType(ToolHandler.TH_TYPE_ERASER);
                    if (handler != null) {
                        setCurrentToolHandler(handler);
                    }
                }
            } else {
                if (currentToolHandler != null && !currentToolHandler.getType().equals(ToolHandler.TH_TYPE_INK)) {
                    setCurrentToolHandler(inkHandler);
                }
            }
        }
        resetHideToolbarsTimer();
        if (mPdfViewCtrl.isDynamicXFA()) {
            DynamicXFAModule dynamicXFAModule = (DynamicXFAModule) getModuleByName(Module.MODULE_NAME_DYNAMICXFA);
            if (dynamicXFAModule == null) return false;
            DynamicXFAWidgetHandler dynamicXFAWidgetHandler = (DynamicXFAWidgetHandler) dynamicXFAModule.getXFAWidgetHandler();
            if (dynamicXFAWidgetHandler == null) return false;
            return dynamicXFAWidgetHandler.onTouchEvent(pageIndex, motionEvent);
        }

        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_REFLOW) return false;
        PanZoomModule panZoomModule = (PanZoomModule) getModuleByName(Module.MODULE_NAME_PANZOOM);
        if (panZoomModule != null) {
            panZoomModule.onTouchEvent(pageIndex, motionEvent);
        }

        if (motionEvent.getPointerCount() > 1) {
            if (mCurToolHandler == null){
                return  false;
            }
        }
//        if (motionEvent.getPointerCount() > 1) {
//            if (mCurToolHandler == null ||  !(mCurToolHandler instanceof InkToolHandler
//                    || mCurToolHandler instanceof EraserToolHandler  || mCurToolHandler instanceof FormFillerToolHandler
//                    || mCurToolHandler instanceof LinkToolHandler)) {
//                return false;
//            }
//        }

        if (getState() == ReadStateConfig.STATE_FILLSIGN) {
            FillSignModule fillSignModule = (FillSignModule) getModuleByName(Module.MODULE_NAME_FIllSIGN);
            if (fillSignModule != null) {
                return fillSignModule.getToolHandler().onTouchEvent(pageIndex, motionEvent);
            } else {
                return defaultTouchEvent(pageIndex, motionEvent);
            }
        } else if (mCurToolHandler != null) {
//            if (getDocumentManager().getCurrentAnnot() != null) return false;
            return mCurToolHandler.onTouchEvent(pageIndex, motionEvent);
        } else {
            return defaultTouchEvent(pageIndex, motionEvent);
        }
    }

    public boolean defaultTouchEvent(int pageIndex, MotionEvent motionEvent) {
        //annot handler
        if (getDocumentManager().onTouchEvent(pageIndex, motionEvent)) return true;

        for (ITouchEventListener l : mTouchEventListeners) {
            if (l.onTouchEvent(pageIndex, motionEvent)) {
                return true;
            }
        }

        //blank selection tool
        ToolHandler blankSelectionTool = getToolHandlerByType(ToolHandler.TH_TYPE_BLANKSELECT);
        if (blankSelectionTool != null && blankSelectionTool.onTouchEvent(pageIndex, motionEvent))
            return true;

        //text selection tool
        ToolHandler textSelectionTool = getToolHandlerByType(ToolHandler.TH_TYPE_TEXTSELECT);
        return textSelectionTool != null && textSelectionTool.onTouchEvent(pageIndex, motionEvent);
    }

    @Override
    public boolean shouldViewCtrlDraw(Annot annot) {
        return getDocumentManager().shouldViewCtrlDraw(annot);
    }

    @Override
    public Annot getFocusAnnot() {
        if (mPdfViewCtrl == null) return null;
        return getDocumentManager().getFocusAnnot();
    }

    @SuppressLint("WrongCall")
    @Override
    public void onDraw(int pageIndex, Canvas canvas) {
        if (mIsCompareDoc) {
            ComparisonModule comparisonModule = (ComparisonModule) getModuleByName(Module.MODULE_NAME_COMPARISON);
            if (comparisonModule == null) return;
            CompareHandler compareHandler = comparisonModule.getCompareHandler();
            if (compareHandler == null) return;
            compareHandler.onDraw(pageIndex, canvas);
            return;
        }

        if (mPdfViewCtrl.isDynamicXFA()) {
            DynamicXFAModule dynamicXFAModule = (DynamicXFAModule) getModuleByName(Module.MODULE_NAME_DYNAMICXFA);
            if (dynamicXFAModule == null) return;
            DynamicXFAWidgetHandler dynamicXFAWidgetHandler = (DynamicXFAWidgetHandler) dynamicXFAModule.getXFAWidgetHandler();
            if (dynamicXFAWidgetHandler == null) return;
            dynamicXFAWidgetHandler.onDraw(pageIndex, canvas);
            return;
        }

        for (ToolHandler handler : mToolHandlerList.values()) {
            handler.onDraw(pageIndex, canvas);
        }

        for (int i = 0; i < mAnnotHandlerList.size(); i++) {
            int type = mAnnotHandlerList.keyAt(i);
            AnnotHandler handler = mAnnotHandlerList.get(type);
            if (handler != null) {
                handler.onDraw(pageIndex, canvas);
            }
        }
    }

    @Override
    public boolean onSingleTapConfirmed(MotionEvent motionEvent) {
        if (mLongPressOnShowPress){
            return true;
        }
        PointF displayViewPt = new PointF(motionEvent.getX(), motionEvent.getY());
        int pageIndex = mPdfViewCtrl.getPageIndex(displayViewPt);
        if (mIsCompareDoc) {
            ComparisonModule comparisonModule = (ComparisonModule) getModuleByName(Module.MODULE_NAME_COMPARISON);
            if (comparisonModule == null) return false;
            CompareHandler compareHandler = comparisonModule.getCompareHandler();
            if (compareHandler == null) return false;
            if (compareHandler.onSingleTapConfirmed(pageIndex, motionEvent)) return true;
        }

        if (mPdfViewCtrl.isDynamicXFA()) {
            DynamicXFAModule dynamicXFAModule = (DynamicXFAModule) getModuleByName(Module.MODULE_NAME_DYNAMICXFA);
            if (dynamicXFAModule == null) return false;
            DynamicXFAWidgetHandler dynamicXFAWidgetHandler = (DynamicXFAWidgetHandler) dynamicXFAModule.getXFAWidgetHandler();
            if (dynamicXFAWidgetHandler == null) return false;
            return dynamicXFAWidgetHandler.onSingleTapConfirmed(pageIndex, motionEvent);
        }

        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_REFLOW) return false;
        if (motionEvent.getPointerCount() > 1) return false;

        if (getState() == ReadStateConfig.STATE_FILLSIGN) {
            FillSignModule fillSignModule = (FillSignModule) getModuleByName(Module.MODULE_NAME_FIllSIGN);
            if (fillSignModule != null) {
                return fillSignModule.getToolHandler().onSingleTapConfirmed(pageIndex, motionEvent);
            } else {
                return defaultSingleTapConfirmed(pageIndex, motionEvent);
            }
        } else if (mCurToolHandler != null) {
            if (getDocumentManager().getCurrentAnnot() != null) {
                return getCurrentAnnotHandler().onSingleTapConfirmed(pageIndex, motionEvent, getDocumentManager().getCurrentAnnot());
            }
            return mCurToolHandler.onSingleTapConfirmed(pageIndex, motionEvent);
        } else {
            return defaultSingleTapConfirmed(pageIndex, motionEvent);
        }
    }

    public boolean defaultSingleTapConfirmed(int pageIndex, MotionEvent motionEvent) {
        if (getDocumentManager().getCurrentAnnot() != null && getCurrentToolHandler() != null) {
            return false;
        }
        //annot handler
        if (getDocumentManager().onSingleTapConfirmed(pageIndex, motionEvent)) return true;

        for (ITouchEventListener l : mTouchEventListeners) {
            if (l.onSingleTapConfirmed(pageIndex, motionEvent)) {
                return true;
            }
        }

        // blank selection tool
        ToolHandler blankSelectionTool = getToolHandlerByType(ToolHandler.TH_TYPE_BLANKSELECT);
        if (blankSelectionTool != null && blankSelectionTool.onSingleTapConfirmed(pageIndex, motionEvent))
            return true;

        //text selection tool
        ToolHandler textSelectionTool = getToolHandlerByType(ToolHandler.TH_TYPE_TEXTSELECT);
        if (textSelectionTool != null && textSelectionTool.onSingleTapConfirmed(pageIndex, motionEvent))
            return true;

        if (getDocumentManager().getCurrentAnnot() != null) {
            getDocumentManager().setCurrentAnnot(null);
            return true;
        }
        return false;
    }

    @Override
    public boolean onDoubleTap(MotionEvent motionEvent) {
        return false;
    }

    @Override
    public boolean onDoubleTapEvent(MotionEvent motionEvent) {
        return false;
    }

    @Override
    public boolean onDown(MotionEvent motionEvent) {
        mLongPressOnShowPress = false;
        return false;
    }

    private boolean mLongPressOnShowPress = false;

    @Override
    public void onShowPress(MotionEvent motionEvent) {
        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_REFLOW || mIsCompareDoc)
            return;
        if (motionEvent.getPointerCount() > 1) {
            return;
        }
        PointF displayViewPt = new PointF(motionEvent.getX(), motionEvent.getY());
        int pageIndex = mPdfViewCtrl.getPageIndex(displayViewPt);
        if (pageIndex < 0) return;

        if (getState() != ReadStateConfig.STATE_FILLSIGN && mCurToolHandler != null) {
            if (isSupportTurnPageTool(mCurToolHandler.getType())) {
                if (mPdfViewCtrl.isTouchPageView(new Point((int)motionEvent.getX(), (int)motionEvent.getY()))
                        && mCurToolHandler.onLongPress(pageIndex, motionEvent)) {
                    mLongPressOnShowPress = true;
                }
            }
        }
    }

    @Override
    public boolean onSingleTapUp(MotionEvent motionEvent) {
        return false;
    }

    @Override
    public boolean onScroll(MotionEvent motionEvent, MotionEvent motionEvent1, float v, float v1) {
        return false;
    }

    @Override
    public void onLongPress(MotionEvent motionEvent) {
        if (mPdfViewCtrl.getPageLayoutMode() == PDFViewCtrl.PAGELAYOUTMODE_REFLOW || mIsCompareDoc)
            return;
        if (motionEvent.getPointerCount() > 1) {
            return;
        }
        PointF displayViewPt = new PointF(motionEvent.getX(), motionEvent.getY());
        int pageIndex = mPdfViewCtrl.getPageIndex(displayViewPt);
        if (pageIndex < 0) return;

        if (getState() == ReadStateConfig.STATE_FILLSIGN) {
            FillSignModule fillSignModule = (FillSignModule) getModuleByName(Module.MODULE_NAME_FIllSIGN);
            if (fillSignModule != null) {
                fillSignModule.getToolHandler().onLongPress(pageIndex, motionEvent);
            } else {
                defaultLongPress(pageIndex, motionEvent);
            }
            return;
        } else if (mCurToolHandler != null) {
            if (getDocumentManager().getCurrentAnnot() != null) {
                getDocumentManager().setCurrentAnnot(null);
                mCurToolHandler.onLongPress(pageIndex, motionEvent);
                return;
            }

            if (!(mCurToolHandler instanceof PolygonToolHandler) && !(mCurToolHandler instanceof PolyLineToolHandler)
                && !(mCurToolHandler instanceof InkToolHandler)){
                //if (handleLongPressWithType(motionEvent, pageIndex, Annot.e_Note)) return;
                //if (handleLongPressWithType(motionEvent, pageIndex, Annot.e_FileAttachment)) return;
            }

            if (!isSupportTurnPageTool(mCurToolHandler.getType())) {
                if (mCurToolHandler.onLongPress(pageIndex, motionEvent)) {
                    return;
                }
            }
        } else {
            if (defaultLongPress(pageIndex, motionEvent)) return;
        }
        return;
    }

    private boolean handleLongPressWithType(MotionEvent motionEvent, int pageIndex, int type) {
        try {
            Annot annot = getDocumentManager().getAnnot(pageIndex, motionEvent, null, mPdfViewCtrl);
            if (annot != null && annot.getType() == type) {
                defaultLongPress(pageIndex, motionEvent);
                return true;
            }
        } catch (Exception ignored) {
        }
        return false;
    }

    public boolean defaultLongPress(int pageIndex, MotionEvent motionEvent) {
        //annot handler
        if (getDocumentManager().onLongPress(pageIndex, motionEvent)) {
            return true;
        }

        for (ITouchEventListener l : mTouchEventListeners) {
            if (l.onLongPress(pageIndex, motionEvent)) {
                return true;
            }
        }

        // blank selection tool
        ToolHandler blankSelectionTool = getToolHandlerByType(ToolHandler.TH_TYPE_BLANKSELECT);
        if (blankSelectionTool != null && blankSelectionTool.onLongPress(pageIndex, motionEvent)) {
            return true;
        }

        //text selection tool
        ToolHandler textSelectionTool = getToolHandlerByType(ToolHandler.TH_TYPE_TEXTSELECT);
        return textSelectionTool != null && textSelectionTool.onLongPress(pageIndex, motionEvent);
    }

    @Override
    public boolean onFling(MotionEvent motionEvent, MotionEvent motionEvent1, float v, float v1) {
        return false;
    }

    @Override
    public boolean onScale(ScaleGestureDetector scaleGestureDetector) {
        return false;
    }

    @Override
    public boolean onScaleBegin(ScaleGestureDetector scaleGestureDetector) {
        return false;
    }

    @Override
    public void onScaleEnd(ScaleGestureDetector scaleGestureDetector) {
        if (AppDisplay.isPad())
            mMainFrame.updateZoomModeSettings();
    }

    private static final float SINGLETAP_BORDER_AREA = 0.2f;

    private boolean isClickBorderArea(PointF point) {
        float leftEdge = mPdfViewCtrl.getWidth() * SINGLETAP_BORDER_AREA;
        float rightEdge = mPdfViewCtrl.getWidth() * (1 - SINGLETAP_BORDER_AREA);
        return point.x < leftEdge || point.x > rightEdge;
    }


    PDFViewCtrl.IDoubleTapEventListener mDoubleTapEventListener = new PDFViewCtrl.IDoubleTapEventListener() {
        @Override
        public boolean onSingleTapConfirmed(MotionEvent motionEvent) {
            return handleSingleTapConfirmed(motionEvent);
        }

        @Override
        public boolean onDoubleTap(MotionEvent motionEvent) {
            return false;
        }

        @Override
        public boolean onDoubleTapEvent(MotionEvent motionEvent) {
            if (AppDisplay.isPad())
                mMainFrame.updateZoomModeSettings();
            return false;
        }
    };

    private PDFViewCtrl.ITouchEventListener mTouchEventListener = new PDFViewCtrl.ITouchEventListener() {
        @Override
        public boolean onTouchEvent(MotionEvent event) {
            ToolHandler toolHandler = getCurrentToolHandler();
            if (toolHandler instanceof InkToolHandler) {
                PointF displayViewPt = new PointF(event.getX(), event.getY());
                int pageIndex = mPdfViewCtrl.getPageIndex(displayViewPt);
                if (pageIndex < 0) {
                    ((InkToolHandler) toolHandler).reset();
                }
            }
            return false;
        }
    };

    private PDFViewCtrl.IDocEventListener mDocEventListener = new DocEventListener() {
        @Override
        public void onDocWillOpen() {
            mIsDocOpened = false;
        }

        @Override
        public void onDocOpened(PDFDoc document, int errCode) {
            switch (errCode) {
                case e_ErrSuccess:
                    mIsDocOpened = true;
                    if (isTwoColumnLeft(document)) {
                        mOldPageLayout = mPdfViewCtrl.getPageLayoutMode();
                        mPdfViewCtrl.setPageLayoutMode(PDFViewCtrl.PAGELAYOUTMODE_FACING);
                        mOldFileIsTwoColumnLeft = true;
                    } else if (mOldFileIsTwoColumnLeft) {
                        mOldFileIsTwoColumnLeft = false;
                        mPdfViewCtrl.setPageLayoutMode(mOldPageLayout);
                    }
                    if (mIsCompareDoc = isCompareDoc()) {
                        mState = ReadStateConfig.STATE_COMPARE;
                    }

                    if (!mPdfViewCtrl.isDynamicXFA()) {
                        doOpenAction(document);
                        getDocumentManager().initDocProperties(document);
                        try {
                            document.enableUpdatePageLabel(true);
                            if (mConfig.optimizations.optimizeEmbeddedFontsForAnnots) {
                                document.loadAnnotsFonts(new Range());
                            }
                        } catch (PDFException e) {
                            e.printStackTrace();
                        }
                    }
                    setFilePath(mPdfViewCtrl.getFilePath());

                    bDocClosed = false;
                    isSaveDocInCurPath = false;
                    mPasswordError = false;
//                    changeState(mState);
                    mMainFrame.onDocOpened(document, errCode);
                    break;
                case e_ErrPassword:
                    String tips;
                    if (mPasswordError) {
                        tips = AppResource.getString(mContext.getApplicationContext(), R.string.rv_tips_password_error);
                    } else {
                        tips = AppResource.getString(mContext.getApplicationContext(), R.string.rv_tips_password);
                    }
                    final UITextEditDialog uiTextEditDialog = new UITextEditDialog(mMainFrame.getAttachedActivity());
                    uiTextEditDialog.getDialog().setCanceledOnTouchOutside(false);
                    uiTextEditDialog.getInputEditText().setInputType(InputType.TYPE_CLASS_TEXT | InputType.TYPE_TEXT_VARIATION_PASSWORD);
                    uiTextEditDialog.setTitle(AppResource.getString(mContext.getApplicationContext(), R.string.rv_password_dialog_title));
                    uiTextEditDialog.getPromptTextView().setText(tips);
                    uiTextEditDialog.show();
                    AppUtil.showSoftInput(uiTextEditDialog.getInputEditText());
                    uiTextEditDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            uiTextEditDialog.dismiss();
                            AppUtil.dismissInputSoft(uiTextEditDialog.getInputEditText());
                            String pw = uiTextEditDialog.getInputEditText().getText().toString();
                            if (mOpenFileType == OPEN_FILE_TYPE_PATH) {
                                mPdfViewCtrl.openDoc(mDocPath, pw.getBytes());
                            } else if (mOpenFileType == OPEN_FILE_TYPE_STREAM) {
                                mPdfViewCtrl.openDoc(mFileReader, pw.getBytes());
                            }
                            DocSaveAsModule module = (DocSaveAsModule) getModuleByName(Module.MODULE_NAME_SAVE_AS);
                            if (module != null)
                                module.setPassword(pw);
                        }
                    });

                    uiTextEditDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            uiTextEditDialog.dismiss();
                            AppUtil.dismissInputSoft(uiTextEditDialog.getInputEditText());
                            mPasswordError = false;
                            bDocClosed = true;
                            openDocumentFailed();
                        }
                    });

                    uiTextEditDialog.getDialog().setOnKeyListener(new DialogInterface.OnKeyListener() {
                        @Override
                        public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                            if (keyCode == KeyEvent.KEYCODE_BACK) {
                                uiTextEditDialog.getDialog().cancel();
                                mPasswordError = false;
                                bDocClosed = true;
                                openDocumentFailed();
                                return true;
                            }
                            return false;
                        }
                    });

                    mPasswordError = true;
                    break;
                default:
                    bDocClosed = true;
                    String message = AppUtil.getMessage(mContext, errCode);
                    UIToast.getInstance(mContext).show(message);
                    openDocumentFailed();
                    break;
            }

            dismissProgressDlg();
        }

        @Override
        public void onDocWillClose(PDFDoc document) {
            AnnotActionHandler annotActionHandler = (AnnotActionHandler) getDocumentManager().getActionCallback();
            if (annotActionHandler != null)
                annotActionHandler.reset();
        }

        @Override
        public void onDocClosed(PDFDoc document, int errCode) {
            dismissProgressDlg();
            if (mPanelManager.isShowingPanel()) {
                mPanelManager.hidePanel();
            }
            bDocClosed = true;
            closeDocumentSucceed();
            documentManager.setViewSignedDocFlag(false);
        }

        @Override
        public void onDocWillSave(PDFDoc document) {
        }

        @Override
        public void onDocSaved(PDFDoc document, int errCode) {
            if (mProgressDlg != null && mProgressDlg.isShowing()) {
                if (errCode == e_ErrSuccess) {
                    if (!isSaveDocInCurPath) {
                        if (AppFileUtil.needScopedStorageAdaptation()) {
                            AppStorageManager.getInstance(mContext).copyDocument(currentFileCachePath, mSavePath, true);
                        }
                        updateThumbnail(mSavePath);
                    } else if (mRealDocPath != null && currentFileCachePath != null) {
                        AppStorageManager.getInstance(mContext).copyDocument(currentFileCachePath, mRealDocPath, false);
                    } else {
                        renameCacheToFile();
                        updateThumbnail(mSavePath);
                    }
                }

                if (isCloseDocAfterSaving) {
                    closeAllDocuments();
                }

                dismissProgressDlg();
            }
        }
    };

    private PDFViewCtrl.IPageEventListener mPageEventListener = new PageEventListener() {

        @Override
        public void onPagesRemoved(boolean success, int[] pageIndexes) {
            mSaveFlag = PDFDoc.e_SaveFlagXRefStream;
        }

    };

    private PDFViewCtrl.IRecoveryEventListener mRecoveryEventListener = new PDFViewCtrl.IRecoveryEventListener() {

        @Override
        public void onWillRecover() {
            mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_recovering);
            showProgressDlg();
            getDocumentManager().mCurAnnot = null;
            getDocumentManager().mCurAnnotHandlerType = Annot.e_UnknownType;
            getDocumentManager().clearUndoRedo();
            getDocumentManager().reInit();
            BlankSelectToolHandler toolHandler = (BlankSelectToolHandler) getToolHandlerByType(ToolHandler.TH_TYPE_BLANKSELECT);
            if (toolHandler != null) {
                toolHandler.dismissMenu();
            }
        }

        @Override
        public void onRecovered() {
            dismissProgressDlg();
            if (!mPdfViewCtrl.isDynamicXFA()) {
                getDocumentManager().initDocProperties(mPdfViewCtrl.getDoc());
            }

            initActionHandler();
        }
    };

    /**
     * Set the attached activity.
     * <p>
     * If you want add a Note, FreeText, FileAttachment annotation; you must set the attached activity.
     * <p>
     * If you want to use the function of adding reply or comment to the annotation or about thumbnail,
     * you must set the attached activity and it must be a FragmentActivity.
     *
     * @param activity The attached activity.
     */
    public void setAttachedActivity(Activity activity) {
        mAttachActivity = activity;
        mMainFrame.setAttachedActivity(activity);
    }

    /**
     * Get the attached activity.
     *
     * @return The attached activity.
     */
    public Activity getAttachedActivity() {
        return mAttachActivity;
    }

    /**
     * Print PDF documents and Static XFA documents(that is xfaDoc.getType() == XFADoc.e_Static)
     * <p>
     * Note: Only when OS version is Kitkat and above (Android API >= 19) the print function can be used
     *
     * @param context      The context to use. it must be instanceof Activity.
     * @param pdfDoc       The {@link PDFDoc} Object, it can not be empty .
     * @param printJobName print job name, it is can be null or empty.
     * @param fileName     The document name which may be shown to the user and
     *                     is the file name if the content it describes is saved as a PDF.
     *                     Cannot be empty.
     * @param callback     print callback {@link IPrintResultCallback}
     */
    public void startPrintJob(Context context, PDFDoc pdfDoc, String printJobName, String fileName, IPrintResultCallback callback) {
        if (pdfDoc == null || pdfDoc.isEmpty()) return;
        new PDFPrint.Builder(context)
                .setAdapter(new PDFPrintAdapter(context, pdfDoc, fileName, callback))
                .setOutputFileName(fileName)
                .setPrintJobName(printJobName)
                .print();
    }

    /**
     * Print Dynamic XFA documents (that is xfaDoc.getType() == XFADoc.e_Dynamic)
     * <p>
     * Note: Only when OS version is Kitkat and above (Android API >= 19) the print function can be used
     *
     * @param context      The context to use. it must be instanceof Activity.
     * @param xfaDoc       The {@link XFADoc} Object, it can not be empty .
     * @param printJobName print job name, it is can be null or empty.
     * @param fileName     The document name which may be shown to the user and
     *                     is the file name if the content it describes is saved as a PDF.
     *                     Cannot be empty.
     * @param callback     print callback {@link IPrintResultCallback}
     */
    public void startPrintJob(Context context, XFADoc xfaDoc, String printJobName, String fileName, IPrintResultCallback callback) {
        if (xfaDoc == null || xfaDoc.isEmpty()) return;
        new PDFPrint.Builder(context)
                .setAdapter(new XFAPrintAdapter(context, xfaDoc, fileName, callback))
                .setOutputFileName(fileName)
                .setPrintJobName(printJobName)
                .print();
    }

    /**
     * Get the interface to handle the creation, display, hiding, layout, and interaction of panels.
     *
     * @return {@link IPanelManager}
     */
    public IPanelManager getPanelManager() {
        return mPanelManager;
    }

    UIExtensionsManager.MenuEventListener mMenuEventListener = new UIExtensionsManager.MenuEventListener() {
        @Override
        public void onTriggerDismissMenu() {
            if (getDocumentManager().getCurrentAnnot() != null) {
                getDocumentManager().setCurrentAnnot(null);
            }
        }
    };

    /**
     * Get a property config
     *
     * @return {@link Config}
     */
    public Config getConfig() {
        return mConfig;
    }

    /**
     * Check whether the document can be modified.
     *
     * @return true means The document can be modified
     */
    public boolean canModifyContents() {
        return getDocumentManager().canModifyContents();
    }

    /**
     * Check whether the document can add annotation
     *
     * @return true means The document can add annotation
     */
    public boolean canAddAnnot() {
        return getDocumentManager().canAddAnnot();
    }

    /**
     * Check whether the document can be assemble.
     *
     * @return true means The document can be assemble
     */
    public boolean canAssemble() {
        return getDocumentManager().canAssemble();
    }

    /**
     * Exit the pan zoom mode.
     */
    public void exitPanZoomMode() {
        PanZoomModule module = (PanZoomModule) getModuleByName(Module.MODULE_NAME_PANZOOM);
        if (module != null) {
            module.exit();
        }
    }

    /**
     * Set the night color mode.
     *
     * @param nightColorMode The night color mode. It should be one of {@link #NIGHTCOLORMODE_DEFAULT}, {@link #NIGHTCOLORMODE_MAPPINGGRAY}.
     */
    public void setNightColorMode(int nightColorMode) {
        mNightColorMode = nightColorMode;
    }

    /**
     * Get the night color mode.
     *
     * @return The {@link } night color mode. It would be one of {@link #NIGHTCOLORMODE_DEFAULT}, {@link #NIGHTCOLORMODE_MAPPINGGRAY}.
     * .
     */
    public int getNightColorMode() {
        return mNightColorMode;
    }

    /**
     * Set the page color mode.
     *
     * @param pageColorMode The page color mode. It should be one of {@link #NIGHTCOLORMODE_DEFAULT}, {@link #NIGHTCOLORMODE_MAPPINGGRAY}.
     */
    public void setPageColorMode(int pageColorMode) {
        mPageColorMode = pageColorMode;
    }

    /**
     * Get the page color mode.
     *
     * @return The {@link } page color mode. It would be one of {@link #NIGHTCOLORMODE_DEFAULT}, {@link #NIGHTCOLORMODE_MAPPINGGRAY}.
     * .
     */
    public int getPageColorMode() {
        return mPageColorMode;
    }

    private void release() {
        BlankSelectToolHandler selectToolHandler = (BlankSelectToolHandler) getToolHandlerByType(ToolHandler.TH_TYPE_BLANKSELECT);
        if (selectToolHandler != null) {
            selectToolHandler.unload();
            unregisterToolHandler(selectToolHandler);
        }

        for (Module module : mModules) {
            if (module instanceof LocalModule) continue;
            module.unloadModule();
        }

        mDocumentModifiedEventListeners.clear();
        mMenuEventListeners.clear();
        mLifecycleEventList.clear();
        mStateChangeEventList.clear();
        mXFAPageEventListeners.clear();
        mXFAWidgetEventListener.clear();
        mInteractionListeners.clear();
        mThemeEventListeners.clear();
        mLayoutChangedListeners.clear();
        mConfigurationChangedListeners.clear();
        mPolicyEventListeners.clear();
        mSaveEventListeners.clear();

        mModules.clear();
        mModules = null;
        mPdfViewCtrl.unregisterDocEventListener(mDocEventListener);
        mPdfViewCtrl.unregisterRecoveryEventListener(mRecoveryEventListener);
        mPdfViewCtrl.unregisterDoubleTapEventListener(mDoubleTapEventListener);
        mPdfViewCtrl.unregisterTouchEventListener(mTouchEventListener);
        mPdfViewCtrl.unregisterPageEventListener(mPageEventListener);
        unregisterMenuEventListener(mMenuEventListener);
        getDocumentManager().destroy();

        mDocumentModifiedEventListeners = null;
        mXFAPageEventListeners = null;
        mXFAWidgetEventListener = null;
        onFinishListener = null;
        mBackEventListener = null;
        mTouchEventListener = null;
        mRecoveryEventListener = null;
        mDoubleTapEventListener = null;
        mPageEventListener = null;
        mDocEventListener = null;
        mMenuEventListener = null;
        mContext = null;
        mActivityLayout.removeAllViews();
        mActivityLayout = null;
        mCurToolHandler = null;
        mConfig = null;
        mPanelManager = null;
        mAttachActivity = null;

        documentManager = null;
        mPdfViewCtrl.release();
        mPdfViewCtrl = null;
        mMainFrame.release();
        mMainFrame = null;
        mParent = null;
        UIPopoverWin.release();
    }

    /**
     * Should be called in {@link Activity#onCreate(Bundle) } or {@link Fragment#onCreate(Bundle)}
     */
    @SuppressWarnings("JavadocReference")
    public void onCreate(Activity act, PDFViewCtrl pdfViewCtrl, Bundle bundle) {
        if (mMainFrame == null) return;
        if (mMainFrame.getAttachedActivity() != null && mMainFrame.getAttachedActivity() != act) {
            for (ILifecycleEventListener listener : mLifecycleEventList) {
                listener.onDestroy(act);
            }
        }
        mMainFrame.setAttachedActivity(act);

        for (ILifecycleEventListener listener : mLifecycleEventList) {
            listener.onCreate(act, bundle);
        }
    }

    /**
     * Should be called in {@link Activity#onConfigurationChanged(Configuration)} or {@link Fragment#onConfigurationChanged(Configuration)}
     *
     * @param act       The current activity
     * @param newConfig The new device configuration.
     */
    public void onConfigurationChanged(Activity act, Configuration newConfig) {
        if (mMainFrame == null || mMainFrame.getAttachedActivity() != act) return;
        if (mFolderSelectDialog != null && mFolderSelectDialog.isShowing()) {
            mFolderSelectDialog.resetWH();
            mFolderSelectDialog.showDialog();
        }

        if (AppDisplay.isPad()) {
            if (mSaveAlertDlg != null && mSaveAlertDlg.isShowing()) {
                Window window = mSaveAlertDlg.getWindow();
                if (window != null) {
                    WindowManager.LayoutParams layoutParams = window.getAttributes();
                    layoutParams.width = Math.min(AppDisplay.getDialogWidth(), AppDisplay.getActivityWidth()) * 3 / 5;
                    window.setAttributes(layoutParams);
                }
            }
        }

        int newNightMode = newConfig.uiMode & Configuration.UI_MODE_NIGHT_MASK;
        if (newNightMode != mCurNightMode) {
            mCurNightMode = newNightMode;
            AppDarkUtil.getInstance(mContext).setCurNightMode(mCurNightMode);
            if (mMainFrame.getPropertyBar() != null)
                mMainFrame.getPropertyBar().updateTheme();
            if (getDocumentManager().getCurrentAnnot() != null)
                getDocumentManager().setCurrentAnnot(null);

            ThemeConfig.getInstance(act).getAdapter().notifyThemeChanged("", -1);
        }

        //for (ConfigurationChangedListener listener : mConfigurationChangedListeners) {  // MOBRD-2722: java.util.ConcurrentModificationException:
        for (int i = 0;i < mConfigurationChangedListeners.size(); i++) {
            ConfigurationChangedListener listener = mConfigurationChangedListeners.get(i);
            listener.onConfigurationChanged(newConfig);
        }
    }

    /**
     * Should be called in {@link Activity#onStart()} or {@link Fragment#onStart()}
     */
    @SuppressWarnings("JavadocReference")
    public void onStart(Activity act) {
        if (mMainFrame == null) return;
        if (mMainFrame.getAttachedActivity() != act) return;
        for (ILifecycleEventListener listener : mLifecycleEventList) {
            listener.onStart(act);
        }
    }

    /**
     * Should be called in {@link Activity#onPause()} or {@link Fragment#onPause()}
     */
    @SuppressWarnings("JavadocReference")
    public void onPause(Activity act) {
        if (mMainFrame == null) return;
        if (mMainFrame.getAttachedActivity() != act) return;
        for (ILifecycleEventListener listener : mLifecycleEventList) {
            listener.onPause(act);
        }
    }

    /**
     * Should be called in {@link Fragment#onHiddenChanged(boolean)}
     *
     * @param hidden True if the fragment is now hidden, false otherwise.
     */
    public void onHiddenChanged(boolean hidden) {
        for (ILifecycleEventListener listener : mLifecycleEventList) {
            listener.onHiddenChanged(hidden);
        }
    }

    /**
     * Should be called in {@link Activity#onResume()} or {@link Fragment#onResume()}
     */
    @SuppressWarnings("JavadocReference")
    public void onResume(Activity act) {
        if (mMainFrame == null) return;
        if (mMainFrame.getAttachedActivity() != act) return;
        for (ILifecycleEventListener listener : mLifecycleEventList) {
            listener.onResume(act);
        }
    }

    /**
     * Should be called in {@link Activity#onStop()} or {@link Fragment#onStop()}
     */
    @SuppressWarnings("JavadocReference")
    public void onStop(Activity act) {
        if (mMainFrame == null) return;
        if (mMainFrame.getAttachedActivity() != act) return;
        for (ILifecycleEventListener listener : mLifecycleEventList) {
            listener.onStop(act);
        }
    }

    /**
     * Should be called in {@link Activity#onDestroy()} or {@link Fragment#onDestroy()}
     */
    @SuppressWarnings("JavadocReference")
    public void onDestroy(Activity act) {
        if (mMainFrame == null) return;
        if (mMainFrame.getAttachedActivity() != act) return;
        for (ILifecycleEventListener listener : mLifecycleEventList) {
            listener.onDestroy(act);
        }
        mMainFrame.setAttachedActivity(null);
        closeAllDocuments();
        release();
        ThemeConfig.getInstance(mContext).getAdapter().unregisterThemeChangeObserver(this);
        AppDialogManager.getInstance().closeAllDialog();
    }

    /**
     * Receive and handle result from activity
     *
     * @param act         The current activity
     * @param requestCode The integer request code originally supplied to
     *                    startActivityForResult(), allowing you to identify who this
     *                    result came from.
     * @param resultCode  The integer result code returned by the child activity
     *                    through its setResult().
     * @param data        An Intent, which can return result data to the caller
     *                    (various data can be attached to Intent "extras").
     * {@link Activity#onActivityResult(int, int, Intent)}
     * {@link Fragment#onActivityResult(int, int, Intent)}
     */
    @SuppressWarnings("JavadocReference")
    public void handleActivityResult(Activity act, int requestCode, int resultCode, Intent data) {
        if (mMainFrame == null) return;
        if (mMainFrame.getAttachedActivity() != act) return;

        mPdfViewCtrl.handleActivityResult(requestCode, resultCode, data);

        for (ILifecycleEventListener listener : mLifecycleEventList) {
            listener.onActivityResult(act, requestCode, resultCode, data);
        }
    }

    /**
     * Register a callback to be invoked when the policy has changed.
     *
     * @param listener The policy event listener to be registered.
     */
    public void registerPolicyEventListener(IPolicyEventListener listener) {
        mPolicyEventListeners.add(listener);
    }

    /**
     * Unregister a callback to be invoked when the policy has changed.
     *
     * @param listener The policy event listener to be unregistered.
     */
    public void unregisterPolicyEventListener(IPolicyEventListener listener) {
        mPolicyEventListeners.remove(listener);
    }

    /**
     * Receive and handle request permissions result from activity
     *
     * @param requestCode The request code passed in {@link Activity#requestPermissions(String[], int)}.
     * @param permissions The requested permissions. Never null.
     * @param grantResults The grant results for the corresponding permissions
     *     which is either {@link android.content.pm.PackageManager#PERMISSION_GRANTED}
     *     or {@link android.content.pm.PackageManager#PERMISSION_DENIED}. Never null.
     *
     */
    public void handleRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        if (requestCode == ActRequestCode.REQ_TTS_READ_PHONE_STATE_PERMISSION) {
            onPhoneStatePermissionResult(permissions, grantResults);
            return;
        }
        for (IPolicyEventListener listener : mPolicyEventListeners) {
            listener.onRequestPermissionsResult(requestCode, permissions, grantResults);
        }
    }

    /**
     * Open a PDF document from a specified PDF file path.
     *
     * @param path     A PDF file path.
     * @param password A byte array which specifies the password used to load the PDF document content. It can be either user password or owner password.
     *                 If the PDF document is not encrypted by password, just pass an empty string.
     */
    public void openDocument(String path, byte[] password) {
        _resetStatusBeforeOpen();
        mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_opening);
        showProgressDlg();
        if (AppFileUtil.needScopedStorageAdaptation()) {
            File file = new File(path);
            if (file.exists() && !file.canRead()) {
                mRealDocPath = path;
                final String originalPath = path;
                final byte[] originalPassword = password;
                AppThreadManager.getInstance().startThread(new Runnable() {
                    @Override
                    public void run() {
                        openDocumentInternal(AppFileUtil.getScopedCachePath(mContext, originalPath), originalPassword);
                    }
                });
                return;
            }
        }
        openDocumentInternal(path, password);
    }

    private void openDocumentInternal(String path, byte[] password) {
        mOpenFileType = OPEN_FILE_TYPE_PATH;
        mFileReader = null;
        setFilePath(path);
        mPdfViewCtrl.openDoc(path, password);
    }

    /**
     * Open a PDF document with a file read callback object.
     *
     * @param fileReader A {@link FileReaderCallback} object which is implemented by user to load a PDF document.
     *                   It should not be null.
     * @param password   A byte array which specifies the password used to load the PDF document content. It can be either user password or owner password.
     *                   If the PDF document is not encrypted by password, just pass an empty string.
     */
    public void openDocument(FileReaderCallback fileReader, byte[] password) {
        _resetStatusBeforeOpen();
        mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_opening);
        showProgressDlg();

        mOpenFileType = OPEN_FILE_TYPE_STREAM;
        mDocPath = "";
        mFileReader = fileReader;
        mPdfViewCtrl.openDoc(fileReader, password);
    }

    @Override
    public boolean registerLifecycleListener(ILifecycleEventListener listener) {
        mLifecycleEventList.add(listener);
        return true;
    }

    @Override
    public boolean unregisterLifecycleListener(ILifecycleEventListener listener) {
        mLifecycleEventList.remove(listener);
        return true;
    }

    @Override
    public boolean registerStateChangeListener(IStateChangeListener listener) {
        mStateChangeEventList.add(listener);
        return false;
    }

    @Override
    public boolean unregisterStateChangeListener(IStateChangeListener listener) {
        mStateChangeEventList.remove(listener);
        return false;
    }

    @Override
    public int getState() {
        return mState;
    }


    @Override
    public void changeState(int state) {
        int oldState = mState;
        mState = state;
        for (IStateChangeListener listener : mStateChangeEventList) {
            listener.onStateChanged(oldState, state);
        }
        if (mIsDocOpened) {
            if (mProgressDlg != null && !mProgressDlg.isShowing()){
                mPdfViewCtrl.postPageContainer();
            }
            startHideToolbarsTimer();
        }
    }

    @Override
    public IMainFrame getMainFrame() {
        return mMainFrame;
    }

    @Override
    public PDFViewCtrl getPDFViewCtrl() {
        return mPdfViewCtrl;
    }

    @Override
    public IBarsHandler getBarManager() {
        return mBaseBarMgr;
    }

    @Override
    public IViewSettingsWindow getSettingWindow() {
        return mMainFrame.getSettingWindow();
    }

    @Override
    public DocumentManager getDocumentManager() {
        return documentManager.on(mPdfViewCtrl);
    }

    @Override
    public RelativeLayout getContentView() {
        return mActivityLayout;
    }

    @Override
    public void backToPrevActivity() {
        if (getCurrentToolHandler() != null) {
            setCurrentToolHandler(null);
        }

        if (getDocumentManager() != null && getDocumentManager().getCurrentAnnot() != null) {
            getDocumentManager().setCurrentAnnot(null);
        }

        if (mPdfViewCtrl.isDynamicXFA()) {
            DynamicXFAModule dynamicXFAModule = (DynamicXFAModule) getModuleByName(Module.MODULE_NAME_DYNAMICXFA);
            if (dynamicXFAModule != null && dynamicXFAModule.getCurrentXFAWidget() != null) {
                dynamicXFAModule.setCurrentXFAWidget(null);
            }
        }

        if (getState() == ReadStateConfig.STATE_FILLSIGN) {
            FillSignToolHandler toolHandler = (FillSignToolHandler) getToolHandlerByType(ToolHandler.TH_TYPE_FILLSIGN);
            if (toolHandler != null) {
                toolHandler.onExitTool(null);
            }
        }

        if (mMainFrame.getAttachedActivity() == null) {
            mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_closing);
            closeAllDocuments();
            return;
        }

        if (mPdfViewCtrl.getDoc() == null || !getDocumentManager().isDocModified()) {
            mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_closing);
            closeAllDocuments();
            return;
        }

        /*        final boolean hideSave = !mPdfViewCtrl.isDynamicXFA() && !getDocumentManager().canModifyContents();*/
        if (/*!hideSave && */mIsAutoSaveDoc) {
            saveToOriginalFile();
            return;
        }

        AlertDialog.Builder builder = new AlertDialog.Builder(mMainFrame.getAttachedActivity());
        String[] items;
//        if (hideSave) {
//            items = new String[]{
//                    AppResource.getString(mContext.getApplicationContext(), R.string.rv_back_save_to_new_file),
//                    AppResource.getString(mContext.getApplicationContext(), R.string.rv_back_discard_modify),
//            };
//        } else {
        items = new String[]{
                AppResource.getString(mContext, R.string.rv_back_save_to_original_file),
                AppResource.getString(mContext, R.string.rv_back_save_to_new_file),
                AppResource.getString(mContext, R.string.rv_back_discard_modify),
        };
//        }

        builder.setItems(items, new DialogInterface.OnClickListener() {
            @Override
            public void onClick(DialogInterface dialog, int which) {
//                if (hideSave) {
//                    which += 1;
//                }
                switch (which) {
                    case 0: // save
                        saveToOriginalFile();
                        break;
                    case 1: // save as
                        mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_saving);
                        onSaveAsClicked();
                        break;
                    case 2: // discard modify
                        mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_closing);
                        isSaveDocInCurPath = false;
                        closeAllDocuments();
                        break;
                    default:
                        break;
                }
                dialog.dismiss();
                mSaveAlertDlg = null;
            }

            void showInputFileNameDialog(final String fileFolder) {
                String oriFileName;
                if (TextUtils.isEmpty(mDocPath)) {
                    oriFileName = AppDmUtil.randomUUID(null);
                } else {
                    oriFileName = AppFileUtil.getFileName(mDocPath);
                }
                final String newFilePath = fileFolder + "/" + oriFileName;
                final String filePath = AppFileUtil.getFileDuplicateName(newFilePath);
                final String fileName = AppFileUtil.getFileNameWithoutExt(filePath);

                final UITextEditDialog rmDialog = new UITextEditDialog(mMainFrame.getAttachedActivity());
                rmDialog.setPattern("[/\\:*?<>|\"\n\t]");
                rmDialog.setTitle(AppResource.getString(mContext.getApplicationContext(), R.string.fx_string_saveas));
                rmDialog.setLengthFilters(UITextEditDialog.MAX_FILE_NAME_LENGTH);
                rmDialog.getPromptTextView().setVisibility(View.GONE);
                rmDialog.getInputEditText().setText(fileName);
                rmDialog.getInputEditText().selectAll();
                rmDialog.show();
                AppUtil.showSoftInput(rmDialog.getInputEditText());

                rmDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        rmDialog.dismiss();
                        String inputName = rmDialog.getInputEditText().getText().toString();
                        String newPath = fileFolder + "/" + inputName;
                        newPath += ".pdf";
                        if (AppFileUtil.existsFileOrDocument(mContext, newPath)) {
                            showAskReplaceDialog(fileFolder, newPath);
                        } else {
                            mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_saving);
                            showProgressDlg();
                            isCloseDocAfterSaving = true;
                            mSavePath = newPath;
                            if (AppFileUtil.needScopedStorageAdaptation()) {
                                newPath = getCacheFile(newPath);
                            }
                            saveDoc(newPath);
                        }
                    }
                });
            }

            void showAskReplaceDialog(final String fileFolder, final String newPath) {
                final UITextEditDialog rmDialog = new UITextEditDialog(mMainFrame.getAttachedActivity(), UIDialog.NO_INPUT);
                rmDialog.setTitle(AppResource.getString(mContext.getApplicationContext(), R.string.fx_string_saveas));
                rmDialog.getPromptTextView().setText(AppResource.getString(mContext.getApplicationContext(), R.string.fx_string_filereplace_warning));
                rmDialog.show();

                rmDialog.getOKButton().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        rmDialog.dismiss();
                        mSavePath = newPath;
                        isCloseDocAfterSaving = true;
                        mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_saving);
                        showProgressDlg();
                        if (newPath.equalsIgnoreCase(mDocPath)) {
                            isSaveDocInCurPath = true;
                            saveDoc(getCacheFile(null));
                        } else {
                            isSaveDocInCurPath = false;
                            String savePath = newPath;
                            if (AppFileUtil.needScopedStorageAdaptation()) {
                                savePath = getCacheFile(newPath);
                            }
                            saveDoc(savePath);
                        }
                    }
                });

                rmDialog.getCancelButton().setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        rmDialog.dismiss();
                        showInputFileNameDialog(fileFolder);
                    }
                });
            }

            void onSaveAsClicked() {
                mFolderSelectDialog = new UIFolderSelectDialog(mMainFrame.getAttachedActivity());
                mFolderSelectDialog.setFileFilter(new FileFilter() {
                    @Override
                    public boolean accept(File pathname) {
                        return !(pathname.isHidden() || !AppFileUtil.canRead(pathname));
                    }
                });
                mFolderSelectDialog.setListener(new MatchDialog.DialogListener() {
                    @Override
                    public void onResult(long btType) {
                    }

                    @Override
                    public void onBackClick() {
                    }

                    @Override
                    public void onTitleRightButtonClick() {
                        String fileFolder = mFolderSelectDialog.getCurrentPath();
                        if (!AppStorageManager.getInstance(mContext).checkDirectoryPermission(fileFolder))
                            return;
                        showInputFileNameDialog(fileFolder);
                        mFolderSelectDialog.dismiss();
                    }
                });
                mFolderSelectDialog.showDialog();
            }
        });

        mSaveAlertDlg = builder.create();
        mSaveAlertDlg.setCanceledOnTouchOutside(true);
        mSaveAlertDlg.show();
        if (AppDisplay.isPad()) {
            Window window = mSaveAlertDlg.getWindow();
            if (window != null) {
                WindowManager.LayoutParams layoutParams = window.getAttributes();
                layoutParams.width = Math.min(AppDisplay.getDialogWidth(), AppDisplay.getActivityWidth()) * 3 / 5;
                window.setAttributes(layoutParams);
            }
        }
    }

    private void saveToOriginalFile() {
        isCloseDocAfterSaving = true;
        mProgressMsg = mContext.getApplicationContext().getString(R.string.fx_string_saving);
        showProgressDlg();
        // save file to the specified file writer
        if (mFileWriter != null) {
            isSaveDocInCurPath = false;
            saveDoc(mFileWriter);
            return;
        }

        //save file to the specified user path or the original file.
        if (mUserSavePath != null && mUserSavePath.length() > 0 && !mUserSavePath.equalsIgnoreCase(mDocPath)) {
            File userSaveFile = new File(mUserSavePath);
            File defaultSaveFile = new File(mDocPath);
            if (userSaveFile.getParent().equalsIgnoreCase(defaultSaveFile.getParent())) {
                isSaveDocInCurPath = true;
                mSavePath = mUserSavePath;
            } else {
                isSaveDocInCurPath = false;
            }
            saveDoc(mUserSavePath);
        } else {
            isSaveDocInCurPath = true;
            saveDoc(getCacheFile(null));
        }
    }

    @Override
    public void setBackEventListener(BackEventListener listener) {
        mBackEventListener = listener;
    }

    @Override
    public BackEventListener getBackEventListener() {
        return mBackEventListener;
    }

    /**
     * Should be call in {@link Activity#onKeyDown(int, KeyEvent)}
     *
     * @param act     The current activity
     * @param keyCode The value in event.getKeyCode().
     * @param event   Description of the key event.
     * @return Return <code>true</code> to prevent this event from being propagated
     * further, or <code>false</code> to indicate that you have not handled
     * this event and it should continue to be propagated.
     * {@link Activity#onKeyDown(int, KeyEvent)}
     */
    public boolean onKeyDown(Activity act, int keyCode, KeyEvent event) {
        if (event.isCtrlPressed()) return false;
        if (AppUtil.isFastDoubleClick()) return true;
        if (mMainFrame.getAttachedActivity() != act) return false;
        if (keyCode == KeyEvent.KEYCODE_BACK) {
            if (onKeyDownCallback(act, keyCode, event)) return true;
            if (backToNormalState()) return true;

            if (event.getRepeatCount() == 0) {
                backToPrevActivity();
                return true;
            }
        }
        return false;
    }

    public boolean onKeyDownCallback(Activity act, int keyCode, KeyEvent event) {
        for (IInteractionEventListener listener : mInteractionListeners) {
            if (listener.onKeyDown(act, keyCode, event))
                return true;
        }
        return false;
    }

    /**
     * Show normal view reading state
     *
     * @return True means back the normal state
     */
    public boolean backToNormalState() {
        ComparisonModule comparisonModule = (ComparisonModule) getModuleByName(Module.MODULE_NAME_COMPARISON);
        if (comparisonModule != null) {
            comparisonModule.onKeyBack();
        }

        FileSpecPanelModule fileSpecPanelModule = (FileSpecPanelModule) getModuleByName(Module.MODULE_NAME_FILE_PANEL);
        if (fileSpecPanelModule != null && fileSpecPanelModule.onKeyBack()) {
            mMainFrame.showToolbars();
            return true;
        }

        SignaturePanelModule signaturePanelModule = (SignaturePanelModule) getModuleByName(Module.MODULE_NAME_SIGNATUREPANEL);
        if (signaturePanelModule != null && signaturePanelModule.onKeyBack()) {
            mMainFrame.showToolbars();
            return true;
        }

        if (!AppDisplay.isPad() && getPanelManager().isShowingPanel()) {
            getPanelManager().hidePanel();
            return true;
        }

        if (mPdfViewCtrl.isDynamicXFA()) {
            DynamicXFAModule dynamicXFAModule = (DynamicXFAModule) getModuleByName(Module.MODULE_NAME_DYNAMICXFA);
            if (dynamicXFAModule != null && dynamicXFAModule.onKeyBack()) {
                changeState(ReadStateConfig.STATE_NORMAL);
                mMainFrame.showToolbars();
                return true;
            }
        }

        SearchModule searchModule = ((SearchModule) getModuleByName(Module.MODULE_NAME_SEARCH));
        if (searchModule != null && searchModule.onKeyBack()) {
            changeState(ReadStateConfig.STATE_NORMAL);
            mMainFrame.showToolbars();
            return true;
        }

        FormFillerModule formFillerModule = (FormFillerModule) getModuleByName(Module.MODULE_NAME_FORMFILLER);
        if (formFillerModule != null && formFillerModule.onKeyBack()) {
            changeState(ReadStateConfig.STATE_NORMAL);
            mMainFrame.showToolbars();
            return true;
        }

        ToolHandler currentToolHandler = getCurrentToolHandler();
        SignatureModule signature_module = (SignatureModule) getModuleByName(Module.MODULE_NAME_PSISIGNATURE);
        if (signature_module != null && currentToolHandler instanceof SignatureToolHandler && signature_module.onKeyBack()) {
            changeState(ReadStateConfig.STATE_NORMAL);
            mMainFrame.showToolbars();
            mPdfViewCtrl.invalidate();
            return true;
        }

        FileAttachmentModule fileAttachmentModule = (FileAttachmentModule) getModuleByName(Module.MODULE_NAME_FILEATTACHMENT);
        if (fileAttachmentModule != null && fileAttachmentModule.onKeyBack()) {
            mMainFrame.showToolbars();
            return true;
        }

        CropModule cropModule = (CropModule) getModuleByName(Module.MODULE_NAME_CROP);
        if (cropModule != null && cropModule.onKeyBack()) {
            mMainFrame.showToolbars();
            return true;
        }

        PanZoomModule panZoomModule = (PanZoomModule) getModuleByName(Module.MODULE_NAME_PANZOOM);
        if (panZoomModule != null && panZoomModule.exit()) {
            mMainFrame.showToolbars();
            return true;
        }

        BlankSelectToolHandler selectToolHandler = (BlankSelectToolHandler) getToolHandlerByType(ToolHandler.TH_TYPE_BLANKSELECT);
        if (selectToolHandler != null && selectToolHandler.onKeyBack()) {
            return true;
        }

        TextSelectModule textSelectModule = (TextSelectModule) getModuleByName(Module.MODULE_NAME_SELECTION);
        if (textSelectModule != null && textSelectModule.onKeyBack()) {
            return true;
        }

        TTSModule ttsModule = (TTSModule) getModuleByName(Module.MODULE_NAME_TTS);
        if (ttsModule != null && ttsModule.onKeyBack()) {
            return true;
        }

        if (getDocumentManager().onKeyBack()) {
            return true;
        }

        if (currentToolHandler != null) {
            setCurrentToolHandler(null);
            return true;
        }

        if (getState() != ReadStateConfig.STATE_NORMAL
                && getState() != ReadStateConfig.STATE_COMPARE
                && getState() != ReadStateConfig.STATE_FILLSIGN) {
            changeState(ReadStateConfig.STATE_NORMAL);
            return true;
        }
        return false;
    }

    /**
     * Set the file path
     */
    public void setFilePath(String path) {
        mDocPath = path;
        MoreMenuModule module = ((MoreMenuModule) getModuleByName(Module.MODULE_MORE_MENU));
        if (module != null) {
            module.setFilePath(path);
        }
    }

    private void doOpenAction(PDFDoc pdfDoc) {
        try {
            if (pdfDoc !=null)
                pdfDoc.doJSOpenAction();
        } catch (PDFException ignored) {
        }
    }

    /**
     * Whether show the top tool bar
     *
     * @param isEnabled <CODE>True</CODE> show the top tool bar, or otherwise.
     */
    public void enableTopToolbar(boolean isEnabled) {
        if (mMainFrame != null) {
            mMainFrame.enableTopToolbar(isEnabled);
        }
    }

    /**
     * Whether show the bottom tool bar
     *
     * @param isEnabled <CODE>True</CODE> show the bottom tool bar, or otherwise.
     */
    public void enableBottomToolbar(boolean isEnabled) {
        if (mMainFrame != null) {
            mMainFrame.enableBottomToolbar(isEnabled);
        }
    }

    /**
     * Returns true if the document is automatically saved.
     *
     * @return True if the document is automatically saved, false otherwise.
     */
    public boolean isAutoSaveDoc() {
        return mIsAutoSaveDoc;
    }

    /**
     * Sets whether the document should be automatically saved.
     *
     * @param autoSaveDoc True auto save document, false otherwise.
     */
    public void setAutoSaveDoc(boolean autoSaveDoc) {
        this.mIsAutoSaveDoc = autoSaveDoc;
    }


    /**
     * Set annotation author
     *
     * @param author the author string to be set
     */
    public void setAnnotAuthor(String author) {
        mAnnotAuthor = author == null ? "" : author;
    }

    /**
     * Whether to update the default properties of creating annot.
     *
     * @return True if you modify some properties of an annot,
     * those properties will be used the next time when you create the same type of annot,
     * false otherwise.
     */
    public boolean canUpdateAnnotDefaultProperties() {
        return mCanUpdateAnnotDefaultProperties;
    }

    /**
     * Set whether to update the default properties of creating annot.
     *
     * @param update True if you modify some properties of an annot,
     *               those properties will be used the next time when you create the same type of annot,
     *               false otherwise.
     */
    public void setUpdateAnnotDefaultProperties(boolean update) {
        mCanUpdateAnnotDefaultProperties = update;
    }

    /**
     * \fn void setInkDrawToolType(int toolType)
     * Set the ink drawing type.
     *
     * @param toolType The ink drawing tool type. Must be one of:
     *                 {@link InkDrawToolType#STYLUS} – draws ink only when using a stylus, or
     *                 {@link InkDrawToolType#STYLUS_OR_FINGER} – automatically selects the drawing tool based on whether a finger or stylus is used.
     */
    public void setInkDrawToolType(int toolType) {
        AppSharedPreferences.getInstance(mContext).setInteger("OnlyStylus", "LocalDrawToolType", toolType);
    }

    /**
     * Get the ink drawing type.
     *
     * @return The ink drawing tool type. Must be one of:
     * {@link InkDrawToolType#STYLUS} – draws ink only when using a stylus, or
     * {@link InkDrawToolType#STYLUS_OR_FINGER} – automatically selects the drawing tool based on whether a finger or stylus is used.
     */
    public int getInkDrawToolType() {
        return AppSharedPreferences.getInstance(mContext).getInteger("OnlyStylus", "LocalDrawToolType", InkDrawToolType.STYLUS_OR_FINGER);
    }

    /**
     * Get annotation author string. The default author is "foxit sdk"
     *
     * @return Annotation author string
     */
    public String getAnnotAuthor() {
        return mAnnotAuthor;
    }

    /**
     * Set the flag to be used when the document has saved
     */
    public void setSaveDocFlag(int flag) {
        mSaveFlag = flag;
    }

    /**
     * Get the flag while saving a document has used
     */
    public int getSaveDocFlag() {
        return mSaveFlag;
    }

    private void openDocumentFailed() {
        if (mMainFrame.getAttachedActivity() != null) {
            if (onFinishListener != null) {
                onFinishListener.onFinish();
            } else {
                mMainFrame.getAttachedActivity().finish();
            }
        }
    }

    private void _resetStatusAfterClose() {
        changeState(ReadStateConfig.STATE_NORMAL);
    }

    private void _resetStatusBeforeOpen() {
//        mMainFrame.showToolbars();
        mState = ReadStateConfig.STATE_NORMAL;
    }

    private void closeAllDocuments() {
        if (!bDocClosed) {
            _closeDocument();
        } else if (mMainFrame.getAttachedActivity() != null) {
            if (onFinishListener != null) {
                onFinishListener.onFinish();
            } else {
                mMainFrame.getAttachedActivity().finish();
            }
        }
    }

    private void showProgressDlg() {
        if (mProgressDlg == null && mMainFrame.getAttachedActivity() != null) {
            mProgressDlg = new ProgressDialog(mMainFrame.getAttachedActivity());
            mProgressDlg.setProgressStyle(ProgressDialog.STYLE_SPINNER);
            mProgressDlg.setCancelable(false);
            mProgressDlg.setIndeterminate(false);
        }

        if (mProgressDlg != null && !mProgressDlg.isShowing()) {
            mProgressDlg.setMessage(mProgressMsg);
            AppDialogManager.getInstance().showAllowManager(mProgressDlg, null);
        }
    }

    private void dismissProgressDlg() {
        if (mProgressDlg != null && mProgressDlg.isShowing()) {
            AppDialogManager.getInstance().dismiss(mProgressDlg);
            mProgressDlg = null;
        }
    }

    private void _closeDocument() {
        showProgressDlg();
        _resetStatusAfterClose();

        mPdfViewCtrl.closeDoc();
        stopHideToolbarsTimer();
        mMainFrame.resetMaskView();
        documentManager.setDocModified(false);
        getDocumentManager().clearUndoRedo();
    }

    private void closeDocumentSucceed() {
        if (mMainFrame != null && mMainFrame.getAttachedActivity() != null) {

            if (onFinishListener != null) {
                onFinishListener.onFinish();
            } else {
                mMainFrame.getAttachedActivity().finish();
            }
        }
    }

    private void renameCacheToFile() {
        if (currentFileCachePath == null) return;
        if (mDocPath.endsWith(".ppdf")) {
            currentFileCachePath = AppFileUtil.replaceFileExtension(currentFileCachePath, ".ppdf");
        }
        File file = new File(currentFileCachePath);
        File docFile = new File(mDocPath);
        if (file.exists()) {
            docFile.delete();
            if (!file.renameTo(docFile))
                UIToast.getInstance(mContext.getApplicationContext()).show(mContext.getApplicationContext().getString(R.string.fx_save_file_failed));
        } else {
            UIToast.getInstance(mContext.getApplicationContext()).show(mContext.getApplicationContext().getString(R.string.fx_save_file_failed));
        }
    }

    private void updateThumbnail(String path) {
        LocalModule module = (LocalModule) getModuleByName(Module.MODULE_NAME_LOCAL);
        if (module != null && path != null) {
            module.updateThumbnail(path);
        }
    }

    private String getCacheFile(String path) {
        mSavePath = path == null ? mDocPath : path;
        File file = new File(mDocPath);
        String dir = file.getParent() + "/";
        while (file.exists()) {
            currentFileCachePath = dir + AppDmUtil.randomUUID(null) + ".pdf";
            file = new File(currentFileCachePath);
        }
        return currentFileCachePath;
    }

    private int mLastReadState;

    private void changeToolBarState(ToolHandler oldToolHandler, ToolHandler newToolHandler) {
            if (newToolHandler instanceof FormFillerToolHandler)
                return;

            if (newToolHandler instanceof RedactToolHandler) {
                triggerDismissMenuEvent();
                mLastReadState = getState();
                changeState(ReadStateConfig.STATE_REDACT);
            }

            if (oldToolHandler instanceof RedactToolHandler
                    && getState() == ReadStateConfig.STATE_REDACT
                    && newToolHandler == null) {
                if (mLastReadState == ReadStateConfig.STATE_FILLSIGN)
                    changeState(ReadStateConfig.STATE_FILLSIGN);
                else
                    changeState(ReadStateConfig.STATE_NORMAL);
            }

            if ((!mConfig.uiSettings.fullscreen && !mMainFrame.isToolbarsVisible())
                    || (mConfig.uiSettings.fullscreen && !mMainFrame.isToolbarsVisible() && !mMainFrame.isShowFullScreenUI())
                    || (mConfig.uiSettings.fullscreen && mMainFrame.isShowFullScreenUI() && newToolHandler instanceof RedactToolHandler))
                mMainFrame.showToolbars();
    }

    /**
     * set the file writer callback where the document will be saved.
     *
     * @param writer File writer callback{@link FileWriterCallback} for the new saved PDF file.
     */
    public void setSaveWriter(FileWriterCallback writer) {
        mUserSavePath = null;
        mFileWriter = writer;
    }

    /**
     * @return Get the file writer callback where the document will be saved.
     */
    public FileWriterCallback getSaveWriter() {
        return mFileWriter;
    }

    /**
     * Set the path where the document will be saved
     */
    public void setSavePath(String savePath) {
        mFileWriter = null;
        mUserSavePath = savePath;
    }

    /**
     * @return the path where the document is saved
     */
    public String getSavePath() {
        return mUserSavePath;
    }

    /**
     * @return the {@link IMenuView} for interacting with menus
     */
    public IMenuView getMenuView() {
        MoreMenuModule module = ((MoreMenuModule) getModuleByName(Module.MODULE_MORE_MENU));
        if (module == null) return null;
        return module.getMenuView();
    }

    @Override
    public MenuViewManager getMenuViewManager() {
        if (mCustomViewMgr == null)
            mCustomViewMgr = new MenuViewManager();
        return mCustomViewMgr;
    }

    /**
     * Check whether the specified annotation module that is loaded.
     */
    public boolean isLoadAnnotModule(Annot annot) {
        String jsonName = AppAnnotUtil.getTypeString(annot);
        return isLoadAnnotModule(jsonName);
    }

    /// @cond DEV

    /**
     * Check whether the specified annotation module that is loaded
     *
     * @param typeString the module string type{@link JsonConstants}
     * @return true means the module is loaded
     */
    public boolean isLoadAnnotModule(String typeString) {
        if (mConfig == null || mConfig.modules == null) return false;
        if (!mConfig.modules.isLoadAnnotations) return false;
        AnnotationsConfig annotConfig = mConfig.modules.annotations;
        switch (typeString) {
            case JsonConstants.TYPE_HIGHLIGHT:
            case JsonConstants.TYPE_AREA_HIGHLIGHT:
                return annotConfig.isLoadHighlight;
            case JsonConstants.TYPE_UNDERLINE:
                return annotConfig.isLoadUnderline;
            case JsonConstants.TYPE_SQUIGGLY:
                return annotConfig.isLoadSquiggly;
            case JsonConstants.TYPE_STRIKEOUT:
                return annotConfig.isLoadStrikeout;
            case JsonConstants.TYPE_CARET:
                return annotConfig.isLoadInsertText;
            case JsonConstants.TYPE_REPLACE:
                return annotConfig.isLoadReplaceText;
            case JsonConstants.TYPE_LINE:
                return annotConfig.isLoadLine;
            case JsonConstants.TYPE_SQUARE:
                return annotConfig.isLoadSquare;
            case JsonConstants.TYPE_CIRCLE:
                return annotConfig.isLoadCircle;
            case JsonConstants.TYPE_LINEARROW:
                return annotConfig.isLoadArrow;
            case JsonConstants.TYPE_INK:
                return annotConfig.isLoadPencil;
            case JsonConstants.TYPE_TYPEWRITER:
                return annotConfig.isLoadTypewriter;
            case JsonConstants.TYPE_TEXTBOX:
                return annotConfig.isLoadTextbox;
            case JsonConstants.TYPE_CALLOUT:
                return annotConfig.isLoadCallout;
            case JsonConstants.TYPE_NOTE:
                return annotConfig.isLoadNote;
            case JsonConstants.TYPE_STAMP:
                return annotConfig.isLoadStamp;
            case JsonConstants.TYPE_POLYGON:
                return annotConfig.isLoadPolygon;
            case JsonConstants.TYPE_POLYGONCLOUD:
                return annotConfig.isLoadCloud;
            case JsonConstants.TYPE_POLYLINE:
                return annotConfig.isLoadPolyLine;
            case JsonConstants.TYPE_LINEDIMENSION:
                return annotConfig.isLoadMeasure;
            case JsonConstants.TYPE_SCREEN_IMAGE:
                return annotConfig.isLoadImage;
            case JsonConstants.TYPE_SOUND:
            case JsonConstants.TYPE_AUDIO:
                return annotConfig.isLoadAudio;
            case JsonConstants.TYPE_VIDEO:
                return annotConfig.isLoadVideo;
            case JsonConstants.TYPE_ATTACHMENT:
                return annotConfig.isLoadFileattachment;
            case JsonConstants.TYPE_REDACTION:
                return annotConfig.isLoadRedaction;
            default:
                return false;
        }
    }
    /// @endcond

    private boolean isTwoColumnLeft(PDFDoc pdfDoc) {
        try {
            if (pdfDoc == null) return false;

            PDFDictionary root = pdfDoc.getCatalog();
            if (root != null && root.hasKey("PageLayout")) {
                PDFObject pageLayout = root.getElement("PageLayout");
                if (pageLayout == null) return false;
                if (pageLayout.getName().equalsIgnoreCase("TwoColumnLeft")) {
                    return true;
                }
            }
        } catch (PDFException ignored) {
        }
        return false;
    }

    private boolean isCompareDoc() {
        if (mPdfViewCtrl == null || mPdfViewCtrl.getDoc() == null) return false;
        try {
            PDFDictionary root = mPdfViewCtrl.getDoc().getCatalog();
            if (root != null) {
                boolean bExistPieceInfo = root.hasKey("PieceInfo");
                if (!bExistPieceInfo) return false;
                PDFDictionary pieceInfo = root.getElement("PieceInfo").getDict();
                if (pieceInfo == null) return false;
                return pieceInfo.hasKey("ComparePDF");
            }
        } catch (PDFException e) {

        }
        return false;
    }

    /**
     * The interface for document modified event listener.
     */
    public interface IDocModifiedEventListener {
        /**
         * Triggered when the document is modified.
         *
         * @param doc A <CODE>PDFDoc</CODE> object which is modified.
         */
        void onDocModified(PDFDoc doc);
    }

    private ArrayList<IDocModifiedEventListener> mDocumentModifiedEventListeners = new ArrayList<IDocModifiedEventListener>();

    /**
     * Register a document modified event listener.
     *
     * @param listener An <CODE>IDocModifiedEventListener</CODE> object to be registered.
     */
    public void registerDocModifiedEventListener(IDocModifiedEventListener listener) {
        mDocumentModifiedEventListeners.add(listener);
    }

    /**
     * Unregister a document modified event listener.
     *
     * @param listener An <CODE>IDocModifiedEventListener</CODE> object to be unregistered.
     */
    public void unregisterDocModifiedEventListener(IDocModifiedEventListener listener) {
        mDocumentModifiedEventListeners.remove(listener);
    }

    protected void onDocumentModified(PDFDoc pdfDoc) {
        for (IDocModifiedEventListener documentEventListener : mDocumentModifiedEventListeners) {
            documentEventListener.onDocModified(pdfDoc);
        }
    }

    private boolean mIsAutoSaveSignedDoc = false;
    private String mSignedDocSavePath = null;

    /**
     * Returns true if the signed document is automatically saved.
     *
     * @return True if the signed document is automatically saved, false otherwise.
     */
    public boolean isAutoSaveSignedDoc() {
        return mIsAutoSaveSignedDoc;
    }

    /**
     * Set to automatically save the signed document.
     * <p>
     * Note: if user don`t call {@link #setSignedDocSavePath(String)} to set the path where the signed document will be saved,
     * the signed document will be saved to the same path as the original file and "-signed" suffix should be added to the filename.
     *
     * @param autoSaveSignedDoc True auto save signed document, false otherwise.
     */
    public void setAutoSaveSignedDoc(boolean autoSaveSignedDoc) {
        this.mIsAutoSaveSignedDoc = autoSaveSignedDoc;
    }

    /**
     * Set the full PDF file path where the signed document will be saved and it works when set to automatically save the signed document.
     */
    public void setSignedDocSavePath(String savePath) {
        mSignedDocSavePath = savePath;
    }

    /**
     * @return the path where the signed document is saved.
     */
    public String getSignedDocSavePath() {
        return mSignedDocSavePath;
    }

    public ToolItemsManager getToolsManager() {
        return mBarToolsManager;
    }

    @Override
    public void onThemeChanged(String type, int color) {
        // MOBRD-6394
        // Caused by java.util.ConcurrentModificationException:
        //  at java.util.ArrayList$Itr.next (ArrayList.java:860)
        //  at com.foxit.uiextensions.UIExtensionsManager.onThemeChanged (UIExtensionsManager.java:3516)
        int count = mThemeEventListeners.size();
        for (int i = 0; i < count; i ++) {
            if (i >= mThemeEventListeners.size())
                break;
            IThemeEventListener listener = mThemeEventListeners.get(i);
            listener.onThemeColorChanged(type, color);
        }
    }

    private IAppInfoProvider mAPPInfoProvider;

    /**
     * This interface is used to set application information
     * @param provider example Set the application version name {@link IAppInfoProvider}
     */
    public void setAPPInfoProvider(IAppInfoProvider provider) {
        mAPPInfoProvider = provider;
    }

    /**
     * Get Interface for applying related information
     * @return Application version name information {@link IAppInfoProvider}
     */
    public IAppInfoProvider getAPPInfoProvider() {
        if(mAPPInfoProvider == null){
            setAPPInfoProvider(InfoProvider);
        }
        return mAPPInfoProvider;
    }



    //For Customize Permission
    private IPermissionProvider mPermissionProvider;

    public void setPermissionProvider(IPermissionProvider provider) {
        mPermissionProvider = provider;
    }

    public IPermissionProvider getPermissionProvider() {
        return mPermissionProvider;
    }

    private ISystemPermissionProvider mSysPermissionProvider;

    public void setSystemPermissionProvider(ISystemPermissionProvider provider) {
        mSysPermissionProvider = provider;
    }

    public ISystemPermissionProvider getSystemPermissionProvider() {
        return mSysPermissionProvider;
    }

    //For customize annotation permission
    protected void setAnnotationPermission(IAnnotationPermission permission) {
        getDocumentManager().setAnnotationPermission(permission);
    }

    protected IAnnotationPermission getAnnotationPermission() {
        return getDocumentManager().getAnnotationPermission();
    }

    /**
     * Add permission control to annotations on the basis of document permission
     *
     * @param permission customize annotations permission {@link IAnnotationsPermission}
     */
    public void setAnnotationsPermission(IAnnotationsPermission permission) {
        getDocumentManager().setAnnotationsPermission(permission);
    }


    /**
     * Get permission control to annotations on the basis of document permission.
     *
     * @return customize annotations permission {@link IAnnotationsPermission}
     */
    public IAnnotationsPermission getAnnotationsPermission() {
        return getDocumentManager().getAnnotationsPermission();
    }

    ArrayList<ITouchEventListener> mTouchEventListeners = new ArrayList<>();

    public void registerTouchEventListener(ITouchEventListener l) {
        mTouchEventListeners.add(l);
    }

    public void unregisterTouchEventListener(ITouchEventListener l) {
        mTouchEventListeners.remove(l);
    }

    /**
     * Receive and handle the single tap event.
     *
     * @param  motionEvent <CODE>MotionEvent</CODE> object which species the event.
     * @return true if the event is consumed, false otherwise.
     */
    public boolean handleSingleTapConfirmed(MotionEvent motionEvent) {
        if (AppDisplay.isPad() && getPanelManager().isShowingPanel())
            getPanelManager().hidePanel();

        if (getDocumentManager().getCurrentAnnot() != null) {
            getDocumentManager().setCurrentAnnot(null);
            return true;
        }
        if (mPdfViewCtrl.isPageFlippingByTouchBorder()
                && isClickBorderArea(new PointF(motionEvent.getX(), motionEvent.getY()))
                && !(getDocumentManager().hasAnnot(new PointF(motionEvent.getX(), motionEvent.getY()))))
            return false;
        if (getState() == ReadStateConfig.STATE_SEARCH) return false;

        PointF stv_pt = new PointF(motionEvent.getX(), motionEvent.getY());
        int pageIndex = mPdfViewCtrl.getPageIndex(stv_pt);
        if (mCurToolHandler != null && pageIndex < 0) {
            return true;
        }

        if (mMainFrame.isToolbarsVisible()) {
            if (mConfig.uiSettings.fullscreen) {
                mMainFrame.hideToolbars();
                stopHideToolbarsTimer();
            }
        } else {
            mMainFrame.showToolbars();
            startHideToolbarsTimer();
        }
        return true;
    }

    /**
     * The interface for signature related event listener.
     */
    public interface ISignatureEventListener {
        /**
         * Triggered when a digital signature is signed.
         *
         * @param success Return true if the signature has been signed, otherwise return false.
         */
        void onDigitalSignatureSigned(boolean success);
    }

    private ArrayList<ISignatureEventListener> mSignatureEventListeners;

    /**
     * Register a signature related event listener.
     *
     * @param listener An <CODE>ISignatureEventListener</CODE> object to be registered.
     */
    public void registerSignatureEventListener(ISignatureEventListener listener) {
        if (mSignatureEventListeners == null) {
            mSignatureEventListeners = new ArrayList<>();
        }
        mSignatureEventListeners.add(listener);
    }

    /**
     * Unregister a signature related event listener.
     *
     * @param listener An <CODE>ISignatureEventListener</CODE> object to be unregistered.
     */
    public void unregisterSignatureEventListener(ISignatureEventListener listener) {
        if (mSignatureEventListeners == null) return;
        mSignatureEventListeners.remove(listener);
    }

    public void onDigitalSignatureSigned(boolean success) {
        if (mSignatureEventListeners == null) return;
        for (ISignatureEventListener eventListener : mSignatureEventListeners) {
            eventListener.onDigitalSignatureSigned(success);
        }
    }

    private ArrayList<IUIInteractionEventListener> mUIInteractionEventListeners;

    /**
     * Register a UI intercation related event listener.
     *
     * @param listener An <CODE>IUIInteractionEventListener</CODE> object to be registered.
     */
    public void registerUIInteractionEventListener(IUIInteractionEventListener listener) {
        if (mUIInteractionEventListeners == null) {
            mUIInteractionEventListeners = new ArrayList<>();
        }
        mUIInteractionEventListeners.add(listener);
    }

    /**
     * Unregister a UI intercation related event listener.
     *
     * @param listener An <CODE>IUIInteractionEventListener</CODE> object to be unregistered.
     */
    public void unregisterUIInteractionEventListener(IUIInteractionEventListener listener) {
        if (mUIInteractionEventListeners == null) return;
        mUIInteractionEventListeners.remove(listener);
    }

    public void onUIInteractElementClicked(String element) {
        if (mUIInteractionEventListeners == null) return;
        for (IUIInteractionEventListener eventListener : mUIInteractionEventListeners) {
            eventListener.onUIElementClicked(element, null);
        }
    }

    private boolean isSupportTurnPageTool(String toolName) {
//        if (AppDevice.isChromeOs(mAttachActivity)) return false;// on the chrome os ,single tap will
//                                                               // trigger onSingleTapConfirmed and onshowpress(SDKRD-7172/SDKRD-7088)

        if (AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_NOTE)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_FILEATTACHMENT)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_TYPEWRITER)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_TEXTBOX)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_STAMP)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_PDFIMAGE)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_CALLOUT)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_HIGHLIGHT)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_UNDERLINE)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_STRIKEOUT)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_SQUIGGLY)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_REPLACE)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPR_INSERTTEXT)
                //                || AppUtil.isEqual(toolName, ToolHandler.)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_DISTANCE)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_SQUARE)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_CIRCLE)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_LINE)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_ARROW)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_POLYLINE)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_POLYGON)
                || AppUtil.isEqual(toolName, ToolHandler.TH_TYPE_POLYGONCLOUD)
        ) {
            return true;
        }
        return false;
    }

    /**
     * Enable or disable small top toolbar.
     *
     * @param isEnabled Whether small top toolbar is enabled or not.
     */
    public void enableSmallTopToolbar(boolean isEnabled){
        if (mMainFrame != null) {
            mMainFrame.enableSmallTopToolbar(isEnabled);
        }
    }

    /**
     * Enable or disable small bottom toolbar.
     *
     * @param isEnabled Whether small bottom toolbar is enabled or not.
     */
    public void enableSmallBottomToolbar(boolean isEnabled){
        if (mMainFrame != null) {
            mMainFrame.enableSmallBottomToolbar(isEnabled);
        }
    }

    public String getRealDocPath(){
        return mRealDocPath;
    }

    private boolean mIsEnableModify = true;

    /**
     * whether the pdf document can be modified
     *
     * @return whether the pdf document can be modified
     */
    public boolean isEnableModification() {
        return mIsEnableModify;
    }

    /**
     * Set whether the pdf document can be modified. The default is allow modification.
     *
     * @param isEnabled whether the pdf document can be modified
     */
    public void enableModification(boolean isEnabled) {
        mIsEnableModify = isEnabled;
    }

    /**
     * measurement units: inch
     */
    public static final int MEASUREMENT_UNIT_INCH = 0;
    /**
     * measurement units: centermeters
     */
    public static final int MEASUREMENT_UNIT_CM = 1;
    /**
     * measurement units: millimeters
     */
    public static final int MEASUREMENT_UNIT_MM = 2;

    private int mMeasurementUnits = MEASUREMENT_UNIT_INCH;

    /**
     * Get measurement units for page size information
     *
     * @return measurement units
     */
    public int getMeasurementUnits() {
        return mMeasurementUnits;
    }

    /**
     * Set measurement units for page size information. The default value is inch.
     *
     * @param unit  measurement units
     */
    public void setMeasurementUnits(int unit) {
        mMeasurementUnits = unit;
    }

    private void saveDoc(String path){
        int saveFlags = mSaveOptions != null ? mSaveOptions.mSaveFlags : mSaveFlag;
        try {
            saveFlags = mPdfViewCtrl.getDoc().getSignatureCount() > 0 ? PDFDoc.e_SaveFlagIncremental : saveFlags;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        mPdfViewCtrl.saveDoc(path, saveFlags);
        mSaveOptions = null;
    }

    private void saveDoc(FileWriterCallback fileWriter){
        int saveFlags = mSaveOptions != null ? mSaveOptions.mSaveFlags : mSaveFlag;
        try {
            saveFlags= mPdfViewCtrl.getDoc().getSignatureCount() > 0 ? PDFDoc.e_SaveFlagIncremental : saveFlags;
        } catch (PDFException e) {
            e.printStackTrace();
        }
        mPdfViewCtrl.saveDoc(fileWriter, saveFlags);
        mSaveOptions = null;
    }

    /**
     * Sets a timeout for Ink drawing. If the drawing action stops and exceeds the timeout,
     * an Ink annotation will be automatically created.
     * <p>
     * The default value is -1, which disables the timeout. If the timeout value is less than or equal to 0, the timeout feature is disabled.
     *
     * @param timeoutMillis The timeout duration in milliseconds. Default is -1 to disable.
     */
    public void setInkDrawingTimeout(long timeoutMillis) {
        mInkTimeOut = timeoutMillis;
    }

    /**
     *  Gets the current timeout value for Ink drawing.
     * <p>
     * The default value is -1, which disables the timeout. If the timeout value is less than or equal to 0, the timeout feature is disabled.
     *
     * @return The timeout duration in milliseconds. Default is -1 to disable.
     */
    public long getInkDrawingTimeout() {
        return mInkTimeOut;
    }

    /// @cond DEV
    private SaveOptions mSaveOptions;

    public static class SaveOptions{
        private final int mSaveFlags;

        public SaveOptions(int saveFlags){
            this.mSaveFlags = saveFlags;
        }

        public int getSaveFlags(){
            return mSaveFlags;
        }
    }

    public void setSaveOptions(SaveOptions options){
        mSaveOptions = options;
    }

    public SaveOptions getSaveOptions(){
        return mSaveOptions;
    }

    /**
     * \fn void registerUISaveasEventListener(int eventType, IUISaveasEventListener listener)
     * Register a document save event listener.
     *
     * @param eventType Save type, please refer to <CODE>IUISaveasEventListener#SaveasEventType</CODE>
     * @param listener An <CODE>IUISaveasEventListener </CODE> object to be registered.
     */
    public void registerUISaveasEventListener(@IUISaveasEventListener.SaveasEventType int eventType, IUISaveasEventListener listener) {
        mSaveEventListeners.put(eventType, listener);
    }

    /**
     * \fn void unregisterUISaveasEventListener(int eventType)
     * Unregister a document save event listener.
     *
     * @param eventType Save event type, please refer to <CODE>IUISaveasEventListener#SaveasEventType</CODE>
     */
    public void unregisterUISaveasEventListener(@IUISaveasEventListener.SaveasEventType int eventType) {
        mSaveEventListeners.remove(eventType);
    }

    /**
     * \fn void getSaveasEventListener(int eventType)
     * Get save listener by event type.
     *
     * @param eventType Save event type, please refer to <CODE>IUISaveasEventListener#SaveasEventType</CODE>
     * @return the save listener
     */
    public IUISaveasEventListener getSaveasEventListener(@IUISaveasEventListener.SaveasEventType int eventType){
        return mSaveEventListeners.get(eventType);
    }

    /**
     * Add create property changed listener.
     * <p>
     * When {@link #canUpdateAnnotDefaultProperties()} return true, this method will take effect,
     * that is when modify some properties of an annot,those properties will be used the next time when
     * you create the same type of annot, this method is used to notify where changes need to be made.
     *
     * @param annotHandlerType the annot handler type, most of the time it is the type of annot{@link AnnotHandler}
     * @param changedListener the changed callback.
     */
    public void addCreatePropertyChangedListener(int annotHandlerType, PropertyBar.CreatePropertyChangedListener changedListener){
        mCreatePropertyChangedListener.put(annotHandlerType, changedListener);
    }

    /**
     * Remove create property changed listener.
     * <p>
     * When {@link #canUpdateAnnotDefaultProperties()} return true, this method will take effect,
     * that is when modify some properties of an annot,those properties will be used the next time when
     * you create the same type of annot, this method is used to notify where changes need to be made.
     *
     * @param annotHandlerType the annot handler type{@link AnnotHandler}
     */
    public void removeCreatePropertyChangedListener(int annotHandlerType) {
        mCreatePropertyChangedListener.remove(annotHandlerType);
    }

    /**
     * Get create property changed listener.
     * <p>
     * When {@link #canUpdateAnnotDefaultProperties()} return true, this method will take effect,
     * that is when modify some properties of an annot,those properties will be used the next time when
     * you create the same type of annot, this method is used to notify where changes need to be made.
     *
     * @param annotHandlerType the annot handler type{@link AnnotHandler}
     * @return the changed callback.
     */
    public PropertyBar.CreatePropertyChangedListener getCreatePropertyChangedListener(int annotHandlerType){
        return mCreatePropertyChangedListener.get(annotHandlerType);
    }
    /// @endcond

    protected interface IAnnotationPermission {
        boolean canModify(Annot annot);

        boolean canDelete(Annot annot);

        boolean canReply(Annot annot);

        boolean canFlatten(Annot annot);

        boolean canAddPopupAnnot(Annot annot);
    }

    IAppInfoProvider InfoProvider = new IAppInfoProvider() {

        @Override
        public String getAppVersion() {
            return AppUtil.getVersionName(mContext);
        }

        @Override
        public String getAppName() {
            return mContext.getString(mContext.getApplicationInfo().labelRes);
        }
    };

    PhoneStateBroadCastReceiver mPhoneStateBroadCastReceiver;
    public PhoneStateBroadCastReceiver getPhoneStateBroadCaseReceiver() {
        if (mPhoneStateBroadCastReceiver == null) {
            mPhoneStateBroadCastReceiver = new PhoneStateBroadCastReceiver();
        }
        return mPhoneStateBroadCastReceiver;
    }

    private Event.Callback mPhoneStateCallback;
    public void requestPhoneStatePermission(String fun, @NonNull Event.Callback callback) {
        mPhoneStateCallback = callback;
        if (Build.VERSION.SDK_INT >= 23) {
            if (AppSharedPreferences.getInstance(mContext).getBoolean("PhoneState", "Requested", false)) {
                // request permission at first time
                callback.result(null, true);
                return;
            }

            AppSharedPreferences.getInstance(mContext).setBoolean("PhoneState", "Requested", true);
            int permissionPhoneState = ContextCompat.checkSelfPermission(mContext, Manifest.permission.READ_PHONE_STATE);
            if (permissionPhoneState != PackageManager.PERMISSION_GRANTED) {
                if (getSystemPermissionProvider() == null) {
                    getAttachedActivity().requestPermissions(new String[]{android.Manifest.permission.READ_PHONE_STATE}, ActRequestCode.REQ_TTS_READ_PHONE_STATE_PERMISSION);
                } else {
                    getSystemPermissionProvider().requestPermission(
                            true,
                            fun,
                            new String[]{Manifest.permission.READ_PHONE_STATE},
                            new ISystemPermissionProvider.IPermissionResult() {
                                @Override
                                public void onResult(boolean allowed) {
                                    if (allowed) {
                                        getAttachedActivity().requestPermissions(new String[]{Manifest.permission.READ_PHONE_STATE}, ActRequestCode.REQ_TTS_READ_PHONE_STATE_PERMISSION);
                                    } else {
                                        _showTips();
                                    }
                                }
                            }
                    );
                }

            } else {
                callback.result(null, true);
            }
        } else {
            callback.result(null,true);
        }
    }

    private void onPhoneStatePermissionResult(@NonNull String[] permissions, @NonNull int[] grantResults) {
        int permissionPhoneState = ContextCompat.checkSelfPermission(mContext, Manifest.permission.READ_PHONE_STATE);
        if (permissionPhoneState != PackageManager.PERMISSION_GRANTED) {
           //show tips
            _showTips();
        } else {
            mPhoneStateCallback.result(null, true);
        }

    }

    private void _showTips() {
        final UITextEditDialog dialog = new UITextEditDialog(getAttachedActivity());
        dialog.setTitle(mContext.getString(R.string.fx_string_prompt));
        dialog.getInputEditText().setVisibility(View.GONE);
        dialog.getPromptTextView().setText(AppResource.getString(mContext, R.string.phone_state_permission_tips));
        dialog.getCancelButton().setVisibility(View.GONE);
        dialog.getOKButton().setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                dialog.dismiss();
                mPhoneStateCallback.result(null, false);
            }
        });
        dialog.getDialog().setCanceledOnTouchOutside(false);
        dialog.show();
    }

}
