/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.views;

import android.app.Activity;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.graphics.Path;
import android.graphics.PointF;
import android.util.AttributeSet;
import android.view.SurfaceHolder;
import android.view.SurfaceHolder.Callback;
import android.view.SurfaceView;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.utils.PointUtils;
import com.luratech.android.appframework.BitmapWithMetadata;

public class DetectionResultView extends SurfaceView implements Callback {
    /**
     * Stroke width of the displayed rectangular area
     */
    private static float STROKE_WIDTH;

    /**
     * Corner points coordinate spanning the whole bitmap
     */
    private PointF[] corners;

    protected int fillColor, lineColor;
    private Paint markerPaint, shadowPaint;

    private SurfaceHolder holder;

    private Bitmap bitmap;

    private float scale;

    private int dx, dy, width, height;

    protected Matrix mDrawMatrix;

    private Path markerPath;

    private Path shadowPath;

    public Matrix getDrawMatrix() {
        return mDrawMatrix;
    }

    public DetectionResultView(Context context) {
        super(context);
        initComponent();
    }

    public DetectionResultView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initComponent();
    }

    protected void initComponent() {
        holder = getHolder();
        holder.addCallback(this);
        Activity activity = (Activity) getContext();
        Resources resources = getResources();

        STROKE_WIDTH = PointUtils.dpToPx(
                activity,
                getResources().getInteger(
                        R.integer.scanner_detection_stroke_width));

        fillColor = resources.getColor(R.color.scan_color_black_80000000);
        lineColor = resources.getColor(R.color.scan_color_blue_ff007aff);
        initPaints();
    }

    private void initPaints() {
        createShadowPaint(fillColor);
        createMarkerPaint(lineColor);
    }

    private void createShadowPaint(int fill) {
        shadowPaint = new Paint();
        shadowPaint.setStyle(Style.FILL);
        shadowPaint.setColor(fill);
    }

    private void createMarkerPaint(int stroke) {
        markerPaint = new Paint();
        markerPaint.setStrokeWidth(STROKE_WIDTH);
        markerPaint.setStrokeCap(Paint.Cap.ROUND);
        markerPaint.setStyle(Style.STROKE);
        markerPaint.setColor(stroke);
    }

    public void setFillColor(int fillColor) {
        this.fillColor = fillColor;
        createShadowPaint(fillColor);
    }

    public void setLineColor(int lineColor) {
        this.lineColor = lineColor;
        createMarkerPaint(lineColor);
    }

    public void setBitmap(BitmapWithMetadata bitmap) {
        this.bitmap = bitmap.getBitmap();
    }

    public void setDimensions(float width, float height) {
        this.width = (int) width;
        this.height = (int) height;
        computeCorners(width, height);
    }

    /**
     * Corner points coordinate spanning the whole bitmap
     *
     * @param width  bitmap width
     * @param height bitmap height
     */
    private void computeCorners(float width, float height) {
        corners = new PointF[]{new PointF(0, 0), new PointF(width, 0),
                new PointF(width, height), new PointF(0, height)};
    }

    /**
     * Set Points to draw (null means draw nothing).
     *
     * @param points points already translated to the image coordinate system and
     *               ordered clock wise (first is upper left)
     */
    public void setPoints(PointF[] points) {
        markerPath = new Path();
        shadowPath = new Path();
        if (points != null) {
            PointF[] cornerPoints = new PointF[]{points[0], points[1],
                    points[2], points[3]};
            PointF lastPoint = cornerPoints[cornerPoints.length - 1];

            markerPath.moveTo(lastPoint.x, lastPoint.y);
            for (PointF point : cornerPoints) {
                markerPath.lineTo(point.x, point.y);
            }

            for (int i = 0; i < cornerPoints.length; ++i) {
                final int j = (i + 1) % 4;

                PointF p1 = cornerPoints[i];
                PointF p2 = cornerPoints[j];
                PointF p3 = corners[(j + 1) % 4];
                PointF p4 = corners[(i + 1) % 4];

                shadowPath.moveTo(p1.x, p1.y);
                shadowPath.lineTo(p2.x, p2.y);
                shadowPath.lineTo(p3.x, p3.y);
                shadowPath.lineTo(p4.x, p4.y);
                shadowPath.lineTo(p1.x, p1.y);
            }
        }
        markerPath.close();
        shadowPath.close();
        invalidate();
    }

    @Override
    protected void onDraw(Canvas canvas) {
        int saveCount = canvas.getSaveCount();
        canvas.save();
        if (bitmap != null) {
            // only crop view has an underlying bitmap, camera has not
            canvas.drawBitmap(bitmap, mDrawMatrix, null);
        }
        canvas.concat(mDrawMatrix);
        drawMe(canvas);
        canvas.restoreToCount(saveCount);
    }

    protected void drawMe(Canvas canvas) {
        if (markerPath != null) {
            canvas.drawPath(markerPath, markerPaint);
        }
        if (shadowPath != null) {
            canvas.drawPath(shadowPath, shadowPaint);
        }
    }

    @Override
    public void surfaceCreated(SurfaceHolder holder) {
        setWillNotDraw(false);

        centerInside();
    }

    @Override
    public void surfaceChanged(SurfaceHolder holder, int format, int width,

                               int height) {
    }

    @Override
    public void surfaceDestroyed(SurfaceHolder holder) {
    }

    /**
     * scaleType=centerInside copied from ImageView.class (not available for
     * surface view)
     */
    private void centerInside() {
        mDrawMatrix = new Matrix(getMatrix());
        int vwidth = getWidth() - getPaddingLeft() - getPaddingRight();
        int vheight = getHeight() - getPaddingTop() - getPaddingBottom();

        int dwidth = width == 0 ? vwidth : width;
        int dheight = height == 0 ? vheight : height;

        if (dwidth <= vwidth && dheight <= vheight) {
            scale = 1.0f;
        } else {
            scale = Math.min((float) vwidth / (float) dwidth, (float) vheight
                    / (float) dheight);
        }

        dx = (int) ((vwidth - dwidth * scale) * 0.5f + 0.5f + getPaddingLeft());
        dy = (int) ((vheight - dheight * scale) * 0.5f + 0.5f + getPaddingTop());

        mDrawMatrix.setScale(scale, scale);
        mDrawMatrix.postTranslate(dx, dy);
    }

}
