/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.views;

import android.app.Activity;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Paint.Style;
import android.graphics.Path;
import android.graphics.Path.Direction;
import android.graphics.PointF;
import android.graphics.RectF;
import android.util.AttributeSet;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.utils.PointUtils;

public class CropableDetectionResultView extends DetectionResultView {
    private float CIRCLE_RADIUS;
    private float RECT_WIDTH;
    private float RECT_HEIGHT;

    private Paint shapePaintFill, shapePaintLine;
    private Path cornerPath;
    private Path[] handlesPath;

    public CropableDetectionResultView(Context context) {
        super(context);
        initComponent();
    }

    public CropableDetectionResultView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initComponent();
    }

    @Override
    protected void initComponent() {
        super.initComponent();

        Activity activity = (Activity) getContext();
        Resources resources = getResources();
        CIRCLE_RADIUS = PointUtils.dpToPx(activity, resources.getInteger(R.integer.editimage_cropview_corner_point_radius));
        RECT_WIDTH = PointUtils.dpToPx(activity, resources.getInteger(R.integer.editimage_cropview_handle_point_width));
        RECT_HEIGHT = PointUtils.dpToPx(activity, resources.getInteger(R.integer.editimage_cropview_handle_point_height));
        int lineColor = resources.getColor(R.color.scan_color_blue_ff179cd8);
        int fillColor = resources.getColor(R.color.scan_color_white);

        shapePaintFill = new Paint();
        shapePaintFill.setStyle(Style.FILL);
        shapePaintFill.setColor(fillColor);
        shapePaintFill.setAntiAlias(true);
        shapePaintLine = new Paint();
        shapePaintLine.setStyle(Style.STROKE);
        shapePaintLine.setAntiAlias(true);
        shapePaintLine.setStrokeWidth(resources.getInteger(R.integer.editimage_cropview_stroke_width));
        shapePaintLine.setColor(lineColor);
        CIRCLE_RADIUS = PointUtils.dpToPx(activity, 6);
        int padding = (int) CIRCLE_RADIUS;
        setPadding(padding, padding, padding, padding);
    }

    @Override
    protected void drawMe(Canvas canvas) {
        super.drawMe(canvas);
        if (cornerPath != null) {
            canvas.drawPath(cornerPath, shapePaintFill);
            canvas.drawPath(cornerPath, shapePaintLine);
        }
        if (handlesPath != null) {
            for (Path path : handlesPath) {
                if (path != null) {
                    canvas.drawPath(path, shapePaintFill);
                    canvas.drawPath(path, shapePaintLine);
                }
            }
        }
    }

    @Override
    public void setPoints(PointF[] points) {
        PointF[] corner = new PointF[]{points[0], points[1], points[2], points[3]};

        cornerPath = new Path();
        //PointF rectOriginOffset = new PointF(CIRCLE_RADIUS * 0.5f, CIRCLE_RADIUS * 0.5f);
        //PointF rectOriginOffset = new PointF(10.0f, 10.0f);
        PointF rectOriginOffset = new PointF(0, 0);
        for (PointF p : corner) {
            if (validPoint(p)) {
                PointF origin = new PointF(p.x - rectOriginOffset.x / 2, p.y - rectOriginOffset.y / 2);
                cornerPath.addCircle(origin.x, origin.y, CIRCLE_RADIUS, Direction.CW);
            }
        }
        cornerPath.close();

        PointF[] handles = new PointF[]{points[4], points[5], points[6],
                points[7]};
        handlesPath = new Path[handles.length];
        int i = 0;
        for (PointF p : handles) {
            if (validPoint(p)) {
                RectF r = new RectF(-RECT_WIDTH, -RECT_HEIGHT, RECT_WIDTH,
                        RECT_HEIGHT);
                float angle = angleForHandle(i, points);

                Matrix matrix = new Matrix();
                matrix.preTranslate(p.x, p.y);
                matrix.preRotate((float) Math.toDegrees(angle));

                handlesPath[i] = new Path();
                handlesPath[i].addRect(r, Direction.CW);
                handlesPath[i].transform(matrix);
                handlesPath[i].close();
            } else {
                handlesPath[i] = null;
            }
            i++;
        }

        super.setPoints(points);
    }

    private boolean validPoint(PointF p) {
        return p.x >= 0 && p.y >= 0;
    }

    private float angleForHandle(int i, PointF[] points) {
        final int i0 = i;
        final int i1 = (i + 1) % 4;

        PointF p0 = new PointF(points[i0].x, points[i0].y);

        PointF p1 = new PointF(points[i1].x, points[i1].y);
        p1.x -= p0.x;
        p1.y -= p0.y;

        PointF line = PointUtils.deltaVector(new PointF(0, 0), p1);
        return (float) Math.atan2(line.y, line.x);
    }

}
