/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.views;

import android.app.Activity;
import android.content.Context;
import android.content.res.Resources;
import android.graphics.PointF;
import android.os.Build;
import android.util.AttributeSet;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.widget.FrameLayout;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.editimage.cropview.CropViewHelper;
import com.foxit.pdfscan.utils.PointUtils;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.luratech.android.appframework.BitmapWithMetadata;

public class CropView extends FrameLayout {
    /**
     * Minimum squared distance to the exact corner point to detect a drag
     * operation
     */
    private static float MIN_SQUARED_DISTANCE_DRAG_DETECT;
    /**
     * Minimum squared distance of a handle point to a corner point, otherwise
     * it is not displayed.
     */
    private static float MIN_SQUARED_DISTANCE_TO_SHOW_HANDLE;
    /**
     * Minimum squared distance between handles, where no further move operation
     * is allowed
     */
    private static int MIN_SQUARED_DISTANCE_BETWEEN_HANDLES;
    /**
     * Index of the currently selected corner point.
     */
    private int selectedPoint;

    /**
     * View responsible for drawing corner- and handle points
     */
    private CropableDetectionResultView cropableDetectionResultView;
    /**
     * The magnifier loupe
     */
    private MagnifierView loupe;

    /**
     * Currently visible cropped area points
     */
    private PointF[] points;

    private CropViewHelper helper;

    /**
     * Metrics of the bitmap to crop
     */
//	private int bitmapWidth, bitmapHeight;
    public int bitmapWidth, bitmapHeight;
    private Activity activity;

    public static PointF[] mFullPoints4 = new PointF[]{
            new PointF(0, 0),
            new PointF(1, 0),
            new PointF(1, 1),
            new PointF(0, 1),
    };

    public static PointF[] mFullPoints8 = new PointF[]{
            new PointF(0, 0),
            new PointF(1, 0),
            new PointF(1, 1),
            new PointF(0, 1),
            new PointF(0.5f, 0),
            new PointF(1, 0.5f),
            new PointF(0.5f, 1),
            new PointF(0, 0.5f)
    };

    public CropView(Context context) {
        super(context);
        initComponent(context);
    }

    public CropView(Context context, AttributeSet attrs) {
        super(context, attrs);
        initComponent(context);
    }

    private void initComponent(Context context) {
        activity = (Activity) getContext();
        Resources resources = getResources();
        MIN_SQUARED_DISTANCE_DRAG_DETECT = resources.getInteger(R.integer.editimage_cropview_min_squared_distance_drag_detect);
        MIN_SQUARED_DISTANCE_DRAG_DETECT = resources.getInteger(R.integer.editimage_cropview_min_squared_distance_drag_detect);
        MIN_SQUARED_DISTANCE_TO_SHOW_HANDLE = resources.getInteger(R.integer.editimage_cropview_min_squared_distance_to_show_handle);
        MIN_SQUARED_DISTANCE_BETWEEN_HANDLES = resources.getInteger(R.integer.editimage_cropview_min_squared_distance_between_handles);

        LayoutInflater inflater = LayoutInflater.from(context);
        View view = inflater.inflate(R.layout.fx_photo2pdf_fragment_editimage_cropview,
                this, true);

        cropableDetectionResultView = (CropableDetectionResultView) view.findViewById(R.id.cropableDetectionResultView);
        cropableDetectionResultView.setLineColor(resources.getColor(com.foxit.uiextensions.R.color.ux_color_comparison_filename));
        // Initially select ALL:
        points = new PointF[0];
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            this.setForceDarkAllowed(false);
        }
        loupe = (MagnifierView) view.findViewById(R.id.loupe);
        loupe.setVisibility(View.INVISIBLE);
    }

    public void setBitmap(BitmapWithMetadata bitmap) {
        bitmapWidth = bitmap.getBitmap().getWidth();
        bitmapHeight = bitmap.getBitmap().getHeight();
//		bitmapWidth = 1000;
//		bitmapHeight = 1000;

        cropableDetectionResultView.setDimensions(bitmapWidth, bitmapHeight);
        cropableDetectionResultView.setBitmap(bitmap);
        helper = new CropViewHelper(bitmapWidth, bitmapHeight);
        loupe.setBitmap(bitmap);

        this.selectedPoint = -1;
    }

    /**
     * Set Initial Points of the rectangular area to crop.
     *
     * @param points points in relative coordinates [0..1].
     */
    public void setPoints(PointF[] points) {
        // Select all when no detection points set.
        if (points.length != 4)
            points = helper.createPointsAtImageCorners();

        points = helper.transformPointsToViewSpace(points);

        this.points = extendArrayForHandlePoints(points);
        updateHandlePoints();

        drawCornersAndHandles();
    }

    public boolean isFull() {
        PointF[] curPts = this.getCroppedPoints();
        if (curPts == null || curPts.length != mFullPoints8.length) {
            return true;
        }

        for (int i = 0; i < mFullPoints8.length; i++) {
            if (!curPts[i].equals(mFullPoints8[i]))
                return false;
        }

        return true;
    }

    @Override
    public boolean onTouchEvent(MotionEvent event) {
        //if (this.getCroppedPoints() != CropViewFragment.croppedPoints) {
        switch (event.getActionMasked()) {
            case MotionEvent.ACTION_DOWN:
                dragStart(event);
                break;
            case MotionEvent.ACTION_UP:
                dragStop();
                break;
            case MotionEvent.ACTION_MOVE:
                dragMove(event);
                break;
            default:
                break;
        }
        return true;
    }

    /**
     * Dragging starts, determine corner point and show loupe
     */
    private void dragStart(MotionEvent event) {
        if (loupe.getVisibility() == View.INVISIBLE
                && event.getPointerCount() == 1) {
            selectedPoint = indexOfPointAt(currentTouch(event));
            if (selectedPoint != -1) {
                showLoupeAtPoint(points[selectedPoint]);
            }
        }
    }

    /**
     * Dragging around, move corner- or handle point
     */
    private void dragMove(MotionEvent event) {
        movePoint(currentTouch(event));
    }

    /**
     * Drag stop, hide loupe
     */
    private void dragStop() {
        loupe.setVisibility(View.INVISIBLE);
        selectedPoint = -1;
    }

    /**
     * Map view coordinate of the drag operation to image coordinate.
     */
    private PointF currentTouch(MotionEvent event) {
        return helper.viewCoordinateToImageCoordinate(
                cropableDetectionResultView.getDrawMatrix(),
                new PointF(event.getX(), event.getY()));
    }

    /**
     * Depending on the dragged item, it is moved.
     */
    private void movePoint(PointF p) {
        if (isInsideView(p) && selectedPoint != -1) {
            if (isCornerPointSelected())
                moveCornerPointTo(p);
            else if (isHandlePointSelected())
                moveHandlePointTo(p);

            updateHandlePoints();
            drawCornersAndHandles();
            showLoupeAtPoint(points[selectedPoint]);
        }
    }

    /**
     * @return is a corner point selected?
     */
    private boolean isCornerPointSelected() {
        return selectedPoint >= 0 && selectedPoint < 4;
    }

    /**
     * @return is a handle point selected?
     */
    private boolean isHandlePointSelected() {
        return selectedPoint >= 4 && selectedPoint < 8;
    }

    private void drawCornersAndHandles() {
        cropableDetectionResultView.setPoints(points);
    }

    /**
     * @return Create a copy of the points array and make room for handle
     * points.
     */
    private PointF[] extendArrayForHandlePoints(PointF[] points) {
        PointF[] result = new PointF[8];
        System.arraycopy(points, 0, result, 0, points.length);
        for (int i = 0; i < 4; ++i)
            result[4 + i] = new PointF(-1, -1);
        return result;
    }

    /**
     * Update handle points depending on the corner points
     */
    private void updateHandlePoints() {
        PointF[] handles = helper.computeCenterPoints(activity, points,
                MIN_SQUARED_DISTANCE_TO_SHOW_HANDLE);

        for (int i = 0; i < handles.length; i++) {
            points[4 + i] = handles[i];
        }
    }

    /**
     * Translate absolute coordinates [0..IMAGE_SIZE] to relative coordinates
     * [0..1].
     */
    public PointF[] getCroppedPoints() {
        return helper.transformPointsToImageSpace(points);
    }

    /**
     * @return corner points of visible cropped area
     */
    private PointF[] getCornerPoints() {
        return new PointF[]{points[0], points[1], points[2], points[3]};
    }

    /**
     * Check if the points still form a convex polygon, then replace selected
     * point.
     *
     * @param point point to replace
     */
    private void moveCornerPointTo(PointF point) {
        if (helper.rectConvexWhenReplacing(selectedPoint, point,
                getCornerPoints())) {
            points[selectedPoint] = point;
        }
    }

    /**
     * <PRE>
     * Updates the points of the handle.
     * The handle is moved and new points c1 and c2 are computed.
     * Indices used in function:
     * c1 -------- Handle -------- c2
     * |							|
     * |							|
     * |							|
     * |							|
     * |							|
     * |							|
     * c0							c3
     * </PRE>
     */
    private void moveHandlePointTo(PointF p) {
        // Compute indices used and get points
        final int corner1Index = selectedPoint - 4;
        final int corner2Index = (selectedPoint - 3) % 4;
        final int corner0Index = (corner1Index + 3) % 4;
        final int corner4Index = (corner2Index + 1) % 4;

        PointF corner0 = points[corner0Index];
        PointF corner1 = points[corner1Index];
        PointF corner2 = points[corner2Index];
        PointF corner3 = points[corner4Index];

        // Compute new position of the handle
        PointF line = PointUtils.deltaVector(corner1, corner2);

        PointF normal = PointUtils.absolute(PointUtils.computeNormal(line));

        PointF oldHandlePoint = points[selectedPoint];

        PointF d = PointUtils.deltaVector(oldHandlePoint, p);

        PointF pointOffset = new PointF(normal.x * d.x, normal.y * d.y);

        PointF newHandlePoint = new PointF(oldHandlePoint.x + pointOffset.x,
                oldHandlePoint.y + pointOffset.y);

        PointF movedLineEnd = new PointF(newHandlePoint.x + line.x,
                newHandlePoint.y + line.y);

        // Compute new points for corner1 and 2
        PointF newCorner1 = helper.intersectionOfLineFrom(newHandlePoint,
                movedLineEnd, corner0, corner1);

        PointF newCorner2 = helper.intersectionOfLineFrom(newHandlePoint,
                movedLineEnd, corner3, corner2);

        // Validate points and update when valid
        if (newCorner1 != null && newCorner2 != null) {
            boolean newPointsInView = isInsideView(newCorner1)
                    && isInsideView(newCorner2);
            boolean flipped = normalsFlipped(corner1, newCorner1, corner0)
                    || normalsFlipped(corner2, newCorner2, corner3);

            if (PointUtils.squaredDistance(activity, newCorner1, corner0) > MIN_SQUARED_DISTANCE_BETWEEN_HANDLES
                    && PointUtils.squaredDistance(activity, newCorner2, corner3) > MIN_SQUARED_DISTANCE_BETWEEN_HANDLES
                    && !flipped && newPointsInView) {
                points[corner1Index] = newCorner1;
                points[corner2Index] = newCorner2;
            }
        }
    }

    /**
     * Make magnifier loupe visible and display area around the given point
     *
     * @param point current point of interest
     */
    private void showLoupeAtPoint(PointF point) {
        loupe.setVisibility(View.VISIBLE);
        loupe.setTouchPoint(point);
    }

    /**
     * Check, if the point is within the bitmap area.<BR>
     * Note: Beware of rounding problems here, therefore we cast to int
     */
    private boolean isInsideView(PointF point) {
        return (int) point.x >= 0 && (int) point.y >= 0
                && (int) point.x <= bitmapWidth + 0.001f
                && (int) point.y <= bitmapHeight + 0.001f;
    }

    /**
     * Determine the selected corner point
     *
     * @param point point of the drag operation
     * @return index of the corner points array
     */
    private int indexOfPointAt(PointF point) {
        for (int i = 0; i < points.length; ++i) {
            PointF pointToCheck = points[i];
            if (PointUtils.squaredDistance(activity, point, pointToCheck) < MIN_SQUARED_DISTANCE_DRAG_DETECT)
                return i;
        }
        return -1;
    }

    private boolean normalsFlipped(PointF oldCorner, PointF newCorner,
                                   PointF reference) {
        PointF oldLineVector = PointUtils.normalize(PointUtils.deltaVector(
                oldCorner, reference));
        PointF newLineVector = PointUtils.normalize(PointUtils.deltaVector(
                newCorner, reference));

        boolean flipped = Math.abs(oldLineVector.x - newLineVector.x) > 0.1f
                || Math.abs(oldLineVector.y - newLineVector.y) > 0.1f;

        return flipped;
    }

}
