/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.utils;

import android.app.Activity;
import android.graphics.PointF;
import android.util.DisplayMetrics;

public class PointUtils {

    /**
     * @param activity
     * @return the squared distance between two points.
     */
    public static float squaredDistance(Activity activity, PointF p, PointF q) {
        float dx = PointUtils.pxToDp(activity, p.x)
                - PointUtils.pxToDp(activity, q.x);
        float dy = PointUtils.pxToDp(activity, p.y)
                - PointUtils.pxToDp(activity, q.y);
        return (dx * dx) + (dy * dy);
    }

    /**
     * @return the difference vector between two points.
     */
    public static PointF deltaVector(PointF p1, PointF p2) {
        return new PointF(p2.x - p1.x, p2.y - p1.y);
    }

    /**
     * @return the squared length of a vector.
     */
    public static float length2(PointF v) {
        return (v.x * v.x) + (v.y * v.y);
    }

    /**
     * @return the length of a vector.
     */
    public static float length(PointF v) {
        return (float) Math.sqrt(length2(v));
    }

    /**
     * @return the mid points for a line connecting two points.
     */
    public static PointF midPoint(PointF p1, PointF p2) {
        PointF delta = deltaVector(p1, p2);
        delta.x *= 0.5f;
        delta.y *= 0.5f;

        return new PointF(p1.x + delta.x, p1.y + delta.y);
    }

    public static PointF normalize(PointF p) {
        float l = length(p);
        return new PointF(p.x / l, p.y / l);
    }

    public static PointF computeNormal(PointF line) {
        return normalize(new PointF(-line.y, line.x));
    }

    public static PointF absolute(PointF line) {
        return new PointF(line.x >= 0 ? line.x : -line.x, line.y >= 0 ? line.y
                : -line.y);
    }

    public static float dpToPx(Activity activity, float dp) {
        DisplayMetrics metrics = new DisplayMetrics();
        activity.getWindowManager().getDefaultDisplay().getMetrics(metrics);
        float logicalDensity = metrics.density;

        return (float) Math.ceil(dp * logicalDensity);
    }

    public static float pxToDp(Activity activity, float px) {
        DisplayMetrics metrics = new DisplayMetrics();
        activity.getWindowManager().getDefaultDisplay().getMetrics(metrics);
        float logicalDensity = metrics.density;

        return (float) Math.ceil(px / logicalDensity);
    }

    public static float getDisplayAspectRatio(Activity activity) {
        DisplayMetrics metrics = new DisplayMetrics();
        activity.getWindowManager().getDefaultDisplay().getMetrics(metrics);
        return metrics.widthPixels / (float) metrics.heightPixels;
    }
}
