/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.utils;


import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.os.Build;
import android.util.DisplayMetrics;
import android.util.Log;
import android.view.Display;
import android.view.WindowManager;

import com.foxit.uiextensions.utils.AppDevice;

import java.lang.reflect.Method;

public class DisplayUtils {

    private Context mContext;
    private DisplayMetrics mMetrics;
    private int mWidthPixels;
    private int mHeightPixels;
    private boolean mPadDevice;

    private static DisplayUtils mAppDisplay = null;

    public static DisplayUtils getInstance(Context context) {
        if (mAppDisplay == null) {
            mAppDisplay = new DisplayUtils(context);
        }
        return mAppDisplay;
    }

    /**
     * Device DPI
     * 120 === LDPI
     * 160 === MDPI
     * 213 === TVDPI
     * 240 === HDPI
     * 320 === XHDPI
     * 480 === XXHDPI
     */
    public DisplayUtils(Context context) {
        mContext = context;
        mMetrics = context.getResources().getDisplayMetrics();
        Log.d("AppDisplay", "DPI:" + mMetrics.densityDpi);
        Display display = ((WindowManager) context.getSystemService(Context.WINDOW_SERVICE)).getDefaultDisplay();
        if (Build.VERSION.SDK_INT < 13) {
            mWidthPixels = mMetrics.widthPixels;
            mHeightPixels = mMetrics.heightPixels;
        } else if (Build.VERSION.SDK_INT == 13) {
            try {
                Method methodWidth = Display.class.getMethod("getRealWidth");
                Method methodHeight = Display.class.getMethod("getRealHeight");
                mWidthPixels = (Integer) methodWidth.invoke(display);
                mHeightPixels = (Integer) methodHeight.invoke(display);
            } catch (Exception e) {
                mWidthPixels = mMetrics.widthPixels;
                mHeightPixels = mMetrics.heightPixels;
            }
        } else if (Build.VERSION.SDK_INT > 13 && Build.VERSION.SDK_INT < 17) {
            try {
                Method methodWidth = Display.class.getMethod("getRawWidth");
                Method methodHeight = Display.class.getMethod("getRawHeight");
                mWidthPixels = (Integer) methodWidth.invoke(display);
                mHeightPixels = (Integer) methodHeight.invoke(display);
            } catch (Exception e) {
                mWidthPixels = mMetrics.widthPixels;
                mHeightPixels = mMetrics.heightPixels;
            }
        } else if (Build.VERSION.SDK_INT >= 17) {
            display.getRealMetrics(mMetrics);

            mWidthPixels = mMetrics.widthPixels;
            mHeightPixels = mMetrics.heightPixels;
        }

        float diagonalPixels = (float) Math.sqrt(Math.pow(getRawScreenWidth(), 2) + Math.pow(getRawScreenHeight(), 2));
        float screenSize = diagonalPixels / mMetrics.densityDpi;
        if (screenSize < 7) {
            mPadDevice = false;
        } else if (screenSize >= 7 && screenSize < 8 && mMetrics.densityDpi < 160) {
            mPadDevice = false;
        } else {
            mPadDevice = true;
        }
    }

    public int dp2px(float value) {
        return (int) (value * mMetrics.density + 0.5);
    }

    public float px2dp(float pxValue) {
        return pxValue / mMetrics.density;
    }

    public int getScreenWidth() {
        mMetrics = mContext.getResources().getDisplayMetrics();
        return mMetrics.widthPixels;
    }

    public int getScreenHeight() {
        mMetrics = mContext.getResources().getDisplayMetrics();
        return mMetrics.heightPixels;
    }

    public int getActivityWidth(Activity act) {
        if (AppUtils.isChromeOs(act)) {
            return act.getResources().getConfiguration().screenWidthDp;
        } else {
            return getScreenWidth();
        }
    }

    public int getActivityHeight(Activity act) {
        if (AppDevice.isChromeOs(act)) {
            return act.getResources().getConfiguration().screenHeightDp;
        } else {
            return getScreenHeight();
        }
    }

    public boolean isLandscape() {
        if (getScreenWidth() > getScreenHeight()) {
            return true;
        } else {
            return false;
        }
    }

    public int getRawScreenWidth() {
        if (isLandscape()) {
            return Math.max(mWidthPixels, mHeightPixels);
        } else {
            return Math.min(mWidthPixels, mHeightPixels);
        }
    }

    public int getRawScreenHeight() {
        if (isLandscape()) {
            return Math.min(mWidthPixels, mHeightPixels);
        } else {
            return Math.max(mWidthPixels, mHeightPixels);
        }
    }

    public boolean isPad() {
        return mPadDevice;
    }

}
