/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.fragment;


import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.graphics.Bitmap;
import android.os.Build;
import android.os.Bundle;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AdapterView;
import android.widget.BaseAdapter;
import android.widget.ImageView;
import android.widget.ListView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import com.foxit.pdfscan.ActRequestCode;
import com.foxit.pdfscan.R;
import com.foxit.pdfscan.ScannerSession;
import com.foxit.pdfscan.activity.EditPDFActivity;
import com.foxit.pdfscan.activity.ScannerCameraActivity;
import com.foxit.pdfscan.utils.ResourceUtils;
import com.foxit.pdfscan.utils.SessionIntentUtils;
import com.foxit.pdfscan.utils.AppUtils;
import com.foxit.pdfscan.views.ProgressBarView;
import com.foxitsoftware.mobile.scanning.DocumentEditingSession;
import com.foxitsoftware.mobile.scanning.DocumentSession;
import com.foxitsoftware.mobile.scanning.DocumentSessionManager;
import com.foxitsoftware.mobile.scanning.DocumentSessionManagerFactory;
import com.luratech.android.appframework.Document;
import com.luratech.android.appframework.DocumentManager;
import com.luratech.android.appframework.DocumentManagerListener;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.fragment.app.DialogFragment;

public class ScannerListFragment extends DialogFragment implements DocumentManagerListener {

    private ImageView mShowNavBtn;
    private TextView mCreateDocBtn;
    private ListView mDocListView;
    private TextView mTvPromptNoDoc;
    private ScannerListAdapter mDocListAdapter;
    private RelativeLayout mDocListBackground;
    private ProgressBarView mDocListProgressView;

    private List<ScannerListAdapter.ScannerFileInfo> mScannerFileInfos = new ArrayList<>();
    private Context mContext;
    private int mLastSelectedPosition = -1;

    public static ScannerListFragment newInstance() {
        return new ScannerListFragment();
    }

    @Override
    public void onCreate(@Nullable Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        int theme;
        if (Build.VERSION.SDK_INT >= 21) {
            theme = android.R.style.Theme_Holo_Light_NoActionBar_Fullscreen;
        } else {
            theme = android.R.style.Theme_Light_NoTitleBar_Fullscreen;
        }
        setStyle(STYLE_NO_TITLE, theme);

        registerBoradcastReceiver();
        mContext = getActivity().getApplicationContext();
    }

    private void registerBoradcastReceiver() {
        IntentFilter filter = new IntentFilter();
        filter.addAction(ScannerSession.INTENT_ACTION_UPDATE_LIST);
        getActivity().registerReceiver(mBroadcastReceiver, filter);
    }

    private BroadcastReceiver mBroadcastReceiver = new BroadcastReceiver() {

        @Override
        public void onReceive(Context context, Intent intent) {
            String action = intent.getAction();
            if (action != null && action.equals(ScannerSession.INTENT_ACTION_UPDATE_LIST)) {
                initListItem();
                initScannerView();
                mDocListAdapter.notifyDataSetChanged();
            }
        }
    };

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container, @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.fx_photo2pdf_creatpage, container, false);

        initView(view);
        initListItem();
        initScannerView();
        DocumentManager.get().setDocumentManagerListener(this);
        return view;
    }

    public void initView(View view) {
        view.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                if (mLastSelectedPosition != -1) {
                    ScannerListAdapter.ScannerFileInfo scannerFileInfo = mScannerFileInfos.get(mLastSelectedPosition);
                    scannerFileInfo.isSelected = false;
                    mDocListAdapter.notifyDataSetChanged();
                    mLastSelectedPosition = -1;
                }
            }
        });

        mShowNavBtn = (ImageView) view.findViewById(R.id.photo2pdf_show_navigation);
        mShowNavBtn.setContentDescription(ResourceUtils.getString(mContext, R.string.atb_hm_toolbar));
        mShowNavBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                ScannerListFragment.this.dismiss();
            }
        });

        mCreateDocBtn = view.findViewById(R.id.photo2pdf_create_document);
        mCreateDocBtn.setContentDescription(ResourceUtils.getString(mContext, R.string.hm_photo2pdf_Scan));
        mCreateDocBtn.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                onCameraBtnClicked("");
            }
        });

        mDocListView = (ListView) view.findViewById(R.id.photo2pdf_document_list);
        mDocListView.setVisibility(View.GONE);
        mDocListAdapter = new ScannerListAdapter(mContext, mScannerFileInfos, new IScannerItemClick() {
            @Override
            public void onClick(int type, int position, View view) {
                if (type == ScannerListAdapter.SHOW_MORE_MENU) {
                    if (mLastSelectedPosition != -1) {
                        ScannerListAdapter.ScannerFileInfo scannerFileInfo = mScannerFileInfos.get(mLastSelectedPosition);
                        scannerFileInfo.isSelected = false;
                    }
                    ScannerListAdapter.ScannerFileInfo scannerFileInfo = mScannerFileInfos.get(position);
                    scannerFileInfo.isSelected = true;

                    mLastSelectedPosition = position;
                    mDocListAdapter.notifyDataSetChanged();
                } else if (type == ScannerListAdapter.DELETE_ITEM) {
                    mLastSelectedPosition = -1;
                    String name = mScannerFileInfos.get(position).name;
                    Document doc = DocumentManager.get().findDocument(name);
                    if (doc != null) {
                        DocumentManager.get().deleteDocument(doc);
                        initListItem();
                        initScannerView();
                        mDocListAdapter.notifyDataSetChanged();
                    }
                }
            }
        });
        mDocListView.setAdapter(mDocListAdapter);

        mDocListView.setOnItemClickListener(new AdapterView.OnItemClickListener() {
            @Override
            public void onItemClick(AdapterView<?> parent, View view, final int position, long id) {
                if (AppUtils.isFastDoubleClick())
                    return;

                if (mLastSelectedPosition != -1) {
                    ScannerListAdapter.ScannerFileInfo scannerFileInfo = mScannerFileInfos.get(mLastSelectedPosition);
                    scannerFileInfo.isSelected = false;
                    mDocListAdapter.notifyDataSetChanged();
                    mLastSelectedPosition = -1;
                } else {
                    mDocListProgressView.start(getActivity(), true);
                    mDocListBackground.setVisibility(View.VISIBLE);

                    final ScannerListAdapter.ScannerFileInfo fileInfo = (ScannerListAdapter.ScannerFileInfo) parent.getItemAtPosition(position);
                    final int pageCount = DocumentManager.get().findDocument(fileInfo.name).getPageCount();

                    Document doc = DocumentManager.get().findDocument(fileInfo.name);
                    DocumentSessionManagerFactory.getInstance().getViewingSessionForDocument(doc, new DocumentSessionManager.ProgressListener() {
                        @Override
                        public void onStartedAction(DocumentSession documentSession, String s) {
                        }

                        @Override
                        public void onUpdateProgress(DocumentSession documentSession, int done, int total) {
                            mDocListProgressView.progress(mContext.getApplicationContext().getString(
                                    R.string.photo2pdf_loading, done, total));
                        }

                        @Override
                        public void onFinishedAction(DocumentSession documentSession, Document document) {
                            if (documentSession.getPageCount() > 0) {
                                mDocListProgressView.progress(
                                        mContext.getApplicationContext().getString(
                                                R.string.photo2pdf_loading, documentSession.getPageCount(), pageCount));
                            }
                            mDocListProgressView.end();
                            mDocListBackground.setVisibility(View.INVISIBLE);
                            Intent intent = new Intent(mContext, EditPDFActivity.class);
                            DocumentEditingSession editingSession = DocumentSessionManagerFactory.getInstance().getEditingSession(documentSession);
                            SessionIntentUtils.saveSessionId(intent, editingSession);
                            intent.putExtra("FILENAME", fileInfo.name);
                            startActivity(intent);

                            DocumentSessionManagerFactory.getInstance().cancelLoading(document);
                        }

                        @Override
                        public void onCanceledAction(DocumentSession documentSession) {
                        }
                    });
                }
            }
        });

        mDocListBackground = (RelativeLayout) view.findViewById(R.id.photo2pdf_listview_progress_background);
        mDocListBackground.setVisibility(View.INVISIBLE);

        mDocListProgressView = (ProgressBarView) view.findViewById(R.id.photo2pdf_listview_progressView);

        mTvPromptNoDoc = view.findViewById(R.id.photo2pdf_prompt_no_document);
        mTvPromptNoDoc.setVisibility(View.GONE);
    }

    private void onCameraBtnClicked(String docPath) {
        if (mLastSelectedPosition != -1) {
            ScannerListAdapter.ScannerFileInfo scannerFileInfo = mScannerFileInfos.get(mLastSelectedPosition);
            scannerFileInfo.isSelected = false;
            mDocListAdapter.notifyDataSetChanged();
            mLastSelectedPosition = -1;
        }

        Intent intent = new Intent(mContext, ScannerCameraActivity.class);
        final DocumentEditingSession createDocSession = DocumentSessionManagerFactory.getInstance().createSession();
        SessionIntentUtils.saveSessionId(intent, createDocSession);
        if (!AppUtils.isEmpty(docPath)) {
            intent.putExtra(ScannerSession.EXTRA_FILE_PATH, docPath);
        }
        startActivityForResult(intent, ActRequestCode.REQ_SCAN_CAMERA);
    }

    private void deleteTempDocs() {
        List<Document> docs = DocumentManager.get().documents();
        for (int i = docs.size() - 1; i >= 0; i--) {
            Document doc = docs.get(i);
            if (AppUtils.isEqual(doc.getDocumentName(), ScannerSession.TEMP_SAVE_IMAGE_NAME)
                    || AppUtils.isEqual(doc.getDocumentName(), ScannerSession.TEMP_SAVE_PDF_NAME)
                    || doc.getDocumentName().startsWith(ScannerSession.TEMP_RENAME_NAME)
            ) {
                DocumentManager.get().deleteDocument(doc);
            }
        }
    }

    @Override
    public void onDocumentAdded(Document document) {
    }

    @Override
    public void onDocumentRemoved(Document document) {
    }

    @Override
    public void onDocumentUpdated(Document document) {
    }

    @Override
    public void onThumbnailGenerated(Document document) {
        updateDocumentList();
    }

    private void updateDocumentList() {
        initListItem();
        initScannerView();
        mDocListAdapter.notifyDataSetChanged();
    }

    private static class ScannerListAdapter extends BaseAdapter {
        private static final int SHOW_MORE_MENU = 0;
        private static final int DELETE_ITEM = 1;

        private Context mContext;
        private List<ScannerFileInfo> mFileInfos = new ArrayList<>();
        private IScannerItemClick mItemClick;

        private ScannerListAdapter(Context context, List<ScannerFileInfo> fileInfos, IScannerItemClick itemClick) {
            this.mContext = context;
            this.mFileInfos = fileInfos;
            this.mItemClick = itemClick;
        }

        @Override
        public int getCount() {
            return mFileInfos.size();
        }

        @Override
        public Object getItem(int position) {
            return mFileInfos.get(position);
        }

        @Override
        public long getItemId(int position) {
            return position;
        }

        @Override
        public View getView(final int position, View convertView, final ViewGroup parent) {
            final ViewHolder holder;
            if (convertView == null) {
                convertView = View.inflate(mContext, R.layout.fx_photo2pdf_creatpage_listview_item, null);
                holder = new ViewHolder(convertView, position);
                convertView.setTag(holder);
            } else {
                holder = (ViewHolder) convertView.getTag();
            }
            final ScannerFileInfo fileInfo = mFileInfos.get(position);
            int delVisibility = fileInfo.isSelected ? View.VISIBLE : View.INVISIBLE;
            holder.delLayout.setVisibility(delVisibility);
            int moreVisibility = fileInfo.isSelected ? View.INVISIBLE : View.VISIBLE;
            holder.itemMore.setVisibility(moreVisibility);
            holder.itemImage.setImageBitmap(fileInfo.thumbBitmap);
            holder.itemName.setText(fileInfo.name);
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy:MM:dd HH:mm:ss");
            Date curDate = fileInfo.createDate;
            String str = formatter.format(curDate);
            holder.itemTime.setText(str);

            holder.itemMore.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (mItemClick != null)
                        mItemClick.onClick(SHOW_MORE_MENU, position, v);
                }
            });

            holder.delLayout.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    if (mItemClick != null)
                        mItemClick.onClick(DELETE_ITEM, position, v);
                }
            });
            return convertView;
        }

        private static class ViewHolder {
            private View delLayout;
            ImageView itemImage;
            TextView itemName;
            TextView itemTime;
            ImageView itemMore;
            int postion;

            private ViewHolder(View view, int postion) {
                delLayout = view.findViewById(R.id.photo2pdf_fb_item_delete_layout);
                itemImage = (ImageView) view.findViewById(R.id.photo2pdf_createpage_listview_imageview);
                itemName = (TextView) view.findViewById(R.id.photo2pdf_createpage_listview_doc_name);
                itemTime = (TextView) view.findViewById(R.id.photo2pdf_createpage_listview_doc_createtime);
                itemMore = (ImageView) view.findViewById(R.id.photo2pdf_createpage_more_button);
                this.postion = postion;
            }
        }

        private static class ScannerFileInfo {
            public String name;
            public Date createDate;
            public Bitmap thumbBitmap;
            public boolean isSelected;
        }

    }

    private void initListItem() {
        mLastSelectedPosition = -1;
        mScannerFileInfos.clear();

        List<Document> documents = DocumentManager.get().documents();
        for (int i = 0; i < documents.size(); i++) {
            Document doc = documents.get(i);
            String name = doc.getDocumentName();
            if (AppUtils.isEqual(name, ScannerSession.TEMP_SAVE_IMAGE_NAME) ||
                    AppUtils.isEqual(name, ScannerSession.TEMP_SAVE_PDF_NAME) ||
                    name.startsWith(ScannerSession.TEMP_RENAME_NAME)
            ) {
                continue;
            }

            Date date = doc.getDate();
            boolean inserted = false;
            for (int j = 0; j < mScannerFileInfos.size(); j++) {
                if (AppUtils.timeCompare(date, mScannerFileInfos.get(j).createDate) >= 0) {
                    ScannerListAdapter.ScannerFileInfo scannerFileInfo = new ScannerListAdapter.ScannerFileInfo();
                    scannerFileInfo.name = name;
                    scannerFileInfo.createDate = date;
                    scannerFileInfo.thumbBitmap = doc.getThumbnail();
                    mScannerFileInfos.add(j, scannerFileInfo);
                    inserted = true;
                    break;
                }
            }
            if (!inserted) {
                ScannerListAdapter.ScannerFileInfo scannerFileInfo = new ScannerListAdapter.ScannerFileInfo();
                scannerFileInfo.name = name;
                scannerFileInfo.createDate = date;
                scannerFileInfo.thumbBitmap = doc.getThumbnail();
                mScannerFileInfos.add(scannerFileInfo);
            }
        }
    }

    private void initScannerView() {
        if (mScannerFileInfos.size() > 0) {
            mDocListView.setVisibility(View.VISIBLE);
            mTvPromptNoDoc.setVisibility(View.GONE);
        } else {
            mDocListView.setVisibility(View.GONE);
            mTvPromptNoDoc.setVisibility(View.VISIBLE);
        }
    }

    @Override
    public void onDismiss(@NonNull DialogInterface dialog) {
        super.onDismiss(dialog);
        if (mOnDismissListener != null)
            mOnDismissListener.onDismiss(dialog);
    }

    @Override
    public void onDestroy() {
        deleteTempDocs();
        ScannerSession.clear();
        DocumentManager.get().removeDocumentManagerListener(this);
        getActivity().unregisterReceiver(mBroadcastReceiver);
        super.onDestroy();
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        super.onActivityResult(requestCode, resultCode, data);
        if (resultCode == Activity.RESULT_OK) {
            if (requestCode == ActRequestCode.REQ_SCAN_CAMERA || requestCode == ActRequestCode.REQ_EDIT_PDF) {
                updateDocumentList();
            }
        }
    }

    private interface IScannerItemClick {
        void onClick(int type, int position, View view);
    }

    private DialogInterface.OnDismissListener mOnDismissListener;

    public void setDismissListener(DialogInterface.OnDismissListener dismissListener) {
        this.mOnDismissListener = dismissListener;
    }

}
