/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.fragment;

import static com.foxit.pdfscan.NumberConstant.ZERO;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.animation.AnimatorSet;
import android.animation.ObjectAnimator;
import android.app.Activity;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.PointF;
import android.graphics.Rect;
import android.graphics.drawable.ColorDrawable;
import android.os.AsyncTask;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.util.Log;
import android.view.Gravity;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.view.View.OnClickListener;
import android.view.ViewGroup;
import android.view.animation.DecelerateInterpolator;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;
import android.widget.Toast;

import androidx.fragment.app.Fragment;
import androidx.recyclerview.widget.ItemTouchHelper;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

import com.foxit.pdfscan.ActRequestCode;
import com.foxit.pdfscan.IPDFScanManagerListener;
import com.foxit.pdfscan.PDFScanManager;
import com.foxit.pdfscan.R;
import com.foxit.pdfscan.ScannerSession;
import com.foxit.pdfscan.activity.EditImageActivity;
import com.foxit.pdfscan.activity.EditPDFActivity;
import com.foxit.pdfscan.activity.FormatActivity;
import com.foxit.pdfscan.format.PageFormat;
import com.foxit.pdfscan.takephotolist.TakePhotoListAdapter;
import com.foxit.pdfscan.takephotolist.TakePhotoListInfo;
import com.foxit.pdfscan.takephotolist.TakePhotoTouchHelper;
import com.foxit.pdfscan.utils.AppUtils;
import com.foxit.pdfscan.utils.DisplayUtils;
import com.foxit.pdfscan.utils.FileUtils;
import com.foxit.pdfscan.utils.IResultCallback;
import com.foxit.pdfscan.utils.ResourceUtils;
import com.foxit.pdfscan.utils.SessionIntentUtils;
import com.foxit.pdfscan.utils.ThreadManager;
import com.foxit.pdfscan.views.CameraSurfaceView;
import com.foxit.pdfscan.views.DetectionResultView;
import com.foxit.pdfscan.views.ProgressBarView;
import com.foxit.pdfscan.views.ResultMessageView;
import com.foxit.pdfscan.views.util.UilLocalAlbum;
import com.foxit.pdfscan.views.util.UilLocalImageHelper;
import com.foxit.pdfscan.views.util.UilPhoto;
import com.foxit.uiextensions.controls.dialog.UITextEditDialog;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppDisplay;
import com.foxit.uiextensions.utils.AppFileUtil;
import com.foxit.uiextensions.utils.AppResource;
import com.foxit.uiextensions.utils.AppSharedPreferences;
import com.foxit.uiextensions.utils.UIToast;
import com.foxitsoftware.mobile.scanning.DocumentEditingSession;
import com.foxitsoftware.mobile.scanning.DocumentSession;
import com.foxitsoftware.mobile.scanning.DocumentSessionManager;
import com.foxitsoftware.mobile.scanning.DocumentSessionManagerFactory;
import com.luratech.android.appframework.BitmapWithMetadata;
import com.luratech.android.appframework.CaptureService;
import com.luratech.android.appframework.CaptureServiceResultListener;
import com.luratech.android.appframework.CaptureServiceShutterListener;
import com.luratech.android.appframework.CaptureServiceUpdateListener;
import com.luratech.android.appframework.DetectionResult;
import com.luratech.android.appframework.Document;
import com.luratech.android.appframework.DocumentManager;
import com.luratech.android.appframework.DocumentManagerListener;
import com.luratech.android.appframework.ImageProcessor;
import com.luratech.android.appframework.ImageProcessorListener;
import com.nostra13.universalimageloader.cache.disc.impl.UnlimitedDiskCache;
import com.nostra13.universalimageloader.cache.disc.naming.Md5FileNameGenerator;
import com.nostra13.universalimageloader.core.ImageLoader;
import com.nostra13.universalimageloader.core.ImageLoaderConfiguration;
import com.nostra13.universalimageloader.core.assist.QueueProcessingType;
import com.nostra13.universalimageloader.core.download.BaseImageDownloader;

import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class ScannerCameraFragment extends Fragment implements
        CaptureServiceResultListener, CaptureServiceUpdateListener,
        CaptureServiceShutterListener, ImageProcessorListener, OnClickListener,
        DocumentSessionManager.ProgressListener, DocumentManagerListener {
    private int mRemakeIndex = -1;

    private static final String TAG = ScannerCameraFragment.class.getSimpleName();
    private static float THRESHOLD = 20;

    private Context mContext;

    private CaptureService mCaptureService;
    private FrameLayout mCameraSurfaceFrame;
    private CameraSurfaceView mCameraSurfaceView;
    private DetectionResultView mDetectionResultView;
    private ResultMessageView mHintMessageView;
    private ProgressBarView mProgressView;

    private TextView mPhotoCountTextView;

    private ImageView mTakePictureBtn;
    private ImageView mChoosePhotoBtn;
    private ImageView mTorchLightBtn;
    private ImageView mContinuousShotBtn;
    private ImageView mAutoDetectBtn;
    private ImageView mPageSizeBtn;
    private FrameLayout mPhotosButton;
    private View mPhotosBar;
//    private TextView mPhotosPromptTextView;
    private TextView mCancelBtn;
    private TextView mDoneBtn;
    private RecyclerView mScanPhotosRecyclerView;
    private ImageView mScanDeleteImg;

    private DocumentEditingSession mDocumentSession;
    private ArrayList<ImageProcessor> mImageProcessors = new ArrayList<>();
    private PageFormat mCurrentPageFormat;

    private String mDocName = "";
    private String mSaveDocName = "";
    private String mFileNameForAddPage;

    private int mButtonClicked = 0;
    private int orientation;

    private boolean mIsAutoDetect = false;
    private boolean mIsContinuousSnap = false;
    private boolean mIsFlashOpen = false;

    private boolean mIsResult = false;
    private int mUpdateIndex = -1;
    private final String PREF_NAME = "foxit_auto_detect_preference";
    private final String KEY_AUTO_DETECT = "auto_detect";

    private TakePhotoListAdapter mTakePhotoListAdapter;
    private ArrayList<TakePhotoListInfo> mTakePhotoListInfo = new ArrayList<>();

    private PDFScanManager mPDFScanManager;

    private boolean mIsUpdate = false;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setRetainInstance(true);
        setHasOptionsMenu(false);
        mContext = getActivity().getApplicationContext();
        Intent intent = getActivity().getIntent();
        try {
            mDocumentSession = (DocumentEditingSession) SessionIntentUtils.getSession(intent);
        } catch (Exception e) {
            mDocumentSession = DocumentSessionManagerFactory.getInstance().createSession();
            e.printStackTrace();
        }
        mPDFScanManager = PDFScanManager.instance();
        DocumentManager.get().setDocumentManagerListener(this);
        mFileNameForAddPage = intent.getStringExtra(ScannerSession.EXTRA_FILE_NAME);
        String tmp = intent.getStringExtra(ScannerSession.EXTRA_FILE_PATH);
        mRemakeIndex = intent.getIntExtra(ScannerSession.EXTRA_REMAKE_INDEX, -1);
        if (!AppUtils.isEmpty(tmp))
            mDocName = tmp;
        if (!AppUtils.isEmpty(mFileNameForAddPage))
            mDocName = mFileNameForAddPage;
        if (AppUtils.isEmpty(mDocName) && mDocumentSession != null && mDocumentSession.getSourceDocument() != null)
            mDocName = mDocumentSession.getSourceDocument().getDocumentName();

        registerScreenBroadcastReceiver();

        AppSharedPreferences.getInstance(getActivity()).getBoolean(PREF_NAME, KEY_AUTO_DETECT, false);
        orientation = getActivity().getRequestedOrientation();
        getActivity().setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);
    }

    View view;

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup parent,
                             Bundle savedInstanceState) {
        view = inflater.inflate(R.layout.fx_photo2pdf_camearing, parent, false);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            view.setForceDarkAllowed(false);
        }
        mCameraSurfaceFrame = (FrameLayout) view.findViewById(R.id.scan_camera_surface_frame);
        mCameraSurfaceView = (CameraSurfaceView) view.findViewById(R.id.scan_camera_surface_view);
        mCameraSurfaceView.setShutterCallback(this);
        mCameraSurfaceView.setCaptureServiceResultListener(this);
        mCameraSurfaceView.setCaptureServiceUpdateListener(this);
        mCaptureService = mCameraSurfaceView.getCaptureService();
        mDetectionResultView = (DetectionResultView) view.findViewById(R.id.scan_auto_detection_result_view);
        mHintMessageView = (ResultMessageView) view.findViewById(R.id.scan_hint_message_view);
        mHintMessageView.setVisibility(View.INVISIBLE);

        mProgressView = (ProgressBarView) view.findViewById(R.id.scan_progress_bar_view);

        mTakePictureBtn = (ImageView) view.findViewById(R.id.scan_take_picture_button);
        mTakePictureBtn.setContentDescription(ResourceUtils.getString(mContext, R.string.hm_photo2pdf_Scan));
        mTakePictureBtn.setOnClickListener(this);

        mChoosePhotoBtn = view.findViewById(R.id.scan_choose_image_button);
        mChoosePhotoBtn.setContentDescription(ResourceUtils.getString(mContext, R.string.atb_scan_add_photos_button));
        mChoosePhotoBtn.setOnClickListener(this);

        mPhotoCountTextView = (TextView) view.findViewById(R.id.scan_doc_preview_button);

        mTorchLightBtn = (ImageView) view.findViewById(R.id.scan_torchLight_button);
        mTorchLightBtn.setContentDescription(ResourceUtils.getString(mContext, R.string.atb_scan_flash_button));
        mTorchLightBtn.setOnClickListener(this);
        mContinuousShotBtn = (ImageView) view.findViewById(R.id.scan_signal_snap_button);
        mContinuousShotBtn.setContentDescription(ResourceUtils.getString(mContext, R.string.atb_scan_continuous_shooting_button));
        mContinuousShotBtn.setOnClickListener(this);

        mAutoDetectBtn = (ImageView) view.findViewById(R.id.scan_auto_detect_button);
        mAutoDetectBtn.setSelected(false);
        mCaptureService.enableAutoTrigger(false);
        mAutoDetectBtn.setContentDescription(ResourceUtils.getString(mContext, R.string.atb_scan_auto_detection_button));
        mAutoDetectBtn.setOnClickListener(this);

        mPageSizeBtn = (ImageView) view.findViewById(R.id.scan_page_size_button);
        mPageSizeBtn.setContentDescription(ResourceUtils.getString(mContext, R.string.atb_scan_page_size_button));
        mPageSizeBtn.setOnClickListener(this);
        mPageSizeBtn.setColorFilter(mContext.getResources().getColor(R.color.scan_color_grey_ff878787));
        mPageSizeBtn.setEnabled(false);
//        mPhotosPromptTextView = view.findViewById(R.id.scan_photos_prompt_text_view);

        mPhotosButton = view.findViewById(R.id.scan_photos_relative_btn);
        mPhotosButton.setVisibility(View.GONE);
        mPhotosButton.findViewById(R.id.scan_photos_relative_btn_iv).setSelected(false); // /\
        mPhotosButton.setOnClickListener(this);

        mPhotosBar = view.findViewById(R.id.scan_photos_relative);
        mPhotosBar.getBackground().setAlpha((int) (255 * 0.8f));

        setItemHiddenWithType();
//        view.findViewById(R.id.scan_bottom_bar).getBackground().setAlpha((int) (255 * 0.8f));
        mCancelBtn = view.findViewById(R.id.scan_cancel_btn);
        mCancelBtn.setOnClickListener(this);

        mDoneBtn = view.findViewById(R.id.scan_done_btn);
        mDoneBtn.setOnClickListener(this);
        mDoneBtn.setAlpha(0.5F);

        if (mRemakeIndex >= 0) {
            // Remake
            mContinuousShotBtn.setColorFilter(mContext.getResources().getColor(R.color.scan_color_grey_ff878787));
            mContinuousShotBtn.setEnabled(false);
            mDoneBtn.setVisibility(View.INVISIBLE);
        } else {
            mContinuousShotBtn.setSelected(false);
        }

        mScanDeleteImg = (ImageView) view.findViewById(R.id.scan_delete_img);

        mScanPhotosRecyclerView = view.findViewById(R.id.scan_photos_recycler_view);
        mScanPhotosRecyclerView.setLayoutManager(new LinearLayoutManager(getActivity(), RecyclerView.HORIZONTAL, false));
        mTakePhotoListAdapter = new TakePhotoListAdapter(getActivity(), mTakePhotoListInfo);
        mScanPhotosRecyclerView.setHasFixedSize(true);
        mScanPhotosRecyclerView.setAdapter(mTakePhotoListAdapter);
        mTakePhotoListAdapter.setOnItemClickListener(new TakePhotoListAdapter.ItemClickListener() {

            @Override
            public void onItemClick(View v, int position, RecyclerView.ViewHolder viewHolder) {
                EditImageFragment.fromWhere = "ScannerCameraActivity";
                Intent intent = new Intent(getActivity(), EditImageActivity.class);
                SessionIntentUtils.saveSessionId(intent, mDocumentSession);
                intent.putExtra(EditImageFragment.EXTRA_PAGE, position);
                startActivityForResult(intent, 0);
//                getActivity().finish();
            }
        });

        TakePhotoTouchHelper.OnDragListener dragListener = new TakePhotoTouchHelper.OnDragListener() {
            @Override
            public void onFinishDrag(final int position, final boolean isDelete) {
                ThreadManager.getInstance().getMainThreadHandler().post(new Runnable() {
                    @Override
                    public void run() {
                        mScanDeleteImg.setVisibility(View.INVISIBLE);
                        if (isDelete) {
                            if (mTakePhotoListInfo.contains(mTakePhotoListInfo.get(position))) {
                                mTakePhotoListInfo.remove(position);
                                mTakePhotoListAdapter.notifyItemRemoved(position);
                                mDocumentSession.removePage(position);
                                if (mTakePhotoListInfo.size() > 0) {
                                    mDoneBtn.setAlpha(1F);
                                } else {
                                    mDoneBtn.setAlpha(0.5F);
                                }
                                photoQuantity();
                            }
                        }
                    }
                });

            }

            @Override
            public void onDeleteBin(boolean isShow, int location) {
                mScanDeleteImg.setVisibility(View.VISIBLE);
                mScanDeleteImg.setX(location - mScanDeleteImg.getWidth() / 2);
                if (isShow) {
                    mScanDeleteImg.setAlpha(1F);
                } else {
                    mScanDeleteImg.setAlpha(0.5F);
                }

            }
        };

        final ItemTouchHelper itemTouchHelper = new ItemTouchHelper(new TakePhotoTouchHelper(mTakePhotoListAdapter).setOnDragListener(dragListener));
        itemTouchHelper.attachToRecyclerView(mScanPhotosRecyclerView);
        photoQuantity();

        if (DisplayUtils.getInstance(mContext).isPad()) {
            ((LinearLayout.LayoutParams) mTorchLightBtn.getLayoutParams()).rightMargin = ResourceUtils.getDimensionPixelSize(mContext, R.dimen.scan_bottombar_button_space_pad);
            ((LinearLayout.LayoutParams) mContinuousShotBtn.getLayoutParams()).rightMargin = ResourceUtils.getDimensionPixelSize(mContext, R.dimen.scan_bottombar_button_space_pad);
            ((LinearLayout.LayoutParams) mAutoDetectBtn.getLayoutParams()).rightMargin = ResourceUtils.getDimensionPixelSize(mContext, R.dimen.scan_bottombar_button_space_pad);
        }

        if (getActivity() != null) {
            mIsContinuousSnap = ScannerSession.isContinuousSnap();
            mContinuousShotBtn.setSelected(mIsContinuousSnap);

            mIsAutoDetect = ScannerSession.isAutoDetect();
            if (mIsAutoDetect) {
                mPageSizeBtn.setEnabled(true);
                mPageSizeBtn.setColorFilter(mContext.getResources().getColor(R.color.scan_color_translucent));
                mAutoDetectBtn.setSelected(true);
                mCaptureService.enableAutoTrigger(true);
                mHintMessageView.setVisibility(View.VISIBLE);
                mDetectionResultView.setVisibility(View.VISIBLE);
            }

            PageFormat pageFormat = ScannerSession.getCurrentPageFormat();
            if (pageFormat != null)
                mCurrentPageFormat = pageFormat;


            if (!getActivity().getPackageManager().hasSystemFeature(PackageManager.FEATURE_CAMERA_FLASH)
                    && Build.MODEL.equals("Nexus 7")) {
                ThemeUtil.setTintList(mTorchLightBtn, ThemeUtil.getEnableIconColor(getActivity()));
            }
        }

        view.setFocusableInTouchMode(true);
        view.requestFocus();
        view.setOnKeyListener(new View.OnKeyListener() {
            @Override
            public boolean onKey(View v, int keyCode, KeyEvent event) {
                if (keyCode == KeyEvent.KEYCODE_BACK && event.getAction() == KeyEvent.ACTION_UP) {
                    onClick(mCancelBtn);
                    return true;
                }
                return false;
            }
        });
        return view;
    }

    private void setItemHiddenWithType() {
        Map<Integer, Boolean> map = mPDFScanManager.getItemHiddenWithType();
        for (Map.Entry<Integer, Boolean> entry : map.entrySet()) {
            int visibility = entry.getValue() ? View.GONE : View.VISIBLE;
            switch (entry.getKey()) {

                case PDFScanManager.FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_FLASH_LIGHT:
                    mTorchLightBtn.setVisibility(visibility);
                    break;
                case PDFScanManager.FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_SINGLE_SHOOTING:
                    mContinuousShotBtn.setVisibility(visibility);
                    break;
                case PDFScanManager.FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_PHOTOS:
                    mChoosePhotoBtn.setVisibility(visibility);
                    break;
                case PDFScanManager.FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_AUTO_DECTION:
                    mAutoDetectBtn.setVisibility(visibility);
                    break;
                case PDFScanManager.FS_SCAN_CAMERA_CONTROLLER_ITEM_TYPE_FORMAT_PAGESIZE:
                    mPageSizeBtn.setVisibility(visibility);
                    break;
            }
        }
    }

    private void photoQuantity() {
        if (mRemakeIndex >= 0) return;
        if (ZERO == mDocumentSession.getPageCount()) {
            mPhotoCountTextView.setVisibility(View.GONE);
//            mPhotosPromptTextView.setVisibility(View.VISIBLE);
//            mScanPhotosRecyclerView.setVisibility(View.GONE);
            mPhotosBar.setVisibility(View.GONE);
            mPhotosButton.setVisibility(View.GONE);
            mPhotosButton.findViewById(R.id.scan_photos_relative_btn_iv).setSelected(false); // /\
            mDoneBtn.setAlpha(0.5F);
        } else {
            mPhotoCountTextView.setVisibility(View.VISIBLE);
//            mPhotosPromptTextView.setVisibility(View.GONE);
//            mScanPhotosRecyclerView.setVisibility(View.VISIBLE);
//            mPhotosBar.setVisibility(View.VISIBLE);
            mPhotosButton.setVisibility(View.VISIBLE);
//            mPhotosButton.findViewById(R.id.scan_photos_relative_btn_iv).setSelected(true); // \/
            mPhotoCountTextView.setText("" + mDocumentSession.getPageCount());
            mDoneBtn.setAlpha(1F);
        }
    }

    @Override
    public void onResume() {
        super.onResume();
        getActivity().setRequestedOrientation(ActivityInfo.SCREEN_ORIENTATION_PORTRAIT);
        if (((ColorDrawable) view.getBackground()).getColor() != AppResource.getColor(mContext, R.color.scan_color_black)) {
            view.setBackgroundColor(AppResource.getColor(mContext, R.color.scan_color_black));
        }

        view.findViewById(R.id.scan_top_bar).setBackgroundColor(AppResource.getColor(mContext, R.color.scan_color_black));
        view.findViewById(R.id.scan_bottom_bar).setBackgroundColor(AppResource.getColor(mContext, R.color.scan_color_black));

        if (mIsResult || mIsUpdate) {
            new EditScanPhotoTask().execute();
            mIsResult = false;
        }
    }

    @Override
    public void onDestroyView() {
        super.onDestroyView();
        getActivity().setRequestedOrientation(orientation);
        if (mScreenReceiver != null) {
            getActivity().unregisterReceiver(mScreenReceiver);
        }
        mScreenReceiver = null;
        AppSharedPreferences.getInstance(getActivity()).setBoolean(PREF_NAME, KEY_AUTO_DETECT, mIsAutoDetect);
        if (mCaptureService != null)
            mCaptureService.close();

    }

    private void resetTakingPicture() {
        mTakePictureBtn.setEnabled(true);
        if (mCaptureService != null) {
            mCaptureService.resetTakingPicture();
        }
    }

    @Override
    public void onResultAvailable(CaptureService service, DetectionResult result) {
        if (mIsAutoDetect) {
            if (getActivity() == null)
                return;
            mHintMessageView.setDetectionResult(result);

            PointF[] points = null;
            if (result.isValid()) {
                mDetectionResultView.setDimensions(mCameraSurfaceFrame.getWidth(), mCameraSurfaceFrame.getHeight());
                // get relative coordinates: exchange X<->Y (rotated picture)
                PointF[] absolutePoints = result.getPoints(
                        mCameraSurfaceFrame.getHeight(),
                        mCameraSurfaceFrame.getWidth());
                // correct aspect ratio (camera surface X coordinate has been
                // adjusted earlier)
                points = mirrorAxis(absolutePoints);
            }
            mDetectionResultView.setPoints(points);
        }
    }

    private PointF[] mirrorAxis(PointF[] points) {
        int shiftedMax = mCameraSurfaceFrame.getWidth();
        for (int i = 0; i < points.length; ++i) {
            points[i] = new PointF(shiftedMax - points[i].y, points[i].x);
        }
        return points;
    }

    // Called by the capture service to indicate the progress of the auto trigger process.
    // The method can be used to update the GUI to visualize the progress of auto triggering.
    @Override
    public void onAutoTriggerProgressUpdate(int done, int total) {
    }

    /**
     * Called as near as possible to the moment when a photo is captured
     * from the sensor.  This is a good opportunity to play a shutter sound
     * or give other feedback of camera operation.  This may be some time
     * after the photo was triggered, but some time before the actual data
     * is available.
     */
    @Override
    public void onShutter() {
        ThreadManager.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                mProgressView.start(getActivity(), false);
            }
        });
    }

    // Called by CaptureService when an image was taken.
    // The DetectionResult may not be valid when image was taken manually
    // or when the capture service failed to detect the document in the final image.
    @Override
    public void onImageTaken(final BitmapWithMetadata bitmap, DetectionResult result) {
        if (getActivity() != null && result != null /* && result.isValid()*/) {
            ImageProcessor imageProcessor = new ImageProcessor();
            if (mIsAutoDetect)
                imageProcessor.setDetectionResult(result);

            int rotation = bitmap.getRotation();
            if (rotation == 90) {
                imageProcessor.rotate(ImageProcessor.RotationDirection.CW);
            } else if (rotation == 270) {
                imageProcessor.rotate(ImageProcessor.RotationDirection.CCW);
            }
            imageProcessor.setImageProcessorListener(this);
            setPageFormat(imageProcessor);
            imageProcessor.setInputImage(bitmap);
            mImageProcessors.add(imageProcessor);
//            Bitmap rotationBitmap = mTakePhotoListAdapter.rotateBitmapByDegree(bitmap.getBitmap(), bitmap.getRotation());
//            mScanPhotosRecyclerView.setVisibility(View.VISIBLE);
//            mPhotosPromptTextView.setVisibility(View.GONE);
            if (mImageProcessors.size() == 1)
                mImageProcessors.get(0).computeResult();
        } else {
            resetTakingPicture();
        }
    }

    // If unwanted camera movement has been detected by MotionDetector the process of taking a camera picture gets cancelled
    // and Camera.takePicture(ShutterCallback, android.hardware.Camera.PictureCallback, android.hardware.Camera.PictureCallback) i
    // s NOT called. You can react on this by implementing this callback.
    @Override
    public void onTakePictureCancelled() {
        resetTakingPicture();
    }

    private void setPageFormat(ImageProcessor imageProcessor) {
        int width = mCurrentPageFormat == null ? 0 : mCurrentPageFormat.getWidth();
        int height = mCurrentPageFormat == null ? 0 : mCurrentPageFormat.getHeight();
        imageProcessor.setPageFormat(width, height);
        imageProcessor.setPageFormatMachtingThreshold(THRESHOLD);
    }

    // Function called when the computation of a new image started.
    @Override
    public void onStartedProcessingImage(ImageProcessor processor) {
        if (mIsAutoDetect) {
            mHintMessageView.setVisibility(View.VISIBLE);
        } else {
            mHintMessageView.setVisibility(View.INVISIBLE);
        }
        mDetectionResultView.setPoints(null);
    }

    // This function is called when the image processor is done computing a preview image.
    @Override
    public void onDoneComputingImage(final ImageProcessor processor, final BitmapWithMetadata bitmap) {
        ThreadManager.getInstance().runOnUiThread(new Runnable() {
            @Override
            public void run() {
                if (mImageProcessors.contains(processor)) {

                    if (mCurrentPageFormat != null) {
                        int rotation = 0;
                        if (mCurrentPageFormat.getWidth() > mCurrentPageFormat.getHeight()) {
                            if (bitmap.getWidth() < bitmap.getHeight()) {
                                rotation = 270;
                            }
                        }
                        if (mCurrentPageFormat.getWidth() < mCurrentPageFormat.getHeight()) {
                            if (bitmap.getWidth() > bitmap.getHeight()) {
                                rotation = 270;
                            }
                        }
                        if (rotation != 0) {
                            Bitmap rotationBitmap = mTakePhotoListAdapter.rotateBitmapByDegree(bitmap.getBitmap(), rotation);
                            bitmap.setBitmap(rotationBitmap);
                        }
                    }

                    retractedShotImage(Bitmap.createBitmap(bitmap.getBitmap()));

                    Bitmap newBitmap = mTakePhotoListAdapter.zoomBitmap(bitmap.getBitmap());
                    mTakePhotoListInfo.add(new TakePhotoListInfo(newBitmap));
                    mTakePhotoListAdapter.notifyItemInserted(mTakePhotoListAdapter.getItemCount() - 1);
                    if (mRemakeIndex >= 0) {
                        mDocumentSession.replacePage(mRemakeIndex, bitmap);
                        mImageProcessors.remove(processor);
                        processor.close();
                        getActivity().setResult(Activity.RESULT_OK);
                        getActivity().finish();
                    } else {
                        mDocumentSession.addPage(bitmap);
                        mImageProcessors.remove(processor);
                        photoQuantity();
                        processor.close();
                    }

                    if (mIsContinuousSnap) {
                        if (mImageProcessors.size() > 0) {
                            mImageProcessors.get(0).computeResult();
                        }
                        resetTakingPicture();
                    } else {
                        try {
                            resetTakingPicture();
                            EditImageFragment.fromWhere = "ScannerCameraActivity";
                            Intent intent = new Intent(getActivity(), EditImageActivity.class);
                            SessionIntentUtils.saveSessionId(intent, mDocumentSession);
                            intent.putExtra(EditImageFragment.EXTRA_PAGE, mDocumentSession.getPageCount() - 1);
                            startActivityForResult(intent, 0);
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }
                } else {
                    resetTakingPicture();
                }
                mProgressView.end();
            }
        });
    }

    @Override
    public void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (resultCode == Activity.RESULT_OK) {
            if (requestCode == ActRequestCode.REQ_SCAN_RC_FORMAT) {
                Bundle extras = data.getExtras();
                if (extras != null) {
                    mCurrentPageFormat = (PageFormat) extras.getSerializable(ScannerSession.EXTRA_FORMAT);
                }
                ScannerSession.setCurrentPageFormat(mCurrentPageFormat);
            }
        }
        if (resultCode == EditPDFFragment.RESULT_SCAN_EDIT_PDF) {
            try {
                mDocumentSession = (DocumentEditingSession) SessionIntentUtils.getSession(data);
            } catch (Exception e) {
                e.printStackTrace();
            }
            mFileNameForAddPage = data.getStringExtra(ScannerSession.EXTRA_FILE_NAME);
            if (!AppUtils.isEmpty(mFileNameForAddPage)) {
                mDocName = mFileNameForAddPage;
            }

            mIsUpdate = data.getBooleanExtra(EditPDFFragment.RESULT_SCAN_UPDATE, false);
        }
        if (resultCode == EditPDFActivity.RESULT_CODE) {
            _finish();
        }
        if (resultCode == EditImageFragment.RESULT_SCAN_IMAGE) {
            mUpdateIndex = data.getIntExtra(EditImageFragment.EXTRA_PAGE, 0);
            mIsResult = data.getBooleanExtra(EditImageFragment.RESULT_SCAN_UPDATE, false);
        }
        if (requestCode == ActRequestCode.REQ_SCAN_PHOTO_LIBRARY && resultCode == Activity.RESULT_OK) {
            new AddScanPhotoTask().execute();
        }
    }

    @Override
    public void onDocumentAdded(Document document) {
        onDocumentMgrEvent(document);
    }

    @Override
    public void onDocumentRemoved(Document document) {
    }

    @Override
    public void onDocumentUpdated(Document document) {
        onDocumentMgrEvent(document);

    }

    @Override
    public void onThumbnailGenerated(Document document) {

    }

    private class EditScanPhotoTask extends AsyncTask<Object, int[], Boolean> {

        @Override
        protected Boolean doInBackground(Object... objects) {
            return null;
        }

        @Override
        protected void onPreExecute() {
            mProgressView.start(getActivity(), false);
        }

        @Override
        protected void onPostExecute(Boolean aBoolean) {
            if (mIsUpdate) {
                mTakePhotoListInfo.clear();
                for (int i = 0; i < mDocumentSession.getPageCount(); i++) {
                    Bitmap newBitmap = mTakePhotoListAdapter.zoomBitmap(mDocumentSession.getImageForPage(i).getBitmap());
                    mTakePhotoListInfo.add(new TakePhotoListInfo(newBitmap));
                }
                mTakePhotoListAdapter.notifyUpdateData();
                mIsUpdate = false;
            } else {
                mTakePhotoListInfo.set(mUpdateIndex, new TakePhotoListInfo(mDocumentSession.getImageForPage(mUpdateIndex).getBitmap()));
                mTakePhotoListAdapter.notifyItemChanged(mUpdateIndex);
            }
            mProgressView.end();
            photoQuantity();
        }
    }

    private class AddScanPhotoTask extends AsyncTask<Object, int[], Boolean> {

        @Override
        protected void onPreExecute() {
            List<UilPhoto> files = UilLocalImageHelper.getInstance(mContext).getCheckedItems();
            int size = files.size();
            if (size > 0) {
                mProgressView.start(getActivity(), true);
            }
        }

        @Override
        protected Boolean doInBackground(Object... objects) {
            try {
                if (UilLocalImageHelper.getInstance(mContext).isResultOk()) {
                    UilLocalImageHelper.getInstance(mContext).setResultOk(false);

                    List<UilPhoto> files = UilLocalImageHelper.getInstance(mContext).getCheckedItems();
                    int size = files.size();

                    int[] pageSize = new int[2];
                    pageSize[0] = size;
                    for (int i = 0; i < size; i++) {
                        Bitmap bitmap = BitmapFactory.decodeFile(files.get(i).getOriginalPath());
                        BitmapWithMetadata bm = new BitmapWithMetadata();
                        bm.setBitmap(bitmap);
                        if (mRemakeIndex >= 0) {
                            mDocumentSession.replacePage(mRemakeIndex, bm);
                        } else {
                            mDocumentSession.addPage(bm);
                        }
                        pageSize[1] = i;
                        publishProgress(pageSize);
//                        bitmap.recycle();
//                        bitmap = null;
                    }
                    files.clear();

                    UilLocalImageHelper.getInstance(mContext).setCurrentSize(5);
                    UilLocalImageHelper.getInstance(mContext).getCheckedItems().clear();

                }
                return true;
            } catch (Exception e) {
                Log.d(TAG, e.getMessage());
            }
            return false;
        }

        @Override
        protected void onProgressUpdate(int[]... values) {
            super.onProgressUpdate(values);
            int[] pages = values[0];
            if (pages.length == 2) {
                int done = pages[1] + 1;
                int total = pages[0];
                mProgressView.progress(mContext.getApplicationContext().getString(
                        R.string.photo2pdf_loading, done, total));
            }
        }

        @Override
        protected void onPostExecute(Boolean result) {
            if (mRemakeIndex >= 0) {
                mProgressView.end();
                getActivity().setResult(Activity.RESULT_OK);
                getActivity().finish();
            } else {
                if (mDocumentSession.getPageCount() != mTakePhotoListAdapter.getItemCount()) {
                    int size = 0;
                    for (int i = mTakePhotoListAdapter.getItemCount(); i < mDocumentSession.getPageCount(); i++) {
                        Bitmap newBitmap = mTakePhotoListAdapter.zoomBitmap(mDocumentSession.getImageForPage(i).getBitmap());
                        mTakePhotoListInfo.add(new TakePhotoListInfo(newBitmap));
                        size++;
                    }
                    mTakePhotoListAdapter.notifyItemRangeInserted(mTakePhotoListInfo.size() - size, size);
                    photoQuantity();
                }
                mProgressView.end();
            }
        }
    }

    private void initUilLibrary(Context context) {
        ImageLoaderConfiguration config = new ImageLoaderConfiguration.Builder(context)
                .threadPriority(Thread.NORM_PRIORITY)
                .denyCacheImageMultipleSizesInMemory()
                .memoryCacheSize((int) Runtime.getRuntime().maxMemory() / 3)
                .diskCacheFileNameGenerator(new Md5FileNameGenerator())
                .diskCacheSize(200 * 1024 * 1024)
                .diskCache(new UnlimitedDiskCache(new File(FileUtils.getAppCacheDir(mContext), "ImageLoader")))
                .tasksProcessingOrder(QueueProcessingType.LIFO)
                .imageDownloader(new BaseImageDownloader(context, 5 * 1000, 5 * 1000))
                .build();

        ImageLoader.getInstance().init(config);
    }

    private void _finish() {
        if (mDocumentSession != null) {
            for (int i =  mDocumentSession.getPageCount() -1;  i >= 0; i--) {
                mDocumentSession.removePage(i);
            }
        }
        if (getActivity() != null) {
            getActivity().finish();
        }
    }


    @Override
    public void onClick(View v) {
        if (mWaiting) return;
        int id = v.getId();
        if (AppUtils.isFastDoubleClick())
            return;
        if (id == R.id.scan_cancel_btn) {
//            if (mDocumentSession.getPageCount() != 0 && mDocumentSession.hasChanges()) {
//                String docName = FileUtils.generateNewDocName();
//                mDocName = docName;
//                Document doc = DocumentManager.get().findDocument(docName);
//                if (doc == null) {
//                    DocumentSessionManagerFactory.getInstance().
//                            saveSessionAsDocument(mDocumentSession, docName, Document.DocumentType.ImageSeries, this);
//                    mButtonClicked = 1;
//                    mProgressView.start(getActivity(), false);
//                }
//            } else {
//            getActivity().finish();
//            }
            if (mRemakeIndex >= 0) {
                getActivity().finish();
            } else if (mDocumentSession.getPageCount() != 0) {
                final UITextEditDialog dialog = new UITextEditDialog(getActivity());
                dialog.setTitle(R.string.exit_scan);
                dialog.getInputEditText().setVisibility(View.GONE);
                dialog.getPromptTextView().setText(R.string.exit_scan_prompt);
                dialog.getOKButton().setText(R.string.scan_discard);
                dialog.getOKButton().setEnabled(true);
                dialog.show();
                dialog.getCancelButton().setOnClickListener(new OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        dialog.dismiss();
                    }
                });
                dialog.getOKButton().setOnClickListener(new OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        dialog.dismiss();
                        _finish();
                    }
                });
            } else {
                getActivity().finish();
            }

        } else if (id == R.id.scan_choose_image_button) {
            if (!ImageLoader.getInstance().isInited())
                initUilLibrary(mContext);

            Intent intent = new Intent(getActivity(), UilLocalAlbum.class);
            intent.putExtra(ScannerSession.EXTRA_SINGLE_CHOOSE, mRemakeIndex >= 0);
            startActivityForResult(intent, ActRequestCode.REQ_SCAN_PHOTO_LIBRARY);
        } else if (id == R.id.scan_take_picture_button) {
            mTakePictureBtn.setEnabled(false);
            mCaptureService.manuallyTakePicture();
        } else if (id == R.id.scan_torchLight_button) {
            if (getActivity().getPackageManager().hasSystemFeature(PackageManager.FEATURE_CAMERA_FLASH)
                    && !Build.MODEL.equals("Nexus 7")) {
                if (mIsFlashOpen) {
                    mTorchLightBtn.setSelected(false);
                    mIsFlashOpen = false;
                    showToast(R.string.atb_scan_close_flash);
                } else {
                    mTorchLightBtn.setSelected(true);
                    mIsFlashOpen = true;
                    showToast(R.string.atb_scan_open_flash);
                }
                mCaptureService.setTorchLight(mIsFlashOpen);
            } else {
                UIToast.getInstance(getContext()).show(R.string.atb_scan_flash_not_available);
            }
        } else if (id == R.id.scan_signal_snap_button) {
            mIsContinuousSnap = !mIsContinuousSnap;
            mContinuousShotBtn.setSelected(mIsContinuousSnap);
            ScannerSession.setContinuousSnap(mIsContinuousSnap);
            int resId = mIsContinuousSnap ? R.string.atb_scan_continuous_shooting : R.string.atb_scan_single_shooting;
            showToast(resId);
        } else if (id == R.id.scan_auto_detect_button) {
            if (mIsAutoDetect) {
                mAutoDetectBtn.setSelected(false);
                mPageSizeBtn.setEnabled(false);
                mPageSizeBtn.setColorFilter(mContext.getResources().getColor(R.color.scan_color_grey_ff878787));
                mHintMessageView.setVisibility(View.INVISIBLE);
                mDetectionResultView.setVisibility(View.INVISIBLE);
                mCaptureService.enableAutoTrigger(false);
                mIsAutoDetect = false;
                showToast(R.string.atb_scan_close_auto_detection);
            } else {
                mPageSizeBtn.setEnabled(true);
                mPageSizeBtn.setColorFilter(mContext.getResources().getColor(R.color.scan_color_translucent));
                mAutoDetectBtn.setSelected(true);
                mCaptureService.enableAutoTrigger(true);
                mHintMessageView.setVisibility(View.VISIBLE);
                mDetectionResultView.setVisibility(View.VISIBLE);
                mIsAutoDetect = true;
                showToast(R.string.atb_scan_open_auto_detection);
            }
            ScannerSession.setAutoDetect(mIsAutoDetect);
        } else if (id == R.id.scan_page_size_button) {
            Intent intent = new Intent(getActivity(), FormatActivity.class);
            if (mCurrentPageFormat != null)
                intent.putExtra(ScannerSession.EXTRA_FORMAT, mCurrentPageFormat);
            startActivityForResult(intent, ActRequestCode.REQ_SCAN_RC_FORMAT);
        } else if (id == R.id.scan_photos_relative_btn) {
            if (mPhotosBar.getVisibility() == View.VISIBLE) {
                mPhotosButton.findViewById(R.id.scan_photos_relative_btn_iv).setSelected(false);
                mPhotosBar.setVisibility(View.GONE);
            } else {
                mPhotosButton.findViewById(R.id.scan_photos_relative_btn_iv).setSelected(true);
                mPhotosBar.setVisibility(View.VISIBLE);
            }

        } else if (id == R.id.scan_done_btn) {
            if (mTakePhotoListInfo.size() > 0) {
                if (mPDFScanManager != null && mPDFScanManager.mScanCallBackListener != null) {
                    mCurOperation = DOC_OPERATION.SAVE_TO_LOCAL;
                    mSaveType = SAVE_AS_PDF;
                    String docName = FileUtils.generateNewDocName();
                    String fileFolder = AppFileUtil.getDiskCachePath(getContext());
                    mSaveAsPath = fileFolder + "/" + docName + "." + "pdf";
                    mSaveDocName = docName;
                    if (mPDFScanManager.mObtainDocumentNameCallBackListener != null) {
                        mPDFScanManager.mObtainDocumentNameCallBackListener.onDocumentName(mSaveDocName);
                    }
                    mProgressView.start(getActivity(), true);
                    saveDocRun.run();
                } else {
                    Intent intent = new Intent(getActivity(), EditPDFActivity.class);
                    SessionIntentUtils.saveSessionId(intent, mDocumentSession);
                    intent.putExtra("mFileNameForEditPhoto", mFileNameForAddPage);
//                    getActivity().setResult(Activity.RESULT_OK, intent);
                    startActivityForResult(intent, 0);
                }
            }
        }
    }

    private ScreenBroadcastReceiver mScreenReceiver;

    private class ScreenBroadcastReceiver extends BroadcastReceiver {
        @Override
        public void onReceive(Context context, Intent intent) {
            if (Intent.ACTION_SCREEN_OFF.equals(intent.getAction())) {
                mTorchLightBtn.setSelected(false);
                mIsFlashOpen = false;
                mCaptureService.setTorchLight(mIsFlashOpen);
            }
        }

    }

    private void registerScreenBroadcastReceiver() {
        mScreenReceiver = new ScreenBroadcastReceiver();
        IntentFilter filter = new IntentFilter();
        filter.addAction(Intent.ACTION_SCREEN_ON);
        filter.addAction(Intent.ACTION_SCREEN_OFF);
        filter.addAction(Intent.ACTION_USER_PRESENT);
        getActivity().registerReceiver(mScreenReceiver, filter);
    }

    @Override
    public void onStartedAction(DocumentSession documentSession, String s) {

    }

    @Override
    public void onUpdateProgress(DocumentSession documentSession, int done, int total) {
        if (getActivity() != null) {
            if (mSaveType == SAVE_AS_PDF) {
                if (mLastOper == OPER_SAVEING_PDF) {
                    mProgressView.progress(mContext.getApplicationContext().getString(R.string.editpdf_saving, done, total));
                } else {
                    mProgressView.progress(ResourceUtils.getString(mContext, R.string.scan_string_saving));
                }
            } else {
                mProgressView.progress(mContext.getApplicationContext().getString(R.string.editpdf_saving, done, total));
            }
        }
    }

    @Override
    public void onFinishedAction(DocumentSession session, Document document) {
        if (getActivity() != null) {
            if (mButtonClicked == 1) {
                mProgressView.end();
                if (session instanceof DocumentEditingSession) {
                    if (((DocumentEditingSession) session).hasChanges()) {
                        DocumentManager mng = DocumentManager.get();
                        mng.addDocument(document);

                        final Intent intent = new Intent(ScannerSession.INTENT_ACTION_UPDATE_LIST);
                        getActivity().runOnUiThread(new Runnable() {
                            @Override
                            public void run() {

                            }
                        });
                        ThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                getActivity().sendBroadcast(intent);
                                getActivity().finish();
                            }
                        }, 300);
                    }
                }
            } else if (mButtonClicked == 2) {
                mProgressView.end();
                if (session instanceof DocumentEditingSession) {
                    if (((DocumentEditingSession) session).hasChanges()) {
                        DocumentManager mng = DocumentManager.get();
                        mng.addDocument(document);

                        final Intent intent = new Intent(ScannerSession.INTENT_ACTION_UPDATE_LIST);
                        ThreadManager.getInstance().getMainThreadHandler().postDelayed(new Runnable() {
                            @Override
                            public void run() {
                                getActivity().sendBroadcast(intent);

                                Intent editIntent = new Intent(getActivity(), EditPDFActivity.class);
                                SessionIntentUtils.saveSessionId(intent, mDocumentSession);
                                editIntent.putExtra("mFileNameForEditPhoto", mDocName);
                                getActivity().setResult(Activity.RESULT_OK, editIntent);
                                startActivity(editIntent);
                            }
                        }, 300);
                    }
                }
            } else {
                if (mLastOper == OPER_LOADING_DOC) {
                    DocumentSessionManagerFactory.getInstance().cancelLoading(document);
                    mDocumentSession = DocumentSessionManagerFactory.getInstance().getEditingSession(session);
                    mLastOper = OPER_NORMAL;
                }
                if (mCurOperation == DOC_OPERATION.SAVE_TO_LOCAL) {
                    saveToPdfFlow(session, document);
                }
            }
        }
    }

    @Override
    public void onCanceledAction(DocumentSession documentSession) {
        if (getActivity() != null) {
            mProgressView.end();
            if (getActivity() != null)
                getActivity().finish();
        }
    }

    private Toast mToast;
    private TextView mTvToast;

    private void showToast(int resId) {
        try {
            if (mToast == null) {
                mToast = new Toast(mContext);
                LayoutInflater inflate = (LayoutInflater) mContext.getSystemService(Context.LAYOUT_INFLATER_SERVICE);
                View toastlayout = inflate.inflate(R.layout.fx_photo2pdf_toast_tips, null);
                mToast.setView(toastlayout);
                mToast.setDuration(Toast.LENGTH_SHORT);
                mTvToast = (TextView) toastlayout.findViewById(R.id.tv_toast_tips);
                mToast.setGravity(Gravity.CENTER, 0, 0);
            }
            mTvToast.setText(ResourceUtils.getString(mContext, resId));
            mToast.show();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private DOC_OPERATION mCurOperation = DOC_OPERATION.NONE;
    private static final int OPER_NORMAL = 0;
    private static final int OPER_EDIT_DOC = 1;
    private static final int OPER_RENAME_DOC = 2;
    private static final int OPER_LOADING_DOC = 3;
    private static final int OPER_SAVEING_IMAGE = 4;
    private static final int OPER_SAVEING_PDF = 5;
    private static final int SAVE_AS_IMAGE = 0;
    private static final int SAVE_AS_PDF = 1;
    private int mSaveType = SAVE_AS_IMAGE;
    private int mLastOper = OPER_NORMAL;
    private String mCurOpStep = "0";
    private String mSaveAsPath;

    enum DOC_OPERATION {
        NONE,
        SAVE_TO_LOCAL,
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////
    // *** CREATE A NEW DOC OR MODIFY AND RENMAE
    // 1.0 start
    // 1.1 save new doc
    // 1.2 add new doc
    // (1.3) delete doc before rename
    // *
    // *** MODIFY ONLY
    // 2.0 start
    // 2.1 save to temp doc
    // 2.2 add temp doc
    // 2.3 delete original doc
    // 2.4 rename temp to original name
    // *
    // *** RENAME ONLY
    // 3.0 start
    // 3.1 rename
    void saveToImagesFlow(DocumentSession session, Document document) {

        switch (mCurOpStep) {
            case "1.0": {
                mCurOpStep = "1.1";
                mLastOper = OPER_SAVEING_IMAGE;
                DocumentSessionManagerFactory.getInstance().
                        saveSessionAsDocument(mDocumentSession, mSaveDocName, Document.DocumentType.ImageSeries, this);
                break;
            }
            case "1.1": {
                mCurOpStep = "1.2";
                DocumentManager.get().addDocument(document);
                break;
            }
            default:
                break;
        }
    }

    ////////////////////////////////////////////////////////////////////////////////////////////////
    // *** CREATE A NEW DOC OR MODIFY AND RENMAE
    // 1.0 start
    // 1.1 save new doc
    // 1.2 add new doc
    // (1.3) delete doc before rename
    // *
    // *** MODIFY ONLY
    // 2.0 start
    // 2.1 save to temp doc
    // 2.2 add temp doc
    // 2.3 delete original doc
    // 2.4 rename temp to original name
    // *
    // *** RENAME ONLY
    // 3.0 start
    // 3.1 rename
    // *
    // *** SAVE TO TEMP PDF FILE
    // a.0 reload document
    // a.1 save to tmp pdf
    // a.2 copy tmp pdf to dest path
    // a.4 delete tmp pdf
    void saveToPdfFlow(DocumentSession session, final Document document) {
        switch (mCurOpStep) {
            case "1.0":
            case "1.1":
            case "2.0":
            case "2.1":
            case "2.2":
            case "2.3":
            case "3.0":
                saveToImagesFlow(session, document);
                break;
            case "1.2":
            case "1.3":
            case "2.4":
            case "3.1": {
                mCurOpStep = "a.0";
                Document doc = DocumentManager.get().findDocument(mSaveDocName);
                if (doc != null) {
                    try {
                        mLastOper = OPER_LOADING_DOC;
                        DocumentSessionManagerFactory.getInstance().
                                getViewingSessionForDocument(doc, this);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
                break;
            }
            case "a.0": {
                mCurOpStep = "a.1";
                mLastOper = OPER_SAVEING_PDF;
                DocumentSessionManagerFactory.getInstance().saveSessionAsDocument(mDocumentSession, ScannerSession.TEMP_SAVE_PDF_NAME, Document.DocumentType.PDF, this);
                UIToast.getInstance(mContext).show(getString(R.string.scan_save_successfully));
                break;
            }
            case "a.1": {
                if (mCurOperation == DOC_OPERATION.SAVE_TO_LOCAL) {
                    mCurOpStep = "a.2";
                    FileUtils.copyFile(document.getFilename(), mSaveAsPath, true, new IResultCallback() {
                        @Override
                        public void onResult(boolean success) {
                            mProgressView.end();
                            mCurOpStep = "a.4";
                            String path = document.getFilename();
                            boolean ret = FileUtils.deleteFile(path);
                            onDocumentMgrEvent(document);
                        }
                    });
                } else {
                    mProgressView.end();
                }
                break;
            }
            case "a.4": {
                _finish();
                int errorCode = IPDFScanManagerListener.e_ErrUnknown;
                if (FileUtils.isFileExist(mSaveAsPath))
                    errorCode = IPDFScanManagerListener.e_ErrSuccess;
                if (mCurOperation == DOC_OPERATION.SAVE_TO_LOCAL) {
                    mPDFScanManager.mScanCallBackListener.doneCallBack(errorCode, mSaveAsPath);
                }
                break;
            }
            default:
                break;
        }
    }


    Runnable saveDocRun = new Runnable() {
        @Override
        public void run() {
            if (mDocumentSession.getPageCount() != 0 && mDocumentSession.hasChanges()) {
                Document doc = DocumentManager.get().findDocument(mSaveDocName);
                if (doc == null) {
                    mCurOpStep = "1.0";
                    saveToPdfFlow(null, null);
                }
            }
        }
    };

    void onDocumentMgrEvent(Document document) {
        if (getActivity() == null)
            return;
        if (mCurOperation == DOC_OPERATION.SAVE_TO_LOCAL) {
            saveToPdfFlow(null, document);
        }
    }

    private boolean mWaiting;
    private Animator mShotAnimator;
    private void retractedShotImage(Bitmap bitmap) {
        if (mShotAnimator != null) {
            mShotAnimator.cancel();
        }

        // add Expanded ImageView
        final ImageView expandedView = new ImageView(mContext);
        expandedView.setScaleType(ImageView.ScaleType.CENTER_CROP);
        expandedView.setImageBitmap(bitmap);

        RelativeLayout.LayoutParams lp = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
        int margin = AppDisplay.dp2px(AppResource.getDimensionPixelSize(mContext, R.dimen.scan_margin_16dp));
        lp.setMargins(margin, margin, margin, margin);
        lp.addRule(RelativeLayout.ABOVE, R.id.scan_bottom_bar);
        lp.addRule(RelativeLayout.BELOW, R.id.scan_top_bar);
        ((FrameLayout) view.findViewById(R.id.scan_photos_view_ly)).addView(expandedView, lp);

        mWaiting = true;
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                mWaiting = false;
                // animator
                View destView = destView = mTakePictureBtn;

                final Rect destRect = new Rect();
                final Rect offsetRect = new Rect();
                destView.getGlobalVisibleRect(destRect);
                view.findViewById(R.id.scan_top_bar).getGlobalVisibleRect(offsetRect);
                float destWidth = AppDisplay.dp2px(56);
                float scale = destWidth / (view.getWidth() - AppDisplay.dp2px(32));
                float offset_x = (view.getWidth() - destWidth) / 2;
                float offset_y = destRect.top - offsetRect.bottom - destWidth;

                Log.d("yy", String.format("scale: %f\n(%d - %d)/2 + %f = %f\nexpandedView %d, %d", scale,
                        destRect.top, offsetRect.bottom, destWidth, offset_y, expandedView.getMeasuredWidth(), expandedView.getMeasuredHeight()));

                expandedView.setPivotX(0);
                expandedView.setPivotY(0);

                AnimatorSet set = new AnimatorSet();
                set.play(ObjectAnimator.ofFloat(expandedView, View.X, offset_x))
                        .with(ObjectAnimator.ofFloat(expandedView, View.Y, offset_y))
                        .with(ObjectAnimator.ofFloat(expandedView, View.SCALE_X, scale))
                        .with(ObjectAnimator.ofFloat(expandedView, View.SCALE_Y, scale));

                set.setDuration(800);
                set.setInterpolator(new DecelerateInterpolator());
                set.addListener(new AnimatorListenerAdapter() {
                    @Override
                    public void onAnimationCancel(Animator animation) {
                        onAnimationEnd(animation);
                    }

                    @Override
                    public void onAnimationEnd(Animator animation) {
                        expandedView.setVisibility(View.GONE);
                        ((FrameLayout) view.findViewById(R.id.scan_photos_view_ly)).removeView(expandedView);
                        mShotAnimator = null;
                    }
                });
                set.start();
                mShotAnimator = set;
            }
        }, 2000);

    }

}
