/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.editpdf;


import android.content.Context;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.recyclerview.widget.RecyclerView;

import com.foxit.pdfscan.R;
import com.foxit.uiextensions.theme.ThemeUtil;
import com.foxit.uiextensions.utils.AppResource;

import java.util.Collections;
import java.util.List;

public class EditThumbAdapter extends RecyclerView.Adapter<EditThumbAdapter.ThumbViewHolder> implements ThumbItemTouchHelper.ItemTouchCallback {

    private Context mContext;
    private List<ScanPDFPage> mPageList;
    private List<ScanPDFPage> mSelectedPageList;

    private boolean mIsEditMode = false;
    private boolean mIsSelectedAll = false;

    public EditThumbAdapter(Context context, List<ScanPDFPage> pageList, List<ScanPDFPage> selectedPageList) {
        this.mContext = context;
        this.mPageList = pageList;
        this.mSelectedPageList = selectedPageList;
    }

    public void setEditMode(boolean isEditMode) {
        this.mIsEditMode = isEditMode;
    }

    public void setSelectedAll(boolean selectedAll) {
        this.mIsSelectedAll = selectedAll;
    }

    public boolean isSelectedAll() {
        return mIsSelectedAll;
    }

    public List<ScanPDFPage> getScanPageList() {
        return mPageList;
    }

    public void addItem(ScanPDFPage page) {
        mPageList.add(page);
    }

    public void removeItem(int index) {
        mPageList.remove(index);
    }

    public void clear() {
        mPageList.clear();
    }

    @NonNull
    @Override
    public ThumbViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        View itemView = LayoutInflater.from(parent.getContext()).inflate(R.layout.fx_photo2pdf_edit_grid_item, parent, false);
        return new ThumbViewHolder(itemView);
    }

    @Override
    public void onBindViewHolder(@NonNull ThumbViewHolder holder, int position) {
        ScanPDFPage pageInfo = mPageList.get(position);
        holder.bind(pageInfo, position);
    }

    @Override
    public int getItemCount() {
        return mPageList.size();
    }

    @Override
    public void onMove(int fromPosition, int toPosition) {
        if (fromPosition < toPosition) {
            for (int i = fromPosition; i < toPosition; i++) {
                swap(i, i + 1);
            }
        } else {
            for (int i = fromPosition; i > toPosition; i--) {
                swap(i, i - 1);
            }
        }

        movePDFPage(fromPosition, toPosition);
        notifyItemMoved(fromPosition, toPosition);
    }

    private void movePDFPage(int fromPosition, int toPosition) {
        ScanPDFPage oldPage = mPageList.get(fromPosition);
        oldPage.getSession().movePage(fromPosition, toPosition);
    }

    private void swap(int dst, int src) {
        Collections.swap(mPageList, dst, src);
    }


    class ThumbViewHolder extends RecyclerView.ViewHolder implements View.OnClickListener, View.OnLongClickListener {
        private CheckBox chooseCheckBox;
        private ImageView image;
        private TextView titleText;

        ThumbViewHolder(@NonNull View itemView) {
            super(itemView);
            titleText = (TextView) itemView.findViewById(R.id.scanner_edit_grid_page_number);
            chooseCheckBox = (CheckBox) itemView.findViewById(R.id.scanner_edit_grid_item_chose);
            image = (ImageView) itemView.findViewById(R.id.scanner_edit_grid_thumb);

            image.setOnClickListener(this);
            image.setOnLongClickListener(this);
        }

        void bind(ScanPDFPage page, final int position) {
            titleText.setText("" + (position + 1));
            image.setImageBitmap(page.getBitmap());
            int visibility = mIsEditMode ? View.VISIBLE : View.GONE;
            if(mIsEditMode){
                chooseCheckBox.getBackground().setColorFilter(
                        new PorterDuffColorFilter(AppResource.getColor(mContext, com.foxit.uiextensions.R.color.b1), PorterDuff.Mode.SRC_IN));
                ThemeUtil.setTintList(chooseCheckBox, ThemeUtil.getCheckboxColor(mContext));
            }
            chooseCheckBox.setVisibility(visibility);
            chooseCheckBox.setChecked(mSelectedPageList.contains(page));
        }

        @Override
        public void onClick(View v) {
            int id = v.getId();
            int position = getAdapterPosition();

            if (id == R.id.scanner_edit_grid_thumb) {
                if (mIsEditMode) {
                    ScanPDFPage page = mPageList.get(position);
                    if (mSelectedPageList.contains(page)) {
                        mSelectedPageList.remove(page);
                    } else {
                        mSelectedPageList.add(page);
                    }
                    notifyItemChanged(position);
                }

                if (mItemClickListener != null)
                    mItemClickListener.onItemClick(v, position, ThumbViewHolder.this);
            }
        }

        @Override
        public boolean onLongClick(View v) {
            int id = v.getId();
            int position = getAdapterPosition();
            if (id == R.id.scanner_edit_grid_thumb) {
                if (mItemClickListener != null)
                    mItemClickListener.onItemLongClick(v, position, ThumbViewHolder.this);
            }
            return false;
        }
    }

    private ThumbItemTouchListener mItemClickListener;

    public void setOnItemClickListener(ThumbItemTouchListener listener) {
        mItemClickListener = listener;
    }

    public interface ThumbItemTouchListener {
        void onItemClick(View v, int position, RecyclerView.ViewHolder viewHolder);

        void onItemLongClick(View v, int position, RecyclerView.ViewHolder viewHolder);
    }

}
