/**
 * Copyright (C) 2003-2025, Foxit Software Inc..
 * All Rights Reserved.
 * <p>
 * http://www.foxitsoftware.com
 * <p>
 * The following code is copyrighted and is the proprietary of Foxit Software Inc.. It is not allowed to
 * distribute any parts of Foxit PDF SDK to third party or public without permission unless an agreement
 * is signed between Foxit Software Inc. and customers to explicitly grant customers permissions.
 * Review legal.txt for additional license and legal information.
 */
package com.foxit.pdfscan.browser;

import android.app.Activity;
import android.content.Context;
import android.os.Build;
import android.util.Log;
import android.view.DragEvent;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnDragListener;
import android.view.View.OnTouchListener;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputMethodManager;
import android.widget.AbsListView;
import android.widget.AbsListView.OnScrollListener;
import android.widget.EditText;
import android.widget.FrameLayout;
import android.widget.ListView;
import android.widget.TextView;
import android.widget.TextView.OnEditorActionListener;

import com.foxit.pdfscan.R;
import com.foxit.pdfscan.dialog.AlreadyExistsDialog;
import com.foxit.pdfscan.views.BrowserViewPager;
import com.foxit.uiextensions.utils.AppDisplay;
import com.luratech.android.appframework.Document;
import com.luratech.android.appframework.DocumentManager;

import androidx.annotation.RequiresApi;
import androidx.viewpager.widget.ViewPager;


public class RenameDocument implements OnTouchListener, ViewPager.OnPageChangeListener,
        OnScrollListener, OnDragListener, OnEditorActionListener {
    private static final String TAG = RenameDocument.class.getSimpleName();

    /**
     * Parent view element
     */
    private ListView listView;

    /**
     * Currently being edited, only one can be edited at a time!
     */
    private EditText editText;
    /**
     * Currently being edited, only one can be edited at a time!
     */
    private TextView nameText;

    private Document document;

    private View lblCreated;

    private View lblPages;

    private View created;

    private View pages;

    public void setDocument(Document document) {
        this.document = document;
    }

    public void setListView(ListView listView) {
        this.listView = listView;
    }

    @RequiresApi(api = Build.VERSION_CODES.JELLY_BEAN)
    public void startEditing(final BrowserViewPager viewPager) {
        stopEditing();
        editText = (EditText) viewPager.findViewById(R.id.edit_document);
        nameText = (TextView) viewPager.findViewById(R.id.browser_item_name);

        lblCreated = viewPager.findViewById(R.id.browser_item_label_created);
        lblPages = viewPager.findViewById(R.id.browser_item_label_pages);
        created = viewPager.findViewById(R.id.browser_item_created);
        pages = viewPager.findViewById(R.id.browser_item_page_count);

        Log.d(TAG, "Start editing");

        editText.setOnEditorActionListener(this);
        listView.setClickable(false);
        listView.setFocusable(false);
        listView.setFocusableInTouchMode(false);

        editText.setText(nameText.getText());

        editText.setX(nameText.getX());
        editText.setY(nameText.getY());
        editText.setLayoutParams(new FrameLayout.LayoutParams(nameText
                .getWidth(), nameText.getMaxHeight()));
        nameText.setVisibility(View.INVISIBLE);
        editText.setVisibility(View.VISIBLE);
        if (AppDisplay.isPad()) {
            editText.setImeOptions(EditorInfo.IME_ACTION_DONE | EditorInfo.IME_FLAG_NO_EXTRACT_UI);
        }

        lblCreated.setVisibility(View.INVISIBLE);
        lblPages.setVisibility(View.INVISIBLE);
        created.setVisibility(View.INVISIBLE);
        pages.setVisibility(View.INVISIBLE);

        editText.setError(null);
        editText.requestLayout();
        InputMethodManager imm = (InputMethodManager) viewPager.getContext()
                .getSystemService(Context.INPUT_METHOD_SERVICE);
        imm.showSoftInput(editText, InputMethodManager.SHOW_IMPLICIT);
        editText.requestFocus();
        editText.setSelection(editText.getText().length());
        viewPager.setOnTouchListener(this);
        viewPager.addOnPageChangeListener(this);
        listView.setOnTouchListener(this);
        listView.setOnScrollListener(this);
        listView.setOnDragListener(this);
    }

    public void stopEditing() {
        if (editText != null && nameText != null) {
            Log.d(TAG, "Stop editing");
            Context context = listView.getContext();

            String newName = editText.getText().toString();

            if (!newName.equals(document.getDocumentName())
                    && DocumentManager.get().findDocument(newName) != null) {
                new AlreadyExistsDialog(context, newName) {

                    @Override
                    protected void positive() {
                    }
                }.show();
            } else {
                if (newName.isEmpty()) {
                    editText.requestFocus();
                    return;
                }
                Log.d(TAG, "Rename Document: " + document.getDocumentName());
                Log.d(TAG, "New name: : " + newName);
                DocumentManager.get().renameDocument(document, newName);
            }
            listView.setClickable(true);
            listView.setFocusable(true);
            listView.setFocusableInTouchMode(true);
            editText.setVisibility(View.INVISIBLE);
            nameText.setVisibility(View.VISIBLE);

            lblCreated.setVisibility(View.VISIBLE);
            lblPages.setVisibility(View.VISIBLE);
            created.setVisibility(View.VISIBLE);
            pages.setVisibility(View.VISIBLE);

            InputMethodManager inputManager = (InputMethodManager) context
                    .getSystemService(Context.INPUT_METHOD_SERVICE);
            if (inputManager != null) {
                View currentFocus = ((Activity) context).getCurrentFocus();
                if (currentFocus != null) {
                    inputManager.hideSoftInputFromWindow(
                            currentFocus.getWindowToken(),
                            InputMethodManager.HIDE_NOT_ALWAYS);
                }
            }
            editText = null;
            nameText = null;
        }
    }

    @Override
    public boolean onTouch(View v, MotionEvent event) {
        stopEditing();
        return false;
    }

    @Override
    public void onPageScrollStateChanged(int arg0) {
    }

    @Override
    public void onPageScrolled(int arg0, float arg1, int arg2) {
    }

    @Override
    public void onPageSelected(int arg0) {
        if (arg0 == 1)
            stopEditing();
    }

    @Override
    public void onScrollStateChanged(AbsListView view, int scrollState) {
    }

    @Override
    public void onScroll(AbsListView view, int firstVisibleItem,
                         int visibleItemCount, int totalItemCount) {
    }

    @Override
    public boolean onDrag(View v, DragEvent event) {
        stopEditing();
        return false;
    }

    @Override
    public boolean onEditorAction(TextView v, int actionId, KeyEvent event) {
        stopEditing();
        return true;
    }

    public boolean isEditingMode() {
        return editText != null && nameText != null;
    }

}
